"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ExcelBuilder = void 0;
var _exceljs = _interopRequireDefault(require("exceljs"));
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); }
class ExcelBuilder {
  constructor() {
    _defineProperty(this, "workbook", void 0);
    _defineProperty(this, "worksheet", void 0);
    _defineProperty(this, "headerRowsCount", void 0);
    this.workbook = new _exceljs.default.Workbook();
    this.worksheet = this.workbook.addWorksheet('Sheet1');
    this.headerRowsCount = 0;
  }
  addHeaders(json) {
    if (!json || json.length <= 0) {
      return this;
    }
    const headerKeys = this.getHeaderKeys(json);
    this.worksheet.columns = Object.keys(headerKeys).map(key => {
      return {
        header: [key],
        key
      };
    });
    this.headerRowsCount++;
    return this;
  }
  addRows(json) {
    if (!json || json.length <= 0) {
      return this;
    }
    for (const data of json) {
      this.worksheet.addRow(data);
    }
    return this;
  }
  updateColumnWidths() {
    for (const column of this.worksheet.columns) {
      if (!column.eachCell) {
        continue;
      }
      let maxLength = 0;
      let count = 1;
      column.eachCell({
        includeEmpty: true
      }, cell => {
        if (count > this.headerRowsCount) {
          const cellLength = cell.value ? cell.value.toString().length : 0;
          if (cellLength > maxLength) {
            maxLength = cellLength;
          }
        }
        count++;
      });
      column.width = this.getColumnWidth(maxLength);
    }
    return this;
  }
  applyFreeze() {
    this.worksheet.views = [{
      state: 'frozen',
      ySplit: this.headerRowsCount
    }];
    return this;
  }
  applyHeaderStyles() {
    for (let i = 1; i <= this.headerRowsCount; i++) {
      const row = this.worksheet.getRow(i);
      row.height = this.worksheet.properties.defaultRowHeight * 2;
      row.font = {
        bold: true
      };
      row.alignment = {
        horizontal: 'left',
        vertical: 'top',
        wrapText: true
      };
    }
    return this;
  }
  applyAutoFilter() {
    const row = this.worksheet.getRow(this.headerRowsCount);
    const firstCell = row.getCell(1);
    let lastCell = firstCell;
    row.eachCell({
      includeEmpty: true
    }, function (cell) {
      lastCell = cell;
    });
    this.worksheet.autoFilter = {
      from: firstCell.$col$row,
      to: lastCell.$col$row
    };
    return this;
  }
  async getWorkbookAsBase64() {
    const buffer = await this.workbook.xlsx.writeBuffer();
    const base64 = buffer.toString('base64');
    return base64;
  }
  getColumnWidth(length) {
    if (length <= 13) {
      return 15;
    }
    if (length <= 18) {
      return 20;
    }
    return 25;
  }
  getHeaderKeys(json) {
    const headerKeys = {};
    for (const data of json) {
      for (const key of Object.keys(data)) {
        if (!headerKeys[key]) {
          headerKeys[key] = true;
        }
      }
    }
    return headerKeys;
  }
}
exports.ExcelBuilder = ExcelBuilder;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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