package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.40.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.40.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@(!`0````````````,`/@`!````0"(```````!``````````%"Q````
M`````````$``.``-`$``&P`:``8````$````0`````````!``````````$``
M````````V`(```````#8`@````````@``````````P````0````8`P``````
M`!@#````````&`,````````<`````````!P``````````0`````````!````
M!````````````````````````````````````#@:````````.!H`````````
M$`````````$````%`````"``````````(``````````@````````43T`````
M``!1/0`````````0`````````0````0`````8`````````!@`````````&``
M````````1@````````!&`````````!`````````!````!@```*BK````````
MJ+L```````"HNP```````(`$````````J`0`````````$`````````(````&
M````V*L```````#8NP```````-B[``````````(``````````@````````@`
M````````!`````0````X`P```````#@#````````.`,```````!0````````
M`%``````````"``````````$````!````(@#````````B`,```````"(`P``
M`````$0`````````1``````````$`````````%/E=&0$````.`,````````X
M`P```````#@#````````4`````````!0``````````@`````````4.5T9`0`
M``"PH````````+"@````````L*````````#L`````````.P`````````!```
M``````!1Y71D!@``````````````````````````````````````````````
M```````````0`````````%+E=&0$````J*L```````"HNP```````*B[````
M````6`0```````!8!`````````$`````````+VQI8C8T+VQD+6QI;G5X+7@X
M-BTV-"YS;RXR```````$````0`````4```!'3E4``@``P`0````#````````
M``*``,`$`````0`````````!``'`!`````D``````````@`!P`0````!````
M``````0````4`````P```$=.50!6%G,U%1O@M#=V<!C>%T6#Q>)CA`0````0
M`````0```$=.50``````!`````0``````````````"L```!$````!`````@`
M```@$!``!4`$A"@"``2`A@"$@HD``@D@!`B(@!!!N@`""$0`````````````
M`$4``````````````$8`````````1P```$@`````````20````````!+````
M``````````!,````30````````!.````3P```%$`````````4@```%,```!4
M``````````````````````````````!5````5@```%<```!8````60``````
M````````6P````````!<``````````````")!W#3:QJ>4*/@^TF93\.;XT?&
M`6:OS&9#1=7LN^.2?+,!BV]OQ=\?%8PES"[%F[9K"4/6(?_D#ONA;&##Y3\L
M#3J7"\_>1RW9<5@<?ZF<F^&G`@I"^M>':W^:?#-BV^W49-1\K4OCP```````
M``````````````````````````````"F`@``$@``````````````````````
M```-`@``$@````````````````````````",````$0``````````````````
M``````"-`@``$@`````````````````````````^`0``$@``````````````
M``````````#H`0``$@````````````````````````!)`@``$@``````````
M```````````````!````(`````````````````````````"X`0``$@``````
M``````````````````!G````$@````````````````````````#%`0``$@``
M``````````````````````"4````$@````````````````````````"&`@``
M$@````````````````````````#8`@``$@````````````````````````#]
M`0``$@````````````````````````#2````$@``````````````````````
M``#B````$@`````````````````````````P`0``$@``````````````````
M``````!Q`0``$@````````````````````````!J`0``$@``````````````
M``````````"*````$0````````````````````````!]`P``$@``````````
M```````````````0````(`````````````````````````"^`@``$@``````
M``````````````````"K`@``$@````````````````````````"]````$@``
M```````````````````````L````(`````````````````````````"=````
M$@````````````````````````"K````$@````````````````````````"I
M`0``$@````````````````````````"P`@``$@``````````````````````
M``!8````(@````````````````````````#H````%@``````````````````
M``````#[````$@````````````````````````!<`@``$@``````````````
M```````````6`0``$@````````````````````````#)`@``$@``````````
M``````````````":`0``$@````````````````````````#V`0``$@``````
M``````````````````!%`0``$@`````````````````````````G`0``$@``
M``````````````````````!^`0``$@````````````````````````"8`@``
M$@````````````````````````#7`0``$@````````````````````````"%
M`0``$@`````````````````````````N`@``$@``````````````````````
M``!Y````$@````````````````````````"3`0``$@``````````````````
M``````",`0``$@````````````````````````"<````$@``````````````
M``````````#*`0``$@`````````````````````````9`@``$@``````````
M``````````````!Z`@``$@````````````````````````"]`0``$@``````
M``````````````````#9````$@````````````````````````#$````$@``
M``````````````````````"B`0``$0````````````````````````!M`@``
M$0````````````````````````"D````$@````````````````````````!B
M`0``$@`````````````````````````+`0``$@``````````````````````
M``!4`0``$@`````````````````````````Z`@``$@``````````````````
M``````!&````$@````````````````````````#A`@``$@``````````````
M``````````"S````$@````````````````````````#1`0``$@``````````
M```````````````.!```$0`7`"#`````````"`````````!A`P``$@`-`*!:
M````````@P(````````@!```$@`-`"!%````````7`0````````8!```$@`-
M`/!)````````P@`````````.`P``$@`-`(!#````````-`````````!5`P``
M$@`-`,!#````````4@$```````#H`P``$``7`"C```````````````````#6
M`P``$``8`%#````````````````````Q`P``$@`-`(!)````````:@``````
M``"&`P``$@`-`(!7````````W`````````#-`P``$@`-`&!8````````1P``
M``````#]`@``$@`-`$!!````````"@````````"A`P``$``7``#`````````
M```````````!!```$@`-`%!!````````)0(```````#O`P``$@`-`$!,````
M````Y@````````"Y`P``$@`-`"!`````````*@````````"C`P``(``7``#`
M``````````````````#;`P``$@`-`#!-``````````@```````#!`P``$``8
M`"C```````````````````!K`P``$@`-`"!,````````(``````````_`P``
M$@`-`%!`````````XP````````"N`P``$@`-`+!8````````X0$```````!3
M````$@`-`"`@````````&`(```````"G`P``$@`-`$`B````````)@``````
M```;`P``$@`-`#!5````````2@(```````"2`P``$0`/``!@````````!```
M````````7U]G;6]N7W-T87)T7U\`7TE435]D97)E9VES=&5R5$U#;&]N951A
M8FQE`%])5$U?<F5G:7-T97)434-L;VYE5&%B;&4`7U]L:6)C7W-T87)T7VUA
M:6X`7U]C>&%?9FEN86QI>F4`7U]R96=I<W1E<E]A=&9O<FL`7U]S=&%C:U]C
M:&M?9F%I;`!?7V5N=FER;VX`<W1R;F-M<`!L<W1A=#8T`&%C8V5S<P!O<&5N
M9&ER`')E861D:7(V-`!S=')L96X`7U]S<')I;G1F7V-H:P!U;FQI;FL`8VQO
M<V5D:7(`<FUD:7(`4$Q?8W5R<F5N=%]C;VYT97AT`&)O;W1?1'EN84QO861E
M<@!097)L7VYE=UA3`%!E<FQ?;F5W6%-?9FQA9W,`<F5A9&QI;FL`4&5R;%]S
M=E]S971P=@!M96UC:'(`4&5R;%]S=E]S971P=FX`7U]M96UM;W9E7V-H:P!R
M96%L;&]C`&=E='!I9`!?7VUE;6-P>5]C:&L`<W1R8VAR`'-T<F1U<`!S=')T
M;VL`<W1R8VUP`'-T<G)C:'(`<W1D97)R`%]?=F9P<FEN=&9?8VAK`&5X:70`
M;'-E96LV-`!R96%D`&=E='5I9`!M:V1I<@!?7V5R<FYO7VQO8V%T:6]N`%]?
M9G!R:6YT9E]C:&L`;W!E;C8T`%!E<FQ?9W9?9F5T8VAP=@!097)L7V=E=%]S
M=@!097)L7V-R;V%K7VYO8V]N=&5X=`!097)L7VUG7W-E=`!097)L7W-Y<U]I
M;FET,P!097)L7V%T9F]R:U]U;FQO8VL`4&5R;%]A=&9O<FM?;&]C:P!03%]D
M;U]U;F1U;7``4&5R;%]S869E<WES;6%L;&]C`'!E<FQ?<&%R<V4`<&5R;%]D
M97-T<G5C=`!P97)L7V9R964`4&5R;%]S>7-?=&5R;0!P97)L7V%L;&]C`'!E
M<FQ?8V]N<W1R=6-T`'!E<FQ?<G5N`&UE;6-P>0!L:6)P97)L+G-O`&QI8F,N
M<V\N-@!P87)?8W5R<F5N=%]E>&5C`'!A<E]B87-E;F%M90!84U]);G1E<FYA
M;'-?4$%27T)/3U0`<&%R7V5N=E]C;&5A;@!P87)?8W5R<F5N=%]E>&5C7W!R
M;V,`<&%R7V1I<FYA;64`<VAA7V9I;F%L`&=E=%]U<V5R;F%M95]F<F]M7V=E
M='!W=6ED`'!A<E]C;&5A;G5P`%])3U]S=&1I;E]U<V5D`%]?9&%T85]S=&%R
M=`!S:&%?=7!D871E`'AS7VEN:70`7U]B<W-?<W1A<G0`<VAA7VEN:70`7V5N
M9`!P87)?;6MT;7!D:7(`7V5D871A`'!A<E]S971U<%]L:6)P871H`'!A<E]F
M:6YD<')O9P!M>5]P87)?<&P`<&%R7V1I90!P87)?:6YI=%]E;G8`1TQ)0D-?
M04))7T147U)%3%(`1TQ)0D-?,BXQ-`!'3$E"0U\R+C,T`$=,24)#7S(N-`!'
M3$E"0U\R+C,S`$=,24)#7S(N,RXR`$=,24)#7S(N,RXT`$=,24)#7S(N,BXU
M`"]U<W(O;&EB+W!E<FPU+S4N-#`O8V]R95]P97)L+T-/4D4``````0`!``(`
M`0`"``,``0`!``(`!``"``(``@`!``$``@`"``$``P`"``(``@`!``$``@`"
M``$`!0`"``,``0`"``$``0`!``$``0`"``(``0`"``(``0`"``(``0`&``(`
M`@`%``(``0`!``(``@`#``(``0`"``(``0`#``$`!P`(``(``@`!``$``0`!
M``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$`
M```!``@`\P(``!``````````0@[]````"0`M!```$````)21E@8```@`/P0`
M`!````"TD98&```'`$H$```0````%&EI#0``!@!5!```$````+.1E@8```4`
M7P0``!````!R&6D)```$`&H$```0````=!EI"0```P!V!```$````'4::0D`
M``(`@@0```````#PO0````````8````!``````````````#XO0````````8`
M```"````````````````O@````````8````#```````````````(O@``````
M``8````$```````````````0O@````````8````%```````````````8O@``
M``````8````&```````````````@O@````````8````'```````````````H
MO@````````8````(```````````````PO@````````8````)````````````
M```XO@````````8````*``````````````!`O@````````8````+````````
M``````!(O@````````8````,``````````````!0O@````````8````-````
M``````````!8O@````````8````.``````````````!@O@````````8````/
M``````````````!HO@````````8````0``````````````!PO@````````8`
M```1``````````````!XO@````````8````2``````````````"`O@``````
M``8````3``````````````"(O@````````8````4``````````````"0O@``
M``````8````6``````````````"8O@````````8````7``````````````"@
MO@````````8````8``````````````"HO@````````8````9````````````
M``"PO@````````8````:``````````````"XO@````````8````;````````
M``````#`O@````````8````<``````````````#(O@````````8````=````
M``````````#0O@````````8````>``````````````#8O@````````8````?
M``````````````#@O@````````8````@``````````````#HO@```````!(`
M```A``````````````#PO@````````8````B``````````````#XO@``````
M``8````C````````````````OP````````8````D```````````````(OP``
M``````8````E```````````````0OP````````8````F```````````````8
MOP````````8````G```````````````@OP````````8````H````````````
M```HOP````````8````I```````````````POP````````8````J````````
M```````XOP````````8````K``````````````!`OP````````8````L````
M``````````!(OP````````8````M``````````````!0OP````````8````N
M``````````````!8OP````````8````O``````````````!@OP````````8`
M```P``````````````!HOP````````8````Q``````````````!POP``````
M``8````R``````````````!XOP````````8````S``````````````"`OP``
M``````8````T``````````````"(OP````````8````U``````````````"0
MOP````````8````V``````````````"8OP````````8````W````````````
M``"@OP````````8````X``````````````"HOP````````8````Y````````
M``````"POP````````8````Z``````````````"XOP````````8````[````
M``````````#`OP````````8````\``````````````#(OP````````8````]
M``````````````#0OP````````8````^``````````````#8OP````````8`
M```_``````````````#@OP````````8```!```````````````#HOP``````
M``8```!!``````````````#POP````````8```!"``````````````#XOP``
M``````8```!#``````````````"HNP```````#\`````````",`````````)
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``#S#Q[Z2(/L"$B+!1F>``!(A<!T`O_02(/$",,``````/,/'OI52(GE0513
M2(/L,(E]W$B-?=Q(B7702(UUT$B)5<AD2(L4)2@```!(B57H2(U5R/\5?Y\`
M`$B+-<"=``!(BSV1G@``2(GR9^C`/```2(L%.9\``(`X``^$'@$``$B+!6&>
M``!D2(L`@(@Z!````XM%W(UX!(U8`TAC_TC!YP/_%>">``!(BTW01(M%W$B)
MQDB)!8Z?``!(BP%(B09(C04@00``2(E&"$B-!4Z?``!(BP!(B4802(T%"D$`
M`$B)1AA!@_@!#XXO`0``2(/!"$&#P`.X!`````\?1```2(LY2&/0@\`!2(/!
M"$B)/-9$.<!UZDB813'`2(GQ2(L]$Y\``$C'!,8`````B=I(C34Z&P``_Q7,
MG```A<`/A(P```!(BSWMG@``3(UEY/\5XYT``$R)YDB-/40_``")PS'`Z"@8
M``!(A<!T=4B+/<2>``#_%7:<``#_%5B=``!(BT7H9$@K!"4H````#X6?````
M2(/$,(G86T%<7</_%?R<``!(B06-G@``2(G'2(7`='3_%4^=``!(BP4HG0``
M9$B+$,:"M@@```#IM_[__TB+/6&>``#_%7N<``#I8O___TB+/5>>``!(A?]T
M"V?HC#4``.ET____2(T]6#X``$R)YNB(%P``2(D%,9X``$B)QTB%P`^$4O__
M_^O1N`0```#I[_[__[\!````_Q7^FP``_Q4@G0``#Q^$``````#S#Q[Z,>U)
MB=%>2(GB2(/D\%!413'`,<E(C3W!_?___Q5[G0``]&8N#Q^$``````!(C3VQ
MG0``2(T%JIT``$@Y^'052(L%#IP``$B%P'0)_^`/'X``````PP\?@`````!(
MC3V!G0``2(TU>IT``$@I_DB)\$C![C](P?@#2`'&2-'^=!1(BP7MFP``2(7`
M=`C_X&8/'T0``,,/'X``````\P\>^H`]/9T```!U,U5(@SWJFP```$B)Y70-
M2(L]!IT``/\5V)L``.AC____Q@44G0```5W#9BX/'X0``````,-F9BX/'X0`
M``````\?0`#S#Q[ZZ6?___\/'X``````54B)Y4%7059(C8U`_?__3(V-P/W_
M_T%505132('LF`(``&1(BQ0E*````$B)5<A(B?I(B<],C4(X#Q^``````$&+
M`$V+$$B#QQ!)@\`(#\B)P$R)UDB)1_!,B=!(P>X@2,'H*$G!ZC@E`/\``$P)
MT$F)\DC!Y@A)P>(8@>8``/\`18G23`G02`GP2(E'^$PYSW6MN_____]F#V^%
M4/W__V8/;YU@_?__2(V%J/W__V9(#V[K9@]OE7#]__](@<%H`@``9@]L[68/
M'X0``````/,/;PAF#V_B9@]O4-A(@\`09@_O0(AF#^_*9@_OP68/;\AF#W/P
M`68/<]$?9@_KP68/V\4/*4`(9@]OPV8/;]Q(.<AUNTR+"DR+6A!(BWH@2(M:
M&$R)R$R)R4R+A4C]__](P>D;2,'@!4@)R`.%0/W__TZ-A`.9>8):C8PXF7F"
M6DR)V$@QV$@C0@A(,=B-/`%(BTH(2(M""$F)^DC!Z`)(P>$>2`G!3(G82<'J
M&XG.2#'P3"'(3#'83`'`28GX2<'@!4T)T$:-)`!,B<A-B<A)P>@"2,'@'DV)
MY4P)P$R+A5#]__])P>T;2#'!08G"2"'Y3XV$`YEY@EI(,?%,`<%-B>!)P>`%
M30GH3(NM6/W__T$!R$B)^4C![P)(P>$>18G`2HVT+IEY@EI("?E-B<5(,<B)
MSTPAX$G![1M,,=!(`?!,B<9(P>8%3`GN3(NM8/W__P'&3(G@2<'L`DC!X!Z)
M]D^-E"J9>8):3`G@28GU2#'!2<'M&T&)Q$PAP4@Q^4P!T4F)\DG!X@5-">I,
MBZUH_?__00'*3(G!2<'H`DC!X1Y%B=)*C;POF7F"6DP)P4V)U4@QR$G![1M!
MB<A((?!,,>!(`?A,B==(P><%3`GO3(NM</W__P''2(GP2,'N`DC!X!Z)_T^-
MI"R9>8):2`GP28G]2#'!2<'M&XG&3"'13#'!3`'A28G\2<'D!4T)[$R+K7C]
M__]!`<Q,B=%)P>H"2,'A'D6)Y$^-A"B9>8):3`G138GE2#'(2<'M&T&)RD@A
M^$@Q\$P!P$V)X$G!X`5-">A,BZV`_?__00'`2(GX2,'O`DC!X!Y%B<!*C;0N
MF7F"6D@)^$V)Q4@QP4G![1N)QTPAX4PQT4@!\4R)QDC!Y@5,">Y,BZV(_?__
M`<Y,B>%)P>P"2,'A'HGV3XV4*IEY@EI,">%)B?5(,<A)P>T;08G,3"'`2#'X
M3`'028GR2<'B!4T)ZDR+K9#]__]!`<),B<!)P>@"2,'@'D6)TDJ-O"^9>8):
M3`G`38G52#'!2<'M&T&)P$@A\4PQX4@!^4R)UTC!YP5,">],BZV8_?__`<](
MB?%(P>X"2,'A'HG_3XVD+)EY@EI("?%)B?U(,<A)P>T;B<Y,(=!,,<!,`>!)
MB?Q)P>0%30GL3(NMH/W__T$!Q$R)T$C!X!Y)P>H"18GD3XV$*)EY@EI,"=!-
MB>5(,<%)P>T;08G"2"'Y2#'Q3`'!38G@2<'@!4T)Z$R+K:C]__]!`<A(B?E(
MP>\"2,'A'D6)P$J-M"Z9>8):2`GY38G%2#'(2<'M&XG/3"'@3#'02`'P3(G&
M2,'F!4P)[DR+K;#]__\!QDR)X$G![`)(P>`>B?9/C90JF7F"6DP)X$F)]4@Q
MP4G![1M!B<1,(<%(,?E,`=%)B?))P>(%30GJ3(NMN/W__T$!RDR)P4C!X1Y)
MP>@"18G22HV\+YEY@EI,"<%-B=5(,<A)P>T;08G(2"'P3#'@2`'X3(G72,'G
M!4P)[TR+K<#]__\!QTB)\$C![@)(P>`>B?]/C:0LF7F"6D@)\$F)_4@QP4G!
M[1N)QDPAT4PQP4P!X4F)_$G!Y`5-">Q&C30A3(G12<'J`DC!X1Y,"=%,BY7(
M_?__2#'(08G,2"'X3XV$$)EY@EI-B?)(,?!)P>H;3`'`38GP2<'@!4T)T$R+
ME=#]__]!`<!(B?A(P>`>2,'O`D6)P$J-M!:9>8):2`GX38G"2#'!2<'J&XG'
M3"'Q3#'A2`'.3(G!2,'A!4P)T4V)\DG![@))P>(>B<E-"?)(`<Y(BXW8_?__
M3#'008GU3"'`28V,#)EY@EI-B>Q(,?A)P>P;2`'(3(GI2,'A!4P)X8G)2`'(
M3(G!2<'H`DC!X1Y!B<1,"<%-B=!%B=)),<A$,<9,BX7@_?__2HV\!Z'KV6Y-
MB>!(`?=,B>9)P>@;2,'F!4P)QHGV2`'W3(GN2,'F'DG![0)!B?A,`Y7H_?__
M3`GN28G-B<E(`XWP_?__23'U1#'H38G%38V4`J'KV6Y,B<!)P>T;2,'@!4P)
MZ(G`20'"3(G@2<'L`DC!X!Y%B=5,">!)B?2)]D@#M?C]__]),<1$,>=-B>Q(
MC;PYH>O9;DR)Z4G![!M(P>$%3`GAB<E(`<],B<%)P>@"2,'A'D&)_$P)P4F)
MP(G`23'(13'038GB3HV$!J'KV6Y,B>9)P>H;2,'F!4P)UHGV20'P3(GN2<'M
M`DC!YAY%B<),">Y)B<U(`X4`_O__B<E),?5(`XT(_O__1#'O38G52(V\.*'K
MV6Y,B=!)P>T;2,'@!4P)Z(G`2`''3(G@2<'L`DC!X!Y!B?U,">!)B?2)]D@#
MM1#^__]),<1%,>!-B>Q.C80!H>O9;DR)Z4G![!M(P>$%3`GAB<E)`<A,B=%)
MP>H"2,'A'D6)Q$P)T4F)PHG`2`.%&/[__TDQRD0QUTV)XDB-O#ZAZ]EN3(GF
M2<'J&TC!Y@5,"=:)]D@!]TR)[DG![0)(P>8>08GZ3`GN28G-B<E),?5(`XT@
M_O__13'H38G53HV$`*'KV6Y,B=!)P>T;2,'@!4P)Z(G`20'`3(G@2<'L`DC!
MX!Y%B<5,">!)B?2)]D@#M2C^__]),<1$,>=-B>Q(C;PYH>O9;DR)Z4G![!M(
MP>$%3`GAB<E(`<],B=%)P>H"2,'A'D&)_$P)T4F)PHG`2`.%,/[__TDQRD4Q
MT$V)XDZ-A`:AZ]EN3(GF2<'J&TC!Y@5,"=:)]DD!\$R)[DG![0)(P>8>18G"
M3`GN28G-B<E),?5$,>]-B=5(C;PXH>O9;DR)T$C!X`5)P>T;2`.-./[__TP)
MZ(G`2`''3(G@2<'L`DC!X!Y!B?U,">!)B?2)]D@#M4#^__]),<1%,>!-B>Q.
MC80!H>O9;DR)Z4G![!M(P>$%3`GAB<E)`<A,B=%)P>H"2,'A'D6)Q$P)T4F)
MPHG`2`.%2/[__TDQRD0QUTV)XDB-O#ZAZ]EN3(GF2<'J&TC!Y@5,"=:)]D@!
M]TR)[DG![0)(P>8>08GZ3`GN28G-B<E),?5%,>A-B=5.C80`H>O9;DR)T$G!
M[1M(P>`%3`GH2`.-4/[__XG`20'`3(G@2<'L`DC!X!Y%B<5,">!)B?2)]D@#
MM5C^__]),<1$,>=-B>Q(C;PYH>O9;DR)Z4G![!M(P>$%3`GAB<E(`<],B=%)
MP>H"2,'A'D&)_$P)T4F)PHG`2`.%8/[__TDQRD4QT$V)XDZ-A`:AZ]EN3(GF
M2<'J&TC!Y@5,"=:)]DD!\$R)[DG![0)(P>8>18G"3`GN28G-B<E),?5$,>]-
MB=5(C;PXH>O9;DR)T$G![1M(P>`%3`GHB<!(`<=,B>!(P>`>2<'L`D&)_4@#
MC6C^__],">!)B?2)]D@#M7#^__]),<1%,>!.C:0!H>O9;DR)Z4V)Z$G!Z!M(
MP>$%3`G!B<E)`<Q,B=%)P>H"2,'A'D6)X$P)T4F)PHG`2`.%>/[__TDQRD0Q
MUTV)PDB-O#ZAZ]EN3(G&2<'J&TC!Y@5,"=9-B>I)P>T"2<'B'@'W2(G.B<E-
M">J)_TPQUDV)UD6)U40QYDF)_$B-M#"AZ]EN2(GX2<'L&TC!X`5,">!$C20&
M3(G&2<'H`KC<O!N/2,'F'DP)QDD)]D&)\$TAPDDA_DT)UDR+E8#^__])`<),
M`=%-B>),`?%-B>9)P>(%2<'N&TT)\D$!RDB)^4C![P)(P>$>18G22`GY2`G.
M28G.B<],(>9-(<9)"?9(B[6(_O__2`'&3`'N38G53`'V38G62<'E!4G![AM-
M"?5)B<Y!`?5,B>9)P>P"2,'F'D6)[4P)YD&)]$D)]DTAUDPAX4P)\4R+M9#^
M__])`<9-`<9-B>A,`?%-B>Y)P>`%2<'N&TT)\$$!R$R)T4G!Z@)(P>$>18G`
M3`G12`G.28G.08G*3"'N32'F20GV2(NUF/[__T@!QD@!_DR)QTP!]DV)QDC!
MYP5)P>X;3`GW28G.`?=,B>Y)P>T"2,'F'HG_3`GN08GU20GV32'&3"'I3`GQ
M3(NUH/[__TD!QDT!YDF)_$P!\4F)_DG!Y`5)P>X;30GT00',3(G!2<'H`DC!
MX1Y%B>1,"<%("<Y)B<Y!B<A((?Y-(>Y)"?9(B[6H_O__2`'&3`'638GB3`'V
M38GF2<'B!4G![AM-"?))B<Y!`?)(B?Y(P>\"2,'F'D6)TD@)_HGW20GV32'F
M2"'Y3`GQ3(NUL/[__TD!QDT![DV)U4P!\4V)UDG!Y05)P>X;30GU00'-3(GA
M2,'A'DG![`)%B>U,">%("<Y)B<Y!B<Q,(=9)(?Y)"?9(B[6X_O__2`'&3`'&
M38GH3`'V38GN2<'@!4G![AM-"?!)B<Y!`?!,B=9)P>H"2,'F'D6)P$P)UD&)
M\DD)]DTA[DPAT4P)\4R+M<#^__])`<9)`?Y,B<=,`?%-B<9(P><%2<'N&TP)
M]P'/3(GI2<'M`DC!X1Z)_TP)Z4@)SDF)SD&)S4PAQDTAUDD)]DB+M<C^__](
M`<9,`>9)B?Q,`?9)B?Y)P>0%2<'N&TT)]$F)SD$!]$R)QDG!Z`)(P>8>18GD
M3`G&08GP20GV22'^3"'!3`GQ3(NUT/[__TD!QDT!UDV)XDP!\4V)YDG!X@5)
MP>X;30GR00'*2(GY2,'O`DC!X1Y%B=)("?E("<Y)B<Z)STPAYDTAQDD)]DB+
MM=C^__](`<9,`>Y-B=5,`?9-B=9)P>4%2<'N&TT)]4F)SD$!]4R)YDG![`)(
MP>8>18GM3`GF08GT20GV32'63"'A3`GQ3(NUX/[__TD!QDT!QDV)Z$P!\4V)
M[DG!X`5)P>X;30GP00'(3(G12<'J`DC!X1Y%B<!,"=%("<Y)B<Y!B<I,(>Y-
M(>9)"?9(B[7H_O__2`'&2`'^3(G'3`'V38G&2,'G!4G![AM,"?=)B<X!]TR)
M[DG![0)(P>8>B?],">Y!B?5)"?9-(<9,(>E,"?%,B[7P_O__20'&30'F28G\
M3`'Q28G^2<'D!4G![AM-"?1!`<Q,B<%)P>@"2,'A'D6)Y$P)P4@)SDF)SD&)
MR$@A_DTA[DD)]DB+M?C^__](`<9,`=9-B>),`?9-B>9)P>(%2<'N&TT)\DF)
MSD$!\DB)_DC![P)(P>8>18G22`G^20GVB?=-(>9((?E,"?%,B[4`____20'&
M30'N38G53`'Q38G62<'E!4G![AM-"?5!`<U,B>%)P>P"2,'A'D6)[4P)X4@)
MSDF)S$&)STPAUDDA_$D)]$B+M0C___](`<9,`<9-B>A,`>9-B>Q)P>`%2<'L
M&TT)X$$!\$R)UDG!Z@)(P>8>18G`3`G628G*08GT20GR32'J3"'A3`G13(N5
M$/___TD!PD@#A1C___],`==-B<),`?A,B[TX____2`'/3(G!2<'J&TC!X05,
M"=&)R4@!STR)Z4G![0)(P>$>08GZ3`GI2`G.28G-08G.3"'&32'E20GU28UT
M!0!,B=!-B=5)P>T;2,'@!4P)Z(G`2`'&3(G`2<'H`DC!X!Y!B?5)"<"XUL%B
MRDPQP3'Y2(N](/___T@!QTP!YTV)[$@!^4R)[TG![!M(P><%3`GG38G4B?](
M`?E)P>0>20''2<'J`HG/30GB38G$18G`33'403'T2(NU*/___T@!QDP!]DF)
M_DP!YDF)_$G![AM)P>0%30GT38GN2<'M`DG!YAY%B>1-"?5,B[4P____3`'F
M08GT20'&30'P38G633'N03'.3(GA30'P38GF2,'A!4G![AM,"?&)R4D!R$B)
M^4C![P)(P>$>18G&2`G/1(G138GJ23'Z3`'Y3(N]0/___T$Q\DJ--!%,B?%-
MB?)(P>$%2<'J&TD!QTP)T8G)2`'.3(GA2<'L`DC!X1Y!B?),">%%B>Q)B?V)
M_TDQS4T!_$R+O5C___]%,<5/C00L38G438G52<'M&TG!Y`5-">Q-B?5)P>X"
M2<'E'D6)Y$T)[DR+K4C___]-`>!%B<1)`<5,`>])B<V)R4TQ]4$Q]4R)YDP!
M[TV)Y4C!Y@5)P>T;3`GNB?9(`?=,B=9)P>H"2,'F'D&)_4P)UDR+E5#___])
M`<),`=%-B?)),?)%,<))`<=.C0013(GI38GJ2<'J&TC!X05,"=&)R4D!R$R)
MX4G![`)(P>$>18G"20G,1(GQ28GVB?9-,>9,`?E,B[UH____03'^2HT\,4R)
MT4V)UDD!QTG![AM(P>$%3`GQB<E(`<],B>E)P>T"2,'A'D&)_DP)Z4R+K6#_
M__])`<5,`>Y-B>5),<U%,<5.C00N3(GV38GU2<'M&TC!Y@5,">Z)]DD!\$R)
MUDG!Z@)(P>8>18G%20GR1(GF28G,B<E,`?Y-,=1,B[UX____03'\2HT\)DR)
M[DV)[$D!QTG![!M(P>8%3`GFB?9(`?=,B?9)P>X"2,'F'D&)_$P)]DR+M7#_
M__])`<9,`?%-B=9),?9%,<9.C00Q3(GA38GF2<'N&TC!X05,"?&)R4D!R$R)
MZ4G![0)(P>$>18G&20G-1(G128GRB?9-,>I,`?E,BWV(03'Z2HT\$4R)\4V)
M\DG!ZAM(P>$%3`G1B<E(`<],B>%)P>P"2,'A'D&)^DP)X4R+98!)`<1,`>9-
MB>Q)`<=),<Q%,<1.C00F3(G638G42<'L&TC!Y@5,">:)]DD!\$R)]DG![@)(
MP>8>18G$20GV1(GN28G-B<E-,?5,`?Y,BWV803']2HT\+DR)YDV)Y4G![1M(
MP>8%3`GNB?9(`?=,B=9)P>H"2,'F'D&)_4P)UDR+59!)`<),`=%-B?)),?)%
M,<).C0013(GI38GJ2<'J&TC!X05,"=&)R4D!R$R)X4G![`)(P>$>18G"20G,
M1(GQ28GV20''33'F3`'YB?9,BWVH03'^3(G720'.3(G12,'O&TD!QTC!X05(
M"?F)R4D!SDR)Z4G![0)(P>$>1(GW3`GI3(MMH$D!Q4P![DV)Y4DQS44QQ4Z-
M!"Y(B?Y)B?U)P>T;2,'F!4P)[HGV20'P3(G62<'J`DC!YAY%B<5)"?)$B>9)
MB<R)R4TQU$P!_D4Q]$V)[DP!YDV)[$G![AM)P>0%30GT28G^18GD3`'F2<'F
M'DC![P)!B?1,"?=,BW6P20'&2`-%N$P!\4V)UDP!R$6)TDDQ_DP!T$4QQDV)
MX$P!\4V)YDG!X`5)P>X;30GP1`'!38GH2<'M`DG!X!Z)R4T)Q4F)^`-Z($TQ
MZ$$!W4B)>B!$,<9)B<A,B6H82`'P2(G.2<'H&P-*"$C!Y@5(B4H(3`G&`?!(
MB0),B>!)P>P"2,'@'DP)X$0!V$B)0A!(BT7(9$@K!"4H````=1)(@<28`@``
M6T%<05U!7D%?7</_%<R%```/'T``54B)Y4%7059!54%44TB#[!A(BP58A```
M2(EUR$R+,$V%]@^$GP````^V!TF)_83`=`0\/75:13'D28L>2(7;=#9-8^1-
MB?=F#Q^$``````!,B>),B>Y(B=__%5F$``"%P'4'0H`\(SUT1DF+7PA)@\<(
M2(7;==E(@\082(G86T%<05U!7D%?7<,/'X``````28G\00^V1"0!28/$`83`
M=`0\/77N12GLZX]F#Q]$``!,B?E(BT7(2HU<(P%,*?%(P?D#B0CKL#';ZZP/
M'T0``%5(B>532(VU4/___TB![*@```!D2(L<)2@```!(B5WH2(G[_Q7GA```
MA<!U.XN5:/___X'B`/```('Z`$```'0(@?H`H```=2&^`@```$B)W_\5`84`
M`(7`#Y3`#[;`ZPD/'X``````,<!(BU7H9$@K%"4H````=09(BUWXR</_%7.$
M``!F9BX/'X0``````%5(B>5!5T%6055!5%-(@>RH````9$B+!"4H````2(E%
MR#'`2(7_=`B`/P!(B?MU*$B+1<AD2"L$)2@````/A30!``!(@<2H````6T%<
M05U!7D%?7<,/'P#_%7J#``!)B<5(A<!TRDR)[_\5D80``$B%P`^$N````(!X
M$RY,C6`3=0Y!@'PD`0!TW&8/'T0``(!X$RX/A+8```!(B=__%1V#``!,B>=)
MB<;_%1&#``!%C7P&`DUC_TR)__\5H((``$V)X4F)V$R)^DF)QDB)Q[X"````
M,<!(C0TW)```_Q7-@P``3(GW2(VU,/____\5W8(``$R)]X/X_W02BX5(____
M)0#P```]`$```'1C_Q5E@@``3(GW_Q6<@@``3(GO_Q7;@P``2(7`#X5*____
M9I!,B>__%6^#``!(B=__%3Z"``#IZ?[__V8/'X0``````$&`?"0!+@^%/O__
M_T&`?"0"``^$_O[__^DM____#Q\`Z(O^___KG/\5ZX(```\?`/,/'OI52(T-
M]&,``$B--8<C``!(B>532(/L"$B+'5:"``!(BQ57@@``9$B+._\5)8,``$R-
M!1HE``!D2(L[13')2(T5>!@``$B+7?A,B<%(C35@(P``R?\E,X(```\?`%5(
MB>5!5D%505132('L`!```$B##"0`2(/L$+K_#P``9$R+)"4H````3(EEV$F)
M]$R-K=#O__](B?M(C3TJ(P``3(GN_Q4.@@``A<!^%HG!@^@!2&/0@+P5T.__
M_P!U0(7`=39(BP6M@0``3(GB2(G>9$B+./\5+8$``$B+1=AD2"L$)2@```!U
M2TB!Q!`0``!;05Q!74%>7<.)P0\?0`!,8_&^+P```$R)[TR)\O\5C(```$B%
MP'2K2(L%6($``$R)\4R)ZDB)WF1(BSC_%7V!``#KIO\5K8$```\?1```53'`
M2(GE05=!5D%505132(G[2(/L*(`^/61,BRPE*````$R);<A)B?4/E,!)`<5,
MB>__%<B```!(C77$2(G?28G$,<#HE_O__TB%P`^$%@$``$F)QDB)Q_\5HH``
M`$PYX`^#X0```$QC1<1,BSW>?P``2<'@`P^V`X3`#X3'`0``/#T/A+\!``!(
MB=@/'T``#[90`4B#P`&$TG0%@/H]=>](*=A(F$V)QDR)1;A-`S=*C7P@`O\5
MY7\``$F)!DB%P'1?28L'3(M%N$B-2P%*BQ0`#[8#B`(\/70JA,!U%>LD9F8N
M#Q^$```````/'T``/#UT$0^V`4B#P@%(@\$!B`*$P'7KQ@(],<!F#Q]$``!!
M#[9,!0"(3`(!2(/``83)=>Y(BT7(9$@K!"4H````#X4G`0``2(/$*%M!7$%=
M05Y!7UW##Q]``#'`9@\?1```00^V5`4`08@4!DB#P`&$TG7NZ[X/'T``3(L]
MX7X``$V+#TF#.0`/A-,```!,B<A%,?9F9BX/'X0```````\?`$B#P`A$B?)!
M@\8!2(,X`'7OC4H#2&/)2,'A`XL%SX```(7`=6M(B<],B4VPQP6Z@````0``
M`$B)3;C_%=!^``!(B<=(A<`/A$;___]-8\9(BTVX2(MUL$G!X`-,B46X3(G"
M_Q4H@```3(M%N$F)!TB)QT&-1@%$B77$2)A(QP3'`````.E=_O__#Q^`````
M`$R)STB)SO\5Y'\``$F)!TB)QTB%P`^$Y_[__TUCQDG!X`/KO@\?0``QP.E7
M_O__N1````!%,?;I3_____\51G\``&9F+@\?A```````#Q\`\P\>^DB+!;U^
M``!(BQ6^?@``2(T--V```$B--<H?``!D2(LX_R5^?P``9@\?1```\P\>^E5(
MB>5!54%44TB![``0``!(@PPD`$B![``0``!(@PPD`$B#["AD2(L$)2@```!(
MB478,<!(C9W0[___3(VEP-____\5['T``$R-#:,?``"Z`1```$B)WT&)P$B-
M#94?``"^`@```#'`_Q7>?@``2(G?N@`0``!,B>;_%55^```Q_X7`>#1$C6@!
M2(G#36/M3(GO_Q5D?0``2(G'2(7`=!E(8]M,B>E,B>9(B=K_%7I]``#&!!@`
M2(G'2(M%V&1(*P0E*````'412('$*"```$B)^%M!7$%=7</_%25^``!F9BX/
M'X0``````&:0\P\>^ND'____D&8/'T0``/,/'OI52(GE05=!5D%505132('L
M`!```$B##"0`2('LN````&1(BP0E*````$B)1<@QP$F)_$F)]4B-/<@>``!(
MC;4L[___Z/3W__^^+P```$R)YTB)P_\5@WT``$B%P'0^,<!,B>9(C3VC'@``
MZ.W[__],B>!(BU7(9$@K%"4H````#X66`0``2(UEV%M!7$%=05Y!7UW##Q^$
M``````!,B>],C2UT'@``_Q5(?0``3(GN2(G'_Q5<?0``28G&2(7`=)Y!@#X`
M#X3U````9I!(A=MT%$R)]DB)W_\5+WT``(7`#X2Y````3(GW_Q5N?```28U4
M!O])B<=).=9R'>LL9F8N#Q^$```````/'T``Q@(`2(/J`4DYUG0%@#HO=.],
MB??_%39\``!)B<=,B>?_%2I\``!)C40'`4@]_P\```^'&____TB#[`A,C;W`
M[___38GP,<!!5+H`$```O@(```!,B?](C0VW'@``3(T-K!T``/\5UGP``$B-
MM3#O__],B?__%>9[``!:687`=1*+A4CO__\E`/```#T`@```=$Y,B>XQ__\5
M:WP``$F)QDB%P`^$J?[__T&`/@`/A0W___](A=MT!8`[+G030;\!````3(TU
M+!T``.E.____D(![`0!TO.OE#Q^$``````"^`0```$R)__\5:GP``(7`=:!,
MB?Y(C3T%'0``Z$_Z__],B?__%=Y[``#I5_[___\5XWL``&9F+@\?A```````
M\P\>^E6^+P```$B)Y5-(B?M(@^P(_Q5U>P``2(U0`4B%P$@/1=I(B=](BUWX
MR?\EE'L``&9F+@\?A```````D/,/'OI52(GE05132('L`!```$B##"0`2(/L
M$&1(BP0E*````$B)1>@QP$B%_P^$W0```(`_`$B)^P^$T0```/\5JWH``$B)
MPDB-0`%(/0`0```/A_````!,C:7@[___3(GG@_@(<@Z)P4B)WL'I`_-(I4B)
M\S')J`1T"8L+B0^Y!````*@"=`Y$#[<$"V9$B00/2(/!`J@!=`</M@0+B`0/
M28U$%/]).<1R(>LD9F8N#Q^$```````/'X0``````,8``$B#Z`%,.>!T!8`X
M+W3OOB\```!,B>?_%7%Z``!(A<!T+$DYQ')73(GG_Q66>@``2(M5Z&1(*Q0E
M*````'5+2('$$!```%M!7%W##Q\`2(M%Z&1(*P0E*````'4M2('$$!```$B-
M/6D;``!;05Q=_R51>@``9@\?A```````Q@``ZZ0/'P`QP.NF_Q5&>@``9F8N
M#Q^$```````/'P#S#Q[Z54B)Y4%7059!54%43(TE.QL``%-(@^P89$B+'"4H
M````2(E=R$B-7<0/'P`QP$B)WDR)Y^@S]/__2(7`=#9(BP67>```2&-5Q$B+
M`$B-!-!(BU`(2(D02(72=-!(BU`02(/`"$B)$$B%TG7PZ[YF#Q]$``!,C27<
M&@``D#'`2(G>3(GGZ./S__](A<!T-DB+!4=X``!(8U7$2(L`2(T$T$B+4`A(
MB1!(A=)TT$B+4!!(@\`(2(D02(72=?#KOF8/'T0``$R-)90:``"0,<!(B=Y,
MB>?HD_/__TB%P'0V2(L%]W<``$AC5<1(BP!(C0302(M0"$B)$$B%TG302(M0
M$$B#P`A(B1!(A=)U\.N^9@\?1```3(TE31H``)`QP$B)WDR)Y^A#\___2(7`
M=#9(BP6G=P``2&-5Q$B+`$B-!-!(BU`(2(D02(72=-!(BU`02(/`"$B)$$B%
MTG7PZ[YF#Q]$``!,C24$&@``D#'`2(G>3(GGZ//R__](A<!T-DB+!5=W``!(
M8U7$2(L`2(T$T$B+4`A(B1!(A=)TT$B+4!!(@\`(2(D02(72=?#KOF8/'T0`
M`$R-+<09``"0,<!(B=Y,B>_HH_+__TB%P'0V2(L%!W<``$AC5<1(BP!(C030
M2(M0"$B)$$B%TG302(M0$$B#P`A(B1!(A=)U\.N^9@\?1```3(TM*1D``)`Q
MP$B)WDR)[^A3\O__2(7`=#9(BP6W=@``2&-5Q$B+`$B-!-!(BU`(2(D02(72
M=-!(BU`02(/`"$B)$$B%TG7PZ[YF#Q]$``!,C3TP&0``D#'`2(G>3(G_Z`/R
M__](A<!T-DB+!6=V``!(8U7$2(L`2(T$T$B+4`A(B1!(A=)TT$B+4!!(@\`(
M2(D02(72=?#KOF8/'T0``$R-->H8``"0,<!(B=Y,B??HL_'__TB%P'0V2(L%
M%W8``$AC5<1(BP!(C0302(M0"$B)$$B%TG302(M0$$B#P`A(B1!(A=)U\.N^
M9@\?1```,<!(B=Y(C3V?&```Z&?Q__](A<!T.DB+!<MU``!(8U7$2(L`2(T$
MT$B+4`A(B1!(A=)TS`\?0`!(BU`02(/`"$B)$$B%TG7PZ[9F#Q]$```QP$B)
MWDB-/>T7``#H%_'__TB%P'0Z2(L%>W4``$AC5<1(BP!(C0302(M0"$B)$$B%
MTG3,#Q]``$B+4!!(@\`(2(D02(72=?#KMF8/'T0``$B)WDB-/0L8``#HR?#_
M_TB%P'0-2(G&3(GW,<#HU_3__S'`2(G>2(T]^1<``.BF\/__2(7`=!%(B<9(
MC3WW%P``,<#HL/3__TB)WDB-/?$7```QP.A_\/__2(G&2(7`=#A,B>\QP.B-
M]/__2(M%R&1(*P0E*````'5$2(/$&$R)YTB--=P7```QP%M!7$%=05Y!7UWI
M7_3__S'`2(G>2(T]KA<``.@N\/__2(7`=+Q(B<9,B?\QP.@\]/__ZZW_%=QU
M```/'T``\P\>^E5(C3TC%P``2(GE2(/L$&1(BP0E*````$B)1?@QP$B-=?3H
MY>___TB%P'0P#[80@/HP#Y3`A-(/E,()T(/P`0^VP$B+5?AD2"L4)2@```!U
M#\G#9@\?A```````,<#KXO\5;G4``&8/'T0``/,/'OI52(GE2('LT````$B)
MM5C___](B95@____2(F-:/___TR)A7#___],B8UX____A,!T(`\I18`/*4V0
M#RE5H`\I7;`/*67`#REMT`\I=>`/*7WP9$B+!"4H````2(F%2/___S'`2(U%
M$$B)^L>%,/___P@```!(B84X____2(V%4/___[X"````2(V-,/___TB)A4#_
M__](BP47=0``QX4T____,````$B+`$B)Q_\5*70``+__````_Q5^<P``9F8N
M#Q^$```````/'P!52(GE055!5%-,C9PD``#__TB![``0``!(@PPD`$PYW'7O
M2(/L&+H"````,?9D3(LL)2@```!,B6W808G]2(V=T/_^__\5AG0``$R-8/],
MB>)(P?H_2,'J,$F-!!0/M\!(*=!)*<0/B*`````/'T``,=),B>9$B>__%5)T
M``!(@_C_#X2K````N@@``0!(B=Y$B>__%>=R``"#^/]T=TB82(/X!W922(U$
M`_A(.=AR2$B-%9ET``#K$`\?@`````!(@^@!2#G8<B](BPA(.0IU[T@IV$P!
MX$B+5=AD2"L4)2@```!U:TB!Q!@``0!;05Q!75W##Q]``$F![````0!)@?P`
M`/__#X5D____2,?`_____^O"2(M%V&1(*P0E*````'4M2(T]0!4``#'`9^C]
M_?__2(M%V&1(*P0E*````'4/2(T]%14``#'`9^C?_?___Q5!<P``9@\?A```
M````\P\>^E5(B>7_%4IS``")Q_\56G(``$B%P'0#2(L`7</S#Q[Z54B)Y4%7
M059(C77$055,C2W>%```051)B?Q,B>]32(/L&&1(BP0E*````$B)1<@QP.@6
M[?__2(7`=`B`.`!(B<-U,4B+1<AD2"L$)2@````/A88```!,B>9(C6783(GO
M,<!;05Q!74%>05]=Z?OP__\/'P!,B>?_%>]Q``!(B=])B<;_%>-Q``!-C7P&
M`DR)__\5=7$``$B#[`A,B?I-B>!32(G'28G&3(T->Q,``$B-#7H4``"^`@``
M`#'`_Q69<@``6%I(BT7(9$@K!"4H````=0A,B?;I??____\5,G(``&8N#Q^$
M``````#S#Q[Z54B)Y4%7059,C;VT_O__055,B?Y!5%-(@>Q(`0``2(F]J/[_
M__,/?@57;@``2(T]\!(``&1(BP0E*````$B)1<A(C07:$@``2,>%\/[__P``
M``!(QX4H____`````$C'A=#^__\`````9D@/;LA(C06J$P``9@]LP69(#V[0
M2(T%HA,```\IA>#^__]F2`]NV/,/?@7S;0``2(T%D!,``&9(#V[@,<!F#VS"
M#Q&%"/____,/?@7:;0``9@]LPP\1A1C____S#WX%SVT``&8/;,0/*87`_O__
MZ'_K__](A<!T#(`X`$B)PP^%+@4``/\54'$``(G'_Q5@<```2(7`#X17!```
M3(LP387V#X1+!```3(GW_Q5B<```3(UL``%,B>],B>O_%?%O``!%#[8&2(F%
MH/[__TF)Q$6$P'1@9F8N#Q^$```````/'P!,B>I%#[;`3(GG,<!)@\8!2(T-
MUA(``+X"````_Q7^<```3#GK3(GJ10^V!D@/0]-(B=!,*>A(@\`"2#G02`]"
MPDDIU4F#Q`)-C6P%_D6$P'6N3(VM`/___TB-/1`2``#K$V8/'T0``$F+?0A)
M@\4(2(7_=#M,B?XQP.B1ZO__28G$2(7`=.&`.`!TW$B)Q^A,Z___A<!TT$R)
MY_\5)W```$F)Q$B%P'4_#Q^``````$R-K>#^__]-BV4`387D="A!@#PD``^$
M-0,``$R)Y^@-Z___A<`/A0T!``!-BV4(28/%"$V%Y'783(GG_Q4_;P``2(N=
MH/[__TF)Q4B)W_\5+&\``$6-M`4$!```36/V3(GW_Q6X;@``4TV)X$R-#<T0
M``!)B<5(C07.$0``3(GRO@(```!02(T-M1$``$R)[S'`_Q78;P``3(GOOL`!
M``#_%2)P``!?05B#^/]U$_\57&\``$2+"$&#^1$/A28#``!(C84P____3(GO
M2(F%F/[__TB)QO\596\``(G#A<!T?TB+A:C^__]-B>B^`@```$B-%0D2``!(
MBPA(BP5W;P``2(LX,<#_%=QM``!%,>202(M%R&1(*P0E*`````^%N00``$B-
M9=A,B>!;05Q!74%>05]=PP\?`$R)YTF#Q0C_%=-N``!)B<1(A<`/A+;^___I
MXO[__V8N#Q^$``````"+A4C___\E`/```#T`0```#X5K____1(NE3/____\5
MQ6X``$$YQ`^%5?___XN%2/___R7_`0``/<`!```/A3____],B??_%75M``!,
MB?Y(C3U=$```28G$,<#HH>C__TB%P`^$D`(``$B+C:C^__](B<9(BSEGZ$7P
M__](B86@_O__2(7`#X1M`@``2(N]H/[___\5B&T``$B)PH/X`WY02(N-H/[_
M_TB82(F5D/[__TB--3D0``!(C7P!_/\5#FX``(7`=2I(BY60_O__@_H$#X1"
M`P``2(N-H/[__XU"^TB8@#P!+P^$+`,```\?0`!,B?Y,C3TE#P``,<!,B?_H
M_.?__TB%P`^$`P(```^V`(3`#X3X`0``/#`/A/`!``!,B?](C35B#P``,<#H
M[^O__TR-/?P/``#_%;IL``!!5TR)\DV)Z%!,C0V2#@``O@(```!,B>=(C0VW
M#P``,<#_%:MM``!86NM+#Q^``````/\5.FT``(,X$75,_Q5W;```@\,!2(/L
M"$V)Z$%73(T-3`X``$R)\DR)YU-(C0V`#P``O@(```!0,<#_%6!M``!(@\0@
MOL`!``!,B>?_%:9M``"#^/]TJ4R)[_\52&P``$R)YDB-/>X-```QP.@_Z___
M3(GG9^C&^?__Z<']__^013'DZ>#\__\/'X0``````$B-G<#^__]F#Q^$````
M``!(BSM(A?]T)$R)_C'`Z-[F__](B<=(A<!T!8`X`'5!2(M["$B#PPA(A?]U
MW+\-````NPT```!,C35J#@``0;T-````_Q5G:P``0;A3````2(F%H/[__TF)
MQ.F"^___9I#_%4)L``!(@\,(2(7`=)%)B<;I+OO__TB+A:C^__]-B>B^`@``
M`$B-%<,.``!(BPA(BP5Q;```2(LX,<#_%=9J``#I]?S__V8/'X0``````$B)
MQV?HY_C__TB+1<AD2"L$)2@````/A:`!``!(C6782(G?6T%<05U!7D%?7?\E
MQ6L```\?1```2(N%J/[__TB+`$B)A:#^___I??W__V:02(N]H/[__S'V,<#_
M%6=K``")QX7`#X3U_?__B86H_O__Z/KV__](A<`/B.']__\QTHN]J/[__TB-
M</9(B86@_O___Q6H:P``B[VH_O__N@8```!(C;6Z_O___Q5`:@``B[VH_O__
M2(N-H/[__TB#^`8/A?L```"!O;K^__\`0T%##X6)_?__9H&]OO[__TA%#X5Z
M_?__,=*)O:C^__](C7'._Q5(:P``B[VH_O__NB@```!(BYV8_O__2(G>_Q7=
M:0``2(/X*`^%I0```$B-!5`-``"^`@```$V)Z$R)\L:%6/___P!(C0T(#0``
M3(T-W`L``$R)YU`QP%/_%?]J``!97NFO_?__2(N%J/[__TB+4`A(A=(/A,3\
M__](B96H_O__2(G7_Q7D:0``2(/X`P^&JOS__TB+E:C^__](C36A#```2(U\
M`OS_%7%J``!(BY6H_O__A<!(#T65H/[__TB)E:#^___I=/S___\51FH``$B+
M1<AD2"L$)2@```!UZTB-/6(,```QP&?HP/3___,/'OI52(T]5PL``$B)Y4%5
M051,C6W44TR)[DB#[!AD2(L<)2@```!(B5W82(L=A6D``&1,BR-!#[:$)+``
M``!!B(0DL0```#'`Z!'D__](A<!T"8`X,0^$@P$``#'`9^B+[___9$R+(TR)
MY[D#````N@$```!(C346#```_Q6M:```28G$2(7`="I(BT`09$B+.TB-%?L+
M``!(BS#_%:5H``!)BT0D$$B+,/9&#D`/A5,!``!D2(L[N0,```"Z`0```$B-
M-=`+``#_%6!H``!)B<1(A<!T,DB-/5(*``!,B>XQP.AWX___2(G"28M$)!!(
MBSA(A=(/A/,```!(B?YD2(L[_Q4^:```9$B+.[H!````2(TU@@L``/\5J&<`
M`&1(BSM(C16&"P``2(G&_Q44:```9$B+`TR)[DB-/>8)``#&@+$`````,<#H
M$./__TB)!;EI``!(B<-(A<!T0;[``0``2(G?_Q5;:0``C5`!@_H!=@F#^!$/
MA9D```!(BT789$@K!"4H````#X6S````2(/$&%M!7$%=7<,/'T``2(L]<6D`
M`&?H4_;__TB)!5QI``!(B<-(A<!UHTB+1=AD2"L$)2@```!U>4B-/88+```Q
MP/\5=F@``&8/'T0``(!X`0`/A'_^___I;O[__Y!(BQ4A:0``2(LRZ*'E___I
M!O___P\?0`!D2(L[_Q4.:```Z9[^__](BU789$@K%"4H````=1Z)P4B+!>EH
M``!(B=I(C3U'"P``2(LP,<#_%0QH``#_%=YG``!F#Q]$``#S#Q[Z54B)Y4%4
M4TB-=>1(B?M(C3T6"0``2(/L$&1(BP0E*````$B)1>@QP.C?X?__2(7`=!P/
MMA"$T@^4P(#Z,`^4P@C0=0I(A=MT!8`[`'4>2(M%Z&1(*P0E*````'5U2(/$
M$%M!7%W#9@\?1```2(G?9^C'Z___OB\```!)B<1(B<?_%09G``!(C5`!2(7`
M3`]%XDR)Y_\5*F<``+H$````2(TU40D``$B)Q_\5%68``(7`=9M(BT7H9$@K
M!"4H````=1!(@\002(G?6T%<7>F:XO___Q7\9@``#Q]``/,/'OI5OX````!(
MB>7_%=UE``!F#V\%!4@``$C'0#``````#Q$`9@]O!0)(``#'0'@`````#Q%`
M$&8/;P7_1P``#Q%`(%W#9@\?A```````\P\>^E5(8\)(B>5!5TF)_T%6055,
MC6\X051)B<132(/L*$B)=<A(BW<H2(M7,(T,QD@Y\4B)3RA(@](`2,'H'4@!
MT$B)1S!(8T=XA<`/A2T!``!!@_P_#XZ$````1(GC3(MUR,'K!HU#_TC!XP:)
M1<1,`?-(B46X#Q^$``````#S00]O!DR)_TF#QD!!#Q%%`/-!#V]&T$$/$440
M\T$/;T;@00\112#S00]O1O!!#Q%%,.C9R?__23G>=<1(BT6X2(M-R$2+=<1(
MP>`&2(U$`4!!]]Y(B47(0<'F!D>-9#3`26/$2(/X"')N2(M]R$F-=T!,B>E(
M@^;X2(L72"GQ28E7.$B+5`?X2"G/28E4!?A(`<A)B?U(@^#X2(/X"'(@2(/@
M^#'228M,%0!(B0P62(/""$@YPG+N#Q^$``````!%B6=X2(/$*%M!7$%=05Y!
M7UW##Q]$``"H!'5M2(7`=-](BTW(#[8108A7.*@"=-`/MU0!_F9!B50%_NO#
M#Q\`0;Y`````2(MUR$F-?`4`02G&13GF10]/]$ECWDB)VO\5DV4``$&+1WA$
M`?!!B4=X@_A`=8],B?](`5W(12GTZ,S(___IBO[__TB+3<B+$4&)5SB+5`'\
M08E4!?SI7____V9F+@\?A```````#Q]``/,/'OI52(GE05=,C7XX059!54%4
M28G\4TB)\TB#[`A,BW8H3(MN,$R)\$C!Z`.#X#^-2`'&1`8X@$ACP4P!^(/Y
M.`^.J`$``+I`````*<IT#S')B<Z#P0'&!#``.=%R\TB)W^@XR/__9@_OP`\1
M0SA)QT<P`````$$/$4<000\11R!,B>I!#[;&38GP3(GW2,'J&$C!X`A,B>E)
MP>@028G13(GR2,'O&`^V]DR)ZDC!Z1!("?!!#[;P#[;)2,'@"$@)\$`/MO=(
MB=](P>`(2`GP00^V]4C!X`A("?`/MO9!#[;12,'@"$@)\$C!X`A("<A(P>`(
M2`G02(E#<.B?Q___2(L#3(GG3(GA3(GB2,'H&$&(!"1(BP-(P>@008A$)`%(
MBP.(9P)(BP-!B$0D`TB+0PA(P>@808A$)`1(BT,(2,'H$$&(1"0%2(M#"(AG
M!DB+0PA!B$0D!TB+0Q!(P>@808A$)`A(BT,02,'H$$&(1"0)2(M#$(AA"DB+
M0Q!!B$0D"TB+0QA(P>@808A$)`Q(BT,82,'H$$&(1"0-2(M#&(AG#DB+0QA(
MB=]!B$0D#TB+0R!(P>@808A$)!!(BT,@2,'H$$&(1"012(M#((AB$DB+0R!!
MB$0D$TB#Q`A;05Q!74%>05]=_R4:8@``9I"Z.````#'V*<J#^@AS*O;"!'5M
MA=(/A'?^___&``#VP@(/A&O^__\QR6:)3!#^Z5_^__\/'T0``(G12,<`````
M`$C'1`CX`````$B-2`A(@^'X2"G(`<*#XOB#^@@/@B_^__^#XO@QP(G'@\`(
M2(DT.3G0<O/I&/[__V8/'T0``,<``````,=$$/P`````Z?_]__]F+@\?A```
M````#Q\`\P\>^DB+#<UB``#_)?=@``````#S#Q[Z2(/L"$B#Q`C#````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````0`"`"5S+R5S`$1Y;F%,;V%D97(Z.F)O;W1?1'EN84QO861E<@!)
M;G1E<FYA;',Z.E!!4CHZ0D]/5``O<')O8R]S96QF+V5X90`O<')O8R\E:2\E
M<P`N`%!!4E]414U0`%!!4E]04D]'3D%-10`Z`"\`4$523#5,24(`4$523$Q)
M0@!015),-4]05`!015),24\`4$%27TE.251)04Q)6D5$`%!!4E]34$%73D5$
M`%!!4E]#3$5!3@!005)?1$5"54<`4$%27T-!0TA%`%!!4E]'3$]"04Q?1$5"
M54<`4$%27T=,3T)!3%]435!$25(`4$%27U1-4$1)4@!005)?1TQ/0D%,7U1%
M35``4$%27T=,3T)!3%]#3$5!3@`Q`&QS965K(&9A:6QE9`!R96%D(&9A:6QE
M9`!,1%],24)205)97U!!5$@`4UE35$5-`"]T;7``5$5-4$1)4@!435``55-%
M4@!54T523D%-10`E,#)X`"5S)7,E<R5S`'!A<BT`<&%R;``N<&%R`'-H;W)T
M(')E860`)7,E<V-A8VAE+25S)7,`)7,E<W1E;7`M)74E<P`E<R5S=&5M<"TE
M=2TE=25S`!@`<&5R;``P`%!!4CHZ4&%C:V5R.CI615)324].`#$N,#8S`"UE
M`"TM`````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E('-U8F1I<F5C=&]R>2`E
M<R!F86EL960@*&5R<FYO/25I*0H````````E<SH@<')I=F%T92!S=6)D:7)E
M8W1O<GD@)7,@:7,@=6YS869E("AP;&5A<V4@<F5M;W9E(&ET(&%N9"!R971R
M>2!Y;W5R(&]P97)A=&EO;BD*````````56YA8FQE('1O(&-R96%T92!C86-H
M92!D:7)E8W1O<GD``````````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E(&-A
M8VAE('-U8F1I<F5C=&]R>2`E<R!F86EL960@*&5R<FYO/25I*0H`````````
M`&5V86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII
M9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*<&%C:V%G92!?
M7W!A<E]P;#L*;7D@*"1005)?34%'24,L("1&24Q%7V]F9G-E=%]S:7IE+"`D
M8V%C:&5?;F%M95]S:7IE*3L*0D5'24X@>PHD4$%27TU!1TE#(#T@(EQN4$%2
M+G!M7&XB.PHD1DE,15]O9F9S971?<VEZ92`](#0[("`@(R!P86-K*").(BD*
M)&-A8VAE7VYA;65?<VEZ92`](#0P.PHD4$M:25!?34%'24,@/2`B4$M<,#`S
M7#`P-"(["GT*<W5B(&9I;F1?<&%R7VUA9VEC"GL*;7D@*"1F:"D@/2!`7SL*
M;7D@)&-H=6YK7W-I>F4@/2`V-"`J(#$P,C0["FUY("1B=68["FUY("1S:7IE
M(#T@+7,@)&9H.PIM>2`D<&]S(#T@*"1S:7IE+3$I("T@*"1S:7IE+3$I("4@
M)&-H=6YK7W-I>F4[("`@("`@(R!.3U1%.B`D<&]S(&ES(&$@;75L=&EP;&4@
M;V8@)&-H=6YK7W-I>F4*=VAI;&4@*"1P;W,@/CT@,"D@>PIS965K("1F:"P@
M)'!O<RP@,#L*<F5A9"`D9F@L("1B=68L("1C:'5N:U]S:7IE("L@;&5N9W1H
M*"1005)?34%'24,I.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B=68L("1005)?
M34%'24,I*2`^/2`P*2!["G)E='5R;B`D<&]S("L@)&D["GT*)'!O<R`M/2`D
M8VAU;FM?<VEZ93L*?0IR971U<FX@+3$["GT*;7D@*"1P87)?=&5M<"P@)'!R
M;V=N86UE+"!`=&UP9FEL92P@)4UO9'5L94-A8VAE*3L*14Y$('L@:68@*"1%
M3E9[4$%27T-,14%.?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@
M1FEL93HZ0F%S96YA;64["G)E<75I<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I
M<B`]($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P87)?=&5M<"D["F]U=',H
M<7%;4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN
M9#HZ9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M9&ER(#H@=6YL:6YK('TL
M("1P87)?=&5M<"D["G)M9&ER("1P87)?=&5M<#L*:68@*"UD("1P87)?=&5M
M<"`F)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY("1T;7`@/2!N97<@1FEL93HZ
M5&5M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A
M<V5N86UE.CID:7)N86UE*"1T;W!D:7(I+`I3549&25@@/3X@)RYC;60G+`I5
M3DQ)3DL@/3X@,"P**3L*;7D@)&9I;&5N86UE(#T@)'1M<"T^9FEL96YA;64[
M"G!R:6YT("1T;7`@/#PB+BXN(CL*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@
M73L@9&\*<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M
M<"<@73L@=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*
M9&]N90IR;2`G)&9I;&5N86UE)PHN+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P
M,"P@)&9I;&5N86UE.PIM>2`D8VUD(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL
M(#(^)C$@)B(["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG
M<F]U;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI
M.PI]"GT@?0I"14=)3B!["DEN=&5R;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F
M:6YE9"`F26YT97)N86QS.CI005(Z.D)/3U0["F5V86P@>PI?<&%R7VEN:71?
M96YV*"D["FUY("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT["FUY("5S>7,@
M/2`H"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G
M.B<I+`I?97AE("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y
M9W=I;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^
M("]>35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE
M*"D["F]U=',H<7%;7"1P<F]G;F%M92`]("(D<')O9VYA;64B72D["E]S971?
M<&%R7W1E;7`H*3L*;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@/2`B)$5.5GM0
M05)?5$5-4'TB72D["FUY("@D<W1A<G1?<&]S+"`D<W1A<G1?;V9?1DE,15]S
M96-T:6]N*3L*34%'24,Z('L*;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@
M>WT["G5N;&5S<R`H;W!E;B!?1D@L("<\.G)A=R<L("1P<F]G;F%M92D@>PIO
M=71S*'%Q6T-A;B=T(')E860@9G)O;2!F:6QE("(D<')O9VYA;64B72D[("`C
M(&1O;B=T('5S92`D(2!H97)E(&%S(&ET(')E<75I<F5S($5R<FYO+G!M"FQA
M<W0@34%'24,["GT*;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R7VUA9VEC*"I?
M1D@I.PII9B`H)&UA9VEC7W!O<R`\(#`I('L*;W5T<RAQ<5M#86XG="!F:6YD
M(&UA9VEC('-T<FEN9R`B)%!!4E]-04=)0R(@:6X@9FEL92`B)'!R;V=N86UE
M(ETI.PIL87-T($U!1TE#.PI]"F]U=',H(D9O=6YD(%!!4B!M86=I8R!A="!P
M;W-I=&EO;B`D;6%G:6-?<&]S(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@
M)$9)3$5?;V9F<V5T7W-I>F4@+2!L96YG=&@H(EPP0T%#2$4B*2P@,#L*<F5A
M9"!?1D@L("1B=68L(&QE;F=T:"@B7#!#04-(12(I.PII9B`H)&)U9B!N92`B
M7#!#04-(12(@)B8@)&)U9B!N92`B7#!#3$5!3B(I('L*;W5T<R@B3F\@8V%C
M:&4@;6%R:V5R(&9O=6YD(BD["FQA<W0@34%'24,["GT*;W5T<RAQ<5M#86-H
M92!M87)K97(@(B1B=68B(&9O=6YD72D["G-E96L@7T9(+"`D;6%G:6-?<&]S
M("T@)$9)3$5?;V9F<V5T7W-I>F4L(#`["G)E860@7T9(+"`D8G5F+"`D1DE,
M15]O9F9S971?<VEZ93L*;7D@)&]F9G-E="`]('5N<&%C:R@B3B(L("1B=68I
M.PIO=71S*")/9F9S970@9G)O;2!S=&%R="!O9B!&24Q%<R!I<R`D;V9F<V5T
M(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@
M+2`D;V9F<V5T+"`P.PHD<W1A<G1?;V9?1DE,15]S96-T:6]N(#T@=&5L;"!?
M1D@["FUY("5R97%U:7)E7VQI<W0["G)E860@7T9(+"`D8G5F+"`T.R`@("`@
M("`@("`@("`@("`@("`@("`@("`@(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B
M"G=H:6QE("@D8G5F(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*
M<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA
M;64@/2`D8G5F.PIO=71S*'%Q6U5N<&%C:VEN9R!&24Q%("(D9G5L;&YA;64B
M+BXN72D["FUY("1C<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX
M?2DO?'P@*2`_("0Q(#H@=6YD968["FUY("@D8F%S96YA;64L("1E>'0I(#T@
M*"1B=68@/7X@;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B
M=68L(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@
M*&1E9FEN960H)&5X="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI
M)"]I*2!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D8W)C)&5X="(L("1B
M=68L(#`W-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@
M/2`D9FEL96YA;64["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE
M?2`]("1F=6QL;F%M93L*?0IE;'-E('L*)$UO9'5L94-A8VAE>R1F=6QL;F%M
M97T@/2!["F)U9B`]/B`D8G5F+`IC<F,@/3X@)&-R8RP*;F%M92`]/B`D9G5L
M;&YA;64L"GT["B1R97%U:7)E7VQI<W1[)&9U;&QN86UE?2LK.PI]"G)E860@
M7T9(+"`D8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*;7D@*"1S96QF
M+"`D;6]D=6QE*2`]($!?.PIR971U<FX@:68@<F5F("1M;V1U;&4@;W(@(21M
M;V1U;&4["FUY("1I;F9O(#T@)$UO9'5L94-A8VAE>R1M;V1U;&5].PIR971U
M<FX@=6YL97-S("1I;F9O.PID96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE
M?3L*)$E.0WLD;6]D=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M;V1U;&4B.PII
M9B`H)$5.5GM005)?0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ1FEL93HZ;F5W
M*2D@>PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@;W(@9&EE(")#
M86XG="!C<F5A=&4@=&5M<"!F:6QE.B`D(2(["B1F:"T^8FEN;6]D92@I.PHD
M9F@M/G!R:6YT*"1I;F9O+3Y[8G5F?2D["B1F:"T^<V5E:R@P+"`P*3L*<F5T
M=7)N("1F:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1I
M;F9O+3Y[8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D["B1I;F9O+3Y[9FEL97T@
M/2`D9FEL96YA;64["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D9FEL96YA;64@
M;W(@9&EE('%Q6T-A;B=T(')E860@(B1F:6QE;F%M92(Z("0A73L*<F5T=7)N
M("1F:#L*?0ID:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C86XG="!F:6YD
M(&UO9'5L92`D;6]D=6QE(2(["GTL($!)3D,I.PIR97%U:7)E(%A33&]A9&5R
M.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E
M<75I<F4@17AP;W)T97(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA
M;&]A9&5R*"D["G)E<75I<F4@24\Z.D9I;&4["G=H:6QE("AM>2`D9FEL96YA
M;64@/2`H<V]R="!K97ES("5R97%U:7)E7VQI<W0I6S!=*2!["G5N;&5S<R`H
M)$E.0WLD9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O0E-$4$%.+RD@>PII
M9B`H)&9I;&5N86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER92`D9FEL96YA
M;64["GT*96QS92!["F1O("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE(#U^
M("]S:71E8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R97%U:7)E7VQI
M<W1[)&9I;&5N86UE?3L*?0II9B`H)&)U9B!N92`D4$M:25!?34%'24,I('L*
M;W5T<RAQ<5M.;R!Z:7`@9F]U;F0@869T97(@1DE,12!S96-T:6]N(&EN(&9I
M;&4@(B1P<F]G;F%M92)=*3L*;&%S="!-04=)0R`["GT*)'-T87)T7W!O<R`]
M("AT96QL(%]&2"D@+2`T.R`@("`@("`@("`@("`@("`C('-T87)T(&]F('II
M<`I]"FUY($!P87)?87)G<SL*;7D@*"1O=70L("1B=6YD;&4L("1L;V=F:"P@
M)&-A8VAE7VYA;64I.PID96QE=&4@)$5.5GM005)?05!07U)%55-%?3L@(R!S
M86YI=&EZ92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*:68@
M*"$D<W1A<G1?<&]S(&]R("@D05)'5ELP72!E<2`G+2UP87(M;W!T:6]N<R<@
M)B8@<VAI9G0I*2!["FUY("5D:7-T7V-M9"`]('%W*`IP("`@8FQI8E]T;U]P
M87(*:2`@(&EN<W1A;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R"G,@("!S:6=N
M7W!A<@IV("`@=F5R:69Y7W!A<@HI.PII9B`H0$%21U8@86YD("1!4D=66S!=
M(&5Q("<M+7)E=7-E)RD@>PIS:&EF="!`05)'5CL*)$5.5GM005)?05!07U)%
M55-%?2`]('-H:69T($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA;"!P87)L(&)E
M:&%V:6]U<@IM>2!`861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6*2!["B1!4D=6
M6S!=(#U^("]>+2A;04E-3T),8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0["FEF
M("@D,2!E<2`G22<I('L*<'5S:"!`861D7W1O7VEN8RP@)#(["GT*96QS:68@
M*"0Q(&5Q("=-)RD@>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF("@D,2!E<2`G
M02<I('L*=6YS:&EF="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF("@D,2!E<2`G
M3R<I('L*)&]U="`]("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I('L*)&)U;F1L
M92`]("=S:71E)SL*?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B=6YD;&4@/2`G
M86QL)SL*?0IE;'-I9B`H)#$@97$@)W$G*2!["B1Q=6EE="`](#$["GT*96QS
M:68@*"0Q(&5Q("=,)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@9&EE
M('%Q6T-A;B=T(&]P96X@;&]G(&9I;&4@(B0R(CH@)"%=.PI]"F5L<VEF("@D
M,2!E<2`G5"<I('L*)&-A8VAE7VYA;64@/2`D,CL*?0IS:&EF="A`05)'5BD[
M"FEF("AM>2`D8VUD(#T@)&1I<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.5GLG
M4$%27U1%35`G?3L*:6YI=%]I;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0["B9[
M(E!!4CHZ1&ES=#HZ)&-M9")]*"D@=6YL97-S($!!4D=6.PHF>R)005(Z.D1I
M<W0Z.B1C;60B?2@D7RD@9F]R($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF="!`
M24Y#+"!`861D7W1O7VEN8SL*?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER92!)
M3SHZ1FEL93L*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T
M.CI32$$["GT*;7D@)'!A<B`]('-H:69T*$!!4D=6*3L*;7D@)'II<#L*:68@
M*&1E9FEN960@)'!A<BD@>PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)'!A<B!O
M<B!D:64@<7%;0V%N)W0@9FEN9"!P87(@9FEL92`B)'!A<B(Z("0A73L*8FQE
M<W,H)&9H+"`G24\Z.D9I;&4G*3L*)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W
M.PHH("1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P87(I(#T]($%R
M8VAI=F4Z.EII<#HZ05I?3TLH*2`I"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG
M('II<"!A<F-H:79E("(D<&%R(ET["GT*;7D@)6UE=&%?<&%R(#T@9&\@>PII
M9B`H)'II<"!A;F0@;7D@)&UE=&$@/2`D>FEP+3YC;VYT96YT<R@G345402YY
M;6PG*2D@>PHD;65T82`]?B!S+RXJ7G!A<CHD+R]M<SL*)&UE=&$@/7X@<R]>
M7%,N*B\O;7,["B1M971A(#U^("]>("`H6UXZ72LI.B`H+BLI)"]M9SL*?0I]
M.PII9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)'!H+"`G/#IR87<G+"`D
M<&%R(&]R(&1I92!Q<5M#86XG="!R96%D('!A<B!F:6QE("(D<&%R(CH@)"%=
M.PIM>2`D8G5F.PIR96%D("1P:"P@)&)U9BP@-#L*9&EE('%Q6R(D<&%R(B!I
M<R!N;W0@82!P87(@9FEL95T@=6YL97-S("1B=68@97$@)%!+6DE07TU!1TE#
M.PIC;&]S92`D<&@["GT*0W)E871E4&%T:"@D;W5T*2`["FUY("1F:"`]($E/
M.CI&:6QE+3YN97<H"B1O=70L"DE/.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ
M1FEL93HZ3U]21%=2*"D@?"!)3SHZ1FEL93HZ3U]44E5.0R@I+`HP-S<W+`HI
M(&]R(&1I92!Q<5M#86XG="!C<F5A=&4@9FEL92`B)&]U="(Z("0A73L*)&9H
M+3YB:6YM;V1E*"D["G-E96L@7T9(+"`P+"`P.PIM>2`D;&]A9&5R.PII9B`H
M9&5F:6YE9"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*2!["G)E860@7T9(+"`D
M;&]A9&5R+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N.PI](&5L<V4@>PIL;V-A
M;"`D+R`]('5N9&5F.PHD;&]A9&5R(#T@/%]&2#X["GT*:68@*"$D14Y6>U!!
M4E]615)"051)37T@86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<0')E;2DO*2![
M"G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ
M4&]D4W1R:7`M/F%P<&QY*%PD;&]A9&5R+"`D,"D["GT*)&9H+3YP<FEN="@D
M;&]A9&5R*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!L;V%D97(@=&\@(B1O
M=70B.B`D(5T["FEF("@D8G5N9&QE*2!["G)E<75I<F4@4$%2.CI(96%V>3L*
M4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PII;FET7VEN8R@I.PIR
M97%U:7)E7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R97`@>R`A+T)31%!!3B\@
M?0IG<F5P('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@;F4@)$-O;F9I
M9SHZ0V]N9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N9FEG.CI#;VYF
M:6=[<')I=FQI8F5X<'TI.PI]($!)3D,["G-\+RM\+WQG+"!S?"\D?'P@9F]R
M96%C:"!`:6YC.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,
M;V%D97(Z.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI
M8E]E>'0@/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T[("`@("`@("`@(R!8
M6%@@;&EB7V5X="!V<R!D;&5X="`_"FUY("5W<FET=&5N.PIF;W)E86-H(&UY
M("1K97D@*'-O<G0@:V5Y<R`E9FEL97,I('L*;7D@*"1F:6QE+"`D;F%M92D[
M"FEF("AD969I;F5D*&UY("1F8R`]("1005(Z.DAE879Y.CI&=6QL0V%C:&5[
M)&ME>7TI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("1F8RD["GT*96QS
M92!["F9O<F5A8V@@;7D@)&1I<B`H0&EN8RD@>PII9B`H)&ME>2`]?B!M?%Y<
M421D:7)<12\H+BHI)'QI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("0Q
M*3L*;&%S=#L*?0II9B`H)&ME>2`]?B!M?%XO;&]A9&5R+UM>+UTK+R@N*BDD
M?"D@>PII9B`H;7D@)')E9B`]("1-;V1U;&5#86-H97LD,7TI('L**"1F:6QE
M+"`D;F%M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"FEF("@M9B`B)&1I<B\D
M,2(I('L**"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D,2D["FQA<W0[
M"GT*?0I]"GT*;F5X="!U;FQE<W,@9&5F:6YE9"`D;F%M93L*;F5X="!I9B`D
M=W)I='1E;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE*2!A;F0@)&9I
M;&4@/7X@+UPN7%$D;&EB7V5X=%Q%)"]I.PIO=71S*'-P<FEN=&8H<7%;4&%C
M:VEN9R!&24Q%("(E<R(N+BY=+"!R968@)&9I;&4@/R`D9FEL92T^>VYA;65]
M(#H@)&9I;&4I*3L*;7D@)&-O;G1E;G0["FEF("AR968H)&9I;&4I*2!["B1C
M;VYT96YT(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIL;V-A;"`D+R`]('5N
M9&5F.PIO<&5N(&UY("1T:"P@)SPZ<F%W)RP@)&9I;&4@;W(@9&EE('%Q6T-A
M;B=T(')E860@(B1F:6QE(CH@)"%=.PHD8V]N=&5N="`](#PD=&@^.PIC;&]S
M92`D=&@["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1C;VYT96YT
M+"`B/&5M8F5D9&5D/B\D;F%M92(I"FEF("$D14Y6>U!!4E]615)"051)37T@
M86YD("1N86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2.CI&:6QT97(Z
M.E!A=&-H0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F:6QE+"`D
M;F%M92D["GT*)&9H+3YP<FEN="@B1DE,12(L"G!A8VLH)TXG+"!L96YG=&@H
M)&YA;64I("L@.2DL"G-P<FEN=&8H(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z
M.F-O;7!U=&5#4D,S,B@D8V]N=&5N="DL("1N86UE*2P*<&%C:R@G3B<L(&QE
M;F=T:"@D8V]N=&5N="DI+`HD8V]N=&5N="D*;W(@9&EE('%Q6T5R<F]R('=R
M:71I;F<@96UB961D960@1DE,12!T;R`B)&]U="(Z("0A73L*;W5T<RAQ<5M7
M<FET=&5N(&%S("(D;F%M92)=*3L*?0I]"FEF("@D>FEP*2!["B1Z:7`M/G=R
M:71E5&]&:6QE2&%N9&QE*"1F:"D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2PIO
M<B!D:64@<7%;17)R;W(@=W)I=&EN9R!Z:7`@<&%R="!O9B`B)&]U=")=.PI]
M"FEF("@D;65T85]P87)[8VQE86Y]*2!["B1F:"T^<')I;G0H(EPP0TQ%04XB
M*3L*?0IE;'-E('L*:68@*"%D969I;F5D("1C86-H95]N86UE*2!["FUY("1C
M='@@/2!$:6=E<W0Z.E-(02T^;F5W*#$I.PIS965K("1F:"P@,"P@,#L*)&-T
M>"T^861D9FEL92@D9F@I.PIS965K("1F:"P@,"P@,CL*)&-A8VAE7VYA;64@
M/2`D8W1X+3YH97AD:6=E<W0["GT*)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A
M8VAE7VYA;64L(#`L("1C86-H95]N86UE7W-I>F4["B1C86-H95]N86UE("X]
M(")<,"(@>"`H)&-A8VAE7VYA;65?<VEZ92`M(&QE;F=T:"`D8V%C:&5?;F%M
M92D["B1F:"T^<')I;G0H)&-A8VAE7VYA;64L"B)<,$-!0TA%(BD["GT*;7D@
M)&]F9G-E="`]("1F:"T^=&5L;"`M(&QE;F=T:"@D;&]A9&5R*3L*)&9H+3YP
M<FEN="AP86-K*"=.)RP@)&]F9G-E="DL"B1005)?34%'24,I.PHD9F@M/F-L
M;W-E(&]R(&1I92!Q<5M%<G)O<B!W<FET:6YG('1R86EL97(@;V8@(B1O=70B
M.B`D(5T["F-H;6]D(#`W-34L("1O=70["F5X:70["GT*>PIL87-T('5N;&5S
M<R!D969I;F5D("1S=&%R=%]P;W,["E]F:7A?<')O9VYA;64H*3L*<F5Q=6ER
M92!005(["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*>PIR97%U
M:7)E($9I;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV93HZ6FEP.PI]"FUY("1F
M:"`]($E/.CI&:6QE+3YN97<[("`@("`@("`@("`@("`@("`@("`@("`@("`@
M("`C($%R8VAI=F4Z.EII<"!O<&5R871E<R!O;B!A;B!)3SHZ2&%N9&QE"B1F
M:"T^9F1O<&5N*&9I;&5N;RA?1D@I+"`G<B<I(&]R(&1I92!Q<5MF9&]P96XH
M*2!F86EL960Z("0A73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H+7,@
M7T9(*3L*;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH)'II<"T^<F5A
M9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!R;V=N86UE*2`]/2!!<F-H:79E.CI:
M:7`Z.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H
M:79E("(D<')O9VYA;64B73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H
M-C0@*B`Q,#(T*3L*<'5S:"!`4$%2.CI,:6)#86-H92P@)'II<#L*)%!!4CHZ
M3&EB0V%C:&5[)'!R;V=N86UE?2`]("1Z:7`["F]U=',H(D5X=')A8W1I;F<@
M>FEP+BXN(BD["FEF("AD969I;F5D("1%3E9[4$%27U1%35!]*2![(",@<VAO
M=6QD(&)E('-E="!A="!T:&ES('!O:6YT(0IF;W)E86-H(&UY("1M96UB97(@
M*"`D>FEP+3YM96UB97)S("D@>PIN97AT(&EF("1M96UB97(M/FES1&ER96-T
M;W)Y.PIM>2`D;65M8F5R7VYA;64@/2`D;65M8F5R+3YF:6QE3F%M93L*;F5X
M="!U;FQE<W,@)&UE;6)E<E]N86UE(#U^(&U["EX*+S]S:&QI8B\**#\Z)$-O
M;F9I9SHZ0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E
M>'1R86-T7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M
M/F-A=&9I;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H
M+68@)&1E<W1?;F%M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E
M9%-I>F4H*2D@>PIO=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I
M;F-E(&ET(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(ETI.PI](&5L
M<V4@>PIO=71S*'%Q6T5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D
M97-T7VYA;64B+BXN72D["B@D;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H
M)&1E<W1?;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@
M<7%;17)R;W(@97AT<F%C=&EN9R!Z:7`@;65M8F5R('1O("(D9&5S=%]N86UE
M(ET["F-H;6]D*#`U-34L("1D97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"([
M"GT*?0I]"GT*=6YL97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2![
M"F1I92`\/"`B+B(@=6YL97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP
M87(@72!;("U)9&ER(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O
M9W)A;2YP;"!="B0P(%L@+4)\+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*
M)$5.5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`
M05)'5BD["GT*<W5B($-R96%T95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E
M<75I<F4@1FEL93HZ0F%S96YA;64["FUY("@D8F%S96YA;64L("1P871H+"`D
M97AT*2`]($9I;&4Z.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN
M*B<I*3L*<F5Q=6ER92!&:6QE.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H
M)'!A=&@I('5N;&5S<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E
M<G)O<@I]"G-U8B!R97%U:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q
M=6ER92!$>6YA3&]A9&5R.PIR97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R
M:6-T.PIR97%U:7)E('=A<FYI;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@
M0V%R<#L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q
M=6ER92!%>'!O<G1E<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I
M<F4@1F-N=&P["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3
M<&5C.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER
M92!)3SHZ2&%N9&QE.PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R
M97-S.CI:;&EB.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E
M<W0Z.E-(03L*<F5Q=6ER92!005(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q
M=6ER92!005(Z.D1I<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP
M.PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@
M871T<FEB=71E<SL*979A;"![(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U
M:7)E(%=I;C,R('T["F5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE
M=F%L('L@<F5Q=6ER92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![
M(')E<75I<F4@5&EE.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E
M<75I<F4@4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@
M>R!R97%U:7)E('5T9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD
M969I;F5D("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O
M*"XK*2\I('L*)'!A<E]T96UP(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY
M("1P871H("@**&UA<"`D14Y6>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2
M(%1%35!$25(@5$5-4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI
M('L*;F5X="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD
M("UW("1P871H.PIM>2`D=7-E<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P
M=W5I9"D@/2!G971P=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E
M9FEN960H)E=I;C,R.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN
M,S(Z.DQO9VEN3F%M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S
M97)N86UE(#T@)'!W=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U53
M15).04U%?2!\?"`D14Y6>U5315)]('Q\("=365-414TG.PI]"FUY("1S=&UP
M9&ER(#T@(B1P871H)'-Y<WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U
M<V5R;F%M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U.PIM>2`D8V%C:&5?9&ER
M.PII9B`H)$5.5GM005)?0TQ%04Y]*2!["B1C86-H95]D:7(@/2`B=&5M<"TD
M)"(["GT*96QS92!["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA;64@
M;W(@9&EE('%Q6T-A;B=T(')E860@(B1P<F]G;F%M92(Z("0A73L*:68@*"AM
M>2`D;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H)&9H*2D@/CT@,"D@>PIS
M965K("1F:"P@)&UA9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S:7IE"BT@;&5N
M9W1H*")<,$-!0TA%(BDL(#`["FUY("1B=68["G)E860@)&9H+"`D8G5F+"!L
M96YG=&@H(EPP0T%#2$4B*3L*:68@*"1B=68@97$@(EPP0T%#2$4B*2!["G-E
M96L@)&9H+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG
M=&@H(EPP0T%#2$4B*0HM("1C86-H95]N86UE7W-I>F4L(#`["G)E860@)&9H
M+"`D8G5F+"`D8V%C:&5?;F%M95]S:7IE.PHD8G5F(#U^(',O7#`O+V<["B1C
M86-H95]D:7(@/2`B8V%C:&4M)&)U9B(["GT*?0IC;&]S92`D9F@["GT*:68@
M*"$D8V%C:&5?9&ER*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["B1%3E9[
M4$%27T-,14%.?2`](#$["GT*)'-T;7!D:7(@+CT@(B1S>7-[7V1E;&EM?21C
M86-H95]D:7(B.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*)$5.5GM005)?5$5-
M4'T@/2`D<W1M<&1I<CL*;&%S=#L*?0HD<&%R7W1E;7`@/2`D,2!I9B`D14Y6
M>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO.PI]"G-U
M8B!?<V%V95]A<R!["FUY("@D;F%M92P@)&-O;G1E;G1S+"`D;6]D92D@/2!`
M7SL*;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M92(["G5N;&5S<R`H
M+64@)&9U;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D9G5L;&YA;64N)"0B
M.PIM>2`D9F@["BAO<&5N("1F:"P@)SXZ<F%W)RP@)'1E;7!N86UE*0HF)B`H
M<')I;G0@)&9H("1C;VYT96YT<RD*)B8@*&-L;W-E("1F:"D*;W(@9&EE('%Q
M6T5R<F]R('=R:71I;F<@(B1T96UP;F%M92(Z("0A73L*8VAM;V0@)&UO9&4L
M("1T96UP;F%M92!I9B!D969I;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE
M+"`D9G5L;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D
M9G5L;&YA;64["GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D
M14Y6>U!!4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O
M*"XK*2\I('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*
M:68@*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[
M4$%27U1%35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA
M;64L(')I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@*R`Q*3L*?0II
M9B`H(21%3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L("1S
M>7-[7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\)RP@)'!R
M;V=N86UE*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D<')O9VYA
M;64D<WES>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)'-Y<WM?97AE?3L*<F5T
M=7)N.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D<WES>W!A=&A?
M<V5P?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF(&5X:7-T<R`D14Y6>U!!
M4E]414U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]414U0?3L*)&1I<B`]?B!S
M+UQ1)'-Y<WM?9&5L:6U]7$4D+R\["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES
M>U]D96QI;7TD<')O9VYA;64D<WES>U]E>&5](BDL(&QA<W0I"FEF("US("(D
M9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2(["B@H)'!R;V=N
M86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@
M+7,@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?
M<')O9VYA;64@>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!
M345].PII9B`H:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI(#P@,"D@
M>PHD<')O9VYA;64@/2`B+B1S>7-[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@
M)'!W9"`]("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I
M"CH@*"AD969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H
M*2`Z(&!P=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<
M+EPN/UQ1)'-Y<WM?9&5L:6U]7$4I+R1P=V0D<WES>U]D96QI;7TO.PHD14Y6
M>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?:6YI=%]E
M;G8@>PII9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2!["G)E
M='5R;CL*?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@,CL*?0IF
M;W(@*'%W*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!
M344@*2`I('L*9&5L971E("1%3E9[(E!!4E\D7R)].PI]"F9O<B`H<7<O(%1-
M4$1)4B!414U0($-,14%.($1%0E5'("\I('L*)$5.5GLB4$%27R1?(GT@/2`D
M14Y6>R)005)?1TQ/0D%,7R1?(GT@:68@97AI<W1S("1%3E9[(E!!4E]'3$]"
M04Q?)%\B?3L*?0II9B`H)$5.5GM005)?5$5-4'TI('L*9&5L971E("1%3E9[
M4$%27T-,14%.?3L*?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*
M:68@*"1L;V=F:"D@>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP
M<FEN="`B0%]<;B(["GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I
M9SL*<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF
M:6=[)%]]+"!Q=R@*87)C:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP
M('-I=&5L:6)E>'`*=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP
M86-K86=E(&UA:6X["G)E<75I<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2
M.CIF:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q6W!A
M<BYP;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@
M<W5C:"!F:6QE(&]R(&1I<F5C=&]R>5QN70IU;FQE<W,@+64@)'!R;V=N86UE
M.PID;R`D<')O9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+
M7T58251?7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/
M4B`]("1`(&EF("1`.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2
M(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@
M)#HZ7U]%4E)/4CL*,3L*7U]%3D1?7PH``$-!0TA%``!P97)L>'-I+F,`````
M`````2-%9P````")J\WO`````/[<NI@`````=E0R$`````#PX=+#````````
M`````````1L#.^P````<````<'___Y`$``"0@?__"`$``)""__\@`0``X)C_
M_U`!``"PF?__?`$``$":__^@`0``P)O__\P!```@G/__[`$```"=__\8`@``
M<)___T@"``"@G___7`(``)"@__^$`@``H*#__Y@"``#0HO__Q`(``!"C___D
M`@``<*3__Q0#``#0J/__1`,``$"I__]H`P``$*K__X0#``!PJ___K`,``)"K
M___,`P``@*S___P#``"`M/__-`0``-"V__]@!```L+?__[@$````N/__V`0`
M`/"Y__\(!0``@+S__S@%````````%``````````!>E(``7@0`1L,!PB0`0``
M%````!P```"`@/__)@````!$!Q``````+````#0```!H@?__3!8```!!#A"&
M`D,-!EZ/`XX$C06,!H,'`R,6"@P'"$$+````*````&0```"(E___RP````!!
M#A"&`D,-!DV/`XX$C06,!H,'`G<*#`<(2`L@````D````"R8__^%`````$$.
M$(8"0PT&3X,#`FL*#`<(00LH````M````)B8__]]`0```$$.$(8"0PT&4(\#
MC@2-!8P&@P<"0`H,!PA$"QP```#@````[)G__UT`````10X0A@)1#09%@P-\
M#`<(*``````!```LFO__VP````!!#A"&`D,-!E>.`XT$C`6#!@)^"@P'"$$+
M```L````+`$``.":__]B`@```$$.$(8"10T&28\#C@2-!8P&@P<#-`$*#`<(
M10L````0````7`$``""=__\J`````````"0```!P`0``/)W__^,`````10X0
MA@)##09AC0.,!(,%`K,*#`<(00L0````F`$```2>__\*`````````"@```"L
M`0```)[__R4"````10X0A@)##09<CP..!(T%C`:#!P)S"@P'"$D+'````-@!
M```$H/__-`````!%#A"&`D@-!D&#`V`,!P@L````^`$``"2@__]2`0```$4.
M$(8"0PT&4XP#@P0"\0H,!PA$"V$*#`<(3PL````L````*`(``%2A__]<!```
M`$4.$(8"0PT&2(\#C@2-!8P&3(,'`Q`$"@P'"$4+```@````6`(``(2E__]J
M`````$4.$(8"2@T&`D<*#`<(2@L````8````?`(``-"E___"`````$4.$(8"
M0PT&````)````)@"``"$IO__5P$```!!#A"&`D,-!F*-`XP$@P4"S0H,!PA%
M"QP```#``@``O*?__R``````10X0A@)##097#`<(````+````.`"``"\I___
MY@````!%#A"&`D,-!DJ/`XX$C05)C`9+@P<"2@H,!PA("P``-````!`#``!\
MJ/__``@```!%#A"&`D,-!D2/`XX$5HT%C`:#!P,2`PH,!PA$"P,5`PH,!PA+
M"P`H````2`,``$2P__]*`@```$4.$(8"2@T&1(T#C`1,@P4#?`$*#`<(10L`
M`"P```!T`P``:++__]P`````10X0A@)##09'C`.#!`):"@P'"$<+`F$*#`<(
M10L``"0```"D`P``V'K__Q@"````10X0A@)##09'C`.#!`-N`0H,!PA!"P`<
M````S`,``/"R__]'`````$4.$(8"2`T&>0P'"````"P```#L`P``(+/__^$!
M````10X0A@)&#09"CP-'C@2-!4:,!DB#!P,P`0H,!PA&"RP````<!```X+3_
M_X,"````10X0A@)##09"CP-*C@2-!8P&1(,'`]`!"@P'"$@+`!````!,!```
M0+?__Q$`````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````,",```````#@(@```````$YA````
M````Z6`````````&80```````%]A`````````0````````#H`@````````$`
M````````\P(````````=`````````(X$````````#```````````(```````
M``T`````````1%T````````9`````````*B[````````&P`````````(````
M`````!H`````````L+L````````<``````````@`````````]?[_;P````#0
M`P````````4`````````Z`T````````&`````````!@%````````"@``````
M``"Q!`````````L`````````&``````````5````````````````````!P``
M``````#H$P````````@`````````,`8````````)`````````!@`````````
M'@`````````(`````````/O__V\``````0``"`````#^__]O`````%@3````
M````____;P`````!`````````/#__V\`````FA(````````D`````````!@:
M````````(P`````````@`````````"4`````````"```````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````````````#8
MNP``````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````C`````````"E!!4BYP;0H``````````/AB
M````````1T-#.B`H1TY5*2`Q-"XR+C$@,C`R-#`X,#4``"YS:'-T<G1A8@`N
M:6YT97)P`"YN;W1E+F=N=2YP<F]P97)T>0`N;F]T92YG;G4N8G5I;&0M:60`
M+FYO=&4N04))+71A9P`N9VYU+FAA<V@`+F1Y;G-Y;0`N9'EN<W1R`"YG;G4N
M=F5R<VEO;@`N9VYU+G9E<G-I;VY?<@`N<F5L82YD>6X`+G)E;'(N9'EN`"YI
M;FET`"YT97AT`"YF:6YI`"YR;V1A=&$`+F5H7V9R86UE7VAD<@`N96A?9G)A
M;64`+FEN:71?87)R87D`+F9I;FE?87)R87D`+F1A=&$N<F5L+G)O`"YD>6YA
M;6EC`"YG;W0`+F1A=&$`+F)S<P`N8V]M;65N=```````````````````````
M````````````````````````````````````````````````````````````
M````````"P````$````"`````````!@#````````&`,````````<````````
M`````````````0```````````````````!,````'`````@`````````X`P``
M`````#@#````````4`````````````````````@````````````````````F
M````!P````(`````````B`,```````"(`P```````"0`````````````````
M```$````````````````````.0````<````"`````````*P#````````K`,`
M```````@````````````````````!````````````````````$<```#V__]O
M`@````````#0`P```````-`#````````1`$````````&``````````@`````
M``````````````!1````"P````(`````````&`4````````8!0```````-`(
M````````!P````$````(`````````!@`````````60````,````"````````
M`.@-````````Z`T```````"Q!````````````````````0``````````````
M`````&$```#___]O`@````````":$@```````)H2````````O``````````&
M``````````(``````````@````````!N````_O__;P(`````````6!,`````
M``!8$P```````)``````````!P````$````(````````````````````?0``
M``0````"`````````.@3````````Z!,````````P!@````````8`````````
M"``````````8`````````(<````3`````@`````````8&@```````!@:````
M````(`````````````````````@`````````"`````````"1`````0````8`
M`````````"``````````(````````!L````````````````````$````````
M````````````EP````$````&`````````"`@````````("`````````A/0``
M````````````````(````````````````````)T````!````!@````````!$
M70```````$1=````````#0````````````````````0`````````````````
M``"C`````0````(``````````&``````````8````````+!`````````````
M```````0````````````````````JP````$````"`````````+"@````````
ML*````````#L````````````````````!````````````````````+D````!
M`````@````````"@H0```````*"A````````8`0```````````````````@`
M``````````````````##````#@````,`````````J+L```````"HJP``````
M``@````````````````````(``````````@`````````SP````\````#````
M`````+"[````````L*L````````(````````````````````"``````````(
M`````````-L````!`````P````````"XNP```````+BK````````(```````
M``````````````@```````````````````#H````!@````,`````````V+L`
M``````#8JP`````````"````````!P`````````(`````````!``````````
M\0````$````#`````````-B]````````V*T````````H`@``````````````
M````"``````````(`````````/8````!`````P``````````P`````````"P
M````````*`````````````````````@```````````````````#\````"```
M``,`````````*,`````````HL````````"@````````````````````(````
M`````````````````0$```$````P````````````````````*+`````````;
M`````````````````````0`````````!``````````$````#````````````
M`````````````$.P````````"@$```````````````````$`````````````
%````````
