# Copyright 2021-2024 NVIDIA Corporation.  All rights reserved.
#
# Please refer to the NVIDIA end user license agreement (EULA) associated
# with this source code for terms and conditions that govern your use of
# this software. Any use, reproduction, disclosure, or distribution of
# this software and related documentation outside the terms of the EULA
# is strictly prohibited.
from typing import List, Tuple, Any, Optional
from enum import IntEnum
import cython
import ctypes
from libc.stdlib cimport calloc, free
from libc cimport string
from libc.stdint cimport int32_t, uint32_t, int64_t, uint64_t
from libc.stddef cimport wchar_t
from libc.limits cimport CHAR_MIN
from libcpp.vector cimport vector
from cpython.buffer cimport PyObject_CheckBuffer, PyObject_GetBuffer, PyBuffer_Release, PyBUF_SIMPLE, PyBUF_ANY_CONTIGUOUS
from cpython.bytes cimport PyBytes_FromStringAndSize

ctypedef unsigned long long signed_char_ptr
ctypedef unsigned long long unsigned_char_ptr
ctypedef unsigned long long char_ptr
ctypedef unsigned long long short_ptr
ctypedef unsigned long long unsigned_short_ptr
ctypedef unsigned long long int_ptr
ctypedef unsigned long long long_int_ptr
ctypedef unsigned long long long_long_int_ptr
ctypedef unsigned long long unsigned_int_ptr
ctypedef unsigned long long unsigned_long_int_ptr
ctypedef unsigned long long unsigned_long_long_int_ptr
ctypedef unsigned long long uint32_t_ptr
ctypedef unsigned long long uint64_t_ptr
ctypedef unsigned long long int32_t_ptr
ctypedef unsigned long long int64_t_ptr
ctypedef unsigned long long unsigned_ptr
ctypedef unsigned long long unsigned_long_long_ptr
ctypedef unsigned long long long_long_ptr
ctypedef unsigned long long size_t_ptr
ctypedef unsigned long long float_ptr
ctypedef unsigned long long double_ptr
ctypedef unsigned long long void_ptr

#: CUDA API version number
CUDA_VERSION = ccuda.CUDA_VERSION

#: CUDA IPC handle size
CU_IPC_HANDLE_SIZE = ccuda.CU_IPC_HANDLE_SIZE

#: Legacy stream handle
#:
#: Stream handle that can be passed as a CUstream to use an implicit stream
#: with legacy synchronization behavior.
#:
#: See details of the \link_sync_behavior
CU_STREAM_LEGACY = ccuda.CU_STREAM_LEGACY

#: Per-thread stream handle
#:
#: Stream handle that can be passed as a CUstream to use an implicit stream
#: with per-thread synchronization behavior.
#:
#: See details of the \link_sync_behavior
CU_STREAM_PER_THREAD = ccuda.CU_STREAM_PER_THREAD

CU_COMPUTE_ACCELERATED_TARGET_BASE = ccuda.CU_COMPUTE_ACCELERATED_TARGET_BASE

#: Conditional node handle flags Default value is applied when graph is
#: launched.
CU_GRAPH_COND_ASSIGN_DEFAULT = ccuda.CU_GRAPH_COND_ASSIGN_DEFAULT

#: This port activates when the kernel has finished executing.
CU_GRAPH_KERNEL_NODE_PORT_DEFAULT = ccuda.CU_GRAPH_KERNEL_NODE_PORT_DEFAULT

#: This port activates when all blocks of the kernel have performed
#: cudaTriggerProgrammaticLaunchCompletion() or have terminated. It must be
#: used with edge type :py:obj:`~.CU_GRAPH_DEPENDENCY_TYPE_PROGRAMMATIC`.
#: See also :py:obj:`~.CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_EVENT`.
CU_GRAPH_KERNEL_NODE_PORT_PROGRAMMATIC = ccuda.CU_GRAPH_KERNEL_NODE_PORT_PROGRAMMATIC

#: This port activates when all blocks of the kernel have begun execution.
#: See also :py:obj:`~.CU_LAUNCH_ATTRIBUTE_LAUNCH_COMPLETION_EVENT`.
CU_GRAPH_KERNEL_NODE_PORT_LAUNCH_ORDER = ccuda.CU_GRAPH_KERNEL_NODE_PORT_LAUNCH_ORDER

CU_KERNEL_NODE_ATTRIBUTE_ACCESS_POLICY_WINDOW = ccuda.CU_KERNEL_NODE_ATTRIBUTE_ACCESS_POLICY_WINDOW

CU_KERNEL_NODE_ATTRIBUTE_COOPERATIVE = ccuda.CU_KERNEL_NODE_ATTRIBUTE_COOPERATIVE

CU_KERNEL_NODE_ATTRIBUTE_CLUSTER_DIMENSION = ccuda.CU_KERNEL_NODE_ATTRIBUTE_CLUSTER_DIMENSION

CU_KERNEL_NODE_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE = ccuda.CU_KERNEL_NODE_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE

CU_KERNEL_NODE_ATTRIBUTE_PRIORITY = ccuda.CU_KERNEL_NODE_ATTRIBUTE_PRIORITY

CU_KERNEL_NODE_ATTRIBUTE_MEM_SYNC_DOMAIN_MAP = ccuda.CU_KERNEL_NODE_ATTRIBUTE_MEM_SYNC_DOMAIN_MAP

CU_KERNEL_NODE_ATTRIBUTE_MEM_SYNC_DOMAIN = ccuda.CU_KERNEL_NODE_ATTRIBUTE_MEM_SYNC_DOMAIN

CU_KERNEL_NODE_ATTRIBUTE_DEVICE_UPDATABLE_KERNEL_NODE = ccuda.CU_KERNEL_NODE_ATTRIBUTE_DEVICE_UPDATABLE_KERNEL_NODE

CU_STREAM_ATTRIBUTE_ACCESS_POLICY_WINDOW = ccuda.CU_STREAM_ATTRIBUTE_ACCESS_POLICY_WINDOW

CU_STREAM_ATTRIBUTE_SYNCHRONIZATION_POLICY = ccuda.CU_STREAM_ATTRIBUTE_SYNCHRONIZATION_POLICY

CU_STREAM_ATTRIBUTE_PRIORITY = ccuda.CU_STREAM_ATTRIBUTE_PRIORITY

CU_STREAM_ATTRIBUTE_MEM_SYNC_DOMAIN_MAP = ccuda.CU_STREAM_ATTRIBUTE_MEM_SYNC_DOMAIN_MAP

CU_STREAM_ATTRIBUTE_MEM_SYNC_DOMAIN = ccuda.CU_STREAM_ATTRIBUTE_MEM_SYNC_DOMAIN

#: If set, host memory is portable between CUDA contexts. Flag for
#: :py:obj:`~.cuMemHostAlloc()`
CU_MEMHOSTALLOC_PORTABLE = ccuda.CU_MEMHOSTALLOC_PORTABLE

#: If set, host memory is mapped into CUDA address space and
#: :py:obj:`~.cuMemHostGetDevicePointer()` may be called on the host
#: pointer. Flag for :py:obj:`~.cuMemHostAlloc()`
CU_MEMHOSTALLOC_DEVICEMAP = ccuda.CU_MEMHOSTALLOC_DEVICEMAP

#: If set, host memory is allocated as write-combined - fast to write,
#: faster to DMA, slow to read except via SSE4 streaming load instruction
#: (MOVNTDQA). Flag for :py:obj:`~.cuMemHostAlloc()`
CU_MEMHOSTALLOC_WRITECOMBINED = ccuda.CU_MEMHOSTALLOC_WRITECOMBINED

#: If set, host memory is portable between CUDA contexts. Flag for
#: :py:obj:`~.cuMemHostRegister()`
CU_MEMHOSTREGISTER_PORTABLE = ccuda.CU_MEMHOSTREGISTER_PORTABLE

#: If set, host memory is mapped into CUDA address space and
#: :py:obj:`~.cuMemHostGetDevicePointer()` may be called on the host
#: pointer. Flag for :py:obj:`~.cuMemHostRegister()`
CU_MEMHOSTREGISTER_DEVICEMAP = ccuda.CU_MEMHOSTREGISTER_DEVICEMAP

#: If set, the passed memory pointer is treated as pointing to some memory-
#: mapped I/O space, e.g. belonging to a third-party PCIe device. On
#: Windows the flag is a no-op. On Linux that memory is marked as non
#: cache-coherent for the GPU and is expected to be physically contiguous.
#: It may return :py:obj:`~.CUDA_ERROR_NOT_PERMITTED` if run as an
#: unprivileged user, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED` on older Linux
#: kernel versions. On all other platforms, it is not supported and
#: :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED` is returned. Flag for
#: :py:obj:`~.cuMemHostRegister()`
CU_MEMHOSTREGISTER_IOMEMORY = ccuda.CU_MEMHOSTREGISTER_IOMEMORY

#: If set, the passed memory pointer is treated as pointing to memory that
#: is considered read-only by the device. On platforms without
#: :py:obj:`~.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS_USES_HOST_PAGE_TABLES`,
#: this flag is required in order to register memory mapped to the CPU as
#: read-only. Support for the use of this flag can be queried from the
#: device attribute
#: :py:obj:`~.CU_DEVICE_ATTRIBUTE_READ_ONLY_HOST_REGISTER_SUPPORTED`. Using
#: this flag with a current context associated with a device that does not
#: have this attribute set will cause :py:obj:`~.cuMemHostRegister` to
#: error with :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`.
CU_MEMHOSTREGISTER_READ_ONLY = ccuda.CU_MEMHOSTREGISTER_READ_ONLY

#: Indicates that the layered sparse CUDA array or CUDA mipmapped array has
#: a single mip tail region for all layers
CU_ARRAY_SPARSE_PROPERTIES_SINGLE_MIPTAIL = ccuda.CU_ARRAY_SPARSE_PROPERTIES_SINGLE_MIPTAIL

#: Size of tensor map descriptor
CU_TENSOR_MAP_NUM_QWORDS = ccuda.CU_TENSOR_MAP_NUM_QWORDS

#: Indicates that the external memory object is a dedicated resource
CUDA_EXTERNAL_MEMORY_DEDICATED = ccuda.CUDA_EXTERNAL_MEMORY_DEDICATED

#: When the `flags` parameter of
#: :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS` contains this flag, it
#: indicates that signaling an external semaphore object should skip
#: performing appropriate memory synchronization operations over all the
#: external memory objects that are imported as
#: :py:obj:`~.CU_EXTERNAL_MEMORY_HANDLE_TYPE_NVSCIBUF`, which otherwise are
#: performed by default to ensure data coherency with other importers of
#: the same NvSciBuf memory objects.
CUDA_EXTERNAL_SEMAPHORE_SIGNAL_SKIP_NVSCIBUF_MEMSYNC = ccuda.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_SKIP_NVSCIBUF_MEMSYNC

#: When the `flags` parameter of
#: :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS` contains this flag, it
#: indicates that waiting on an external semaphore object should skip
#: performing appropriate memory synchronization operations over all the
#: external memory objects that are imported as
#: :py:obj:`~.CU_EXTERNAL_MEMORY_HANDLE_TYPE_NVSCIBUF`, which otherwise are
#: performed by default to ensure data coherency with other importers of
#: the same NvSciBuf memory objects.
CUDA_EXTERNAL_SEMAPHORE_WAIT_SKIP_NVSCIBUF_MEMSYNC = ccuda.CUDA_EXTERNAL_SEMAPHORE_WAIT_SKIP_NVSCIBUF_MEMSYNC

#: When `flags` of :py:obj:`~.cuDeviceGetNvSciSyncAttributes` is set to
#: this, it indicates that application needs signaler specific
#: NvSciSyncAttr to be filled by
#: :py:obj:`~.cuDeviceGetNvSciSyncAttributes`.
CUDA_NVSCISYNC_ATTR_SIGNAL = ccuda.CUDA_NVSCISYNC_ATTR_SIGNAL

#: When `flags` of :py:obj:`~.cuDeviceGetNvSciSyncAttributes` is set to
#: this, it indicates that application needs waiter specific NvSciSyncAttr
#: to be filled by :py:obj:`~.cuDeviceGetNvSciSyncAttributes`.
CUDA_NVSCISYNC_ATTR_WAIT = ccuda.CUDA_NVSCISYNC_ATTR_WAIT

#: This flag if set indicates that the memory will be used as a tile pool.
CU_MEM_CREATE_USAGE_TILE_POOL = ccuda.CU_MEM_CREATE_USAGE_TILE_POOL

#: If set, each kernel launched as part of
#: :py:obj:`~.cuLaunchCooperativeKernelMultiDevice` only waits for prior
#: work in the stream corresponding to that GPU to complete before the
#: kernel begins execution.
CUDA_COOPERATIVE_LAUNCH_MULTI_DEVICE_NO_PRE_LAUNCH_SYNC = ccuda.CUDA_COOPERATIVE_LAUNCH_MULTI_DEVICE_NO_PRE_LAUNCH_SYNC

#: If set, any subsequent work pushed in a stream that participated in a
#: call to :py:obj:`~.cuLaunchCooperativeKernelMultiDevice` will only wait
#: for the kernel launched on the GPU corresponding to that stream to
#: complete before it begins execution.
CUDA_COOPERATIVE_LAUNCH_MULTI_DEVICE_NO_POST_LAUNCH_SYNC = ccuda.CUDA_COOPERATIVE_LAUNCH_MULTI_DEVICE_NO_POST_LAUNCH_SYNC

#: If set, the CUDA array is a collection of layers, where each layer is
#: either a 1D or a 2D array and the Depth member of
#: CUDA_ARRAY3D_DESCRIPTOR specifies the number of layers, not the depth of
#: a 3D array.
CUDA_ARRAY3D_LAYERED = ccuda.CUDA_ARRAY3D_LAYERED

#: Deprecated, use CUDA_ARRAY3D_LAYERED
CUDA_ARRAY3D_2DARRAY = ccuda.CUDA_ARRAY3D_2DARRAY

#: This flag must be set in order to bind a surface reference to the CUDA
#: array
CUDA_ARRAY3D_SURFACE_LDST = ccuda.CUDA_ARRAY3D_SURFACE_LDST

#: If set, the CUDA array is a collection of six 2D arrays, representing
#: faces of a cube. The width of such a CUDA array must be equal to its
#: height, and Depth must be six. If :py:obj:`~.CUDA_ARRAY3D_LAYERED` flag
#: is also set, then the CUDA array is a collection of cubemaps and Depth
#: must be a multiple of six.
CUDA_ARRAY3D_CUBEMAP = ccuda.CUDA_ARRAY3D_CUBEMAP

#: This flag must be set in order to perform texture gather operations on a
#: CUDA array.
CUDA_ARRAY3D_TEXTURE_GATHER = ccuda.CUDA_ARRAY3D_TEXTURE_GATHER

#: This flag if set indicates that the CUDA array is a DEPTH_TEXTURE.
CUDA_ARRAY3D_DEPTH_TEXTURE = ccuda.CUDA_ARRAY3D_DEPTH_TEXTURE

#: This flag indicates that the CUDA array may be bound as a color target
#: in an external graphics API
CUDA_ARRAY3D_COLOR_ATTACHMENT = ccuda.CUDA_ARRAY3D_COLOR_ATTACHMENT

#: This flag if set indicates that the CUDA array or CUDA mipmapped array
#: is a sparse CUDA array or CUDA mipmapped array respectively
CUDA_ARRAY3D_SPARSE = ccuda.CUDA_ARRAY3D_SPARSE

#: This flag if set indicates that the CUDA array or CUDA mipmapped array
#: will allow deferred memory mapping
CUDA_ARRAY3D_DEFERRED_MAPPING = ccuda.CUDA_ARRAY3D_DEFERRED_MAPPING

#: Override the texref format with a format inferred from the array. Flag
#: for :py:obj:`~.cuTexRefSetArray()`
CU_TRSA_OVERRIDE_FORMAT = ccuda.CU_TRSA_OVERRIDE_FORMAT

#: Read the texture as integers rather than promoting the values to floats
#: in the range [0,1]. Flag for :py:obj:`~.cuTexRefSetFlags()` and
#: :py:obj:`~.cuTexObjectCreate()`
CU_TRSF_READ_AS_INTEGER = ccuda.CU_TRSF_READ_AS_INTEGER

#: Use normalized texture coordinates in the range [0,1) instead of
#: [0,dim). Flag for :py:obj:`~.cuTexRefSetFlags()` and
#: :py:obj:`~.cuTexObjectCreate()`
CU_TRSF_NORMALIZED_COORDINATES = ccuda.CU_TRSF_NORMALIZED_COORDINATES

#: Perform sRGB->linear conversion during texture read. Flag for
#: :py:obj:`~.cuTexRefSetFlags()` and :py:obj:`~.cuTexObjectCreate()`
CU_TRSF_SRGB = ccuda.CU_TRSF_SRGB

#: Disable any trilinear filtering optimizations. Flag for
#: :py:obj:`~.cuTexRefSetFlags()` and :py:obj:`~.cuTexObjectCreate()`
CU_TRSF_DISABLE_TRILINEAR_OPTIMIZATION = ccuda.CU_TRSF_DISABLE_TRILINEAR_OPTIMIZATION

#: Enable seamless cube map filtering. Flag for
#: :py:obj:`~.cuTexObjectCreate()`
CU_TRSF_SEAMLESS_CUBEMAP = ccuda.CU_TRSF_SEAMLESS_CUBEMAP

#: C++ compile time constant for CU_LAUNCH_PARAM_END
CU_LAUNCH_PARAM_END_AS_INT = ccuda.CU_LAUNCH_PARAM_END_AS_INT

#: End of array terminator for the `extra` parameter to
#: :py:obj:`~.cuLaunchKernel`
CU_LAUNCH_PARAM_END = ccuda.CU_LAUNCH_PARAM_END

#: C++ compile time constant for CU_LAUNCH_PARAM_BUFFER_POINTER
CU_LAUNCH_PARAM_BUFFER_POINTER_AS_INT = ccuda.CU_LAUNCH_PARAM_BUFFER_POINTER_AS_INT

#: Indicator that the next value in the `extra` parameter to
#: :py:obj:`~.cuLaunchKernel` will be a pointer to a buffer containing all
#: kernel parameters used for launching kernel `f`. This buffer needs to
#: honor all alignment/padding requirements of the individual parameters.
#: If :py:obj:`~.CU_LAUNCH_PARAM_BUFFER_SIZE` is not also specified in the
#: `extra` array, then :py:obj:`~.CU_LAUNCH_PARAM_BUFFER_POINTER` will have
#: no effect.
CU_LAUNCH_PARAM_BUFFER_POINTER = ccuda.CU_LAUNCH_PARAM_BUFFER_POINTER

#: C++ compile time constant for CU_LAUNCH_PARAM_BUFFER_SIZE
CU_LAUNCH_PARAM_BUFFER_SIZE_AS_INT = ccuda.CU_LAUNCH_PARAM_BUFFER_SIZE_AS_INT

#: Indicator that the next value in the `extra` parameter to
#: :py:obj:`~.cuLaunchKernel` will be a pointer to a size_t which contains
#: the size of the buffer specified with
#: :py:obj:`~.CU_LAUNCH_PARAM_BUFFER_POINTER`. It is required that
#: :py:obj:`~.CU_LAUNCH_PARAM_BUFFER_POINTER` also be specified in the
#: `extra` array if the value associated with
#: :py:obj:`~.CU_LAUNCH_PARAM_BUFFER_SIZE` is not zero.
CU_LAUNCH_PARAM_BUFFER_SIZE = ccuda.CU_LAUNCH_PARAM_BUFFER_SIZE

#: For texture references loaded into the module, use default texunit from
#: texture reference.
CU_PARAM_TR_DEFAULT = ccuda.CU_PARAM_TR_DEFAULT

#: Device that represents the CPU
CU_DEVICE_CPU = ccuda.CU_DEVICE_CPU

#: Device that represents an invalid device
CU_DEVICE_INVALID = ccuda.CU_DEVICE_INVALID

RESOURCE_ABI_VERSION = ccuda.RESOURCE_ABI_VERSION

RESOURCE_ABI_EXTERNAL_BYTES = ccuda.RESOURCE_ABI_EXTERNAL_BYTES

#: Maximum number of planes per frame
MAX_PLANES = ccuda.MAX_PLANES

#: Indicates that timeout for :py:obj:`~.cuEGLStreamConsumerAcquireFrame`
#: is infinite.
CUDA_EGL_INFINITE_TIMEOUT = ccuda.CUDA_EGL_INFINITE_TIMEOUT

class CUipcMem_flags(IntEnum):
    """
    CUDA Ipc Mem Flags
    """

    #: Automatically enable peer access between remote devices as needed
    CU_IPC_MEM_LAZY_ENABLE_PEER_ACCESS = ccuda.CUipcMem_flags_enum.CU_IPC_MEM_LAZY_ENABLE_PEER_ACCESS

class CUmemAttach_flags(IntEnum):
    """
    CUDA Mem Attach Flags
    """

    #: Memory can be accessed by any stream on any device
    CU_MEM_ATTACH_GLOBAL = ccuda.CUmemAttach_flags_enum.CU_MEM_ATTACH_GLOBAL

    #: Memory cannot be accessed by any stream on any device
    CU_MEM_ATTACH_HOST = ccuda.CUmemAttach_flags_enum.CU_MEM_ATTACH_HOST

    #: Memory can only be accessed by a single stream on the associated
    #: device
    CU_MEM_ATTACH_SINGLE = ccuda.CUmemAttach_flags_enum.CU_MEM_ATTACH_SINGLE

class CUctx_flags(IntEnum):
    """
    Context creation flags
    """

    #: Automatic scheduling
    CU_CTX_SCHED_AUTO = ccuda.CUctx_flags_enum.CU_CTX_SCHED_AUTO

    #: Set spin as default scheduling
    CU_CTX_SCHED_SPIN = ccuda.CUctx_flags_enum.CU_CTX_SCHED_SPIN

    #: Set yield as default scheduling
    CU_CTX_SCHED_YIELD = ccuda.CUctx_flags_enum.CU_CTX_SCHED_YIELD

    #: Set blocking synchronization as default scheduling
    CU_CTX_SCHED_BLOCKING_SYNC = ccuda.CUctx_flags_enum.CU_CTX_SCHED_BLOCKING_SYNC

    #: Set blocking synchronization as default scheduling [Deprecated]
    CU_CTX_BLOCKING_SYNC = ccuda.CUctx_flags_enum.CU_CTX_BLOCKING_SYNC
    CU_CTX_SCHED_MASK = ccuda.CUctx_flags_enum.CU_CTX_SCHED_MASK

    #: [Deprecated]
    CU_CTX_MAP_HOST = ccuda.CUctx_flags_enum.CU_CTX_MAP_HOST

    #: Keep local memory allocation after launch
    CU_CTX_LMEM_RESIZE_TO_MAX = ccuda.CUctx_flags_enum.CU_CTX_LMEM_RESIZE_TO_MAX

    #: Trigger coredumps from exceptions in this context
    CU_CTX_COREDUMP_ENABLE = ccuda.CUctx_flags_enum.CU_CTX_COREDUMP_ENABLE

    #: Enable user pipe to trigger coredumps in this context
    CU_CTX_USER_COREDUMP_ENABLE = ccuda.CUctx_flags_enum.CU_CTX_USER_COREDUMP_ENABLE

    #: Ensure synchronous memory operations on this context will
    #: synchronize
    CU_CTX_SYNC_MEMOPS = ccuda.CUctx_flags_enum.CU_CTX_SYNC_MEMOPS
    CU_CTX_FLAGS_MASK = ccuda.CUctx_flags_enum.CU_CTX_FLAGS_MASK

class CUevent_sched_flags(IntEnum):
    """
    Event sched flags
    """

    #: Automatic scheduling
    CU_EVENT_SCHED_AUTO = ccuda.CUevent_sched_flags_enum.CU_EVENT_SCHED_AUTO

    #: Set spin as default scheduling
    CU_EVENT_SCHED_SPIN = ccuda.CUevent_sched_flags_enum.CU_EVENT_SCHED_SPIN

    #: Set yield as default scheduling
    CU_EVENT_SCHED_YIELD = ccuda.CUevent_sched_flags_enum.CU_EVENT_SCHED_YIELD

    #: Set blocking synchronization as default scheduling
    CU_EVENT_SCHED_BLOCKING_SYNC = ccuda.CUevent_sched_flags_enum.CU_EVENT_SCHED_BLOCKING_SYNC

class cl_event_flags(IntEnum):
    """
    NVCL event scheduling flags
    """

    #: Automatic scheduling
    NVCL_EVENT_SCHED_AUTO = ccuda.cl_event_flags_enum.NVCL_EVENT_SCHED_AUTO

    #: Set spin as default scheduling
    NVCL_EVENT_SCHED_SPIN = ccuda.cl_event_flags_enum.NVCL_EVENT_SCHED_SPIN

    #: Set yield as default scheduling
    NVCL_EVENT_SCHED_YIELD = ccuda.cl_event_flags_enum.NVCL_EVENT_SCHED_YIELD

    #: Set blocking synchronization as default scheduling
    NVCL_EVENT_SCHED_BLOCKING_SYNC = ccuda.cl_event_flags_enum.NVCL_EVENT_SCHED_BLOCKING_SYNC

class cl_context_flags(IntEnum):
    """
    NVCL context scheduling flags
    """

    #: Automatic scheduling
    NVCL_CTX_SCHED_AUTO = ccuda.cl_context_flags_enum.NVCL_CTX_SCHED_AUTO

    #: Set spin as default scheduling
    NVCL_CTX_SCHED_SPIN = ccuda.cl_context_flags_enum.NVCL_CTX_SCHED_SPIN

    #: Set yield as default scheduling
    NVCL_CTX_SCHED_YIELD = ccuda.cl_context_flags_enum.NVCL_CTX_SCHED_YIELD

    #: Set blocking synchronization as default scheduling
    NVCL_CTX_SCHED_BLOCKING_SYNC = ccuda.cl_context_flags_enum.NVCL_CTX_SCHED_BLOCKING_SYNC

class CUstream_flags(IntEnum):
    """
    Stream creation flags
    """

    #: Default stream flag
    CU_STREAM_DEFAULT = ccuda.CUstream_flags_enum.CU_STREAM_DEFAULT

    #: Stream does not synchronize with stream 0 (the NULL stream)
    CU_STREAM_NON_BLOCKING = ccuda.CUstream_flags_enum.CU_STREAM_NON_BLOCKING

class CUevent_flags(IntEnum):
    """
    Event creation flags
    """

    #: Default event flag
    CU_EVENT_DEFAULT = ccuda.CUevent_flags_enum.CU_EVENT_DEFAULT

    #: Event uses blocking synchronization
    CU_EVENT_BLOCKING_SYNC = ccuda.CUevent_flags_enum.CU_EVENT_BLOCKING_SYNC

    #: Event will not record timing data
    CU_EVENT_DISABLE_TIMING = ccuda.CUevent_flags_enum.CU_EVENT_DISABLE_TIMING

    #: Event is suitable for interprocess use. CU_EVENT_DISABLE_TIMING must
    #: be set
    CU_EVENT_INTERPROCESS = ccuda.CUevent_flags_enum.CU_EVENT_INTERPROCESS

class CUevent_record_flags(IntEnum):
    """
    Event record flags
    """

    #: Default event record flag
    CU_EVENT_RECORD_DEFAULT = ccuda.CUevent_record_flags_enum.CU_EVENT_RECORD_DEFAULT

    #: When using stream capture, create an event record node instead of
    #: the default behavior. This flag is invalid when used outside of
    #: capture.
    CU_EVENT_RECORD_EXTERNAL = ccuda.CUevent_record_flags_enum.CU_EVENT_RECORD_EXTERNAL

class CUevent_wait_flags(IntEnum):
    """
    Event wait flags
    """

    #: Default event wait flag
    CU_EVENT_WAIT_DEFAULT = ccuda.CUevent_wait_flags_enum.CU_EVENT_WAIT_DEFAULT

    #: When using stream capture, create an event wait node instead of the
    #: default behavior. This flag is invalid when used outside of capture.
    CU_EVENT_WAIT_EXTERNAL = ccuda.CUevent_wait_flags_enum.CU_EVENT_WAIT_EXTERNAL

class CUstreamWaitValue_flags(IntEnum):
    """
    Flags for :py:obj:`~.cuStreamWaitValue32` and
    :py:obj:`~.cuStreamWaitValue64`
    """

    #: Wait until (int32_t)(*addr - value) >= 0 (or int64_t for 64 bit
    #: values). Note this is a cyclic comparison which ignores wraparound.
    #: (Default behavior.)
    CU_STREAM_WAIT_VALUE_GEQ = ccuda.CUstreamWaitValue_flags_enum.CU_STREAM_WAIT_VALUE_GEQ

    #: Wait until *addr == value.
    CU_STREAM_WAIT_VALUE_EQ = ccuda.CUstreamWaitValue_flags_enum.CU_STREAM_WAIT_VALUE_EQ

    #: Wait until (*addr & value) != 0.
    CU_STREAM_WAIT_VALUE_AND = ccuda.CUstreamWaitValue_flags_enum.CU_STREAM_WAIT_VALUE_AND

    #: Wait until ~(*addr | value) != 0. Support for this operation can be
    #: queried with :py:obj:`~.cuDeviceGetAttribute()` and
    #: :py:obj:`~.CU_DEVICE_ATTRIBUTE_CAN_USE_STREAM_WAIT_VALUE_NOR`.
    CU_STREAM_WAIT_VALUE_NOR = ccuda.CUstreamWaitValue_flags_enum.CU_STREAM_WAIT_VALUE_NOR

    #: Follow the wait operation with a flush of outstanding remote writes.
    #: This means that, if a remote write operation is guaranteed to have
    #: reached the device before the wait can be satisfied, that write is
    #: guaranteed to be visible to downstream device work. The device is
    #: permitted to reorder remote writes internally. For example, this
    #: flag would be required if two remote writes arrive in a defined
    #: order, the wait is satisfied by the second write, and downstream
    #: work needs to observe the first write. Support for this operation is
    #: restricted to selected platforms and can be queried with
    #: :py:obj:`~.CU_DEVICE_ATTRIBUTE_CAN_FLUSH_REMOTE_WRITES`.
    CU_STREAM_WAIT_VALUE_FLUSH = ccuda.CUstreamWaitValue_flags_enum.CU_STREAM_WAIT_VALUE_FLUSH

class CUstreamWriteValue_flags(IntEnum):
    """
    Flags for :py:obj:`~.cuStreamWriteValue32`
    """

    #: Default behavior
    CU_STREAM_WRITE_VALUE_DEFAULT = ccuda.CUstreamWriteValue_flags_enum.CU_STREAM_WRITE_VALUE_DEFAULT

    #: Permits the write to be reordered with writes which were issued
    #: before it, as a performance optimization. Normally,
    #: :py:obj:`~.cuStreamWriteValue32` will provide a memory fence before
    #: the write, which has similar semantics to __threadfence_system() but
    #: is scoped to the stream rather than a CUDA thread. This flag is not
    #: supported in the v2 API.
    CU_STREAM_WRITE_VALUE_NO_MEMORY_BARRIER = ccuda.CUstreamWriteValue_flags_enum.CU_STREAM_WRITE_VALUE_NO_MEMORY_BARRIER

class CUstreamBatchMemOpType(IntEnum):
    """
    Operations for :py:obj:`~.cuStreamBatchMemOp`
    """

    #: Represents a :py:obj:`~.cuStreamWaitValue32` operation
    CU_STREAM_MEM_OP_WAIT_VALUE_32 = ccuda.CUstreamBatchMemOpType_enum.CU_STREAM_MEM_OP_WAIT_VALUE_32

    #: Represents a :py:obj:`~.cuStreamWriteValue32` operation
    CU_STREAM_MEM_OP_WRITE_VALUE_32 = ccuda.CUstreamBatchMemOpType_enum.CU_STREAM_MEM_OP_WRITE_VALUE_32

    #: This has the same effect as :py:obj:`~.CU_STREAM_WAIT_VALUE_FLUSH`,
    #: but as a standalone operation.
    CU_STREAM_MEM_OP_FLUSH_REMOTE_WRITES = ccuda.CUstreamBatchMemOpType_enum.CU_STREAM_MEM_OP_FLUSH_REMOTE_WRITES

    #: Represents a :py:obj:`~.cuStreamWaitValue64` operation
    CU_STREAM_MEM_OP_WAIT_VALUE_64 = ccuda.CUstreamBatchMemOpType_enum.CU_STREAM_MEM_OP_WAIT_VALUE_64

    #: Represents a :py:obj:`~.cuStreamWriteValue64` operation
    CU_STREAM_MEM_OP_WRITE_VALUE_64 = ccuda.CUstreamBatchMemOpType_enum.CU_STREAM_MEM_OP_WRITE_VALUE_64

    #: Insert a memory barrier of the specified type
    CU_STREAM_MEM_OP_BARRIER = ccuda.CUstreamBatchMemOpType_enum.CU_STREAM_MEM_OP_BARRIER

class CUstreamMemoryBarrier_flags(IntEnum):
    """
    Flags for :py:obj:`~.cuStreamMemoryBarrier`
    """

    #: System-wide memory barrier.
    CU_STREAM_MEMORY_BARRIER_TYPE_SYS = ccuda.CUstreamMemoryBarrier_flags_enum.CU_STREAM_MEMORY_BARRIER_TYPE_SYS

    #: Limit memory barrier scope to the GPU.
    CU_STREAM_MEMORY_BARRIER_TYPE_GPU = ccuda.CUstreamMemoryBarrier_flags_enum.CU_STREAM_MEMORY_BARRIER_TYPE_GPU

class CUoccupancy_flags(IntEnum):
    """
    Occupancy calculator flag
    """

    #: Default behavior
    CU_OCCUPANCY_DEFAULT = ccuda.CUoccupancy_flags_enum.CU_OCCUPANCY_DEFAULT

    #: Assume global caching is enabled and cannot be automatically turned
    #: off
    CU_OCCUPANCY_DISABLE_CACHING_OVERRIDE = ccuda.CUoccupancy_flags_enum.CU_OCCUPANCY_DISABLE_CACHING_OVERRIDE

class CUstreamUpdateCaptureDependencies_flags(IntEnum):
    """
    Flags for :py:obj:`~.cuStreamUpdateCaptureDependencies`
    """

    #: Add new nodes to the dependency set
    CU_STREAM_ADD_CAPTURE_DEPENDENCIES = ccuda.CUstreamUpdateCaptureDependencies_flags_enum.CU_STREAM_ADD_CAPTURE_DEPENDENCIES

    #: Replace the dependency set with the new nodes
    CU_STREAM_SET_CAPTURE_DEPENDENCIES = ccuda.CUstreamUpdateCaptureDependencies_flags_enum.CU_STREAM_SET_CAPTURE_DEPENDENCIES

class CUasyncNotificationType(IntEnum):
    """
    Types of async notification that can be sent
    """
    CU_ASYNC_NOTIFICATION_TYPE_OVER_BUDGET = ccuda.CUasyncNotificationType_enum.CU_ASYNC_NOTIFICATION_TYPE_OVER_BUDGET

class CUarray_format(IntEnum):
    """
    Array formats
    """

    #: Unsigned 8-bit integers
    CU_AD_FORMAT_UNSIGNED_INT8 = ccuda.CUarray_format_enum.CU_AD_FORMAT_UNSIGNED_INT8

    #: Unsigned 16-bit integers
    CU_AD_FORMAT_UNSIGNED_INT16 = ccuda.CUarray_format_enum.CU_AD_FORMAT_UNSIGNED_INT16

    #: Unsigned 32-bit integers
    CU_AD_FORMAT_UNSIGNED_INT32 = ccuda.CUarray_format_enum.CU_AD_FORMAT_UNSIGNED_INT32

    #: Signed 8-bit integers
    CU_AD_FORMAT_SIGNED_INT8 = ccuda.CUarray_format_enum.CU_AD_FORMAT_SIGNED_INT8

    #: Signed 16-bit integers
    CU_AD_FORMAT_SIGNED_INT16 = ccuda.CUarray_format_enum.CU_AD_FORMAT_SIGNED_INT16

    #: Signed 32-bit integers
    CU_AD_FORMAT_SIGNED_INT32 = ccuda.CUarray_format_enum.CU_AD_FORMAT_SIGNED_INT32

    #: 16-bit floating point
    CU_AD_FORMAT_HALF = ccuda.CUarray_format_enum.CU_AD_FORMAT_HALF

    #: 32-bit floating point
    CU_AD_FORMAT_FLOAT = ccuda.CUarray_format_enum.CU_AD_FORMAT_FLOAT

    #: 4 channel unsigned normalized block-compressed (BC1 compression)
    #: format
    CU_AD_FORMAT_BC1_UNORM = ccuda.CUarray_format_enum.CU_AD_FORMAT_BC1_UNORM

    #: 4 channel unsigned normalized block-compressed (BC1 compression)
    #: format with sRGB encoding
    CU_AD_FORMAT_BC1_UNORM_SRGB = ccuda.CUarray_format_enum.CU_AD_FORMAT_BC1_UNORM_SRGB

    #: 4 channel unsigned normalized block-compressed (BC2 compression)
    #: format
    CU_AD_FORMAT_BC2_UNORM = ccuda.CUarray_format_enum.CU_AD_FORMAT_BC2_UNORM

    #: 4 channel unsigned normalized block-compressed (BC2 compression)
    #: format with sRGB encoding
    CU_AD_FORMAT_BC2_UNORM_SRGB = ccuda.CUarray_format_enum.CU_AD_FORMAT_BC2_UNORM_SRGB

    #: 4 channel unsigned normalized block-compressed (BC3 compression)
    #: format
    CU_AD_FORMAT_BC3_UNORM = ccuda.CUarray_format_enum.CU_AD_FORMAT_BC3_UNORM

    #: 4 channel unsigned normalized block-compressed (BC3 compression)
    #: format with sRGB encoding
    CU_AD_FORMAT_BC3_UNORM_SRGB = ccuda.CUarray_format_enum.CU_AD_FORMAT_BC3_UNORM_SRGB

    #: 1 channel unsigned normalized block-compressed (BC4 compression)
    #: format
    CU_AD_FORMAT_BC4_UNORM = ccuda.CUarray_format_enum.CU_AD_FORMAT_BC4_UNORM

    #: 1 channel signed normalized block-compressed (BC4 compression)
    #: format
    CU_AD_FORMAT_BC4_SNORM = ccuda.CUarray_format_enum.CU_AD_FORMAT_BC4_SNORM

    #: 2 channel unsigned normalized block-compressed (BC5 compression)
    #: format
    CU_AD_FORMAT_BC5_UNORM = ccuda.CUarray_format_enum.CU_AD_FORMAT_BC5_UNORM

    #: 2 channel signed normalized block-compressed (BC5 compression)
    #: format
    CU_AD_FORMAT_BC5_SNORM = ccuda.CUarray_format_enum.CU_AD_FORMAT_BC5_SNORM

    #: 3 channel unsigned half-float block-compressed (BC6H compression)
    #: format
    CU_AD_FORMAT_BC6H_UF16 = ccuda.CUarray_format_enum.CU_AD_FORMAT_BC6H_UF16

    #: 3 channel signed half-float block-compressed (BC6H compression)
    #: format
    CU_AD_FORMAT_BC6H_SF16 = ccuda.CUarray_format_enum.CU_AD_FORMAT_BC6H_SF16

    #: 4 channel unsigned normalized block-compressed (BC7 compression)
    #: format
    CU_AD_FORMAT_BC7_UNORM = ccuda.CUarray_format_enum.CU_AD_FORMAT_BC7_UNORM

    #: 4 channel unsigned normalized block-compressed (BC7 compression)
    #: format with sRGB encoding
    CU_AD_FORMAT_BC7_UNORM_SRGB = ccuda.CUarray_format_enum.CU_AD_FORMAT_BC7_UNORM_SRGB

    #: 8-bit YUV planar format, with 4:2:0 sampling
    CU_AD_FORMAT_NV12 = ccuda.CUarray_format_enum.CU_AD_FORMAT_NV12

    #: 1 channel unsigned 8-bit normalized integer
    CU_AD_FORMAT_UNORM_INT8X1 = ccuda.CUarray_format_enum.CU_AD_FORMAT_UNORM_INT8X1

    #: 2 channel unsigned 8-bit normalized integer
    CU_AD_FORMAT_UNORM_INT8X2 = ccuda.CUarray_format_enum.CU_AD_FORMAT_UNORM_INT8X2

    #: 4 channel unsigned 8-bit normalized integer
    CU_AD_FORMAT_UNORM_INT8X4 = ccuda.CUarray_format_enum.CU_AD_FORMAT_UNORM_INT8X4

    #: 1 channel unsigned 16-bit normalized integer
    CU_AD_FORMAT_UNORM_INT16X1 = ccuda.CUarray_format_enum.CU_AD_FORMAT_UNORM_INT16X1

    #: 2 channel unsigned 16-bit normalized integer
    CU_AD_FORMAT_UNORM_INT16X2 = ccuda.CUarray_format_enum.CU_AD_FORMAT_UNORM_INT16X2

    #: 4 channel unsigned 16-bit normalized integer
    CU_AD_FORMAT_UNORM_INT16X4 = ccuda.CUarray_format_enum.CU_AD_FORMAT_UNORM_INT16X4

    #: 1 channel signed 8-bit normalized integer
    CU_AD_FORMAT_SNORM_INT8X1 = ccuda.CUarray_format_enum.CU_AD_FORMAT_SNORM_INT8X1

    #: 2 channel signed 8-bit normalized integer
    CU_AD_FORMAT_SNORM_INT8X2 = ccuda.CUarray_format_enum.CU_AD_FORMAT_SNORM_INT8X2

    #: 4 channel signed 8-bit normalized integer
    CU_AD_FORMAT_SNORM_INT8X4 = ccuda.CUarray_format_enum.CU_AD_FORMAT_SNORM_INT8X4

    #: 1 channel signed 16-bit normalized integer
    CU_AD_FORMAT_SNORM_INT16X1 = ccuda.CUarray_format_enum.CU_AD_FORMAT_SNORM_INT16X1

    #: 2 channel signed 16-bit normalized integer
    CU_AD_FORMAT_SNORM_INT16X2 = ccuda.CUarray_format_enum.CU_AD_FORMAT_SNORM_INT16X2

    #: 4 channel signed 16-bit normalized integer
    CU_AD_FORMAT_SNORM_INT16X4 = ccuda.CUarray_format_enum.CU_AD_FORMAT_SNORM_INT16X4

class CUaddress_mode(IntEnum):
    """
    Texture reference addressing modes
    """

    #: Wrapping address mode
    CU_TR_ADDRESS_MODE_WRAP = ccuda.CUaddress_mode_enum.CU_TR_ADDRESS_MODE_WRAP

    #: Clamp to edge address mode
    CU_TR_ADDRESS_MODE_CLAMP = ccuda.CUaddress_mode_enum.CU_TR_ADDRESS_MODE_CLAMP

    #: Mirror address mode
    CU_TR_ADDRESS_MODE_MIRROR = ccuda.CUaddress_mode_enum.CU_TR_ADDRESS_MODE_MIRROR

    #: Border address mode
    CU_TR_ADDRESS_MODE_BORDER = ccuda.CUaddress_mode_enum.CU_TR_ADDRESS_MODE_BORDER

class CUfilter_mode(IntEnum):
    """
    Texture reference filtering modes
    """

    #: Point filter mode
    CU_TR_FILTER_MODE_POINT = ccuda.CUfilter_mode_enum.CU_TR_FILTER_MODE_POINT

    #: Linear filter mode
    CU_TR_FILTER_MODE_LINEAR = ccuda.CUfilter_mode_enum.CU_TR_FILTER_MODE_LINEAR

class CUdevice_attribute(IntEnum):
    """
    Device properties
    """

    #: Maximum number of threads per block
    CU_DEVICE_ATTRIBUTE_MAX_THREADS_PER_BLOCK = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_THREADS_PER_BLOCK

    #: Maximum block dimension X
    CU_DEVICE_ATTRIBUTE_MAX_BLOCK_DIM_X = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_BLOCK_DIM_X

    #: Maximum block dimension Y
    CU_DEVICE_ATTRIBUTE_MAX_BLOCK_DIM_Y = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_BLOCK_DIM_Y

    #: Maximum block dimension Z
    CU_DEVICE_ATTRIBUTE_MAX_BLOCK_DIM_Z = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_BLOCK_DIM_Z

    #: Maximum grid dimension X
    CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_X = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_X

    #: Maximum grid dimension Y
    CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_Y = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_Y

    #: Maximum grid dimension Z
    CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_Z = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_Z

    #: Maximum shared memory available per block in bytes
    CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_BLOCK = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_BLOCK

    #: Deprecated, use CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_BLOCK
    CU_DEVICE_ATTRIBUTE_SHARED_MEMORY_PER_BLOCK = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_SHARED_MEMORY_PER_BLOCK

    #: Memory available on device for constant variables in a CUDA C kernel
    #: in bytes
    CU_DEVICE_ATTRIBUTE_TOTAL_CONSTANT_MEMORY = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_TOTAL_CONSTANT_MEMORY

    #: Warp size in threads
    CU_DEVICE_ATTRIBUTE_WARP_SIZE = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_WARP_SIZE

    #: Maximum pitch in bytes allowed by memory copies
    CU_DEVICE_ATTRIBUTE_MAX_PITCH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_PITCH

    #: Maximum number of 32-bit registers available per block
    CU_DEVICE_ATTRIBUTE_MAX_REGISTERS_PER_BLOCK = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_REGISTERS_PER_BLOCK

    #: Deprecated, use CU_DEVICE_ATTRIBUTE_MAX_REGISTERS_PER_BLOCK
    CU_DEVICE_ATTRIBUTE_REGISTERS_PER_BLOCK = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_REGISTERS_PER_BLOCK

    #: Typical clock frequency in kilohertz
    CU_DEVICE_ATTRIBUTE_CLOCK_RATE = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CLOCK_RATE

    #: Alignment requirement for textures
    CU_DEVICE_ATTRIBUTE_TEXTURE_ALIGNMENT = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_TEXTURE_ALIGNMENT

    #: Device can possibly copy memory and execute a kernel concurrently.
    #: Deprecated. Use instead CU_DEVICE_ATTRIBUTE_ASYNC_ENGINE_COUNT.
    CU_DEVICE_ATTRIBUTE_GPU_OVERLAP = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_GPU_OVERLAP

    #: Number of multiprocessors on device
    CU_DEVICE_ATTRIBUTE_MULTIPROCESSOR_COUNT = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MULTIPROCESSOR_COUNT

    #: Specifies whether there is a run time limit on kernels
    CU_DEVICE_ATTRIBUTE_KERNEL_EXEC_TIMEOUT = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_KERNEL_EXEC_TIMEOUT

    #: Device is integrated with host memory
    CU_DEVICE_ATTRIBUTE_INTEGRATED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_INTEGRATED

    #: Device can map host memory into CUDA address space
    CU_DEVICE_ATTRIBUTE_CAN_MAP_HOST_MEMORY = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CAN_MAP_HOST_MEMORY

    #: Compute mode (See :py:obj:`~.CUcomputemode` for details)
    CU_DEVICE_ATTRIBUTE_COMPUTE_MODE = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_COMPUTE_MODE

    #: Maximum 1D texture width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_WIDTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_WIDTH

    #: Maximum 2D texture width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_WIDTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_WIDTH

    #: Maximum 2D texture height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_HEIGHT = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_HEIGHT

    #: Maximum 3D texture width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_WIDTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_WIDTH

    #: Maximum 3D texture height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_HEIGHT = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_HEIGHT

    #: Maximum 3D texture depth
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_DEPTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_DEPTH

    #: Maximum 2D layered texture width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_WIDTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_WIDTH

    #: Deprecated, use CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_WIDTH
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_ARRAY_WIDTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_ARRAY_WIDTH

    #: Maximum 2D layered texture height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_HEIGHT = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_HEIGHT

    #: Deprecated, use CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_HEIGHT
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_ARRAY_HEIGHT = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_ARRAY_HEIGHT

    #: Maximum layers in a 2D layered texture
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_LAYERS = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_LAYERS

    #: Deprecated, use CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_LAYERS
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_ARRAY_NUMSLICES = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_ARRAY_NUMSLICES

    #: Alignment requirement for surfaces
    CU_DEVICE_ATTRIBUTE_SURFACE_ALIGNMENT = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_SURFACE_ALIGNMENT

    #: Device can possibly execute multiple kernels concurrently
    CU_DEVICE_ATTRIBUTE_CONCURRENT_KERNELS = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CONCURRENT_KERNELS

    #: Device has ECC support enabled
    CU_DEVICE_ATTRIBUTE_ECC_ENABLED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_ECC_ENABLED

    #: PCI bus ID of the device
    CU_DEVICE_ATTRIBUTE_PCI_BUS_ID = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_PCI_BUS_ID

    #: PCI device ID of the device
    CU_DEVICE_ATTRIBUTE_PCI_DEVICE_ID = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_PCI_DEVICE_ID

    #: Device is using TCC driver model
    CU_DEVICE_ATTRIBUTE_TCC_DRIVER = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_TCC_DRIVER

    #: Peak memory clock frequency in kilohertz
    CU_DEVICE_ATTRIBUTE_MEMORY_CLOCK_RATE = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MEMORY_CLOCK_RATE

    #: Global memory bus width in bits
    CU_DEVICE_ATTRIBUTE_GLOBAL_MEMORY_BUS_WIDTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_GLOBAL_MEMORY_BUS_WIDTH

    #: Size of L2 cache in bytes
    CU_DEVICE_ATTRIBUTE_L2_CACHE_SIZE = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_L2_CACHE_SIZE

    #: Maximum resident threads per multiprocessor
    CU_DEVICE_ATTRIBUTE_MAX_THREADS_PER_MULTIPROCESSOR = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_THREADS_PER_MULTIPROCESSOR

    #: Number of asynchronous engines
    CU_DEVICE_ATTRIBUTE_ASYNC_ENGINE_COUNT = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_ASYNC_ENGINE_COUNT

    #: Device shares a unified address space with the host
    CU_DEVICE_ATTRIBUTE_UNIFIED_ADDRESSING = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_UNIFIED_ADDRESSING

    #: Maximum 1D layered texture width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_LAYERED_WIDTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_LAYERED_WIDTH

    #: Maximum layers in a 1D layered texture
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_LAYERED_LAYERS = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_LAYERED_LAYERS

    #: Deprecated, do not use.
    CU_DEVICE_ATTRIBUTE_CAN_TEX2D_GATHER = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CAN_TEX2D_GATHER

    #: Maximum 2D texture width if CUDA_ARRAY3D_TEXTURE_GATHER is set
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_GATHER_WIDTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_GATHER_WIDTH

    #: Maximum 2D texture height if CUDA_ARRAY3D_TEXTURE_GATHER is set
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_GATHER_HEIGHT = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_GATHER_HEIGHT

    #: Alternate maximum 3D texture width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_WIDTH_ALTERNATE = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_WIDTH_ALTERNATE

    #: Alternate maximum 3D texture height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_HEIGHT_ALTERNATE = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_HEIGHT_ALTERNATE

    #: Alternate maximum 3D texture depth
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_DEPTH_ALTERNATE = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_DEPTH_ALTERNATE

    #: PCI domain ID of the device
    CU_DEVICE_ATTRIBUTE_PCI_DOMAIN_ID = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_PCI_DOMAIN_ID

    #: Pitch alignment requirement for textures
    CU_DEVICE_ATTRIBUTE_TEXTURE_PITCH_ALIGNMENT = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_TEXTURE_PITCH_ALIGNMENT

    #: Maximum cubemap texture width/height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURECUBEMAP_WIDTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURECUBEMAP_WIDTH

    #: Maximum cubemap layered texture width/height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURECUBEMAP_LAYERED_WIDTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURECUBEMAP_LAYERED_WIDTH

    #: Maximum layers in a cubemap layered texture
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURECUBEMAP_LAYERED_LAYERS = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURECUBEMAP_LAYERED_LAYERS

    #: Maximum 1D surface width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE1D_WIDTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE1D_WIDTH

    #: Maximum 2D surface width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_WIDTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_WIDTH

    #: Maximum 2D surface height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_HEIGHT = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_HEIGHT

    #: Maximum 3D surface width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE3D_WIDTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE3D_WIDTH

    #: Maximum 3D surface height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE3D_HEIGHT = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE3D_HEIGHT

    #: Maximum 3D surface depth
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE3D_DEPTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE3D_DEPTH

    #: Maximum 1D layered surface width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE1D_LAYERED_WIDTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE1D_LAYERED_WIDTH

    #: Maximum layers in a 1D layered surface
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE1D_LAYERED_LAYERS = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE1D_LAYERED_LAYERS

    #: Maximum 2D layered surface width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_LAYERED_WIDTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_LAYERED_WIDTH

    #: Maximum 2D layered surface height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_LAYERED_HEIGHT = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_LAYERED_HEIGHT

    #: Maximum layers in a 2D layered surface
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_LAYERED_LAYERS = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_LAYERED_LAYERS

    #: Maximum cubemap surface width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACECUBEMAP_WIDTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACECUBEMAP_WIDTH

    #: Maximum cubemap layered surface width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACECUBEMAP_LAYERED_WIDTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACECUBEMAP_LAYERED_WIDTH

    #: Maximum layers in a cubemap layered surface
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACECUBEMAP_LAYERED_LAYERS = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACECUBEMAP_LAYERED_LAYERS

    #: Deprecated, do not use. Use cudaDeviceGetTexture1DLinearMaxWidth()
    #: or :py:obj:`~.cuDeviceGetTexture1DLinearMaxWidth()` instead.
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_LINEAR_WIDTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_LINEAR_WIDTH

    #: Maximum 2D linear texture width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LINEAR_WIDTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LINEAR_WIDTH

    #: Maximum 2D linear texture height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LINEAR_HEIGHT = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LINEAR_HEIGHT

    #: Maximum 2D linear texture pitch in bytes
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LINEAR_PITCH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LINEAR_PITCH

    #: Maximum mipmapped 2D texture width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_MIPMAPPED_WIDTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_MIPMAPPED_WIDTH

    #: Maximum mipmapped 2D texture height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_MIPMAPPED_HEIGHT = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_MIPMAPPED_HEIGHT

    #: Major compute capability version number
    CU_DEVICE_ATTRIBUTE_COMPUTE_CAPABILITY_MAJOR = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_COMPUTE_CAPABILITY_MAJOR

    #: Minor compute capability version number
    CU_DEVICE_ATTRIBUTE_COMPUTE_CAPABILITY_MINOR = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_COMPUTE_CAPABILITY_MINOR

    #: Maximum mipmapped 1D texture width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_MIPMAPPED_WIDTH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_MIPMAPPED_WIDTH

    #: Device supports stream priorities
    CU_DEVICE_ATTRIBUTE_STREAM_PRIORITIES_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_STREAM_PRIORITIES_SUPPORTED

    #: Device supports caching globals in L1
    CU_DEVICE_ATTRIBUTE_GLOBAL_L1_CACHE_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_GLOBAL_L1_CACHE_SUPPORTED

    #: Device supports caching locals in L1
    CU_DEVICE_ATTRIBUTE_LOCAL_L1_CACHE_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_LOCAL_L1_CACHE_SUPPORTED

    #: Maximum shared memory available per multiprocessor in bytes
    CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_MULTIPROCESSOR = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_MULTIPROCESSOR

    #: Maximum number of 32-bit registers available per multiprocessor
    CU_DEVICE_ATTRIBUTE_MAX_REGISTERS_PER_MULTIPROCESSOR = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_REGISTERS_PER_MULTIPROCESSOR

    #: Device can allocate managed memory on this system
    CU_DEVICE_ATTRIBUTE_MANAGED_MEMORY = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MANAGED_MEMORY

    #: Device is on a multi-GPU board
    CU_DEVICE_ATTRIBUTE_MULTI_GPU_BOARD = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MULTI_GPU_BOARD

    #: Unique id for a group of devices on the same multi-GPU board
    CU_DEVICE_ATTRIBUTE_MULTI_GPU_BOARD_GROUP_ID = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MULTI_GPU_BOARD_GROUP_ID

    #: Link between the device and the host supports native atomic
    #: operations (this is a placeholder attribute, and is not supported on
    #: any current hardware)
    CU_DEVICE_ATTRIBUTE_HOST_NATIVE_ATOMIC_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_HOST_NATIVE_ATOMIC_SUPPORTED

    #: Ratio of single precision performance (in floating-point operations
    #: per second) to double precision performance
    CU_DEVICE_ATTRIBUTE_SINGLE_TO_DOUBLE_PRECISION_PERF_RATIO = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_SINGLE_TO_DOUBLE_PRECISION_PERF_RATIO

    #: Device supports coherently accessing pageable memory without calling
    #: cudaHostRegister on it
    CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS

    #: Device can coherently access managed memory concurrently with the
    #: CPU
    CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS

    #: Device supports compute preemption.
    CU_DEVICE_ATTRIBUTE_COMPUTE_PREEMPTION_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_COMPUTE_PREEMPTION_SUPPORTED

    #: Device can access host registered memory at the same virtual address
    #: as the CPU
    CU_DEVICE_ATTRIBUTE_CAN_USE_HOST_POINTER_FOR_REGISTERED_MEM = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CAN_USE_HOST_POINTER_FOR_REGISTERED_MEM

    #: Deprecated, along with v1 MemOps API, :py:obj:`~.cuStreamBatchMemOp`
    #: and related APIs are supported.
    CU_DEVICE_ATTRIBUTE_CAN_USE_STREAM_MEM_OPS_V1 = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CAN_USE_STREAM_MEM_OPS_V1

    #: Deprecated, along with v1 MemOps API, 64-bit operations are
    #: supported in :py:obj:`~.cuStreamBatchMemOp` and related APIs.
    CU_DEVICE_ATTRIBUTE_CAN_USE_64_BIT_STREAM_MEM_OPS_V1 = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CAN_USE_64_BIT_STREAM_MEM_OPS_V1

    #: Deprecated, along with v1 MemOps API,
    #: :py:obj:`~.CU_STREAM_WAIT_VALUE_NOR` is supported.
    CU_DEVICE_ATTRIBUTE_CAN_USE_STREAM_WAIT_VALUE_NOR_V1 = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CAN_USE_STREAM_WAIT_VALUE_NOR_V1

    #: Device supports launching cooperative kernels via
    #: :py:obj:`~.cuLaunchCooperativeKernel`
    CU_DEVICE_ATTRIBUTE_COOPERATIVE_LAUNCH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_COOPERATIVE_LAUNCH

    #: Deprecated, :py:obj:`~.cuLaunchCooperativeKernelMultiDevice` is
    #: deprecated.
    CU_DEVICE_ATTRIBUTE_COOPERATIVE_MULTI_DEVICE_LAUNCH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_COOPERATIVE_MULTI_DEVICE_LAUNCH

    #: Maximum optin shared memory per block
    CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_BLOCK_OPTIN = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_BLOCK_OPTIN

    #: The :py:obj:`~.CU_STREAM_WAIT_VALUE_FLUSH` flag and the
    #: :py:obj:`~.CU_STREAM_MEM_OP_FLUSH_REMOTE_WRITES` MemOp are supported
    #: on the device. See :py:obj:`~.Stream Memory Operations` for
    #: additional details.
    CU_DEVICE_ATTRIBUTE_CAN_FLUSH_REMOTE_WRITES = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CAN_FLUSH_REMOTE_WRITES

    #: Device supports host memory registration via
    #: :py:obj:`~.cudaHostRegister`.
    CU_DEVICE_ATTRIBUTE_HOST_REGISTER_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_HOST_REGISTER_SUPPORTED

    #: Device accesses pageable memory via the host's page tables.
    CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS_USES_HOST_PAGE_TABLES = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS_USES_HOST_PAGE_TABLES

    #: The host can directly access managed memory on the device without
    #: migration.
    CU_DEVICE_ATTRIBUTE_DIRECT_MANAGED_MEM_ACCESS_FROM_HOST = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_DIRECT_MANAGED_MEM_ACCESS_FROM_HOST

    #: Deprecated, Use
    #: CU_DEVICE_ATTRIBUTE_VIRTUAL_MEMORY_MANAGEMENT_SUPPORTED
    CU_DEVICE_ATTRIBUTE_VIRTUAL_ADDRESS_MANAGEMENT_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_VIRTUAL_ADDRESS_MANAGEMENT_SUPPORTED

    #: Device supports virtual memory management APIs like
    #: :py:obj:`~.cuMemAddressReserve`, :py:obj:`~.cuMemCreate`,
    #: :py:obj:`~.cuMemMap` and related APIs
    CU_DEVICE_ATTRIBUTE_VIRTUAL_MEMORY_MANAGEMENT_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_VIRTUAL_MEMORY_MANAGEMENT_SUPPORTED

    #: Device supports exporting memory to a posix file descriptor with
    #: :py:obj:`~.cuMemExportToShareableHandle`, if requested via
    #: :py:obj:`~.cuMemCreate`
    CU_DEVICE_ATTRIBUTE_HANDLE_TYPE_POSIX_FILE_DESCRIPTOR_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_HANDLE_TYPE_POSIX_FILE_DESCRIPTOR_SUPPORTED

    #: Device supports exporting memory to a Win32 NT handle with
    #: :py:obj:`~.cuMemExportToShareableHandle`, if requested via
    #: :py:obj:`~.cuMemCreate`
    CU_DEVICE_ATTRIBUTE_HANDLE_TYPE_WIN32_HANDLE_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_HANDLE_TYPE_WIN32_HANDLE_SUPPORTED

    #: Device supports exporting memory to a Win32 KMT handle with
    #: :py:obj:`~.cuMemExportToShareableHandle`, if requested via
    #: :py:obj:`~.cuMemCreate`
    CU_DEVICE_ATTRIBUTE_HANDLE_TYPE_WIN32_KMT_HANDLE_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_HANDLE_TYPE_WIN32_KMT_HANDLE_SUPPORTED

    #: Maximum number of blocks per multiprocessor
    CU_DEVICE_ATTRIBUTE_MAX_BLOCKS_PER_MULTIPROCESSOR = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_BLOCKS_PER_MULTIPROCESSOR

    #: Device supports compression of memory
    CU_DEVICE_ATTRIBUTE_GENERIC_COMPRESSION_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_GENERIC_COMPRESSION_SUPPORTED

    #: Maximum L2 persisting lines capacity setting in bytes.
    CU_DEVICE_ATTRIBUTE_MAX_PERSISTING_L2_CACHE_SIZE = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_PERSISTING_L2_CACHE_SIZE

    #: Maximum value of :py:obj:`~.CUaccessPolicyWindow.num_bytes`.
    CU_DEVICE_ATTRIBUTE_MAX_ACCESS_POLICY_WINDOW_SIZE = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_ACCESS_POLICY_WINDOW_SIZE

    #: Device supports specifying the GPUDirect RDMA flag with
    #: :py:obj:`~.cuMemCreate`
    CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_WITH_CUDA_VMM_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_WITH_CUDA_VMM_SUPPORTED

    #: Shared memory reserved by CUDA driver per block in bytes
    CU_DEVICE_ATTRIBUTE_RESERVED_SHARED_MEMORY_PER_BLOCK = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_RESERVED_SHARED_MEMORY_PER_BLOCK

    #: Device supports sparse CUDA arrays and sparse CUDA mipmapped arrays
    CU_DEVICE_ATTRIBUTE_SPARSE_CUDA_ARRAY_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_SPARSE_CUDA_ARRAY_SUPPORTED

    #: Device supports using the :py:obj:`~.cuMemHostRegister` flag
    #: :py:obj:`~.CU_MEMHOSTERGISTER_READ_ONLY` to register memory that
    #: must be mapped as read-only to the GPU
    CU_DEVICE_ATTRIBUTE_READ_ONLY_HOST_REGISTER_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_READ_ONLY_HOST_REGISTER_SUPPORTED

    #: External timeline semaphore interop is supported on the device
    CU_DEVICE_ATTRIBUTE_TIMELINE_SEMAPHORE_INTEROP_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_TIMELINE_SEMAPHORE_INTEROP_SUPPORTED

    #: Device supports using the :py:obj:`~.cuMemAllocAsync` and
    #: :py:obj:`~.cuMemPool` family of APIs
    CU_DEVICE_ATTRIBUTE_MEMORY_POOLS_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MEMORY_POOLS_SUPPORTED

    #: Device supports GPUDirect RDMA APIs, like nvidia_p2p_get_pages (see
    #: https://docs.nvidia.com/cuda/gpudirect-rdma for more information)
    CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_SUPPORTED

    #: The returned attribute shall be interpreted as a bitmask, where the
    #: individual bits are described by the
    #: :py:obj:`~.CUflushGPUDirectRDMAWritesOptions` enum
    CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_FLUSH_WRITES_OPTIONS = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_FLUSH_WRITES_OPTIONS

    #: GPUDirect RDMA writes to the device do not need to be flushed for
    #: consumers within the scope indicated by the returned attribute. See
    #: :py:obj:`~.CUGPUDirectRDMAWritesOrdering` for the numerical values
    #: returned here.
    CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_WRITES_ORDERING = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_WRITES_ORDERING

    #: Handle types supported with mempool based IPC
    CU_DEVICE_ATTRIBUTE_MEMPOOL_SUPPORTED_HANDLE_TYPES = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MEMPOOL_SUPPORTED_HANDLE_TYPES

    #: Indicates device supports cluster launch
    CU_DEVICE_ATTRIBUTE_CLUSTER_LAUNCH = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CLUSTER_LAUNCH

    #: Device supports deferred mapping CUDA arrays and CUDA mipmapped
    #: arrays
    CU_DEVICE_ATTRIBUTE_DEFERRED_MAPPING_CUDA_ARRAY_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_DEFERRED_MAPPING_CUDA_ARRAY_SUPPORTED

    #: 64-bit operations are supported in :py:obj:`~.cuStreamBatchMemOp`
    #: and related MemOp APIs.
    CU_DEVICE_ATTRIBUTE_CAN_USE_64_BIT_STREAM_MEM_OPS = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CAN_USE_64_BIT_STREAM_MEM_OPS

    #: :py:obj:`~.CU_STREAM_WAIT_VALUE_NOR` is supported by MemOp APIs.
    CU_DEVICE_ATTRIBUTE_CAN_USE_STREAM_WAIT_VALUE_NOR = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CAN_USE_STREAM_WAIT_VALUE_NOR

    #: Device supports buffer sharing with dma_buf mechanism.
    CU_DEVICE_ATTRIBUTE_DMA_BUF_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_DMA_BUF_SUPPORTED

    #: Device supports IPC Events.
    CU_DEVICE_ATTRIBUTE_IPC_EVENT_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_IPC_EVENT_SUPPORTED

    #: Number of memory domains the device supports.
    CU_DEVICE_ATTRIBUTE_MEM_SYNC_DOMAIN_COUNT = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MEM_SYNC_DOMAIN_COUNT

    #: Device supports accessing memory using Tensor Map.
    CU_DEVICE_ATTRIBUTE_TENSOR_MAP_ACCESS_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_TENSOR_MAP_ACCESS_SUPPORTED

    #: Device supports exporting memory to a fabric handle with
    #: :py:obj:`~.cuMemExportToShareableHandle()` or requested with
    #: :py:obj:`~.cuMemCreate()`
    CU_DEVICE_ATTRIBUTE_HANDLE_TYPE_FABRIC_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_HANDLE_TYPE_FABRIC_SUPPORTED

    #: Device supports unified function pointers.
    CU_DEVICE_ATTRIBUTE_UNIFIED_FUNCTION_POINTERS = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_UNIFIED_FUNCTION_POINTERS
    CU_DEVICE_ATTRIBUTE_NUMA_CONFIG = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_NUMA_CONFIG
    CU_DEVICE_ATTRIBUTE_NUMA_ID = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_NUMA_ID

    #: Device supports switch multicast and reduction operations.
    CU_DEVICE_ATTRIBUTE_MULTICAST_SUPPORTED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MULTICAST_SUPPORTED

    #: Indicates if contexts created on this device will be shared via MPS
    CU_DEVICE_ATTRIBUTE_MPS_ENABLED = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MPS_ENABLED

    #: NUMA ID of the host node closest to the device. Returns -1 when
    #: system does not support NUMA.
    CU_DEVICE_ATTRIBUTE_HOST_NUMA_ID = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_HOST_NUMA_ID
    CU_DEVICE_ATTRIBUTE_MAX = ccuda.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX

class CUpointer_attribute(IntEnum):
    """
    Pointer information
    """

    #: The :py:obj:`~.CUcontext` on which a pointer was allocated or
    #: registered
    CU_POINTER_ATTRIBUTE_CONTEXT = ccuda.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_CONTEXT

    #: The :py:obj:`~.CUmemorytype` describing the physical location of a
    #: pointer
    CU_POINTER_ATTRIBUTE_MEMORY_TYPE = ccuda.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_MEMORY_TYPE

    #: The address at which a pointer's memory may be accessed on the
    #: device
    CU_POINTER_ATTRIBUTE_DEVICE_POINTER = ccuda.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_DEVICE_POINTER

    #: The address at which a pointer's memory may be accessed on the host
    CU_POINTER_ATTRIBUTE_HOST_POINTER = ccuda.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_HOST_POINTER

    #: A pair of tokens for use with the nv-p2p.h Linux kernel interface
    CU_POINTER_ATTRIBUTE_P2P_TOKENS = ccuda.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_P2P_TOKENS

    #: Synchronize every synchronous memory operation initiated on this
    #: region
    CU_POINTER_ATTRIBUTE_SYNC_MEMOPS = ccuda.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_SYNC_MEMOPS

    #: A process-wide unique ID for an allocated memory region
    CU_POINTER_ATTRIBUTE_BUFFER_ID = ccuda.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_BUFFER_ID

    #: Indicates if the pointer points to managed memory
    CU_POINTER_ATTRIBUTE_IS_MANAGED = ccuda.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_IS_MANAGED

    #: A device ordinal of a device on which a pointer was allocated or
    #: registered
    CU_POINTER_ATTRIBUTE_DEVICE_ORDINAL = ccuda.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_DEVICE_ORDINAL

    #: 1 if this pointer maps to an allocation that is suitable for
    #: :py:obj:`~.cudaIpcGetMemHandle`, 0 otherwise
    CU_POINTER_ATTRIBUTE_IS_LEGACY_CUDA_IPC_CAPABLE = ccuda.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_IS_LEGACY_CUDA_IPC_CAPABLE

    #: Starting address for this requested pointer
    CU_POINTER_ATTRIBUTE_RANGE_START_ADDR = ccuda.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_RANGE_START_ADDR

    #: Size of the address range for this requested pointer
    CU_POINTER_ATTRIBUTE_RANGE_SIZE = ccuda.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_RANGE_SIZE

    #: 1 if this pointer is in a valid address range that is mapped to a
    #: backing allocation, 0 otherwise
    CU_POINTER_ATTRIBUTE_MAPPED = ccuda.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_MAPPED

    #: Bitmask of allowed :py:obj:`~.CUmemAllocationHandleType` for this
    #: allocation
    CU_POINTER_ATTRIBUTE_ALLOWED_HANDLE_TYPES = ccuda.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_ALLOWED_HANDLE_TYPES

    #: 1 if the memory this pointer is referencing can be used with the
    #: GPUDirect RDMA API
    CU_POINTER_ATTRIBUTE_IS_GPU_DIRECT_RDMA_CAPABLE = ccuda.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_IS_GPU_DIRECT_RDMA_CAPABLE

    #: Returns the access flags the device associated with the current
    #: context has on the corresponding memory referenced by the pointer
    #: given
    CU_POINTER_ATTRIBUTE_ACCESS_FLAGS = ccuda.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_ACCESS_FLAGS

    #: Returns the mempool handle for the allocation if it was allocated
    #: from a mempool. Otherwise returns NULL.
    CU_POINTER_ATTRIBUTE_MEMPOOL_HANDLE = ccuda.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_MEMPOOL_HANDLE

    #: Size of the actual underlying mapping that the pointer belongs to
    CU_POINTER_ATTRIBUTE_MAPPING_SIZE = ccuda.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_MAPPING_SIZE

    #: The start address of the mapping that the pointer belongs to
    CU_POINTER_ATTRIBUTE_MAPPING_BASE_ADDR = ccuda.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_MAPPING_BASE_ADDR

    #: A process-wide unique id corresponding to the physical allocation
    #: the pointer belongs to
    CU_POINTER_ATTRIBUTE_MEMORY_BLOCK_ID = ccuda.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_MEMORY_BLOCK_ID

class CUfunction_attribute(IntEnum):
    """
    Function properties
    """

    #: The maximum number of threads per block, beyond which a launch of
    #: the function would fail. This number depends on both the function
    #: and the device on which the function is currently loaded.
    CU_FUNC_ATTRIBUTE_MAX_THREADS_PER_BLOCK = ccuda.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_MAX_THREADS_PER_BLOCK

    #: The size in bytes of statically-allocated shared memory required by
    #: this function. This does not include dynamically-allocated shared
    #: memory requested by the user at runtime.
    CU_FUNC_ATTRIBUTE_SHARED_SIZE_BYTES = ccuda.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_SHARED_SIZE_BYTES

    #: The size in bytes of user-allocated constant memory required by this
    #: function.
    CU_FUNC_ATTRIBUTE_CONST_SIZE_BYTES = ccuda.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_CONST_SIZE_BYTES

    #: The size in bytes of local memory used by each thread of this
    #: function.
    CU_FUNC_ATTRIBUTE_LOCAL_SIZE_BYTES = ccuda.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_LOCAL_SIZE_BYTES

    #: The number of registers used by each thread of this function.
    CU_FUNC_ATTRIBUTE_NUM_REGS = ccuda.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_NUM_REGS

    #: The PTX virtual architecture version for which the function was
    #: compiled. This value is the major PTX version * 10 + the minor PTX
    #: version, so a PTX version 1.3 function would return the value 13.
    #: Note that this may return the undefined value of 0 for cubins
    #: compiled prior to CUDA 3.0.
    CU_FUNC_ATTRIBUTE_PTX_VERSION = ccuda.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_PTX_VERSION

    #: The binary architecture version for which the function was compiled.
    #: This value is the major binary version * 10 + the minor binary
    #: version, so a binary version 1.3 function would return the value 13.
    #: Note that this will return a value of 10 for legacy cubins that do
    #: not have a properly-encoded binary architecture version.
    CU_FUNC_ATTRIBUTE_BINARY_VERSION = ccuda.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_BINARY_VERSION

    #: The attribute to indicate whether the function has been compiled
    #: with user specified option "-Xptxas --dlcm=ca" set .
    CU_FUNC_ATTRIBUTE_CACHE_MODE_CA = ccuda.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_CACHE_MODE_CA

    #: The maximum size in bytes of dynamically-allocated shared memory
    #: that can be used by this function. If the user-specified dynamic
    #: shared memory size is larger than this value, the launch will fail.
    #: See :py:obj:`~.cuFuncSetAttribute`, :py:obj:`~.cuKernelSetAttribute`
    CU_FUNC_ATTRIBUTE_MAX_DYNAMIC_SHARED_SIZE_BYTES = ccuda.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_MAX_DYNAMIC_SHARED_SIZE_BYTES

    #: On devices where the L1 cache and shared memory use the same
    #: hardware resources, this sets the shared memory carveout preference,
    #: in percent of the total shared memory. Refer to
    #: :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_MULTIPROCESSOR`.
    #: This is only a hint, and the driver can choose a different ratio if
    #: required to execute the function. See
    #: :py:obj:`~.cuFuncSetAttribute`, :py:obj:`~.cuKernelSetAttribute`
    CU_FUNC_ATTRIBUTE_PREFERRED_SHARED_MEMORY_CARVEOUT = ccuda.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_PREFERRED_SHARED_MEMORY_CARVEOUT

    #: If this attribute is set, the kernel must launch with a valid
    #: cluster size specified. See :py:obj:`~.cuFuncSetAttribute`,
    #: :py:obj:`~.cuKernelSetAttribute`
    CU_FUNC_ATTRIBUTE_CLUSTER_SIZE_MUST_BE_SET = ccuda.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_CLUSTER_SIZE_MUST_BE_SET

    #: The required cluster width in blocks. The values must either all be
    #: 0 or all be positive. The validity of the cluster dimensions is
    #: otherwise checked at launch time.
    #:
    #: If the value is set during compile time, it cannot be set at
    #: runtime. Setting it at runtime will return CUDA_ERROR_NOT_PERMITTED.
    #: See :py:obj:`~.cuFuncSetAttribute`, :py:obj:`~.cuKernelSetAttribute`
    CU_FUNC_ATTRIBUTE_REQUIRED_CLUSTER_WIDTH = ccuda.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_REQUIRED_CLUSTER_WIDTH

    #: The required cluster height in blocks. The values must either all be
    #: 0 or all be positive. The validity of the cluster dimensions is
    #: otherwise checked at launch time.
    #:
    #: If the value is set during compile time, it cannot be set at
    #: runtime. Setting it at runtime should return
    #: CUDA_ERROR_NOT_PERMITTED. See :py:obj:`~.cuFuncSetAttribute`,
    #: :py:obj:`~.cuKernelSetAttribute`
    CU_FUNC_ATTRIBUTE_REQUIRED_CLUSTER_HEIGHT = ccuda.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_REQUIRED_CLUSTER_HEIGHT

    #: The required cluster depth in blocks. The values must either all be
    #: 0 or all be positive. The validity of the cluster dimensions is
    #: otherwise checked at launch time.
    #:
    #: If the value is set during compile time, it cannot be set at
    #: runtime. Setting it at runtime should return
    #: CUDA_ERROR_NOT_PERMITTED. See :py:obj:`~.cuFuncSetAttribute`,
    #: :py:obj:`~.cuKernelSetAttribute`
    CU_FUNC_ATTRIBUTE_REQUIRED_CLUSTER_DEPTH = ccuda.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_REQUIRED_CLUSTER_DEPTH

    #: Whether the function can be launched with non-portable cluster size.
    #: 1 is allowed, 0 is disallowed. A non-portable cluster size may only
    #: function on the specific SKUs the program is tested on. The launch
    #: might fail if the program is run on a different hardware platform.
    #:
    #: CUDA API provides cudaOccupancyMaxActiveClusters to assist with
    #: checking whether the desired size can be launched on the current
    #: device.
    #:
    #: Portable Cluster Size
    #:
    #: A portable cluster size is guaranteed to be functional on all
    #: compute capabilities higher than the target compute capability. The
    #: portable cluster size for sm_90 is 8 blocks per cluster. This value
    #: may increase for future compute capabilities.
    #:
    #: The specific hardware unit may support higher cluster sizes that’s
    #: not guaranteed to be portable. See :py:obj:`~.cuFuncSetAttribute`,
    #: :py:obj:`~.cuKernelSetAttribute`
    CU_FUNC_ATTRIBUTE_NON_PORTABLE_CLUSTER_SIZE_ALLOWED = ccuda.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_NON_PORTABLE_CLUSTER_SIZE_ALLOWED

    #: The block scheduling policy of a function. The value type is
    #: CUclusterSchedulingPolicy / cudaClusterSchedulingPolicy. See
    #: :py:obj:`~.cuFuncSetAttribute`, :py:obj:`~.cuKernelSetAttribute`
    CU_FUNC_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE = ccuda.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE
    CU_FUNC_ATTRIBUTE_MAX = ccuda.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_MAX

class CUfunc_cache(IntEnum):
    """
    Function cache configurations
    """

    #: no preference for shared memory or L1 (default)
    CU_FUNC_CACHE_PREFER_NONE = ccuda.CUfunc_cache_enum.CU_FUNC_CACHE_PREFER_NONE

    #: prefer larger shared memory and smaller L1 cache
    CU_FUNC_CACHE_PREFER_SHARED = ccuda.CUfunc_cache_enum.CU_FUNC_CACHE_PREFER_SHARED

    #: prefer larger L1 cache and smaller shared memory
    CU_FUNC_CACHE_PREFER_L1 = ccuda.CUfunc_cache_enum.CU_FUNC_CACHE_PREFER_L1

    #: prefer equal sized L1 cache and shared memory
    CU_FUNC_CACHE_PREFER_EQUAL = ccuda.CUfunc_cache_enum.CU_FUNC_CACHE_PREFER_EQUAL

class CUsharedconfig(IntEnum):
    """
    [Deprecated]  Shared memory configurations
    """

    #: set default shared memory bank size
    CU_SHARED_MEM_CONFIG_DEFAULT_BANK_SIZE = ccuda.CUsharedconfig_enum.CU_SHARED_MEM_CONFIG_DEFAULT_BANK_SIZE

    #: set shared memory bank width to four bytes
    CU_SHARED_MEM_CONFIG_FOUR_BYTE_BANK_SIZE = ccuda.CUsharedconfig_enum.CU_SHARED_MEM_CONFIG_FOUR_BYTE_BANK_SIZE

    #: set shared memory bank width to eight bytes
    CU_SHARED_MEM_CONFIG_EIGHT_BYTE_BANK_SIZE = ccuda.CUsharedconfig_enum.CU_SHARED_MEM_CONFIG_EIGHT_BYTE_BANK_SIZE

class CUshared_carveout(IntEnum):
    """
    Shared memory carveout configurations. These may be passed to
    :py:obj:`~.cuFuncSetAttribute` or :py:obj:`~.cuKernelSetAttribute`
    """

    #: No preference for shared memory or L1 (default)
    CU_SHAREDMEM_CARVEOUT_DEFAULT = ccuda.CUshared_carveout_enum.CU_SHAREDMEM_CARVEOUT_DEFAULT

    #: Prefer maximum available L1 cache, minimum shared memory
    CU_SHAREDMEM_CARVEOUT_MAX_L1 = ccuda.CUshared_carveout_enum.CU_SHAREDMEM_CARVEOUT_MAX_L1

    #: Prefer maximum available shared memory, minimum L1 cache
    CU_SHAREDMEM_CARVEOUT_MAX_SHARED = ccuda.CUshared_carveout_enum.CU_SHAREDMEM_CARVEOUT_MAX_SHARED

class CUmemorytype(IntEnum):
    """
    Memory types
    """

    #: Host memory
    CU_MEMORYTYPE_HOST = ccuda.CUmemorytype_enum.CU_MEMORYTYPE_HOST

    #: Device memory
    CU_MEMORYTYPE_DEVICE = ccuda.CUmemorytype_enum.CU_MEMORYTYPE_DEVICE

    #: Array memory
    CU_MEMORYTYPE_ARRAY = ccuda.CUmemorytype_enum.CU_MEMORYTYPE_ARRAY

    #: Unified device or host memory
    CU_MEMORYTYPE_UNIFIED = ccuda.CUmemorytype_enum.CU_MEMORYTYPE_UNIFIED

class CUcomputemode(IntEnum):
    """
    Compute Modes
    """

    #: Default compute mode (Multiple contexts allowed per device)
    CU_COMPUTEMODE_DEFAULT = ccuda.CUcomputemode_enum.CU_COMPUTEMODE_DEFAULT

    #: Compute-prohibited mode (No contexts can be created on this device
    #: at this time)
    CU_COMPUTEMODE_PROHIBITED = ccuda.CUcomputemode_enum.CU_COMPUTEMODE_PROHIBITED

    #: Compute-exclusive-process mode (Only one context used by a single
    #: process can be present on this device at a time)
    CU_COMPUTEMODE_EXCLUSIVE_PROCESS = ccuda.CUcomputemode_enum.CU_COMPUTEMODE_EXCLUSIVE_PROCESS

class CUmem_advise(IntEnum):
    """
    Memory advise values
    """

    #: Data will mostly be read and only occasionally be written to
    CU_MEM_ADVISE_SET_READ_MOSTLY = ccuda.CUmem_advise_enum.CU_MEM_ADVISE_SET_READ_MOSTLY

    #: Undo the effect of :py:obj:`~.CU_MEM_ADVISE_SET_READ_MOSTLY`
    CU_MEM_ADVISE_UNSET_READ_MOSTLY = ccuda.CUmem_advise_enum.CU_MEM_ADVISE_UNSET_READ_MOSTLY

    #: Set the preferred location for the data as the specified device
    CU_MEM_ADVISE_SET_PREFERRED_LOCATION = ccuda.CUmem_advise_enum.CU_MEM_ADVISE_SET_PREFERRED_LOCATION

    #: Clear the preferred location for the data
    CU_MEM_ADVISE_UNSET_PREFERRED_LOCATION = ccuda.CUmem_advise_enum.CU_MEM_ADVISE_UNSET_PREFERRED_LOCATION

    #: Data will be accessed by the specified device, so prevent page
    #: faults as much as possible
    CU_MEM_ADVISE_SET_ACCESSED_BY = ccuda.CUmem_advise_enum.CU_MEM_ADVISE_SET_ACCESSED_BY

    #: Let the Unified Memory subsystem decide on the page faulting policy
    #: for the specified device
    CU_MEM_ADVISE_UNSET_ACCESSED_BY = ccuda.CUmem_advise_enum.CU_MEM_ADVISE_UNSET_ACCESSED_BY

class CUmem_range_attribute(IntEnum):
    """

    """

    #: Whether the range will mostly be read and only occasionally be
    #: written to
    CU_MEM_RANGE_ATTRIBUTE_READ_MOSTLY = ccuda.CUmem_range_attribute_enum.CU_MEM_RANGE_ATTRIBUTE_READ_MOSTLY

    #: The preferred location of the range
    CU_MEM_RANGE_ATTRIBUTE_PREFERRED_LOCATION = ccuda.CUmem_range_attribute_enum.CU_MEM_RANGE_ATTRIBUTE_PREFERRED_LOCATION

    #: Memory range has :py:obj:`~.CU_MEM_ADVISE_SET_ACCESSED_BY` set for
    #: specified device
    CU_MEM_RANGE_ATTRIBUTE_ACCESSED_BY = ccuda.CUmem_range_attribute_enum.CU_MEM_RANGE_ATTRIBUTE_ACCESSED_BY

    #: The last location to which the range was prefetched
    CU_MEM_RANGE_ATTRIBUTE_LAST_PREFETCH_LOCATION = ccuda.CUmem_range_attribute_enum.CU_MEM_RANGE_ATTRIBUTE_LAST_PREFETCH_LOCATION

    #: The preferred location type of the range
    CU_MEM_RANGE_ATTRIBUTE_PREFERRED_LOCATION_TYPE = ccuda.CUmem_range_attribute_enum.CU_MEM_RANGE_ATTRIBUTE_PREFERRED_LOCATION_TYPE

    #: The preferred location id of the range
    CU_MEM_RANGE_ATTRIBUTE_PREFERRED_LOCATION_ID = ccuda.CUmem_range_attribute_enum.CU_MEM_RANGE_ATTRIBUTE_PREFERRED_LOCATION_ID

    #: The last location type to which the range was prefetched
    CU_MEM_RANGE_ATTRIBUTE_LAST_PREFETCH_LOCATION_TYPE = ccuda.CUmem_range_attribute_enum.CU_MEM_RANGE_ATTRIBUTE_LAST_PREFETCH_LOCATION_TYPE

    #: The last location id to which the range was prefetched
    CU_MEM_RANGE_ATTRIBUTE_LAST_PREFETCH_LOCATION_ID = ccuda.CUmem_range_attribute_enum.CU_MEM_RANGE_ATTRIBUTE_LAST_PREFETCH_LOCATION_ID

class CUjit_option(IntEnum):
    """
    Online compiler and linker options
    """

    #: Max number of registers that a thread may use.
    #: Option type: unsigned int
    #: Applies to: compiler only
    CU_JIT_MAX_REGISTERS = ccuda.CUjit_option_enum.CU_JIT_MAX_REGISTERS

    #: IN: Specifies minimum number of threads per block to target
    #: compilation for
    #: OUT: Returns the number of threads the compiler actually targeted.
    #: This restricts the resource utilization of the compiler (e.g. max
    #: registers) such that a block with the given number of threads should
    #: be able to launch based on register limitations. Note, this option
    #: does not currently take into account any other resource limitations,
    #: such as shared memory utilization.
    #: Cannot be combined with :py:obj:`~.CU_JIT_TARGET`.
    #: Option type: unsigned int
    #: Applies to: compiler only
    CU_JIT_THREADS_PER_BLOCK = ccuda.CUjit_option_enum.CU_JIT_THREADS_PER_BLOCK

    #: Overwrites the option value with the total wall clock time, in
    #: milliseconds, spent in the compiler and linker
    #: Option type: float
    #: Applies to: compiler and linker
    CU_JIT_WALL_TIME = ccuda.CUjit_option_enum.CU_JIT_WALL_TIME

    #: Pointer to a buffer in which to print any log messages that are
    #: informational in nature (the buffer size is specified via option
    #: :py:obj:`~.CU_JIT_INFO_LOG_BUFFER_SIZE_BYTES`)
    #: Option type: char *
    #: Applies to: compiler and linker
    CU_JIT_INFO_LOG_BUFFER = ccuda.CUjit_option_enum.CU_JIT_INFO_LOG_BUFFER

    #: IN: Log buffer size in bytes. Log messages will be capped at this
    #: size (including null terminator)
    #: OUT: Amount of log buffer filled with messages
    #: Option type: unsigned int
    #: Applies to: compiler and linker
    CU_JIT_INFO_LOG_BUFFER_SIZE_BYTES = ccuda.CUjit_option_enum.CU_JIT_INFO_LOG_BUFFER_SIZE_BYTES

    #: Pointer to a buffer in which to print any log messages that reflect
    #: errors (the buffer size is specified via option
    #: :py:obj:`~.CU_JIT_ERROR_LOG_BUFFER_SIZE_BYTES`)
    #: Option type: char *
    #: Applies to: compiler and linker
    CU_JIT_ERROR_LOG_BUFFER = ccuda.CUjit_option_enum.CU_JIT_ERROR_LOG_BUFFER

    #: IN: Log buffer size in bytes. Log messages will be capped at this
    #: size (including null terminator)
    #: OUT: Amount of log buffer filled with messages
    #: Option type: unsigned int
    #: Applies to: compiler and linker
    CU_JIT_ERROR_LOG_BUFFER_SIZE_BYTES = ccuda.CUjit_option_enum.CU_JIT_ERROR_LOG_BUFFER_SIZE_BYTES

    #: Level of optimizations to apply to generated code (0 - 4), with 4
    #: being the default and highest level of optimizations.
    #: Option type: unsigned int
    #: Applies to: compiler only
    CU_JIT_OPTIMIZATION_LEVEL = ccuda.CUjit_option_enum.CU_JIT_OPTIMIZATION_LEVEL

    #: No option value required. Determines the target based on the current
    #: attached context (default)
    #: Option type: No option value needed
    #: Applies to: compiler and linker
    CU_JIT_TARGET_FROM_CUCONTEXT = ccuda.CUjit_option_enum.CU_JIT_TARGET_FROM_CUCONTEXT

    #: Target is chosen based on supplied :py:obj:`~.CUjit_target`. Cannot
    #: be combined with :py:obj:`~.CU_JIT_THREADS_PER_BLOCK`.
    #: Option type: unsigned int for enumerated type
    #: :py:obj:`~.CUjit_target`
    #: Applies to: compiler and linker
    CU_JIT_TARGET = ccuda.CUjit_option_enum.CU_JIT_TARGET

    #: Specifies choice of fallback strategy if matching cubin is not
    #: found. Choice is based on supplied :py:obj:`~.CUjit_fallback`. This
    #: option cannot be used with cuLink* APIs as the linker requires exact
    #: matches.
    #: Option type: unsigned int for enumerated type
    #: :py:obj:`~.CUjit_fallback`
    #: Applies to: compiler only
    CU_JIT_FALLBACK_STRATEGY = ccuda.CUjit_option_enum.CU_JIT_FALLBACK_STRATEGY

    #: Specifies whether to create debug information in output (-g) (0:
    #: false, default)
    #: Option type: int
    #: Applies to: compiler and linker
    CU_JIT_GENERATE_DEBUG_INFO = ccuda.CUjit_option_enum.CU_JIT_GENERATE_DEBUG_INFO

    #: Generate verbose log messages (0: false, default)
    #: Option type: int
    #: Applies to: compiler and linker
    CU_JIT_LOG_VERBOSE = ccuda.CUjit_option_enum.CU_JIT_LOG_VERBOSE

    #: Generate line number information (-lineinfo) (0: false, default)
    #: Option type: int
    #: Applies to: compiler only
    CU_JIT_GENERATE_LINE_INFO = ccuda.CUjit_option_enum.CU_JIT_GENERATE_LINE_INFO

    #: Specifies whether to enable caching explicitly (-dlcm)
    #: Choice is based on supplied :py:obj:`~.CUjit_cacheMode_enum`.
    #: Option type: unsigned int for enumerated type
    #: :py:obj:`~.CUjit_cacheMode_enum`
    #: Applies to: compiler only
    CU_JIT_CACHE_MODE = ccuda.CUjit_option_enum.CU_JIT_CACHE_MODE

    #: [Deprecated]
    CU_JIT_NEW_SM3X_OPT = ccuda.CUjit_option_enum.CU_JIT_NEW_SM3X_OPT

    #: This jit option is used for internal purpose only.
    CU_JIT_FAST_COMPILE = ccuda.CUjit_option_enum.CU_JIT_FAST_COMPILE

    #: Array of device symbol names that will be relocated to the
    #: corresponding host addresses stored in
    #: :py:obj:`~.CU_JIT_GLOBAL_SYMBOL_ADDRESSES`.
    #: Must contain :py:obj:`~.CU_JIT_GLOBAL_SYMBOL_COUNT` entries.
    #: When loading a device module, driver will relocate all encountered
    #: unresolved symbols to the host addresses.
    #: It is only allowed to register symbols that correspond to unresolved
    #: global variables.
    #: It is illegal to register the same device symbol at multiple
    #: addresses.
    #: Option type: const char **
    #: Applies to: dynamic linker only
    CU_JIT_GLOBAL_SYMBOL_NAMES = ccuda.CUjit_option_enum.CU_JIT_GLOBAL_SYMBOL_NAMES

    #: Array of host addresses that will be used to relocate corresponding
    #: device symbols stored in :py:obj:`~.CU_JIT_GLOBAL_SYMBOL_NAMES`.
    #: Must contain :py:obj:`~.CU_JIT_GLOBAL_SYMBOL_COUNT` entries.
    #: Option type: void **
    #: Applies to: dynamic linker only
    CU_JIT_GLOBAL_SYMBOL_ADDRESSES = ccuda.CUjit_option_enum.CU_JIT_GLOBAL_SYMBOL_ADDRESSES

    #: Number of entries in :py:obj:`~.CU_JIT_GLOBAL_SYMBOL_NAMES` and
    #: :py:obj:`~.CU_JIT_GLOBAL_SYMBOL_ADDRESSES` arrays.
    #: Option type: unsigned int
    #: Applies to: dynamic linker only
    CU_JIT_GLOBAL_SYMBOL_COUNT = ccuda.CUjit_option_enum.CU_JIT_GLOBAL_SYMBOL_COUNT

    #: [Deprecated]
    #:
    #: Only valid with LTO-IR compiled with toolkits prior to CUDA 12.0
    CU_JIT_LTO = ccuda.CUjit_option_enum.CU_JIT_LTO

    #: [Deprecated]
    #:
    #: Only valid with LTO-IR compiled with toolkits prior to CUDA 12.0
    CU_JIT_FTZ = ccuda.CUjit_option_enum.CU_JIT_FTZ

    #: [Deprecated]
    #:
    #: Only valid with LTO-IR compiled with toolkits prior to CUDA 12.0
    CU_JIT_PREC_DIV = ccuda.CUjit_option_enum.CU_JIT_PREC_DIV

    #: [Deprecated]
    #:
    #: Only valid with LTO-IR compiled with toolkits prior to CUDA 12.0
    CU_JIT_PREC_SQRT = ccuda.CUjit_option_enum.CU_JIT_PREC_SQRT

    #: [Deprecated]
    #:
    #: Only valid with LTO-IR compiled with toolkits prior to CUDA 12.0
    CU_JIT_FMA = ccuda.CUjit_option_enum.CU_JIT_FMA

    #: [Deprecated]
    #:
    #: Only valid with LTO-IR compiled with toolkits prior to CUDA 12.0
    CU_JIT_REFERENCED_KERNEL_NAMES = ccuda.CUjit_option_enum.CU_JIT_REFERENCED_KERNEL_NAMES

    #: [Deprecated]
    #:
    #: Only valid with LTO-IR compiled with toolkits prior to CUDA 12.0
    CU_JIT_REFERENCED_KERNEL_COUNT = ccuda.CUjit_option_enum.CU_JIT_REFERENCED_KERNEL_COUNT

    #: [Deprecated]
    #:
    #: Only valid with LTO-IR compiled with toolkits prior to CUDA 12.0
    CU_JIT_REFERENCED_VARIABLE_NAMES = ccuda.CUjit_option_enum.CU_JIT_REFERENCED_VARIABLE_NAMES

    #: [Deprecated]
    #:
    #: Only valid with LTO-IR compiled with toolkits prior to CUDA 12.0
    CU_JIT_REFERENCED_VARIABLE_COUNT = ccuda.CUjit_option_enum.CU_JIT_REFERENCED_VARIABLE_COUNT

    #: [Deprecated]
    #:
    #: Only valid with LTO-IR compiled with toolkits prior to CUDA 12.0
    CU_JIT_OPTIMIZE_UNUSED_DEVICE_VARIABLES = ccuda.CUjit_option_enum.CU_JIT_OPTIMIZE_UNUSED_DEVICE_VARIABLES

    #: Generate position independent code (0: false)
    #: Option type: int
    #: Applies to: compiler only
    CU_JIT_POSITION_INDEPENDENT_CODE = ccuda.CUjit_option_enum.CU_JIT_POSITION_INDEPENDENT_CODE

    #: This option hints to the JIT compiler the minimum number of CTAs
    #: from the kernel’s grid to be mapped to a SM. This option is ignored
    #: when used together with :py:obj:`~.CU_JIT_MAX_REGISTERS` or
    #: :py:obj:`~.CU_JIT_THREADS_PER_BLOCK`. Optimizations based on this
    #: option need :py:obj:`~.CU_JIT_MAX_THREADS_PER_BLOCK` to be specified
    #: as well. For kernels already using PTX directive .minnctapersm, this
    #: option will be ignored by default. Use
    #: :py:obj:`~.CU_JIT_OVERRIDE_DIRECTIVE_VALUES` to let this option take
    #: precedence over the PTX directive. Option type: unsigned int
    #: Applies to: compiler only
    CU_JIT_MIN_CTA_PER_SM = ccuda.CUjit_option_enum.CU_JIT_MIN_CTA_PER_SM

    #: Maximum number threads in a thread block, computed as the product of
    #: the maximum extent specifed for each dimension of the block. This
    #: limit is guaranteed not to be exeeded in any invocation of the
    #: kernel. Exceeding the the maximum number of threads results in
    #: runtime error or kernel launch failure. For kernels already using
    #: PTX directive .maxntid, this option will be ignored by default. Use
    #: :py:obj:`~.CU_JIT_OVERRIDE_DIRECTIVE_VALUES` to let this option take
    #: precedence over the PTX directive. Option type: int
    #: Applies to: compiler only
    CU_JIT_MAX_THREADS_PER_BLOCK = ccuda.CUjit_option_enum.CU_JIT_MAX_THREADS_PER_BLOCK

    #: This option lets the values specified using
    #: :py:obj:`~.CU_JIT_MAX_REGISTERS`,
    #: :py:obj:`~.CU_JIT_THREADS_PER_BLOCK`,
    #: :py:obj:`~.CU_JIT_MAX_THREADS_PER_BLOCK` and
    #: :py:obj:`~.CU_JIT_MIN_CTA_PER_SM` take precedence over any PTX
    #: directives. (0: Disable, default; 1: Enable) Option type: int
    #: Applies to: compiler only
    CU_JIT_OVERRIDE_DIRECTIVE_VALUES = ccuda.CUjit_option_enum.CU_JIT_OVERRIDE_DIRECTIVE_VALUES
    CU_JIT_NUM_OPTIONS = ccuda.CUjit_option_enum.CU_JIT_NUM_OPTIONS

class CUjit_target(IntEnum):
    """
    Online compilation targets
    """

    #: Compute device class 3.0
    CU_TARGET_COMPUTE_30 = ccuda.CUjit_target_enum.CU_TARGET_COMPUTE_30

    #: Compute device class 3.2
    CU_TARGET_COMPUTE_32 = ccuda.CUjit_target_enum.CU_TARGET_COMPUTE_32

    #: Compute device class 3.5
    CU_TARGET_COMPUTE_35 = ccuda.CUjit_target_enum.CU_TARGET_COMPUTE_35

    #: Compute device class 3.7
    CU_TARGET_COMPUTE_37 = ccuda.CUjit_target_enum.CU_TARGET_COMPUTE_37

    #: Compute device class 5.0
    CU_TARGET_COMPUTE_50 = ccuda.CUjit_target_enum.CU_TARGET_COMPUTE_50

    #: Compute device class 5.2
    CU_TARGET_COMPUTE_52 = ccuda.CUjit_target_enum.CU_TARGET_COMPUTE_52

    #: Compute device class 5.3
    CU_TARGET_COMPUTE_53 = ccuda.CUjit_target_enum.CU_TARGET_COMPUTE_53

    #: Compute device class 6.0.
    CU_TARGET_COMPUTE_60 = ccuda.CUjit_target_enum.CU_TARGET_COMPUTE_60

    #: Compute device class 6.1.
    CU_TARGET_COMPUTE_61 = ccuda.CUjit_target_enum.CU_TARGET_COMPUTE_61

    #: Compute device class 6.2.
    CU_TARGET_COMPUTE_62 = ccuda.CUjit_target_enum.CU_TARGET_COMPUTE_62

    #: Compute device class 7.0.
    CU_TARGET_COMPUTE_70 = ccuda.CUjit_target_enum.CU_TARGET_COMPUTE_70

    #: Compute device class 7.2.
    CU_TARGET_COMPUTE_72 = ccuda.CUjit_target_enum.CU_TARGET_COMPUTE_72

    #: Compute device class 7.5.
    CU_TARGET_COMPUTE_75 = ccuda.CUjit_target_enum.CU_TARGET_COMPUTE_75

    #: Compute device class 8.0.
    CU_TARGET_COMPUTE_80 = ccuda.CUjit_target_enum.CU_TARGET_COMPUTE_80

    #: Compute device class 8.6.
    CU_TARGET_COMPUTE_86 = ccuda.CUjit_target_enum.CU_TARGET_COMPUTE_86

    #: Compute device class 8.7.
    CU_TARGET_COMPUTE_87 = ccuda.CUjit_target_enum.CU_TARGET_COMPUTE_87

    #: Compute device class 8.9.
    CU_TARGET_COMPUTE_89 = ccuda.CUjit_target_enum.CU_TARGET_COMPUTE_89

    #: Compute device class 9.0. Compute device class 9.0. with accelerated
    #: features.
    CU_TARGET_COMPUTE_90 = ccuda.CUjit_target_enum.CU_TARGET_COMPUTE_90
    CU_TARGET_COMPUTE_90A = ccuda.CUjit_target_enum.CU_TARGET_COMPUTE_90A

class CUjit_fallback(IntEnum):
    """
    Cubin matching fallback strategies
    """

    #: Prefer to compile ptx if exact binary match not found
    CU_PREFER_PTX = ccuda.CUjit_fallback_enum.CU_PREFER_PTX

    #: Prefer to fall back to compatible binary code if exact match not
    #: found
    CU_PREFER_BINARY = ccuda.CUjit_fallback_enum.CU_PREFER_BINARY

class CUjit_cacheMode(IntEnum):
    """
    Caching modes for dlcm
    """

    #: Compile with no -dlcm flag specified
    CU_JIT_CACHE_OPTION_NONE = ccuda.CUjit_cacheMode_enum.CU_JIT_CACHE_OPTION_NONE

    #: Compile with L1 cache disabled
    CU_JIT_CACHE_OPTION_CG = ccuda.CUjit_cacheMode_enum.CU_JIT_CACHE_OPTION_CG

    #: Compile with L1 cache enabled
    CU_JIT_CACHE_OPTION_CA = ccuda.CUjit_cacheMode_enum.CU_JIT_CACHE_OPTION_CA

class CUjitInputType(IntEnum):
    """
    Device code formats
    """

    #: Compiled device-class-specific device code
    #: Applicable options: none
    CU_JIT_INPUT_CUBIN = ccuda.CUjitInputType_enum.CU_JIT_INPUT_CUBIN

    #: PTX source code
    #: Applicable options: PTX compiler options
    CU_JIT_INPUT_PTX = ccuda.CUjitInputType_enum.CU_JIT_INPUT_PTX

    #: Bundle of multiple cubins and/or PTX of some device code
    #: Applicable options: PTX compiler options,
    #: :py:obj:`~.CU_JIT_FALLBACK_STRATEGY`
    CU_JIT_INPUT_FATBINARY = ccuda.CUjitInputType_enum.CU_JIT_INPUT_FATBINARY

    #: Host object with embedded device code
    #: Applicable options: PTX compiler options,
    #: :py:obj:`~.CU_JIT_FALLBACK_STRATEGY`
    CU_JIT_INPUT_OBJECT = ccuda.CUjitInputType_enum.CU_JIT_INPUT_OBJECT

    #: Archive of host objects with embedded device code
    #: Applicable options: PTX compiler options,
    #: :py:obj:`~.CU_JIT_FALLBACK_STRATEGY`
    CU_JIT_INPUT_LIBRARY = ccuda.CUjitInputType_enum.CU_JIT_INPUT_LIBRARY

    #: [Deprecated]
    #:
    #: Only valid with LTO-IR compiled with toolkits prior to CUDA 12.0
    CU_JIT_INPUT_NVVM = ccuda.CUjitInputType_enum.CU_JIT_INPUT_NVVM
    CU_JIT_NUM_INPUT_TYPES = ccuda.CUjitInputType_enum.CU_JIT_NUM_INPUT_TYPES

class CUgraphicsRegisterFlags(IntEnum):
    """
    Flags to register a graphics resource
    """
    CU_GRAPHICS_REGISTER_FLAGS_NONE = ccuda.CUgraphicsRegisterFlags_enum.CU_GRAPHICS_REGISTER_FLAGS_NONE
    CU_GRAPHICS_REGISTER_FLAGS_READ_ONLY = ccuda.CUgraphicsRegisterFlags_enum.CU_GRAPHICS_REGISTER_FLAGS_READ_ONLY
    CU_GRAPHICS_REGISTER_FLAGS_WRITE_DISCARD = ccuda.CUgraphicsRegisterFlags_enum.CU_GRAPHICS_REGISTER_FLAGS_WRITE_DISCARD
    CU_GRAPHICS_REGISTER_FLAGS_SURFACE_LDST = ccuda.CUgraphicsRegisterFlags_enum.CU_GRAPHICS_REGISTER_FLAGS_SURFACE_LDST
    CU_GRAPHICS_REGISTER_FLAGS_TEXTURE_GATHER = ccuda.CUgraphicsRegisterFlags_enum.CU_GRAPHICS_REGISTER_FLAGS_TEXTURE_GATHER

class CUgraphicsMapResourceFlags(IntEnum):
    """
    Flags for mapping and unmapping interop resources
    """
    CU_GRAPHICS_MAP_RESOURCE_FLAGS_NONE = ccuda.CUgraphicsMapResourceFlags_enum.CU_GRAPHICS_MAP_RESOURCE_FLAGS_NONE
    CU_GRAPHICS_MAP_RESOURCE_FLAGS_READ_ONLY = ccuda.CUgraphicsMapResourceFlags_enum.CU_GRAPHICS_MAP_RESOURCE_FLAGS_READ_ONLY
    CU_GRAPHICS_MAP_RESOURCE_FLAGS_WRITE_DISCARD = ccuda.CUgraphicsMapResourceFlags_enum.CU_GRAPHICS_MAP_RESOURCE_FLAGS_WRITE_DISCARD

class CUarray_cubemap_face(IntEnum):
    """
    Array indices for cube faces
    """

    #: Positive X face of cubemap
    CU_CUBEMAP_FACE_POSITIVE_X = ccuda.CUarray_cubemap_face_enum.CU_CUBEMAP_FACE_POSITIVE_X

    #: Negative X face of cubemap
    CU_CUBEMAP_FACE_NEGATIVE_X = ccuda.CUarray_cubemap_face_enum.CU_CUBEMAP_FACE_NEGATIVE_X

    #: Positive Y face of cubemap
    CU_CUBEMAP_FACE_POSITIVE_Y = ccuda.CUarray_cubemap_face_enum.CU_CUBEMAP_FACE_POSITIVE_Y

    #: Negative Y face of cubemap
    CU_CUBEMAP_FACE_NEGATIVE_Y = ccuda.CUarray_cubemap_face_enum.CU_CUBEMAP_FACE_NEGATIVE_Y

    #: Positive Z face of cubemap
    CU_CUBEMAP_FACE_POSITIVE_Z = ccuda.CUarray_cubemap_face_enum.CU_CUBEMAP_FACE_POSITIVE_Z

    #: Negative Z face of cubemap
    CU_CUBEMAP_FACE_NEGATIVE_Z = ccuda.CUarray_cubemap_face_enum.CU_CUBEMAP_FACE_NEGATIVE_Z

class CUlimit(IntEnum):
    """
    Limits
    """

    #: GPU thread stack size
    CU_LIMIT_STACK_SIZE = ccuda.CUlimit_enum.CU_LIMIT_STACK_SIZE

    #: GPU printf FIFO size
    CU_LIMIT_PRINTF_FIFO_SIZE = ccuda.CUlimit_enum.CU_LIMIT_PRINTF_FIFO_SIZE

    #: GPU malloc heap size
    CU_LIMIT_MALLOC_HEAP_SIZE = ccuda.CUlimit_enum.CU_LIMIT_MALLOC_HEAP_SIZE

    #: GPU device runtime launch synchronize depth
    CU_LIMIT_DEV_RUNTIME_SYNC_DEPTH = ccuda.CUlimit_enum.CU_LIMIT_DEV_RUNTIME_SYNC_DEPTH

    #: GPU device runtime pending launch count
    CU_LIMIT_DEV_RUNTIME_PENDING_LAUNCH_COUNT = ccuda.CUlimit_enum.CU_LIMIT_DEV_RUNTIME_PENDING_LAUNCH_COUNT

    #: A value between 0 and 128 that indicates the maximum fetch
    #: granularity of L2 (in Bytes). This is a hint
    CU_LIMIT_MAX_L2_FETCH_GRANULARITY = ccuda.CUlimit_enum.CU_LIMIT_MAX_L2_FETCH_GRANULARITY

    #: A size in bytes for L2 persisting lines cache size
    CU_LIMIT_PERSISTING_L2_CACHE_SIZE = ccuda.CUlimit_enum.CU_LIMIT_PERSISTING_L2_CACHE_SIZE
    CU_LIMIT_MAX = ccuda.CUlimit_enum.CU_LIMIT_MAX

class CUresourcetype(IntEnum):
    """
    Resource types
    """

    #: Array resource
    CU_RESOURCE_TYPE_ARRAY = ccuda.CUresourcetype_enum.CU_RESOURCE_TYPE_ARRAY

    #: Mipmapped array resource
    CU_RESOURCE_TYPE_MIPMAPPED_ARRAY = ccuda.CUresourcetype_enum.CU_RESOURCE_TYPE_MIPMAPPED_ARRAY

    #: Linear resource
    CU_RESOURCE_TYPE_LINEAR = ccuda.CUresourcetype_enum.CU_RESOURCE_TYPE_LINEAR

    #: Pitch 2D resource
    CU_RESOURCE_TYPE_PITCH2D = ccuda.CUresourcetype_enum.CU_RESOURCE_TYPE_PITCH2D

class CUaccessProperty(IntEnum):
    """
    Specifies performance hint with :py:obj:`~.CUaccessPolicyWindow`
    for hitProp and missProp members.
    """

    #: Normal cache persistence.
    CU_ACCESS_PROPERTY_NORMAL = ccuda.CUaccessProperty_enum.CU_ACCESS_PROPERTY_NORMAL

    #: Streaming access is less likely to persit from cache.
    CU_ACCESS_PROPERTY_STREAMING = ccuda.CUaccessProperty_enum.CU_ACCESS_PROPERTY_STREAMING

    #: Persisting access is more likely to persist in cache.
    CU_ACCESS_PROPERTY_PERSISTING = ccuda.CUaccessProperty_enum.CU_ACCESS_PROPERTY_PERSISTING

class CUgraphConditionalNodeType(IntEnum):
    """
    Conditional node types
    """

    #: Conditional 'if' Node. Body executed once if condition value is non-
    #: zero.
    CU_GRAPH_COND_TYPE_IF = ccuda.CUgraphConditionalNodeType_enum.CU_GRAPH_COND_TYPE_IF

    #: Conditional 'while' Node. Body executed repeatedly while condition
    #: value is non-zero.
    CU_GRAPH_COND_TYPE_WHILE = ccuda.CUgraphConditionalNodeType_enum.CU_GRAPH_COND_TYPE_WHILE

class CUgraphNodeType(IntEnum):
    """
    Graph node types
    """

    #: GPU kernel node
    CU_GRAPH_NODE_TYPE_KERNEL = ccuda.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_KERNEL

    #: Memcpy node
    CU_GRAPH_NODE_TYPE_MEMCPY = ccuda.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_MEMCPY

    #: Memset node
    CU_GRAPH_NODE_TYPE_MEMSET = ccuda.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_MEMSET

    #: Host (executable) node
    CU_GRAPH_NODE_TYPE_HOST = ccuda.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_HOST

    #: Node which executes an embedded graph
    CU_GRAPH_NODE_TYPE_GRAPH = ccuda.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_GRAPH

    #: Empty (no-op) node
    CU_GRAPH_NODE_TYPE_EMPTY = ccuda.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_EMPTY

    #: External event wait node
    CU_GRAPH_NODE_TYPE_WAIT_EVENT = ccuda.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_WAIT_EVENT

    #: External event record node
    CU_GRAPH_NODE_TYPE_EVENT_RECORD = ccuda.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_EVENT_RECORD

    #: External semaphore signal node
    CU_GRAPH_NODE_TYPE_EXT_SEMAS_SIGNAL = ccuda.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_EXT_SEMAS_SIGNAL

    #: External semaphore wait node
    CU_GRAPH_NODE_TYPE_EXT_SEMAS_WAIT = ccuda.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_EXT_SEMAS_WAIT

    #: Memory Allocation Node
    CU_GRAPH_NODE_TYPE_MEM_ALLOC = ccuda.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_MEM_ALLOC

    #: Memory Free Node
    CU_GRAPH_NODE_TYPE_MEM_FREE = ccuda.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_MEM_FREE

    #: Batch MemOp Node
    CU_GRAPH_NODE_TYPE_BATCH_MEM_OP = ccuda.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_BATCH_MEM_OP

    #: Conditional Node                                         May be used
    #: to implement a conditional execution path or loop
    #:                                         inside of a graph. The
    #: graph(s) contained within the body of the conditional node
    #:                                         can be selectively executed
    #: or iterated upon based on the value of a conditional
    #:                                         variable.
    #:
    #:                                         Handles must be created in
    #: advance of creating the node
    #:                                         using
    #: :py:obj:`~.cuGraphConditionalHandleCreate`.
    #:
    #:                                         The following restrictions
    #: apply to graphs which contain conditional nodes:
    #:                                          The graph cannot be used in
    #: a child node.
    #:                                          Only one instantiation of
    #: the graph may exist at any point in time.
    #:                                          The graph cannot be cloned.
    #:
    #:                                         To set the control value,
    #: supply a default value when creating the handle and/or
    #:                                         call
    #: :py:obj:`~.cudaGraphSetConditional` from device code.
    CU_GRAPH_NODE_TYPE_CONDITIONAL = ccuda.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_CONDITIONAL

class CUgraphDependencyType(IntEnum):
    """
    Type annotations that can be applied to graph edges as part of
    :py:obj:`~.CUgraphEdgeData`.
    """

    #: This is an ordinary dependency.
    CU_GRAPH_DEPENDENCY_TYPE_DEFAULT = ccuda.CUgraphDependencyType_enum.CU_GRAPH_DEPENDENCY_TYPE_DEFAULT

    #: This dependency type allows the downstream node to use
    #: `cudaGridDependencySynchronize()`. It may only be used between
    #: kernel nodes, and must be used with either the
    #: :py:obj:`~.CU_GRAPH_KERNEL_NODE_PORT_PROGRAMMATIC` or
    #: :py:obj:`~.CU_GRAPH_KERNEL_NODE_PORT_LAUNCH_ORDER` outgoing port.
    CU_GRAPH_DEPENDENCY_TYPE_PROGRAMMATIC = ccuda.CUgraphDependencyType_enum.CU_GRAPH_DEPENDENCY_TYPE_PROGRAMMATIC

class CUgraphInstantiateResult(IntEnum):
    """
    Graph instantiation results
    """

    #: Instantiation succeeded
    CUDA_GRAPH_INSTANTIATE_SUCCESS = ccuda.CUgraphInstantiateResult_enum.CUDA_GRAPH_INSTANTIATE_SUCCESS

    #: Instantiation failed for an unexpected reason which is described in
    #: the return value of the function
    CUDA_GRAPH_INSTANTIATE_ERROR = ccuda.CUgraphInstantiateResult_enum.CUDA_GRAPH_INSTANTIATE_ERROR

    #: Instantiation failed due to invalid structure, such as cycles
    CUDA_GRAPH_INSTANTIATE_INVALID_STRUCTURE = ccuda.CUgraphInstantiateResult_enum.CUDA_GRAPH_INSTANTIATE_INVALID_STRUCTURE

    #: Instantiation for device launch failed because the graph contained
    #: an unsupported operation
    CUDA_GRAPH_INSTANTIATE_NODE_OPERATION_NOT_SUPPORTED = ccuda.CUgraphInstantiateResult_enum.CUDA_GRAPH_INSTANTIATE_NODE_OPERATION_NOT_SUPPORTED

    #: Instantiation for device launch failed due to the nodes belonging to
    #: different contexts
    CUDA_GRAPH_INSTANTIATE_MULTIPLE_CTXS_NOT_SUPPORTED = ccuda.CUgraphInstantiateResult_enum.CUDA_GRAPH_INSTANTIATE_MULTIPLE_CTXS_NOT_SUPPORTED

class CUsynchronizationPolicy(IntEnum):
    """

    """
    CU_SYNC_POLICY_AUTO = ccuda.CUsynchronizationPolicy_enum.CU_SYNC_POLICY_AUTO
    CU_SYNC_POLICY_SPIN = ccuda.CUsynchronizationPolicy_enum.CU_SYNC_POLICY_SPIN
    CU_SYNC_POLICY_YIELD = ccuda.CUsynchronizationPolicy_enum.CU_SYNC_POLICY_YIELD
    CU_SYNC_POLICY_BLOCKING_SYNC = ccuda.CUsynchronizationPolicy_enum.CU_SYNC_POLICY_BLOCKING_SYNC

class CUclusterSchedulingPolicy(IntEnum):
    """
    Cluster scheduling policies. These may be passed to
    :py:obj:`~.cuFuncSetAttribute` or :py:obj:`~.cuKernelSetAttribute`
    """

    #: the default policy
    CU_CLUSTER_SCHEDULING_POLICY_DEFAULT = ccuda.CUclusterSchedulingPolicy_enum.CU_CLUSTER_SCHEDULING_POLICY_DEFAULT

    #: spread the blocks within a cluster to the SMs
    CU_CLUSTER_SCHEDULING_POLICY_SPREAD = ccuda.CUclusterSchedulingPolicy_enum.CU_CLUSTER_SCHEDULING_POLICY_SPREAD

    #: allow the hardware to load-balance the blocks in a cluster to the
    #: SMs
    CU_CLUSTER_SCHEDULING_POLICY_LOAD_BALANCING = ccuda.CUclusterSchedulingPolicy_enum.CU_CLUSTER_SCHEDULING_POLICY_LOAD_BALANCING

class CUlaunchMemSyncDomain(IntEnum):
    """
    Memory Synchronization Domain  A kernel can be launched in a
    specified memory synchronization domain that affects all memory
    operations issued by that kernel. A memory barrier issued in one
    domain will only order memory operations in that domain, thus
    eliminating latency increase from memory barriers ordering
    unrelated traffic.  By default, kernels are launched in domain 0.
    Kernel launched with :py:obj:`~.CU_LAUNCH_MEM_SYNC_DOMAIN_REMOTE`
    will have a different domain ID. User may also alter the domain ID
    with :py:obj:`~.CUlaunchMemSyncDomainMap` for a specific stream /
    graph node / kernel launch. See
    :py:obj:`~.CU_LAUNCH_ATTRIBUTE_MEM_SYNC_DOMAIN`,
    :py:obj:`~.cuStreamSetAttribute`, :py:obj:`~.cuLaunchKernelEx`,
    :py:obj:`~.cuGraphKernelNodeSetAttribute`.  Memory operations done
    in kernels launched in different domains are considered system-
    scope distanced. In other words, a GPU scoped memory
    synchronization is not sufficient for memory order to be observed
    by kernels in another memory synchronization domain even if they
    are on the same GPU.
    """

    #: Launch kernels in the default domain
    CU_LAUNCH_MEM_SYNC_DOMAIN_DEFAULT = ccuda.CUlaunchMemSyncDomain_enum.CU_LAUNCH_MEM_SYNC_DOMAIN_DEFAULT

    #: Launch kernels in the remote domain
    CU_LAUNCH_MEM_SYNC_DOMAIN_REMOTE = ccuda.CUlaunchMemSyncDomain_enum.CU_LAUNCH_MEM_SYNC_DOMAIN_REMOTE

class CUlaunchAttributeID(IntEnum):
    """
    Launch attributes enum; used as id field of
    :py:obj:`~.CUlaunchAttribute`
    """

    #: Ignored entry, for convenient composition
    CU_LAUNCH_ATTRIBUTE_IGNORE = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_IGNORE

    #: Valid for streams, graph nodes, launches. See
    #: :py:obj:`~.CUlaunchAttributeValue.accessPolicyWindow`.
    CU_LAUNCH_ATTRIBUTE_ACCESS_POLICY_WINDOW = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_ACCESS_POLICY_WINDOW

    #: Valid for graph nodes, launches. See
    #: :py:obj:`~.CUlaunchAttributeValue.cooperative`.
    CU_LAUNCH_ATTRIBUTE_COOPERATIVE = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_COOPERATIVE

    #: Valid for streams. See
    #: :py:obj:`~.CUlaunchAttributeValue.syncPolicy`.
    CU_LAUNCH_ATTRIBUTE_SYNCHRONIZATION_POLICY = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_SYNCHRONIZATION_POLICY

    #: Valid for graph nodes, launches. See
    #: :py:obj:`~.CUlaunchAttributeValue.clusterDim`.
    CU_LAUNCH_ATTRIBUTE_CLUSTER_DIMENSION = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_CLUSTER_DIMENSION

    #: Valid for graph nodes, launches. See
    #: :py:obj:`~.CUlaunchAttributeValue.clusterSchedulingPolicyPreference`.
    CU_LAUNCH_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE

    #: Valid for launches. Setting
    #: :py:obj:`~.CUlaunchAttributeValue.programmaticStreamSerializationAllowed`
    #: to non-0 signals that the kernel will use programmatic means to
    #: resolve its stream dependency, so that the CUDA runtime should
    #: opportunistically allow the grid's execution to overlap with the
    #: previous kernel in the stream, if that kernel requests the overlap.
    #: The dependent launches can choose to wait on the dependency using
    #: the programmatic sync (cudaGridDependencySynchronize() or equivalent
    #: PTX instructions).
    CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_STREAM_SERIALIZATION = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_STREAM_SERIALIZATION

    #: Valid for launches. Set
    #: :py:obj:`~.CUlaunchAttributeValue.programmaticEvent` to record the
    #: event. Event recorded through this launch attribute is guaranteed to
    #: only trigger after all block in the associated kernel trigger the
    #: event. A block can trigger the event through PTX launchdep.release
    #: or CUDA builtin function cudaTriggerProgrammaticLaunchCompletion().
    #: A trigger can also be inserted at the beginning of each block's
    #: execution if triggerAtBlockStart is set to non-0. The dependent
    #: launches can choose to wait on the dependency using the programmatic
    #: sync (cudaGridDependencySynchronize() or equivalent PTX
    #: instructions). Note that dependents (including the CPU thread
    #: calling :py:obj:`~.cuEventSynchronize()`) are not guaranteed to
    #: observe the release precisely when it is released. For example,
    #: :py:obj:`~.cuEventSynchronize()` may only observe the event trigger
    #: long after the associated kernel has completed. This recording type
    #: is primarily meant for establishing programmatic dependency between
    #: device tasks. Note also this type of dependency allows, but does not
    #: guarantee, concurrent execution of tasks.
    #:  The event supplied must not be an interprocess or interop event.
    #: The event must disable timing (i.e. must be created with the
    #: :py:obj:`~.CU_EVENT_DISABLE_TIMING` flag set).
    CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_EVENT = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_EVENT

    #: Valid for streams, graph nodes, launches. See
    #: :py:obj:`~.CUlaunchAttributeValue.priority`.
    CU_LAUNCH_ATTRIBUTE_PRIORITY = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_PRIORITY

    #: Valid for streams, graph nodes, launches. See
    #: :py:obj:`~.CUlaunchAttributeValue.memSyncDomainMap`.
    CU_LAUNCH_ATTRIBUTE_MEM_SYNC_DOMAIN_MAP = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_MEM_SYNC_DOMAIN_MAP

    #: Valid for streams, graph nodes, launches. See
    #: :py:obj:`~.CUlaunchAttributeValue.memSyncDomain`.
    CU_LAUNCH_ATTRIBUTE_MEM_SYNC_DOMAIN = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_MEM_SYNC_DOMAIN

    #: Valid for launches. Set
    #: :py:obj:`~.CUlaunchAttributeValue.launchCompletionEvent` to record
    #: the event.
    #:  Nominally, the event is triggered once all blocks of the kernel
    #: have begun execution. Currently this is a best effort. If a kernel B
    #: has a launch completion dependency on a kernel A, B may wait until A
    #: is complete. Alternatively, blocks of B may begin before all blocks
    #: of A have begun, for example if B can claim execution resources
    #: unavailable to A (e.g. they run on different GPUs) or if B is a
    #: higher priority than A. Exercise caution if such an ordering
    #: inversion could lead to deadlock.
    #:  A launch completion event is nominally similar to a programmatic
    #: event with `triggerAtBlockStart` set except that it is not visible
    #: to `cudaGridDependencySynchronize()` and can be used with compute
    #: capability less than 9.0.
    #:  The event supplied must not be an interprocess or interop event.
    #: The event must disable timing (i.e. must be created with the
    #: :py:obj:`~.CU_EVENT_DISABLE_TIMING` flag set).
    CU_LAUNCH_ATTRIBUTE_LAUNCH_COMPLETION_EVENT = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_LAUNCH_COMPLETION_EVENT

    #: Valid for graph nodes, launches. This attribute is graphs-only, and
    #: passing it to a launch in a non-capturing stream will result in an
    #: error.
    #: :py:obj:`~.CUlaunchAttributeValue`::deviceUpdatableKernelNode::deviceUpdatable
    #: can only be set to 0 or 1. Setting the field to 1 indicates that the
    #: corresponding kernel node should be device-updatable. On success, a
    #: handle will be returned via
    #: :py:obj:`~.CUlaunchAttributeValue`::deviceUpdatableKernelNode::devNode
    #: which can be passed to the various device-side update functions to
    #: update the node's kernel parameters from within another kernel. For
    #: more information on the types of device updates that can be made, as
    #: well as the relevant limitations thereof, see
    #: :py:obj:`~.cudaGraphKernelNodeUpdatesApply`.
    #:  Nodes which are device-updatable have additional restrictions
    #: compared to regular kernel nodes. Firstly, device-updatable nodes
    #: cannot be removed from their graph via
    #: :py:obj:`~.cuGraphDestroyNode`. Additionally, once opted-in to this
    #: functionality, a node cannot opt out, and any attempt to set the
    #: deviceUpdatable attribute to 0 will result in an error. Device-
    #: updatable kernel nodes also cannot have their attributes copied
    #: to/from another kernel node via
    #: :py:obj:`~.cuGraphKernelNodeCopyAttributes`. Graphs containing one
    #: or more device-updatable nodes also do not allow multiple
    #: instantiation, and neither the graph nor its instantiated version
    #: can be passed to :py:obj:`~.cuGraphExecUpdate`.
    #:  If a graph contains device-updatable nodes and updates those nodes
    #: from the device from within the graph, the graph must be uploaded
    #: with :py:obj:`~.cuGraphUpload` before it is launched. For such a
    #: graph, if host-side executable graph updates are made to the device-
    #: updatable nodes, the graph must be uploaded before it is launched
    #: again.
    CU_LAUNCH_ATTRIBUTE_DEVICE_UPDATABLE_KERNEL_NODE = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_DEVICE_UPDATABLE_KERNEL_NODE

class CUstreamCaptureStatus(IntEnum):
    """
    Possible stream capture statuses returned by
    :py:obj:`~.cuStreamIsCapturing`
    """

    #: Stream is not capturing
    CU_STREAM_CAPTURE_STATUS_NONE = ccuda.CUstreamCaptureStatus_enum.CU_STREAM_CAPTURE_STATUS_NONE

    #: Stream is actively capturing
    CU_STREAM_CAPTURE_STATUS_ACTIVE = ccuda.CUstreamCaptureStatus_enum.CU_STREAM_CAPTURE_STATUS_ACTIVE

    #: Stream is part of a capture sequence that has been invalidated, but
    #: not terminated
    CU_STREAM_CAPTURE_STATUS_INVALIDATED = ccuda.CUstreamCaptureStatus_enum.CU_STREAM_CAPTURE_STATUS_INVALIDATED

class CUstreamCaptureMode(IntEnum):
    """
    Possible modes for stream capture thread interactions. For more
    details see :py:obj:`~.cuStreamBeginCapture` and
    :py:obj:`~.cuThreadExchangeStreamCaptureMode`
    """
    CU_STREAM_CAPTURE_MODE_GLOBAL = ccuda.CUstreamCaptureMode_enum.CU_STREAM_CAPTURE_MODE_GLOBAL
    CU_STREAM_CAPTURE_MODE_THREAD_LOCAL = ccuda.CUstreamCaptureMode_enum.CU_STREAM_CAPTURE_MODE_THREAD_LOCAL
    CU_STREAM_CAPTURE_MODE_RELAXED = ccuda.CUstreamCaptureMode_enum.CU_STREAM_CAPTURE_MODE_RELAXED

class CUdriverProcAddress_flags(IntEnum):
    """
    Flags to specify search options. For more details see
    :py:obj:`~.cuGetProcAddress`
    """

    #: Default search mode for driver symbols.
    CU_GET_PROC_ADDRESS_DEFAULT = ccuda.CUdriverProcAddress_flags_enum.CU_GET_PROC_ADDRESS_DEFAULT

    #: Search for legacy versions of driver symbols.
    CU_GET_PROC_ADDRESS_LEGACY_STREAM = ccuda.CUdriverProcAddress_flags_enum.CU_GET_PROC_ADDRESS_LEGACY_STREAM

    #: Search for per-thread versions of driver symbols.
    CU_GET_PROC_ADDRESS_PER_THREAD_DEFAULT_STREAM = ccuda.CUdriverProcAddress_flags_enum.CU_GET_PROC_ADDRESS_PER_THREAD_DEFAULT_STREAM

class CUdriverProcAddressQueryResult(IntEnum):
    """
    Flags to indicate search status. For more details see
    :py:obj:`~.cuGetProcAddress`
    """

    #: Symbol was succesfully found
    CU_GET_PROC_ADDRESS_SUCCESS = ccuda.CUdriverProcAddressQueryResult_enum.CU_GET_PROC_ADDRESS_SUCCESS

    #: Symbol was not found in search
    CU_GET_PROC_ADDRESS_SYMBOL_NOT_FOUND = ccuda.CUdriverProcAddressQueryResult_enum.CU_GET_PROC_ADDRESS_SYMBOL_NOT_FOUND

    #: Symbol was found but version supplied was not sufficient
    CU_GET_PROC_ADDRESS_VERSION_NOT_SUFFICIENT = ccuda.CUdriverProcAddressQueryResult_enum.CU_GET_PROC_ADDRESS_VERSION_NOT_SUFFICIENT

class CUexecAffinityType(IntEnum):
    """
    Execution Affinity Types
    """

    #: Create a context with limited SMs.
    CU_EXEC_AFFINITY_TYPE_SM_COUNT = ccuda.CUexecAffinityType_enum.CU_EXEC_AFFINITY_TYPE_SM_COUNT
    CU_EXEC_AFFINITY_TYPE_MAX = ccuda.CUexecAffinityType_enum.CU_EXEC_AFFINITY_TYPE_MAX

class CUlibraryOption(IntEnum):
    """
    Library options to be specified with
    :py:obj:`~.cuLibraryLoadData()` or
    :py:obj:`~.cuLibraryLoadFromFile()`
    """
    CU_LIBRARY_HOST_UNIVERSAL_FUNCTION_AND_DATA_TABLE = ccuda.CUlibraryOption_enum.CU_LIBRARY_HOST_UNIVERSAL_FUNCTION_AND_DATA_TABLE

    #: Specifes that the argument `code` passed to
    #: :py:obj:`~.cuLibraryLoadData()` will be preserved. Specifying this
    #: option will let the driver know that `code` can be accessed at any
    #: point until :py:obj:`~.cuLibraryUnload()`. The default behavior is
    #: for the driver to allocate and maintain its own copy of `code`. Note
    #: that this is only a memory usage optimization hint and the driver
    #: can choose to ignore it if required. Specifying this option with
    #: :py:obj:`~.cuLibraryLoadFromFile()` is invalid and will return
    #: :py:obj:`~.CUDA_ERROR_INVALID_VALUE`.
    CU_LIBRARY_BINARY_IS_PRESERVED = ccuda.CUlibraryOption_enum.CU_LIBRARY_BINARY_IS_PRESERVED
    CU_LIBRARY_NUM_OPTIONS = ccuda.CUlibraryOption_enum.CU_LIBRARY_NUM_OPTIONS

class CUresult(IntEnum):
    """
    Error codes
    """

    #: The API call returned with no errors. In the case of query calls,
    #: this also means that the operation being queried is complete (see
    #: :py:obj:`~.cuEventQuery()` and :py:obj:`~.cuStreamQuery()`).
    CUDA_SUCCESS = ccuda.cudaError_enum.CUDA_SUCCESS

    #: This indicates that one or more of the parameters passed to the API
    #: call is not within an acceptable range of values.
    CUDA_ERROR_INVALID_VALUE = ccuda.cudaError_enum.CUDA_ERROR_INVALID_VALUE

    #: The API call failed because it was unable to allocate enough memory
    #: or other resources to perform the requested operation.
    CUDA_ERROR_OUT_OF_MEMORY = ccuda.cudaError_enum.CUDA_ERROR_OUT_OF_MEMORY

    #: This indicates that the CUDA driver has not been initialized with
    #: :py:obj:`~.cuInit()` or that initialization has failed.
    CUDA_ERROR_NOT_INITIALIZED = ccuda.cudaError_enum.CUDA_ERROR_NOT_INITIALIZED

    #: This indicates that the CUDA driver is in the process of shutting
    #: down.
    CUDA_ERROR_DEINITIALIZED = ccuda.cudaError_enum.CUDA_ERROR_DEINITIALIZED

    #: This indicates profiler is not initialized for this run. This can
    #: happen when the application is running with external profiling tools
    #: like visual profiler.
    CUDA_ERROR_PROFILER_DISABLED = ccuda.cudaError_enum.CUDA_ERROR_PROFILER_DISABLED

    #: [Deprecated]
    CUDA_ERROR_PROFILER_NOT_INITIALIZED = ccuda.cudaError_enum.CUDA_ERROR_PROFILER_NOT_INITIALIZED

    #: [Deprecated]
    CUDA_ERROR_PROFILER_ALREADY_STARTED = ccuda.cudaError_enum.CUDA_ERROR_PROFILER_ALREADY_STARTED

    #: [Deprecated]
    CUDA_ERROR_PROFILER_ALREADY_STOPPED = ccuda.cudaError_enum.CUDA_ERROR_PROFILER_ALREADY_STOPPED

    #: This indicates that the CUDA driver that the application has loaded
    #: is a stub library. Applications that run with the stub rather than a
    #: real driver loaded will result in CUDA API returning this error.
    CUDA_ERROR_STUB_LIBRARY = ccuda.cudaError_enum.CUDA_ERROR_STUB_LIBRARY

    #: This indicates that requested CUDA device is unavailable at the
    #: current time. Devices are often unavailable due to use of
    #: :py:obj:`~.CU_COMPUTEMODE_EXCLUSIVE_PROCESS` or
    #: :py:obj:`~.CU_COMPUTEMODE_PROHIBITED`.
    CUDA_ERROR_DEVICE_UNAVAILABLE = ccuda.cudaError_enum.CUDA_ERROR_DEVICE_UNAVAILABLE

    #: This indicates that no CUDA-capable devices were detected by the
    #: installed CUDA driver.
    CUDA_ERROR_NO_DEVICE = ccuda.cudaError_enum.CUDA_ERROR_NO_DEVICE

    #: This indicates that the device ordinal supplied by the user does not
    #: correspond to a valid CUDA device or that the action requested is
    #: invalid for the specified device.
    CUDA_ERROR_INVALID_DEVICE = ccuda.cudaError_enum.CUDA_ERROR_INVALID_DEVICE

    #: This error indicates that the Grid license is not applied.
    CUDA_ERROR_DEVICE_NOT_LICENSED = ccuda.cudaError_enum.CUDA_ERROR_DEVICE_NOT_LICENSED

    #: This indicates that the device kernel image is invalid. This can
    #: also indicate an invalid CUDA module.
    CUDA_ERROR_INVALID_IMAGE = ccuda.cudaError_enum.CUDA_ERROR_INVALID_IMAGE

    #: This most frequently indicates that there is no context bound to the
    #: current thread. This can also be returned if the context passed to
    #: an API call is not a valid handle (such as a context that has had
    #: :py:obj:`~.cuCtxDestroy()` invoked on it). This can also be returned
    #: if a user mixes different API versions (i.e. 3010 context with 3020
    #: API calls). See :py:obj:`~.cuCtxGetApiVersion()` for more details.
    #: This can also be returned if the green context passed to an API call
    #: was not converted to a :py:obj:`~.CUcontext` using
    #: :py:obj:`~.cuCtxFromGreenCtx` API.
    CUDA_ERROR_INVALID_CONTEXT = ccuda.cudaError_enum.CUDA_ERROR_INVALID_CONTEXT

    #: This indicated that the context being supplied as a parameter to the
    #: API call was already the active context. [Deprecated]
    CUDA_ERROR_CONTEXT_ALREADY_CURRENT = ccuda.cudaError_enum.CUDA_ERROR_CONTEXT_ALREADY_CURRENT

    #: This indicates that a map or register operation has failed.
    CUDA_ERROR_MAP_FAILED = ccuda.cudaError_enum.CUDA_ERROR_MAP_FAILED

    #: This indicates that an unmap or unregister operation has failed.
    CUDA_ERROR_UNMAP_FAILED = ccuda.cudaError_enum.CUDA_ERROR_UNMAP_FAILED

    #: This indicates that the specified array is currently mapped and thus
    #: cannot be destroyed.
    CUDA_ERROR_ARRAY_IS_MAPPED = ccuda.cudaError_enum.CUDA_ERROR_ARRAY_IS_MAPPED

    #: This indicates that the resource is already mapped.
    CUDA_ERROR_ALREADY_MAPPED = ccuda.cudaError_enum.CUDA_ERROR_ALREADY_MAPPED

    #: This indicates that there is no kernel image available that is
    #: suitable for the device. This can occur when a user specifies code
    #: generation options for a particular CUDA source file that do not
    #: include the corresponding device configuration.
    CUDA_ERROR_NO_BINARY_FOR_GPU = ccuda.cudaError_enum.CUDA_ERROR_NO_BINARY_FOR_GPU

    #: This indicates that a resource has already been acquired.
    CUDA_ERROR_ALREADY_ACQUIRED = ccuda.cudaError_enum.CUDA_ERROR_ALREADY_ACQUIRED

    #: This indicates that a resource is not mapped.
    CUDA_ERROR_NOT_MAPPED = ccuda.cudaError_enum.CUDA_ERROR_NOT_MAPPED

    #: This indicates that a mapped resource is not available for access as
    #: an array.
    CUDA_ERROR_NOT_MAPPED_AS_ARRAY = ccuda.cudaError_enum.CUDA_ERROR_NOT_MAPPED_AS_ARRAY

    #: This indicates that a mapped resource is not available for access as
    #: a pointer.
    CUDA_ERROR_NOT_MAPPED_AS_POINTER = ccuda.cudaError_enum.CUDA_ERROR_NOT_MAPPED_AS_POINTER

    #: This indicates that an uncorrectable ECC error was detected during
    #: execution.
    CUDA_ERROR_ECC_UNCORRECTABLE = ccuda.cudaError_enum.CUDA_ERROR_ECC_UNCORRECTABLE

    #: This indicates that the :py:obj:`~.CUlimit` passed to the API call
    #: is not supported by the active device.
    CUDA_ERROR_UNSUPPORTED_LIMIT = ccuda.cudaError_enum.CUDA_ERROR_UNSUPPORTED_LIMIT

    #: This indicates that the :py:obj:`~.CUcontext` passed to the API call
    #: can only be bound to a single CPU thread at a time but is already
    #: bound to a CPU thread.
    CUDA_ERROR_CONTEXT_ALREADY_IN_USE = ccuda.cudaError_enum.CUDA_ERROR_CONTEXT_ALREADY_IN_USE

    #: This indicates that peer access is not supported across the given
    #: devices.
    CUDA_ERROR_PEER_ACCESS_UNSUPPORTED = ccuda.cudaError_enum.CUDA_ERROR_PEER_ACCESS_UNSUPPORTED

    #: This indicates that a PTX JIT compilation failed.
    CUDA_ERROR_INVALID_PTX = ccuda.cudaError_enum.CUDA_ERROR_INVALID_PTX

    #: This indicates an error with OpenGL or DirectX context.
    CUDA_ERROR_INVALID_GRAPHICS_CONTEXT = ccuda.cudaError_enum.CUDA_ERROR_INVALID_GRAPHICS_CONTEXT

    #: This indicates that an uncorrectable NVLink error was detected
    #: during the execution.
    CUDA_ERROR_NVLINK_UNCORRECTABLE = ccuda.cudaError_enum.CUDA_ERROR_NVLINK_UNCORRECTABLE

    #: This indicates that the PTX JIT compiler library was not found.
    CUDA_ERROR_JIT_COMPILER_NOT_FOUND = ccuda.cudaError_enum.CUDA_ERROR_JIT_COMPILER_NOT_FOUND

    #: This indicates that the provided PTX was compiled with an
    #: unsupported toolchain.
    CUDA_ERROR_UNSUPPORTED_PTX_VERSION = ccuda.cudaError_enum.CUDA_ERROR_UNSUPPORTED_PTX_VERSION

    #: This indicates that the PTX JIT compilation was disabled.
    CUDA_ERROR_JIT_COMPILATION_DISABLED = ccuda.cudaError_enum.CUDA_ERROR_JIT_COMPILATION_DISABLED

    #: This indicates that the :py:obj:`~.CUexecAffinityType` passed to the
    #: API call is not supported by the active device.
    CUDA_ERROR_UNSUPPORTED_EXEC_AFFINITY = ccuda.cudaError_enum.CUDA_ERROR_UNSUPPORTED_EXEC_AFFINITY

    #: This indicates that the code to be compiled by the PTX JIT contains
    #: unsupported call to cudaDeviceSynchronize.
    CUDA_ERROR_UNSUPPORTED_DEVSIDE_SYNC = ccuda.cudaError_enum.CUDA_ERROR_UNSUPPORTED_DEVSIDE_SYNC

    #: This indicates that the device kernel source is invalid. This
    #: includes compilation/linker errors encountered in device code or
    #: user error.
    CUDA_ERROR_INVALID_SOURCE = ccuda.cudaError_enum.CUDA_ERROR_INVALID_SOURCE

    #: This indicates that the file specified was not found.
    CUDA_ERROR_FILE_NOT_FOUND = ccuda.cudaError_enum.CUDA_ERROR_FILE_NOT_FOUND

    #: This indicates that a link to a shared object failed to resolve.
    CUDA_ERROR_SHARED_OBJECT_SYMBOL_NOT_FOUND = ccuda.cudaError_enum.CUDA_ERROR_SHARED_OBJECT_SYMBOL_NOT_FOUND

    #: This indicates that initialization of a shared object failed.
    CUDA_ERROR_SHARED_OBJECT_INIT_FAILED = ccuda.cudaError_enum.CUDA_ERROR_SHARED_OBJECT_INIT_FAILED

    #: This indicates that an OS call failed.
    CUDA_ERROR_OPERATING_SYSTEM = ccuda.cudaError_enum.CUDA_ERROR_OPERATING_SYSTEM

    #: This indicates that a resource handle passed to the API call was not
    #: valid. Resource handles are opaque types like :py:obj:`~.CUstream`
    #: and :py:obj:`~.CUevent`.
    CUDA_ERROR_INVALID_HANDLE = ccuda.cudaError_enum.CUDA_ERROR_INVALID_HANDLE

    #: This indicates that a resource required by the API call is not in a
    #: valid state to perform the requested operation.
    CUDA_ERROR_ILLEGAL_STATE = ccuda.cudaError_enum.CUDA_ERROR_ILLEGAL_STATE

    #: This indicates an attempt was made to introspect an object in a way
    #: that would discard semantically important information. This is
    #: either due to the object using funtionality newer than the API
    #: version used to introspect it or omission of optional return
    #: arguments.
    CUDA_ERROR_LOSSY_QUERY = ccuda.cudaError_enum.CUDA_ERROR_LOSSY_QUERY

    #: This indicates that a named symbol was not found. Examples of
    #: symbols are global/constant variable names, driver function names,
    #: texture names, and surface names.
    CUDA_ERROR_NOT_FOUND = ccuda.cudaError_enum.CUDA_ERROR_NOT_FOUND

    #: This indicates that asynchronous operations issued previously have
    #: not completed yet. This result is not actually an error, but must be
    #: indicated differently than :py:obj:`~.CUDA_SUCCESS` (which indicates
    #: completion). Calls that may return this value include
    #: :py:obj:`~.cuEventQuery()` and :py:obj:`~.cuStreamQuery()`.
    CUDA_ERROR_NOT_READY = ccuda.cudaError_enum.CUDA_ERROR_NOT_READY

    #: While executing a kernel, the device encountered a load or store
    #: instruction on an invalid memory address. This leaves the process in
    #: an inconsistent state and any further CUDA work will return the same
    #: error. To continue using CUDA, the process must be terminated and
    #: relaunched.
    CUDA_ERROR_ILLEGAL_ADDRESS = ccuda.cudaError_enum.CUDA_ERROR_ILLEGAL_ADDRESS

    #: This indicates that a launch did not occur because it did not have
    #: appropriate resources. This error usually indicates that the user
    #: has attempted to pass too many arguments to the device kernel, or
    #: the kernel launch specifies too many threads for the kernel's
    #: register count. Passing arguments of the wrong size (i.e. a 64-bit
    #: pointer when a 32-bit int is expected) is equivalent to passing too
    #: many arguments and can also result in this error.
    CUDA_ERROR_LAUNCH_OUT_OF_RESOURCES = ccuda.cudaError_enum.CUDA_ERROR_LAUNCH_OUT_OF_RESOURCES

    #: This indicates that the device kernel took too long to execute. This
    #: can only occur if timeouts are enabled - see the device attribute
    #: :py:obj:`~.CU_DEVICE_ATTRIBUTE_KERNEL_EXEC_TIMEOUT` for more
    #: information. This leaves the process in an inconsistent state and
    #: any further CUDA work will return the same error. To continue using
    #: CUDA, the process must be terminated and relaunched.
    CUDA_ERROR_LAUNCH_TIMEOUT = ccuda.cudaError_enum.CUDA_ERROR_LAUNCH_TIMEOUT

    #: This error indicates a kernel launch that uses an incompatible
    #: texturing mode.
    CUDA_ERROR_LAUNCH_INCOMPATIBLE_TEXTURING = ccuda.cudaError_enum.CUDA_ERROR_LAUNCH_INCOMPATIBLE_TEXTURING

    #: This error indicates that a call to
    #: :py:obj:`~.cuCtxEnablePeerAccess()` is trying to re-enable peer
    #: access to a context which has already had peer access to it enabled.
    CUDA_ERROR_PEER_ACCESS_ALREADY_ENABLED = ccuda.cudaError_enum.CUDA_ERROR_PEER_ACCESS_ALREADY_ENABLED

    #: This error indicates that :py:obj:`~.cuCtxDisablePeerAccess()` is
    #: trying to disable peer access which has not been enabled yet via
    #: :py:obj:`~.cuCtxEnablePeerAccess()`.
    CUDA_ERROR_PEER_ACCESS_NOT_ENABLED = ccuda.cudaError_enum.CUDA_ERROR_PEER_ACCESS_NOT_ENABLED

    #: This error indicates that the primary context for the specified
    #: device has already been initialized.
    CUDA_ERROR_PRIMARY_CONTEXT_ACTIVE = ccuda.cudaError_enum.CUDA_ERROR_PRIMARY_CONTEXT_ACTIVE

    #: This error indicates that the context current to the calling thread
    #: has been destroyed using :py:obj:`~.cuCtxDestroy`, or is a primary
    #: context which has not yet been initialized.
    CUDA_ERROR_CONTEXT_IS_DESTROYED = ccuda.cudaError_enum.CUDA_ERROR_CONTEXT_IS_DESTROYED

    #: A device-side assert triggered during kernel execution. The context
    #: cannot be used anymore, and must be destroyed. All existing device
    #: memory allocations from this context are invalid and must be
    #: reconstructed if the program is to continue using CUDA.
    CUDA_ERROR_ASSERT = ccuda.cudaError_enum.CUDA_ERROR_ASSERT

    #: This error indicates that the hardware resources required to enable
    #: peer access have been exhausted for one or more of the devices
    #: passed to :py:obj:`~.cuCtxEnablePeerAccess()`.
    CUDA_ERROR_TOO_MANY_PEERS = ccuda.cudaError_enum.CUDA_ERROR_TOO_MANY_PEERS

    #: This error indicates that the memory range passed to
    #: :py:obj:`~.cuMemHostRegister()` has already been registered.
    CUDA_ERROR_HOST_MEMORY_ALREADY_REGISTERED = ccuda.cudaError_enum.CUDA_ERROR_HOST_MEMORY_ALREADY_REGISTERED

    #: This error indicates that the pointer passed to
    #: :py:obj:`~.cuMemHostUnregister()` does not correspond to any
    #: currently registered memory region.
    CUDA_ERROR_HOST_MEMORY_NOT_REGISTERED = ccuda.cudaError_enum.CUDA_ERROR_HOST_MEMORY_NOT_REGISTERED

    #: While executing a kernel, the device encountered a stack error. This
    #: can be due to stack corruption or exceeding the stack size limit.
    #: This leaves the process in an inconsistent state and any further
    #: CUDA work will return the same error. To continue using CUDA, the
    #: process must be terminated and relaunched.
    CUDA_ERROR_HARDWARE_STACK_ERROR = ccuda.cudaError_enum.CUDA_ERROR_HARDWARE_STACK_ERROR

    #: While executing a kernel, the device encountered an illegal
    #: instruction. This leaves the process in an inconsistent state and
    #: any further CUDA work will return the same error. To continue using
    #: CUDA, the process must be terminated and relaunched.
    CUDA_ERROR_ILLEGAL_INSTRUCTION = ccuda.cudaError_enum.CUDA_ERROR_ILLEGAL_INSTRUCTION

    #: While executing a kernel, the device encountered a load or store
    #: instruction on a memory address which is not aligned. This leaves
    #: the process in an inconsistent state and any further CUDA work will
    #: return the same error. To continue using CUDA, the process must be
    #: terminated and relaunched.
    CUDA_ERROR_MISALIGNED_ADDRESS = ccuda.cudaError_enum.CUDA_ERROR_MISALIGNED_ADDRESS

    #: While executing a kernel, the device encountered an instruction
    #: which can only operate on memory locations in certain address spaces
    #: (global, shared, or local), but was supplied a memory address not
    #: belonging to an allowed address space. This leaves the process in an
    #: inconsistent state and any further CUDA work will return the same
    #: error. To continue using CUDA, the process must be terminated and
    #: relaunched.
    CUDA_ERROR_INVALID_ADDRESS_SPACE = ccuda.cudaError_enum.CUDA_ERROR_INVALID_ADDRESS_SPACE

    #: While executing a kernel, the device program counter wrapped its
    #: address space. This leaves the process in an inconsistent state and
    #: any further CUDA work will return the same error. To continue using
    #: CUDA, the process must be terminated and relaunched.
    CUDA_ERROR_INVALID_PC = ccuda.cudaError_enum.CUDA_ERROR_INVALID_PC

    #: An exception occurred on the device while executing a kernel. Common
    #: causes include dereferencing an invalid device pointer and accessing
    #: out of bounds shared memory. Less common cases can be system
    #: specific - more information about these cases can be found in the
    #: system specific user guide. This leaves the process in an
    #: inconsistent state and any further CUDA work will return the same
    #: error. To continue using CUDA, the process must be terminated and
    #: relaunched.
    CUDA_ERROR_LAUNCH_FAILED = ccuda.cudaError_enum.CUDA_ERROR_LAUNCH_FAILED

    #: This error indicates that the number of blocks launched per grid for
    #: a kernel that was launched via either
    #: :py:obj:`~.cuLaunchCooperativeKernel` or
    #: :py:obj:`~.cuLaunchCooperativeKernelMultiDevice` exceeds the maximum
    #: number of blocks as allowed by
    #: :py:obj:`~.cuOccupancyMaxActiveBlocksPerMultiprocessor` or
    #: :py:obj:`~.cuOccupancyMaxActiveBlocksPerMultiprocessorWithFlags`
    #: times the number of multiprocessors as specified by the device
    #: attribute :py:obj:`~.CU_DEVICE_ATTRIBUTE_MULTIPROCESSOR_COUNT`.
    CUDA_ERROR_COOPERATIVE_LAUNCH_TOO_LARGE = ccuda.cudaError_enum.CUDA_ERROR_COOPERATIVE_LAUNCH_TOO_LARGE

    #: This error indicates that the attempted operation is not permitted.
    CUDA_ERROR_NOT_PERMITTED = ccuda.cudaError_enum.CUDA_ERROR_NOT_PERMITTED

    #: This error indicates that the attempted operation is not supported
    #: on the current system or device.
    CUDA_ERROR_NOT_SUPPORTED = ccuda.cudaError_enum.CUDA_ERROR_NOT_SUPPORTED

    #: This error indicates that the system is not yet ready to start any
    #: CUDA work. To continue using CUDA, verify the system configuration
    #: is in a valid state and all required driver daemons are actively
    #: running. More information about this error can be found in the
    #: system specific user guide.
    CUDA_ERROR_SYSTEM_NOT_READY = ccuda.cudaError_enum.CUDA_ERROR_SYSTEM_NOT_READY

    #: This error indicates that there is a mismatch between the versions
    #: of the display driver and the CUDA driver. Refer to the
    #: compatibility documentation for supported versions.
    CUDA_ERROR_SYSTEM_DRIVER_MISMATCH = ccuda.cudaError_enum.CUDA_ERROR_SYSTEM_DRIVER_MISMATCH

    #: This error indicates that the system was upgraded to run with
    #: forward compatibility but the visible hardware detected by CUDA does
    #: not support this configuration. Refer to the compatibility
    #: documentation for the supported hardware matrix or ensure that only
    #: supported hardware is visible during initialization via the
    #: CUDA_VISIBLE_DEVICES environment variable.
    CUDA_ERROR_COMPAT_NOT_SUPPORTED_ON_DEVICE = ccuda.cudaError_enum.CUDA_ERROR_COMPAT_NOT_SUPPORTED_ON_DEVICE

    #: This error indicates that the MPS client failed to connect to the
    #: MPS control daemon or the MPS server.
    CUDA_ERROR_MPS_CONNECTION_FAILED = ccuda.cudaError_enum.CUDA_ERROR_MPS_CONNECTION_FAILED

    #: This error indicates that the remote procedural call between the MPS
    #: server and the MPS client failed.
    CUDA_ERROR_MPS_RPC_FAILURE = ccuda.cudaError_enum.CUDA_ERROR_MPS_RPC_FAILURE

    #: This error indicates that the MPS server is not ready to accept new
    #: MPS client requests. This error can be returned when the MPS server
    #: is in the process of recovering from a fatal failure.
    CUDA_ERROR_MPS_SERVER_NOT_READY = ccuda.cudaError_enum.CUDA_ERROR_MPS_SERVER_NOT_READY

    #: This error indicates that the hardware resources required to create
    #: MPS client have been exhausted.
    CUDA_ERROR_MPS_MAX_CLIENTS_REACHED = ccuda.cudaError_enum.CUDA_ERROR_MPS_MAX_CLIENTS_REACHED

    #: This error indicates the the hardware resources required to support
    #: device connections have been exhausted.
    CUDA_ERROR_MPS_MAX_CONNECTIONS_REACHED = ccuda.cudaError_enum.CUDA_ERROR_MPS_MAX_CONNECTIONS_REACHED

    #: This error indicates that the MPS client has been terminated by the
    #: server. To continue using CUDA, the process must be terminated and
    #: relaunched.
    CUDA_ERROR_MPS_CLIENT_TERMINATED = ccuda.cudaError_enum.CUDA_ERROR_MPS_CLIENT_TERMINATED

    #: This error indicates that the module is using CUDA Dynamic
    #: Parallelism, but the current configuration, like MPS, does not
    #: support it.
    CUDA_ERROR_CDP_NOT_SUPPORTED = ccuda.cudaError_enum.CUDA_ERROR_CDP_NOT_SUPPORTED

    #: This error indicates that a module contains an unsupported
    #: interaction between different versions of CUDA Dynamic Parallelism.
    CUDA_ERROR_CDP_VERSION_MISMATCH = ccuda.cudaError_enum.CUDA_ERROR_CDP_VERSION_MISMATCH

    #: This error indicates that the operation is not permitted when the
    #: stream is capturing.
    CUDA_ERROR_STREAM_CAPTURE_UNSUPPORTED = ccuda.cudaError_enum.CUDA_ERROR_STREAM_CAPTURE_UNSUPPORTED

    #: This error indicates that the current capture sequence on the stream
    #: has been invalidated due to a previous error.
    CUDA_ERROR_STREAM_CAPTURE_INVALIDATED = ccuda.cudaError_enum.CUDA_ERROR_STREAM_CAPTURE_INVALIDATED

    #: This error indicates that the operation would have resulted in a
    #: merge of two independent capture sequences.
    CUDA_ERROR_STREAM_CAPTURE_MERGE = ccuda.cudaError_enum.CUDA_ERROR_STREAM_CAPTURE_MERGE

    #: This error indicates that the capture was not initiated in this
    #: stream.
    CUDA_ERROR_STREAM_CAPTURE_UNMATCHED = ccuda.cudaError_enum.CUDA_ERROR_STREAM_CAPTURE_UNMATCHED

    #: This error indicates that the capture sequence contains a fork that
    #: was not joined to the primary stream.
    CUDA_ERROR_STREAM_CAPTURE_UNJOINED = ccuda.cudaError_enum.CUDA_ERROR_STREAM_CAPTURE_UNJOINED

    #: This error indicates that a dependency would have been created which
    #: crosses the capture sequence boundary. Only implicit in-stream
    #: ordering dependencies are allowed to cross the boundary.
    CUDA_ERROR_STREAM_CAPTURE_ISOLATION = ccuda.cudaError_enum.CUDA_ERROR_STREAM_CAPTURE_ISOLATION

    #: This error indicates a disallowed implicit dependency on a current
    #: capture sequence from cudaStreamLegacy.
    CUDA_ERROR_STREAM_CAPTURE_IMPLICIT = ccuda.cudaError_enum.CUDA_ERROR_STREAM_CAPTURE_IMPLICIT

    #: This error indicates that the operation is not permitted on an event
    #: which was last recorded in a capturing stream.
    CUDA_ERROR_CAPTURED_EVENT = ccuda.cudaError_enum.CUDA_ERROR_CAPTURED_EVENT

    #: A stream capture sequence not initiated with the
    #: :py:obj:`~.CU_STREAM_CAPTURE_MODE_RELAXED` argument to
    #: :py:obj:`~.cuStreamBeginCapture` was passed to
    #: :py:obj:`~.cuStreamEndCapture` in a different thread.
    CUDA_ERROR_STREAM_CAPTURE_WRONG_THREAD = ccuda.cudaError_enum.CUDA_ERROR_STREAM_CAPTURE_WRONG_THREAD

    #: This error indicates that the timeout specified for the wait
    #: operation has lapsed.
    CUDA_ERROR_TIMEOUT = ccuda.cudaError_enum.CUDA_ERROR_TIMEOUT

    #: This error indicates that the graph update was not performed because
    #: it included changes which violated constraints specific to
    #: instantiated graph update.
    CUDA_ERROR_GRAPH_EXEC_UPDATE_FAILURE = ccuda.cudaError_enum.CUDA_ERROR_GRAPH_EXEC_UPDATE_FAILURE

    #: This indicates that an async error has occurred in a device outside
    #: of CUDA. If CUDA was waiting for an external device's signal before
    #: consuming shared data, the external device signaled an error
    #: indicating that the data is not valid for consumption. This leaves
    #: the process in an inconsistent state and any further CUDA work will
    #: return the same error. To continue using CUDA, the process must be
    #: terminated and relaunched.
    CUDA_ERROR_EXTERNAL_DEVICE = ccuda.cudaError_enum.CUDA_ERROR_EXTERNAL_DEVICE

    #: Indicates a kernel launch error due to cluster misconfiguration.
    CUDA_ERROR_INVALID_CLUSTER_SIZE = ccuda.cudaError_enum.CUDA_ERROR_INVALID_CLUSTER_SIZE

    #: Indiciates a function handle is not loaded when calling an API that
    #: requires a loaded function.
    CUDA_ERROR_FUNCTION_NOT_LOADED = ccuda.cudaError_enum.CUDA_ERROR_FUNCTION_NOT_LOADED

    #: This error indicates one or more resources passed in are not valid
    #: resource types for the operation.
    CUDA_ERROR_INVALID_RESOURCE_TYPE = ccuda.cudaError_enum.CUDA_ERROR_INVALID_RESOURCE_TYPE

    #: This error indicates one or more resources are insufficient or non-
    #: applicable for the operation.
    CUDA_ERROR_INVALID_RESOURCE_CONFIGURATION = ccuda.cudaError_enum.CUDA_ERROR_INVALID_RESOURCE_CONFIGURATION

    #: This indicates that an unknown internal error has occurred.
    CUDA_ERROR_UNKNOWN = ccuda.cudaError_enum.CUDA_ERROR_UNKNOWN

class CUdevice_P2PAttribute(IntEnum):
    """
    P2P Attributes
    """

    #: A relative value indicating the performance of the link between two
    #: devices
    CU_DEVICE_P2P_ATTRIBUTE_PERFORMANCE_RANK = ccuda.CUdevice_P2PAttribute_enum.CU_DEVICE_P2P_ATTRIBUTE_PERFORMANCE_RANK

    #: P2P Access is enable
    CU_DEVICE_P2P_ATTRIBUTE_ACCESS_SUPPORTED = ccuda.CUdevice_P2PAttribute_enum.CU_DEVICE_P2P_ATTRIBUTE_ACCESS_SUPPORTED

    #: Atomic operation over the link supported
    CU_DEVICE_P2P_ATTRIBUTE_NATIVE_ATOMIC_SUPPORTED = ccuda.CUdevice_P2PAttribute_enum.CU_DEVICE_P2P_ATTRIBUTE_NATIVE_ATOMIC_SUPPORTED

    #: [Deprecated]
    CU_DEVICE_P2P_ATTRIBUTE_ACCESS_ACCESS_SUPPORTED = ccuda.CUdevice_P2PAttribute_enum.CU_DEVICE_P2P_ATTRIBUTE_ACCESS_ACCESS_SUPPORTED

    #: Accessing CUDA arrays over the link supported
    CU_DEVICE_P2P_ATTRIBUTE_CUDA_ARRAY_ACCESS_SUPPORTED = ccuda.CUdevice_P2PAttribute_enum.CU_DEVICE_P2P_ATTRIBUTE_CUDA_ARRAY_ACCESS_SUPPORTED

class CUresourceViewFormat(IntEnum):
    """
    Resource view format
    """

    #: No resource view format (use underlying resource format)
    CU_RES_VIEW_FORMAT_NONE = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_NONE

    #: 1 channel unsigned 8-bit integers
    CU_RES_VIEW_FORMAT_UINT_1X8 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UINT_1X8

    #: 2 channel unsigned 8-bit integers
    CU_RES_VIEW_FORMAT_UINT_2X8 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UINT_2X8

    #: 4 channel unsigned 8-bit integers
    CU_RES_VIEW_FORMAT_UINT_4X8 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UINT_4X8

    #: 1 channel signed 8-bit integers
    CU_RES_VIEW_FORMAT_SINT_1X8 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SINT_1X8

    #: 2 channel signed 8-bit integers
    CU_RES_VIEW_FORMAT_SINT_2X8 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SINT_2X8

    #: 4 channel signed 8-bit integers
    CU_RES_VIEW_FORMAT_SINT_4X8 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SINT_4X8

    #: 1 channel unsigned 16-bit integers
    CU_RES_VIEW_FORMAT_UINT_1X16 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UINT_1X16

    #: 2 channel unsigned 16-bit integers
    CU_RES_VIEW_FORMAT_UINT_2X16 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UINT_2X16

    #: 4 channel unsigned 16-bit integers
    CU_RES_VIEW_FORMAT_UINT_4X16 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UINT_4X16

    #: 1 channel signed 16-bit integers
    CU_RES_VIEW_FORMAT_SINT_1X16 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SINT_1X16

    #: 2 channel signed 16-bit integers
    CU_RES_VIEW_FORMAT_SINT_2X16 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SINT_2X16

    #: 4 channel signed 16-bit integers
    CU_RES_VIEW_FORMAT_SINT_4X16 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SINT_4X16

    #: 1 channel unsigned 32-bit integers
    CU_RES_VIEW_FORMAT_UINT_1X32 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UINT_1X32

    #: 2 channel unsigned 32-bit integers
    CU_RES_VIEW_FORMAT_UINT_2X32 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UINT_2X32

    #: 4 channel unsigned 32-bit integers
    CU_RES_VIEW_FORMAT_UINT_4X32 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UINT_4X32

    #: 1 channel signed 32-bit integers
    CU_RES_VIEW_FORMAT_SINT_1X32 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SINT_1X32

    #: 2 channel signed 32-bit integers
    CU_RES_VIEW_FORMAT_SINT_2X32 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SINT_2X32

    #: 4 channel signed 32-bit integers
    CU_RES_VIEW_FORMAT_SINT_4X32 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SINT_4X32

    #: 1 channel 16-bit floating point
    CU_RES_VIEW_FORMAT_FLOAT_1X16 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_FLOAT_1X16

    #: 2 channel 16-bit floating point
    CU_RES_VIEW_FORMAT_FLOAT_2X16 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_FLOAT_2X16

    #: 4 channel 16-bit floating point
    CU_RES_VIEW_FORMAT_FLOAT_4X16 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_FLOAT_4X16

    #: 1 channel 32-bit floating point
    CU_RES_VIEW_FORMAT_FLOAT_1X32 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_FLOAT_1X32

    #: 2 channel 32-bit floating point
    CU_RES_VIEW_FORMAT_FLOAT_2X32 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_FLOAT_2X32

    #: 4 channel 32-bit floating point
    CU_RES_VIEW_FORMAT_FLOAT_4X32 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_FLOAT_4X32

    #: Block compressed 1
    CU_RES_VIEW_FORMAT_UNSIGNED_BC1 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UNSIGNED_BC1

    #: Block compressed 2
    CU_RES_VIEW_FORMAT_UNSIGNED_BC2 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UNSIGNED_BC2

    #: Block compressed 3
    CU_RES_VIEW_FORMAT_UNSIGNED_BC3 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UNSIGNED_BC3

    #: Block compressed 4 unsigned
    CU_RES_VIEW_FORMAT_UNSIGNED_BC4 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UNSIGNED_BC4

    #: Block compressed 4 signed
    CU_RES_VIEW_FORMAT_SIGNED_BC4 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SIGNED_BC4

    #: Block compressed 5 unsigned
    CU_RES_VIEW_FORMAT_UNSIGNED_BC5 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UNSIGNED_BC5

    #: Block compressed 5 signed
    CU_RES_VIEW_FORMAT_SIGNED_BC5 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SIGNED_BC5

    #: Block compressed 6 unsigned half-float
    CU_RES_VIEW_FORMAT_UNSIGNED_BC6H = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UNSIGNED_BC6H

    #: Block compressed 6 signed half-float
    CU_RES_VIEW_FORMAT_SIGNED_BC6H = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SIGNED_BC6H

    #: Block compressed 7
    CU_RES_VIEW_FORMAT_UNSIGNED_BC7 = ccuda.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UNSIGNED_BC7

class CUtensorMapDataType(IntEnum):
    """
    Tensor map data type
    """
    CU_TENSOR_MAP_DATA_TYPE_UINT8 = ccuda.CUtensorMapDataType_enum.CU_TENSOR_MAP_DATA_TYPE_UINT8
    CU_TENSOR_MAP_DATA_TYPE_UINT16 = ccuda.CUtensorMapDataType_enum.CU_TENSOR_MAP_DATA_TYPE_UINT16
    CU_TENSOR_MAP_DATA_TYPE_UINT32 = ccuda.CUtensorMapDataType_enum.CU_TENSOR_MAP_DATA_TYPE_UINT32
    CU_TENSOR_MAP_DATA_TYPE_INT32 = ccuda.CUtensorMapDataType_enum.CU_TENSOR_MAP_DATA_TYPE_INT32
    CU_TENSOR_MAP_DATA_TYPE_UINT64 = ccuda.CUtensorMapDataType_enum.CU_TENSOR_MAP_DATA_TYPE_UINT64
    CU_TENSOR_MAP_DATA_TYPE_INT64 = ccuda.CUtensorMapDataType_enum.CU_TENSOR_MAP_DATA_TYPE_INT64
    CU_TENSOR_MAP_DATA_TYPE_FLOAT16 = ccuda.CUtensorMapDataType_enum.CU_TENSOR_MAP_DATA_TYPE_FLOAT16
    CU_TENSOR_MAP_DATA_TYPE_FLOAT32 = ccuda.CUtensorMapDataType_enum.CU_TENSOR_MAP_DATA_TYPE_FLOAT32
    CU_TENSOR_MAP_DATA_TYPE_FLOAT64 = ccuda.CUtensorMapDataType_enum.CU_TENSOR_MAP_DATA_TYPE_FLOAT64
    CU_TENSOR_MAP_DATA_TYPE_BFLOAT16 = ccuda.CUtensorMapDataType_enum.CU_TENSOR_MAP_DATA_TYPE_BFLOAT16
    CU_TENSOR_MAP_DATA_TYPE_FLOAT32_FTZ = ccuda.CUtensorMapDataType_enum.CU_TENSOR_MAP_DATA_TYPE_FLOAT32_FTZ
    CU_TENSOR_MAP_DATA_TYPE_TFLOAT32 = ccuda.CUtensorMapDataType_enum.CU_TENSOR_MAP_DATA_TYPE_TFLOAT32
    CU_TENSOR_MAP_DATA_TYPE_TFLOAT32_FTZ = ccuda.CUtensorMapDataType_enum.CU_TENSOR_MAP_DATA_TYPE_TFLOAT32_FTZ

class CUtensorMapInterleave(IntEnum):
    """
    Tensor map interleave layout type
    """
    CU_TENSOR_MAP_INTERLEAVE_NONE = ccuda.CUtensorMapInterleave_enum.CU_TENSOR_MAP_INTERLEAVE_NONE
    CU_TENSOR_MAP_INTERLEAVE_16B = ccuda.CUtensorMapInterleave_enum.CU_TENSOR_MAP_INTERLEAVE_16B
    CU_TENSOR_MAP_INTERLEAVE_32B = ccuda.CUtensorMapInterleave_enum.CU_TENSOR_MAP_INTERLEAVE_32B

class CUtensorMapSwizzle(IntEnum):
    """
    Tensor map swizzling mode of shared memory banks
    """
    CU_TENSOR_MAP_SWIZZLE_NONE = ccuda.CUtensorMapSwizzle_enum.CU_TENSOR_MAP_SWIZZLE_NONE
    CU_TENSOR_MAP_SWIZZLE_32B = ccuda.CUtensorMapSwizzle_enum.CU_TENSOR_MAP_SWIZZLE_32B
    CU_TENSOR_MAP_SWIZZLE_64B = ccuda.CUtensorMapSwizzle_enum.CU_TENSOR_MAP_SWIZZLE_64B
    CU_TENSOR_MAP_SWIZZLE_128B = ccuda.CUtensorMapSwizzle_enum.CU_TENSOR_MAP_SWIZZLE_128B

class CUtensorMapL2promotion(IntEnum):
    """
    Tensor map L2 promotion type
    """
    CU_TENSOR_MAP_L2_PROMOTION_NONE = ccuda.CUtensorMapL2promotion_enum.CU_TENSOR_MAP_L2_PROMOTION_NONE
    CU_TENSOR_MAP_L2_PROMOTION_L2_64B = ccuda.CUtensorMapL2promotion_enum.CU_TENSOR_MAP_L2_PROMOTION_L2_64B
    CU_TENSOR_MAP_L2_PROMOTION_L2_128B = ccuda.CUtensorMapL2promotion_enum.CU_TENSOR_MAP_L2_PROMOTION_L2_128B
    CU_TENSOR_MAP_L2_PROMOTION_L2_256B = ccuda.CUtensorMapL2promotion_enum.CU_TENSOR_MAP_L2_PROMOTION_L2_256B

class CUtensorMapFloatOOBfill(IntEnum):
    """
    Tensor map out-of-bounds fill type
    """
    CU_TENSOR_MAP_FLOAT_OOB_FILL_NONE = ccuda.CUtensorMapFloatOOBfill_enum.CU_TENSOR_MAP_FLOAT_OOB_FILL_NONE
    CU_TENSOR_MAP_FLOAT_OOB_FILL_NAN_REQUEST_ZERO_FMA = ccuda.CUtensorMapFloatOOBfill_enum.CU_TENSOR_MAP_FLOAT_OOB_FILL_NAN_REQUEST_ZERO_FMA

class CUDA_POINTER_ATTRIBUTE_ACCESS_FLAGS(IntEnum):
    """
    Access flags that specify the level of access the current context's
    device has on the memory referenced.
    """

    #: No access, meaning the device cannot access this memory at all, thus
    #: must be staged through accessible memory in order to complete
    #: certain operations
    CU_POINTER_ATTRIBUTE_ACCESS_FLAG_NONE = ccuda.CUDA_POINTER_ATTRIBUTE_ACCESS_FLAGS_enum.CU_POINTER_ATTRIBUTE_ACCESS_FLAG_NONE

    #: Read-only access, meaning writes to this memory are considered
    #: invalid accesses and thus return error in that case.
    CU_POINTER_ATTRIBUTE_ACCESS_FLAG_READ = ccuda.CUDA_POINTER_ATTRIBUTE_ACCESS_FLAGS_enum.CU_POINTER_ATTRIBUTE_ACCESS_FLAG_READ

    #: Read-write access, the device has full read-write access to the
    #: memory
    CU_POINTER_ATTRIBUTE_ACCESS_FLAG_READWRITE = ccuda.CUDA_POINTER_ATTRIBUTE_ACCESS_FLAGS_enum.CU_POINTER_ATTRIBUTE_ACCESS_FLAG_READWRITE

class CUexternalMemoryHandleType(IntEnum):
    """
    External memory handle types
    """

    #: Handle is an opaque file descriptor
    CU_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_FD = ccuda.CUexternalMemoryHandleType_enum.CU_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_FD

    #: Handle is an opaque shared NT handle
    CU_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32 = ccuda.CUexternalMemoryHandleType_enum.CU_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32

    #: Handle is an opaque, globally shared handle
    CU_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_KMT = ccuda.CUexternalMemoryHandleType_enum.CU_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_KMT

    #: Handle is a D3D12 heap object
    CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_HEAP = ccuda.CUexternalMemoryHandleType_enum.CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_HEAP

    #: Handle is a D3D12 committed resource
    CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_RESOURCE = ccuda.CUexternalMemoryHandleType_enum.CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_RESOURCE

    #: Handle is a shared NT handle to a D3D11 resource
    CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_RESOURCE = ccuda.CUexternalMemoryHandleType_enum.CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_RESOURCE

    #: Handle is a globally shared handle to a D3D11 resource
    CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_RESOURCE_KMT = ccuda.CUexternalMemoryHandleType_enum.CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_RESOURCE_KMT

    #: Handle is an NvSciBuf object
    CU_EXTERNAL_MEMORY_HANDLE_TYPE_NVSCIBUF = ccuda.CUexternalMemoryHandleType_enum.CU_EXTERNAL_MEMORY_HANDLE_TYPE_NVSCIBUF

class CUexternalSemaphoreHandleType(IntEnum):
    """
    External semaphore handle types
    """

    #: Handle is an opaque file descriptor
    CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_FD = ccuda.CUexternalSemaphoreHandleType_enum.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_FD

    #: Handle is an opaque shared NT handle
    CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32 = ccuda.CUexternalSemaphoreHandleType_enum.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32

    #: Handle is an opaque, globally shared handle
    CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32_KMT = ccuda.CUexternalSemaphoreHandleType_enum.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32_KMT

    #: Handle is a shared NT handle referencing a D3D12 fence object
    CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D12_FENCE = ccuda.CUexternalSemaphoreHandleType_enum.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D12_FENCE

    #: Handle is a shared NT handle referencing a D3D11 fence object
    CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D11_FENCE = ccuda.CUexternalSemaphoreHandleType_enum.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D11_FENCE

    #: Opaque handle to NvSciSync Object
    CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_NVSCISYNC = ccuda.CUexternalSemaphoreHandleType_enum.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_NVSCISYNC

    #: Handle is a shared NT handle referencing a D3D11 keyed mutex object
    CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D11_KEYED_MUTEX = ccuda.CUexternalSemaphoreHandleType_enum.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D11_KEYED_MUTEX

    #: Handle is a globally shared handle referencing a D3D11 keyed mutex
    #: object
    CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D11_KEYED_MUTEX_KMT = ccuda.CUexternalSemaphoreHandleType_enum.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D11_KEYED_MUTEX_KMT

    #: Handle is an opaque file descriptor referencing a timeline semaphore
    CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_TIMELINE_SEMAPHORE_FD = ccuda.CUexternalSemaphoreHandleType_enum.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_TIMELINE_SEMAPHORE_FD

    #: Handle is an opaque shared NT handle referencing a timeline
    #: semaphore
    CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_TIMELINE_SEMAPHORE_WIN32 = ccuda.CUexternalSemaphoreHandleType_enum.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_TIMELINE_SEMAPHORE_WIN32

class CUmemAllocationHandleType(IntEnum):
    """
    Flags for specifying particular handle types
    """

    #: Does not allow any export mechanism. >
    CU_MEM_HANDLE_TYPE_NONE = ccuda.CUmemAllocationHandleType_enum.CU_MEM_HANDLE_TYPE_NONE

    #: Allows a file descriptor to be used for exporting. Permitted only on
    #: POSIX systems. (int)
    CU_MEM_HANDLE_TYPE_POSIX_FILE_DESCRIPTOR = ccuda.CUmemAllocationHandleType_enum.CU_MEM_HANDLE_TYPE_POSIX_FILE_DESCRIPTOR

    #: Allows a Win32 NT handle to be used for exporting. (HANDLE)
    CU_MEM_HANDLE_TYPE_WIN32 = ccuda.CUmemAllocationHandleType_enum.CU_MEM_HANDLE_TYPE_WIN32

    #: Allows a Win32 KMT handle to be used for exporting. (D3DKMT_HANDLE)
    CU_MEM_HANDLE_TYPE_WIN32_KMT = ccuda.CUmemAllocationHandleType_enum.CU_MEM_HANDLE_TYPE_WIN32_KMT

    #: Allows a fabric handle to be used for exporting. (CUmemFabricHandle)
    CU_MEM_HANDLE_TYPE_FABRIC = ccuda.CUmemAllocationHandleType_enum.CU_MEM_HANDLE_TYPE_FABRIC
    CU_MEM_HANDLE_TYPE_MAX = ccuda.CUmemAllocationHandleType_enum.CU_MEM_HANDLE_TYPE_MAX

class CUmemAccess_flags(IntEnum):
    """
    Specifies the memory protection flags for mapping.
    """

    #: Default, make the address range not accessible
    CU_MEM_ACCESS_FLAGS_PROT_NONE = ccuda.CUmemAccess_flags_enum.CU_MEM_ACCESS_FLAGS_PROT_NONE

    #: Make the address range read accessible
    CU_MEM_ACCESS_FLAGS_PROT_READ = ccuda.CUmemAccess_flags_enum.CU_MEM_ACCESS_FLAGS_PROT_READ

    #: Make the address range read-write accessible
    CU_MEM_ACCESS_FLAGS_PROT_READWRITE = ccuda.CUmemAccess_flags_enum.CU_MEM_ACCESS_FLAGS_PROT_READWRITE
    CU_MEM_ACCESS_FLAGS_PROT_MAX = ccuda.CUmemAccess_flags_enum.CU_MEM_ACCESS_FLAGS_PROT_MAX

class CUmemLocationType(IntEnum):
    """
    Specifies the type of location
    """
    CU_MEM_LOCATION_TYPE_INVALID = ccuda.CUmemLocationType_enum.CU_MEM_LOCATION_TYPE_INVALID

    #: Location is a device location, thus id is a device ordinal
    CU_MEM_LOCATION_TYPE_DEVICE = ccuda.CUmemLocationType_enum.CU_MEM_LOCATION_TYPE_DEVICE

    #: Location is host, id is ignored
    CU_MEM_LOCATION_TYPE_HOST = ccuda.CUmemLocationType_enum.CU_MEM_LOCATION_TYPE_HOST

    #: Location is a host NUMA node, thus id is a host NUMA node id
    CU_MEM_LOCATION_TYPE_HOST_NUMA = ccuda.CUmemLocationType_enum.CU_MEM_LOCATION_TYPE_HOST_NUMA

    #: Location is a host NUMA node of the current thread, id is ignored
    CU_MEM_LOCATION_TYPE_HOST_NUMA_CURRENT = ccuda.CUmemLocationType_enum.CU_MEM_LOCATION_TYPE_HOST_NUMA_CURRENT
    CU_MEM_LOCATION_TYPE_MAX = ccuda.CUmemLocationType_enum.CU_MEM_LOCATION_TYPE_MAX

class CUmemAllocationType(IntEnum):
    """
    Defines the allocation types available
    """
    CU_MEM_ALLOCATION_TYPE_INVALID = ccuda.CUmemAllocationType_enum.CU_MEM_ALLOCATION_TYPE_INVALID

    #: This allocation type is 'pinned', i.e. cannot migrate from its
    #: current location while the application is actively using it
    CU_MEM_ALLOCATION_TYPE_PINNED = ccuda.CUmemAllocationType_enum.CU_MEM_ALLOCATION_TYPE_PINNED
    CU_MEM_ALLOCATION_TYPE_MAX = ccuda.CUmemAllocationType_enum.CU_MEM_ALLOCATION_TYPE_MAX

class CUmemAllocationGranularity_flags(IntEnum):
    """
    Flag for requesting different optimal and required granularities
    for an allocation.
    """

    #: Minimum required granularity for allocation
    CU_MEM_ALLOC_GRANULARITY_MINIMUM = ccuda.CUmemAllocationGranularity_flags_enum.CU_MEM_ALLOC_GRANULARITY_MINIMUM

    #: Recommended granularity for allocation for best performance
    CU_MEM_ALLOC_GRANULARITY_RECOMMENDED = ccuda.CUmemAllocationGranularity_flags_enum.CU_MEM_ALLOC_GRANULARITY_RECOMMENDED

class CUmemRangeHandleType(IntEnum):
    """
    Specifies the handle type for address range
    """
    CU_MEM_RANGE_HANDLE_TYPE_DMA_BUF_FD = ccuda.CUmemRangeHandleType_enum.CU_MEM_RANGE_HANDLE_TYPE_DMA_BUF_FD
    CU_MEM_RANGE_HANDLE_TYPE_MAX = ccuda.CUmemRangeHandleType_enum.CU_MEM_RANGE_HANDLE_TYPE_MAX

class CUarraySparseSubresourceType(IntEnum):
    """
    Sparse subresource types
    """
    CU_ARRAY_SPARSE_SUBRESOURCE_TYPE_SPARSE_LEVEL = ccuda.CUarraySparseSubresourceType_enum.CU_ARRAY_SPARSE_SUBRESOURCE_TYPE_SPARSE_LEVEL
    CU_ARRAY_SPARSE_SUBRESOURCE_TYPE_MIPTAIL = ccuda.CUarraySparseSubresourceType_enum.CU_ARRAY_SPARSE_SUBRESOURCE_TYPE_MIPTAIL

class CUmemOperationType(IntEnum):
    """
    Memory operation types
    """
    CU_MEM_OPERATION_TYPE_MAP = ccuda.CUmemOperationType_enum.CU_MEM_OPERATION_TYPE_MAP
    CU_MEM_OPERATION_TYPE_UNMAP = ccuda.CUmemOperationType_enum.CU_MEM_OPERATION_TYPE_UNMAP

class CUmemHandleType(IntEnum):
    """
    Memory handle types
    """
    CU_MEM_HANDLE_TYPE_GENERIC = ccuda.CUmemHandleType_enum.CU_MEM_HANDLE_TYPE_GENERIC

class CUmemAllocationCompType(IntEnum):
    """
    Specifies compression attribute for an allocation.
    """

    #: Allocating non-compressible memory
    CU_MEM_ALLOCATION_COMP_NONE = ccuda.CUmemAllocationCompType_enum.CU_MEM_ALLOCATION_COMP_NONE

    #: Allocating compressible memory
    CU_MEM_ALLOCATION_COMP_GENERIC = ccuda.CUmemAllocationCompType_enum.CU_MEM_ALLOCATION_COMP_GENERIC

class CUmulticastGranularity_flags(IntEnum):
    """
    Flags for querying different granularities for a multicast object
    """

    #: Minimum required granularity
    CU_MULTICAST_GRANULARITY_MINIMUM = ccuda.CUmulticastGranularity_flags_enum.CU_MULTICAST_GRANULARITY_MINIMUM

    #: Recommended granularity for best performance
    CU_MULTICAST_GRANULARITY_RECOMMENDED = ccuda.CUmulticastGranularity_flags_enum.CU_MULTICAST_GRANULARITY_RECOMMENDED

class CUgraphExecUpdateResult(IntEnum):
    """
    CUDA Graph Update error types
    """

    #: The update succeeded
    CU_GRAPH_EXEC_UPDATE_SUCCESS = ccuda.CUgraphExecUpdateResult_enum.CU_GRAPH_EXEC_UPDATE_SUCCESS

    #: The update failed for an unexpected reason which is described in the
    #: return value of the function
    CU_GRAPH_EXEC_UPDATE_ERROR = ccuda.CUgraphExecUpdateResult_enum.CU_GRAPH_EXEC_UPDATE_ERROR

    #: The update failed because the topology changed
    CU_GRAPH_EXEC_UPDATE_ERROR_TOPOLOGY_CHANGED = ccuda.CUgraphExecUpdateResult_enum.CU_GRAPH_EXEC_UPDATE_ERROR_TOPOLOGY_CHANGED

    #: The update failed because a node type changed
    CU_GRAPH_EXEC_UPDATE_ERROR_NODE_TYPE_CHANGED = ccuda.CUgraphExecUpdateResult_enum.CU_GRAPH_EXEC_UPDATE_ERROR_NODE_TYPE_CHANGED

    #: The update failed because the function of a kernel node changed
    #: (CUDA driver < 11.2)
    CU_GRAPH_EXEC_UPDATE_ERROR_FUNCTION_CHANGED = ccuda.CUgraphExecUpdateResult_enum.CU_GRAPH_EXEC_UPDATE_ERROR_FUNCTION_CHANGED

    #: The update failed because the parameters changed in a way that is
    #: not supported
    CU_GRAPH_EXEC_UPDATE_ERROR_PARAMETERS_CHANGED = ccuda.CUgraphExecUpdateResult_enum.CU_GRAPH_EXEC_UPDATE_ERROR_PARAMETERS_CHANGED

    #: The update failed because something about the node is not supported
    CU_GRAPH_EXEC_UPDATE_ERROR_NOT_SUPPORTED = ccuda.CUgraphExecUpdateResult_enum.CU_GRAPH_EXEC_UPDATE_ERROR_NOT_SUPPORTED

    #: The update failed because the function of a kernel node changed in
    #: an unsupported way
    CU_GRAPH_EXEC_UPDATE_ERROR_UNSUPPORTED_FUNCTION_CHANGE = ccuda.CUgraphExecUpdateResult_enum.CU_GRAPH_EXEC_UPDATE_ERROR_UNSUPPORTED_FUNCTION_CHANGE

    #: The update failed because the node attributes changed in a way that
    #: is not supported
    CU_GRAPH_EXEC_UPDATE_ERROR_ATTRIBUTES_CHANGED = ccuda.CUgraphExecUpdateResult_enum.CU_GRAPH_EXEC_UPDATE_ERROR_ATTRIBUTES_CHANGED

class CUmemPool_attribute(IntEnum):
    """
    CUDA memory pool attributes
    """

    #: (value type = int) Allow cuMemAllocAsync to use memory
    #: asynchronously freed in another streams as long as a stream ordering
    #: dependency of the allocating stream on the free action exists. Cuda
    #: events and null stream interactions can create the required stream
    #: ordered dependencies. (default enabled)
    CU_MEMPOOL_ATTR_REUSE_FOLLOW_EVENT_DEPENDENCIES = ccuda.CUmemPool_attribute_enum.CU_MEMPOOL_ATTR_REUSE_FOLLOW_EVENT_DEPENDENCIES

    #: (value type = int) Allow reuse of already completed frees when there
    #: is no dependency between the free and allocation. (default enabled)
    CU_MEMPOOL_ATTR_REUSE_ALLOW_OPPORTUNISTIC = ccuda.CUmemPool_attribute_enum.CU_MEMPOOL_ATTR_REUSE_ALLOW_OPPORTUNISTIC

    #: (value type = int) Allow cuMemAllocAsync to insert new stream
    #: dependencies in order to establish the stream ordering required to
    #: reuse a piece of memory released by cuFreeAsync (default enabled).
    CU_MEMPOOL_ATTR_REUSE_ALLOW_INTERNAL_DEPENDENCIES = ccuda.CUmemPool_attribute_enum.CU_MEMPOOL_ATTR_REUSE_ALLOW_INTERNAL_DEPENDENCIES

    #: (value type = cuuint64_t) Amount of reserved memory in bytes to hold
    #: onto before trying to release memory back to the OS. When more than
    #: the release threshold bytes of memory are held by the memory pool,
    #: the allocator will try to release memory back to the OS on the next
    #: call to stream, event or context synchronize. (default 0)
    CU_MEMPOOL_ATTR_RELEASE_THRESHOLD = ccuda.CUmemPool_attribute_enum.CU_MEMPOOL_ATTR_RELEASE_THRESHOLD

    #: (value type = cuuint64_t) Amount of backing memory currently
    #: allocated for the mempool.
    CU_MEMPOOL_ATTR_RESERVED_MEM_CURRENT = ccuda.CUmemPool_attribute_enum.CU_MEMPOOL_ATTR_RESERVED_MEM_CURRENT

    #: (value type = cuuint64_t) High watermark of backing memory allocated
    #: for the mempool since the last time it was reset. High watermark can
    #: only be reset to zero.
    CU_MEMPOOL_ATTR_RESERVED_MEM_HIGH = ccuda.CUmemPool_attribute_enum.CU_MEMPOOL_ATTR_RESERVED_MEM_HIGH

    #: (value type = cuuint64_t) Amount of memory from the pool that is
    #: currently in use by the application.
    CU_MEMPOOL_ATTR_USED_MEM_CURRENT = ccuda.CUmemPool_attribute_enum.CU_MEMPOOL_ATTR_USED_MEM_CURRENT

    #: (value type = cuuint64_t) High watermark of the amount of memory
    #: from the pool that was in use by the application since the last time
    #: it was reset. High watermark can only be reset to zero.
    CU_MEMPOOL_ATTR_USED_MEM_HIGH = ccuda.CUmemPool_attribute_enum.CU_MEMPOOL_ATTR_USED_MEM_HIGH

class CUgraphMem_attribute(IntEnum):
    """

    """

    #: (value type = cuuint64_t) Amount of memory, in bytes, currently
    #: associated with graphs
    CU_GRAPH_MEM_ATTR_USED_MEM_CURRENT = ccuda.CUgraphMem_attribute_enum.CU_GRAPH_MEM_ATTR_USED_MEM_CURRENT

    #: (value type = cuuint64_t) High watermark of memory, in bytes,
    #: associated with graphs since the last time it was reset. High
    #: watermark can only be reset to zero.
    CU_GRAPH_MEM_ATTR_USED_MEM_HIGH = ccuda.CUgraphMem_attribute_enum.CU_GRAPH_MEM_ATTR_USED_MEM_HIGH

    #: (value type = cuuint64_t) Amount of memory, in bytes, currently
    #: allocated for use by the CUDA graphs asynchronous allocator.
    CU_GRAPH_MEM_ATTR_RESERVED_MEM_CURRENT = ccuda.CUgraphMem_attribute_enum.CU_GRAPH_MEM_ATTR_RESERVED_MEM_CURRENT

    #: (value type = cuuint64_t) High watermark of memory, in bytes,
    #: currently allocated for use by the CUDA graphs asynchronous
    #: allocator.
    CU_GRAPH_MEM_ATTR_RESERVED_MEM_HIGH = ccuda.CUgraphMem_attribute_enum.CU_GRAPH_MEM_ATTR_RESERVED_MEM_HIGH

class CUflushGPUDirectRDMAWritesOptions(IntEnum):
    """
    Bitmasks for
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_FLUSH_WRITES_OPTIONS`
    """

    #: :py:obj:`~.cuFlushGPUDirectRDMAWrites()` and its CUDA Runtime API
    #: counterpart are supported on the device.
    CU_FLUSH_GPU_DIRECT_RDMA_WRITES_OPTION_HOST = ccuda.CUflushGPUDirectRDMAWritesOptions_enum.CU_FLUSH_GPU_DIRECT_RDMA_WRITES_OPTION_HOST

    #: The :py:obj:`~.CU_STREAM_WAIT_VALUE_FLUSH` flag and the
    #: :py:obj:`~.CU_STREAM_MEM_OP_FLUSH_REMOTE_WRITES` MemOp are supported
    #: on the device.
    CU_FLUSH_GPU_DIRECT_RDMA_WRITES_OPTION_MEMOPS = ccuda.CUflushGPUDirectRDMAWritesOptions_enum.CU_FLUSH_GPU_DIRECT_RDMA_WRITES_OPTION_MEMOPS

class CUGPUDirectRDMAWritesOrdering(IntEnum):
    """
    Platform native ordering for GPUDirect RDMA writes
    """

    #: The device does not natively support ordering of remote writes.
    #: :py:obj:`~.cuFlushGPUDirectRDMAWrites()` can be leveraged if
    #: supported.
    CU_GPU_DIRECT_RDMA_WRITES_ORDERING_NONE = ccuda.CUGPUDirectRDMAWritesOrdering_enum.CU_GPU_DIRECT_RDMA_WRITES_ORDERING_NONE

    #: Natively, the device can consistently consume remote writes,
    #: although other CUDA devices may not.
    CU_GPU_DIRECT_RDMA_WRITES_ORDERING_OWNER = ccuda.CUGPUDirectRDMAWritesOrdering_enum.CU_GPU_DIRECT_RDMA_WRITES_ORDERING_OWNER

    #: Any CUDA device in the system can consistently consume remote writes
    #: to this device.
    CU_GPU_DIRECT_RDMA_WRITES_ORDERING_ALL_DEVICES = ccuda.CUGPUDirectRDMAWritesOrdering_enum.CU_GPU_DIRECT_RDMA_WRITES_ORDERING_ALL_DEVICES

class CUflushGPUDirectRDMAWritesScope(IntEnum):
    """
    The scopes for :py:obj:`~.cuFlushGPUDirectRDMAWrites`
    """

    #: Blocks until remote writes are visible to the CUDA device context
    #: owning the data.
    CU_FLUSH_GPU_DIRECT_RDMA_WRITES_TO_OWNER = ccuda.CUflushGPUDirectRDMAWritesScope_enum.CU_FLUSH_GPU_DIRECT_RDMA_WRITES_TO_OWNER

    #: Blocks until remote writes are visible to all CUDA device contexts.
    CU_FLUSH_GPU_DIRECT_RDMA_WRITES_TO_ALL_DEVICES = ccuda.CUflushGPUDirectRDMAWritesScope_enum.CU_FLUSH_GPU_DIRECT_RDMA_WRITES_TO_ALL_DEVICES

class CUflushGPUDirectRDMAWritesTarget(IntEnum):
    """
    The targets for :py:obj:`~.cuFlushGPUDirectRDMAWrites`
    """

    #: Sets the target for :py:obj:`~.cuFlushGPUDirectRDMAWrites()` to the
    #: currently active CUDA device context.
    CU_FLUSH_GPU_DIRECT_RDMA_WRITES_TARGET_CURRENT_CTX = ccuda.CUflushGPUDirectRDMAWritesTarget_enum.CU_FLUSH_GPU_DIRECT_RDMA_WRITES_TARGET_CURRENT_CTX

class CUgraphDebugDot_flags(IntEnum):
    """
    The additional write options for :py:obj:`~.cuGraphDebugDotPrint`
    """

    #: Output all debug data as if every debug flag is enabled
    CU_GRAPH_DEBUG_DOT_FLAGS_VERBOSE = ccuda.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_VERBOSE

    #: Use CUDA Runtime structures for output
    CU_GRAPH_DEBUG_DOT_FLAGS_RUNTIME_TYPES = ccuda.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_RUNTIME_TYPES

    #: Adds CUDA_KERNEL_NODE_PARAMS values to output
    CU_GRAPH_DEBUG_DOT_FLAGS_KERNEL_NODE_PARAMS = ccuda.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_KERNEL_NODE_PARAMS

    #: Adds CUDA_MEMCPY3D values to output
    CU_GRAPH_DEBUG_DOT_FLAGS_MEMCPY_NODE_PARAMS = ccuda.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_MEMCPY_NODE_PARAMS

    #: Adds CUDA_MEMSET_NODE_PARAMS values to output
    CU_GRAPH_DEBUG_DOT_FLAGS_MEMSET_NODE_PARAMS = ccuda.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_MEMSET_NODE_PARAMS

    #: Adds CUDA_HOST_NODE_PARAMS values to output
    CU_GRAPH_DEBUG_DOT_FLAGS_HOST_NODE_PARAMS = ccuda.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_HOST_NODE_PARAMS

    #: Adds CUevent handle from record and wait nodes to output
    CU_GRAPH_DEBUG_DOT_FLAGS_EVENT_NODE_PARAMS = ccuda.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_EVENT_NODE_PARAMS

    #: Adds CUDA_EXT_SEM_SIGNAL_NODE_PARAMS values to output
    CU_GRAPH_DEBUG_DOT_FLAGS_EXT_SEMAS_SIGNAL_NODE_PARAMS = ccuda.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_EXT_SEMAS_SIGNAL_NODE_PARAMS

    #: Adds CUDA_EXT_SEM_WAIT_NODE_PARAMS values to output
    CU_GRAPH_DEBUG_DOT_FLAGS_EXT_SEMAS_WAIT_NODE_PARAMS = ccuda.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_EXT_SEMAS_WAIT_NODE_PARAMS

    #: Adds CUkernelNodeAttrValue values to output
    CU_GRAPH_DEBUG_DOT_FLAGS_KERNEL_NODE_ATTRIBUTES = ccuda.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_KERNEL_NODE_ATTRIBUTES

    #: Adds node handles and every kernel function handle to output
    CU_GRAPH_DEBUG_DOT_FLAGS_HANDLES = ccuda.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_HANDLES

    #: Adds memory alloc node parameters to output
    CU_GRAPH_DEBUG_DOT_FLAGS_MEM_ALLOC_NODE_PARAMS = ccuda.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_MEM_ALLOC_NODE_PARAMS

    #: Adds memory free node parameters to output
    CU_GRAPH_DEBUG_DOT_FLAGS_MEM_FREE_NODE_PARAMS = ccuda.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_MEM_FREE_NODE_PARAMS

    #: Adds batch mem op node parameters to output
    CU_GRAPH_DEBUG_DOT_FLAGS_BATCH_MEM_OP_NODE_PARAMS = ccuda.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_BATCH_MEM_OP_NODE_PARAMS

    #: Adds edge numbering information
    CU_GRAPH_DEBUG_DOT_FLAGS_EXTRA_TOPO_INFO = ccuda.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_EXTRA_TOPO_INFO

    #: Adds conditional node parameters to output
    CU_GRAPH_DEBUG_DOT_FLAGS_CONDITIONAL_NODE_PARAMS = ccuda.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_CONDITIONAL_NODE_PARAMS

class CUuserObject_flags(IntEnum):
    """
    Flags for user objects for graphs
    """

    #: Indicates the destructor execution is not synchronized by any CUDA
    #: handle.
    CU_USER_OBJECT_NO_DESTRUCTOR_SYNC = ccuda.CUuserObject_flags_enum.CU_USER_OBJECT_NO_DESTRUCTOR_SYNC

class CUuserObjectRetain_flags(IntEnum):
    """
    Flags for retaining user object references for graphs
    """

    #: Transfer references from the caller rather than creating new
    #: references.
    CU_GRAPH_USER_OBJECT_MOVE = ccuda.CUuserObjectRetain_flags_enum.CU_GRAPH_USER_OBJECT_MOVE

class CUgraphInstantiate_flags(IntEnum):
    """
    Flags for instantiating a graph
    """

    #: Automatically free memory allocated in a graph before relaunching.
    CUDA_GRAPH_INSTANTIATE_FLAG_AUTO_FREE_ON_LAUNCH = ccuda.CUgraphInstantiate_flags_enum.CUDA_GRAPH_INSTANTIATE_FLAG_AUTO_FREE_ON_LAUNCH

    #: Automatically upload the graph after instantiation. Only supported
    #: by :py:obj:`~.cuGraphInstantiateWithParams`. The upload will be
    #: performed using the stream provided in `instantiateParams`.
    CUDA_GRAPH_INSTANTIATE_FLAG_UPLOAD = ccuda.CUgraphInstantiate_flags_enum.CUDA_GRAPH_INSTANTIATE_FLAG_UPLOAD

    #: Instantiate the graph to be launchable from the device. This flag
    #: can only be used on platforms which support unified addressing. This
    #: flag cannot be used in conjunction with
    #: CUDA_GRAPH_INSTANTIATE_FLAG_AUTO_FREE_ON_LAUNCH.
    CUDA_GRAPH_INSTANTIATE_FLAG_DEVICE_LAUNCH = ccuda.CUgraphInstantiate_flags_enum.CUDA_GRAPH_INSTANTIATE_FLAG_DEVICE_LAUNCH

    #: Run the graph using the per-node priority attributes rather than the
    #: priority of the stream it is launched into.
    CUDA_GRAPH_INSTANTIATE_FLAG_USE_NODE_PRIORITY = ccuda.CUgraphInstantiate_flags_enum.CUDA_GRAPH_INSTANTIATE_FLAG_USE_NODE_PRIORITY

class CUdeviceNumaConfig(IntEnum):
    """

    """

    #: The GPU is not a NUMA node
    CU_DEVICE_NUMA_CONFIG_NONE = ccuda.CUdeviceNumaConfig_enum.CU_DEVICE_NUMA_CONFIG_NONE

    #: The GPU is a NUMA node, CU_DEVICE_ATTRIBUTE_NUMA_ID contains its
    #: NUMA ID
    CU_DEVICE_NUMA_CONFIG_NUMA_NODE = ccuda.CUdeviceNumaConfig_enum.CU_DEVICE_NUMA_CONFIG_NUMA_NODE

class CUmoduleLoadingMode(IntEnum):
    """
    CUDA Lazy Loading status
    """

    #: Lazy Kernel Loading is not enabled
    CU_MODULE_EAGER_LOADING = ccuda.CUmoduleLoadingMode_enum.CU_MODULE_EAGER_LOADING

    #: Lazy Kernel Loading is enabled
    CU_MODULE_LAZY_LOADING = ccuda.CUmoduleLoadingMode_enum.CU_MODULE_LAZY_LOADING

class CUfunctionLoadingState(IntEnum):
    """

    """
    CU_FUNCTION_LOADING_STATE_UNLOADED = ccuda.CUfunctionLoadingState_enum.CU_FUNCTION_LOADING_STATE_UNLOADED
    CU_FUNCTION_LOADING_STATE_LOADED = ccuda.CUfunctionLoadingState_enum.CU_FUNCTION_LOADING_STATE_LOADED
    CU_FUNCTION_LOADING_STATE_MAX = ccuda.CUfunctionLoadingState_enum.CU_FUNCTION_LOADING_STATE_MAX

class CUcoredumpSettings(IntEnum):
    """
    Flags for choosing a coredump attribute to get/set
    """
    CU_COREDUMP_ENABLE_ON_EXCEPTION = ccuda.CUcoredumpSettings_enum.CU_COREDUMP_ENABLE_ON_EXCEPTION
    CU_COREDUMP_TRIGGER_HOST = ccuda.CUcoredumpSettings_enum.CU_COREDUMP_TRIGGER_HOST
    CU_COREDUMP_LIGHTWEIGHT = ccuda.CUcoredumpSettings_enum.CU_COREDUMP_LIGHTWEIGHT
    CU_COREDUMP_ENABLE_USER_TRIGGER = ccuda.CUcoredumpSettings_enum.CU_COREDUMP_ENABLE_USER_TRIGGER
    CU_COREDUMP_FILE = ccuda.CUcoredumpSettings_enum.CU_COREDUMP_FILE
    CU_COREDUMP_PIPE = ccuda.CUcoredumpSettings_enum.CU_COREDUMP_PIPE
    CU_COREDUMP_MAX = ccuda.CUcoredumpSettings_enum.CU_COREDUMP_MAX

class CUgreenCtxCreate_flags(IntEnum):
    """

    """

    #: Required. Creates a default stream to use inside the green context
    CU_GREEN_CTX_DEFAULT_STREAM = ccuda.CUgreenCtxCreate_flags.CU_GREEN_CTX_DEFAULT_STREAM

class CUdevResourceType(IntEnum):
    """
    Type of resource
    """
    CU_DEV_RESOURCE_TYPE_INVALID = ccuda.CUdevResourceType.CU_DEV_RESOURCE_TYPE_INVALID

    #: Streaming multiprocessors related information
    CU_DEV_RESOURCE_TYPE_SM = ccuda.CUdevResourceType.CU_DEV_RESOURCE_TYPE_SM

class CUoutput_mode(IntEnum):
    """
    Profiler Output Modes
    """

    #: Output mode Key-Value pair format.
    CU_OUT_KEY_VALUE_PAIR = ccuda.CUoutput_mode_enum.CU_OUT_KEY_VALUE_PAIR

    #: Output mode Comma separated values format.
    CU_OUT_CSV = ccuda.CUoutput_mode_enum.CU_OUT_CSV

class CUeglFrameType(IntEnum):
    """
    CUDA EglFrame type - array or pointer
    """

    #: Frame type CUDA array
    CU_EGL_FRAME_TYPE_ARRAY = ccuda.CUeglFrameType_enum.CU_EGL_FRAME_TYPE_ARRAY

    #: Frame type pointer
    CU_EGL_FRAME_TYPE_PITCH = ccuda.CUeglFrameType_enum.CU_EGL_FRAME_TYPE_PITCH

class CUeglResourceLocationFlags(IntEnum):
    """
    Resource location flags- sysmem or vidmem  For CUDA context on
    iGPU, since video and system memory are equivalent - these flags
    will not have an effect on the execution.  For CUDA context on
    dGPU, applications can use the flag
    :py:obj:`~.CUeglResourceLocationFlags` to give a hint about the
    desired location.  :py:obj:`~.CU_EGL_RESOURCE_LOCATION_SYSMEM` -
    the frame data is made resident on the system memory to be accessed
    by CUDA.  :py:obj:`~.CU_EGL_RESOURCE_LOCATION_VIDMEM` - the frame
    data is made resident on the dedicated video memory to be accessed
    by CUDA.  There may be an additional latency due to new allocation
    and data migration, if the frame is produced on a different memory.
    """

    #: Resource location sysmem
    CU_EGL_RESOURCE_LOCATION_SYSMEM = ccuda.CUeglResourceLocationFlags_enum.CU_EGL_RESOURCE_LOCATION_SYSMEM

    #: Resource location vidmem
    CU_EGL_RESOURCE_LOCATION_VIDMEM = ccuda.CUeglResourceLocationFlags_enum.CU_EGL_RESOURCE_LOCATION_VIDMEM

class CUeglColorFormat(IntEnum):
    """
    CUDA EGL Color Format - The different planar and multiplanar
    formats currently supported for CUDA_EGL interops. Three channel
    formats are currently not supported for
    :py:obj:`~.CU_EGL_FRAME_TYPE_ARRAY`
    """

    #: Y, U, V in three surfaces, each in a separate surface, U/V width =
    #: 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YUV420_PLANAR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV420_PLANAR

    #: Y, UV in two surfaces (UV as one surface) with VU byte ordering,
    #: width, height ratio same as YUV420Planar.
    CU_EGL_COLOR_FORMAT_YUV420_SEMIPLANAR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV420_SEMIPLANAR

    #: Y, U, V each in a separate surface, U/V width = 1/2 Y width, U/V
    #: height = Y height.
    CU_EGL_COLOR_FORMAT_YUV422_PLANAR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV422_PLANAR

    #: Y, UV in two surfaces with VU byte ordering, width, height ratio
    #: same as YUV422Planar.
    CU_EGL_COLOR_FORMAT_YUV422_SEMIPLANAR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV422_SEMIPLANAR

    #: R/G/B three channels in one surface with BGR byte ordering. Only
    #: pitch linear format supported.
    CU_EGL_COLOR_FORMAT_RGB = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_RGB

    #: R/G/B three channels in one surface with RGB byte ordering. Only
    #: pitch linear format supported.
    CU_EGL_COLOR_FORMAT_BGR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BGR

    #: R/G/B/A four channels in one surface with BGRA byte ordering.
    CU_EGL_COLOR_FORMAT_ARGB = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_ARGB

    #: R/G/B/A four channels in one surface with ABGR byte ordering.
    CU_EGL_COLOR_FORMAT_RGBA = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_RGBA

    #: single luminance channel in one surface.
    CU_EGL_COLOR_FORMAT_L = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_L

    #: single color channel in one surface.
    CU_EGL_COLOR_FORMAT_R = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_R

    #: Y, U, V in three surfaces, each in a separate surface, U/V width = Y
    #: width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YUV444_PLANAR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV444_PLANAR

    #: Y, UV in two surfaces (UV as one surface) with VU byte ordering,
    #: width, height ratio same as YUV444Planar.
    CU_EGL_COLOR_FORMAT_YUV444_SEMIPLANAR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV444_SEMIPLANAR

    #: Y, U, V in one surface, interleaved as UYVY in one channel.
    CU_EGL_COLOR_FORMAT_YUYV_422 = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUYV_422

    #: Y, U, V in one surface, interleaved as YUYV in one channel.
    CU_EGL_COLOR_FORMAT_UYVY_422 = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_UYVY_422

    #: R/G/B/A four channels in one surface with RGBA byte ordering.
    CU_EGL_COLOR_FORMAT_ABGR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_ABGR

    #: R/G/B/A four channels in one surface with ARGB byte ordering.
    CU_EGL_COLOR_FORMAT_BGRA = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BGRA

    #: Alpha color format - one channel in one surface.
    CU_EGL_COLOR_FORMAT_A = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_A

    #: R/G color format - two channels in one surface with GR byte ordering
    CU_EGL_COLOR_FORMAT_RG = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_RG

    #: Y, U, V, A four channels in one surface, interleaved as VUYA.
    CU_EGL_COLOR_FORMAT_AYUV = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_AYUV

    #: Y, VU in two surfaces (VU as one surface) with UV byte ordering, U/V
    #: width = Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YVU444_SEMIPLANAR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU444_SEMIPLANAR

    #: Y, VU in two surfaces (VU as one surface) with UV byte ordering, U/V
    #: width = 1/2 Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YVU422_SEMIPLANAR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU422_SEMIPLANAR

    #: Y, VU in two surfaces (VU as one surface) with UV byte ordering, U/V
    #: width = 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YVU420_SEMIPLANAR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU420_SEMIPLANAR

    #: Y10, V10U10 in two surfaces (VU as one surface) with UV byte
    #: ordering, U/V width = Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_444_SEMIPLANAR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_444_SEMIPLANAR

    #: Y10, V10U10 in two surfaces (VU as one surface) with UV byte
    #: ordering, U/V width = 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_420_SEMIPLANAR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_420_SEMIPLANAR

    #: Y12, V12U12 in two surfaces (VU as one surface) with UV byte
    #: ordering, U/V width = Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_Y12V12U12_444_SEMIPLANAR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y12V12U12_444_SEMIPLANAR

    #: Y12, V12U12 in two surfaces (VU as one surface) with UV byte
    #: ordering, U/V width = 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_Y12V12U12_420_SEMIPLANAR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y12V12U12_420_SEMIPLANAR

    #: Extended Range Y, U, V in one surface, interleaved as YVYU in one
    #: channel.
    CU_EGL_COLOR_FORMAT_VYUY_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_VYUY_ER

    #: Extended Range Y, U, V in one surface, interleaved as YUYV in one
    #: channel.
    CU_EGL_COLOR_FORMAT_UYVY_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_UYVY_ER

    #: Extended Range Y, U, V in one surface, interleaved as UYVY in one
    #: channel.
    CU_EGL_COLOR_FORMAT_YUYV_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUYV_ER

    #: Extended Range Y, U, V in one surface, interleaved as VYUY in one
    #: channel.
    CU_EGL_COLOR_FORMAT_YVYU_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVYU_ER

    #: Extended Range Y, U, V three channels in one surface, interleaved as
    #: VUY. Only pitch linear format supported.
    CU_EGL_COLOR_FORMAT_YUV_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV_ER

    #: Extended Range Y, U, V, A four channels in one surface, interleaved
    #: as AVUY.
    CU_EGL_COLOR_FORMAT_YUVA_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUVA_ER

    #: Extended Range Y, U, V, A four channels in one surface, interleaved
    #: as VUYA.
    CU_EGL_COLOR_FORMAT_AYUV_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_AYUV_ER

    #: Extended Range Y, U, V in three surfaces, U/V width = Y width, U/V
    #: height = Y height.
    CU_EGL_COLOR_FORMAT_YUV444_PLANAR_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV444_PLANAR_ER

    #: Extended Range Y, U, V in three surfaces, U/V width = 1/2 Y width,
    #: U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YUV422_PLANAR_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV422_PLANAR_ER

    #: Extended Range Y, U, V in three surfaces, U/V width = 1/2 Y width,
    #: U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YUV420_PLANAR_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV420_PLANAR_ER

    #: Extended Range Y, UV in two surfaces (UV as one surface) with VU
    #: byte ordering, U/V width = Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YUV444_SEMIPLANAR_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV444_SEMIPLANAR_ER

    #: Extended Range Y, UV in two surfaces (UV as one surface) with VU
    #: byte ordering, U/V width = 1/2 Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YUV422_SEMIPLANAR_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV422_SEMIPLANAR_ER

    #: Extended Range Y, UV in two surfaces (UV as one surface) with VU
    #: byte ordering, U/V width = 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YUV420_SEMIPLANAR_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV420_SEMIPLANAR_ER

    #: Extended Range Y, V, U in three surfaces, U/V width = Y width, U/V
    #: height = Y height.
    CU_EGL_COLOR_FORMAT_YVU444_PLANAR_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU444_PLANAR_ER

    #: Extended Range Y, V, U in three surfaces, U/V width = 1/2 Y width,
    #: U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YVU422_PLANAR_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU422_PLANAR_ER

    #: Extended Range Y, V, U in three surfaces, U/V width = 1/2 Y width,
    #: U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YVU420_PLANAR_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU420_PLANAR_ER

    #: Extended Range Y, VU in two surfaces (VU as one surface) with UV
    #: byte ordering, U/V width = Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YVU444_SEMIPLANAR_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU444_SEMIPLANAR_ER

    #: Extended Range Y, VU in two surfaces (VU as one surface) with UV
    #: byte ordering, U/V width = 1/2 Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YVU422_SEMIPLANAR_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU422_SEMIPLANAR_ER

    #: Extended Range Y, VU in two surfaces (VU as one surface) with UV
    #: byte ordering, U/V width = 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YVU420_SEMIPLANAR_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU420_SEMIPLANAR_ER

    #: Bayer format - one channel in one surface with interleaved RGGB
    #: ordering.
    CU_EGL_COLOR_FORMAT_BAYER_RGGB = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_RGGB

    #: Bayer format - one channel in one surface with interleaved BGGR
    #: ordering.
    CU_EGL_COLOR_FORMAT_BAYER_BGGR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_BGGR

    #: Bayer format - one channel in one surface with interleaved GRBG
    #: ordering.
    CU_EGL_COLOR_FORMAT_BAYER_GRBG = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_GRBG

    #: Bayer format - one channel in one surface with interleaved GBRG
    #: ordering.
    CU_EGL_COLOR_FORMAT_BAYER_GBRG = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_GBRG

    #: Bayer10 format - one channel in one surface with interleaved RGGB
    #: ordering. Out of 16 bits, 10 bits used 6 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER10_RGGB = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER10_RGGB

    #: Bayer10 format - one channel in one surface with interleaved BGGR
    #: ordering. Out of 16 bits, 10 bits used 6 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER10_BGGR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER10_BGGR

    #: Bayer10 format - one channel in one surface with interleaved GRBG
    #: ordering. Out of 16 bits, 10 bits used 6 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER10_GRBG = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER10_GRBG

    #: Bayer10 format - one channel in one surface with interleaved GBRG
    #: ordering. Out of 16 bits, 10 bits used 6 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER10_GBRG = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER10_GBRG

    #: Bayer12 format - one channel in one surface with interleaved RGGB
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER12_RGGB = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER12_RGGB

    #: Bayer12 format - one channel in one surface with interleaved BGGR
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER12_BGGR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER12_BGGR

    #: Bayer12 format - one channel in one surface with interleaved GRBG
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER12_GRBG = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER12_GRBG

    #: Bayer12 format - one channel in one surface with interleaved GBRG
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER12_GBRG = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER12_GBRG

    #: Bayer14 format - one channel in one surface with interleaved RGGB
    #: ordering. Out of 16 bits, 14 bits used 2 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER14_RGGB = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER14_RGGB

    #: Bayer14 format - one channel in one surface with interleaved BGGR
    #: ordering. Out of 16 bits, 14 bits used 2 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER14_BGGR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER14_BGGR

    #: Bayer14 format - one channel in one surface with interleaved GRBG
    #: ordering. Out of 16 bits, 14 bits used 2 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER14_GRBG = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER14_GRBG

    #: Bayer14 format - one channel in one surface with interleaved GBRG
    #: ordering. Out of 16 bits, 14 bits used 2 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER14_GBRG = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER14_GBRG

    #: Bayer20 format - one channel in one surface with interleaved RGGB
    #: ordering. Out of 32 bits, 20 bits used 12 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER20_RGGB = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER20_RGGB

    #: Bayer20 format - one channel in one surface with interleaved BGGR
    #: ordering. Out of 32 bits, 20 bits used 12 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER20_BGGR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER20_BGGR

    #: Bayer20 format - one channel in one surface with interleaved GRBG
    #: ordering. Out of 32 bits, 20 bits used 12 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER20_GRBG = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER20_GRBG

    #: Bayer20 format - one channel in one surface with interleaved GBRG
    #: ordering. Out of 32 bits, 20 bits used 12 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER20_GBRG = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER20_GBRG

    #: Y, V, U in three surfaces, each in a separate surface, U/V width = Y
    #: width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YVU444_PLANAR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU444_PLANAR

    #: Y, V, U in three surfaces, each in a separate surface, U/V width =
    #: 1/2 Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YVU422_PLANAR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU422_PLANAR

    #: Y, V, U in three surfaces, each in a separate surface, U/V width =
    #: 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YVU420_PLANAR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU420_PLANAR

    #: Nvidia proprietary Bayer ISP format - one channel in one surface
    #: with interleaved RGGB ordering and mapped to opaque integer
    #: datatype.
    CU_EGL_COLOR_FORMAT_BAYER_ISP_RGGB = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_ISP_RGGB

    #: Nvidia proprietary Bayer ISP format - one channel in one surface
    #: with interleaved BGGR ordering and mapped to opaque integer
    #: datatype.
    CU_EGL_COLOR_FORMAT_BAYER_ISP_BGGR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_ISP_BGGR

    #: Nvidia proprietary Bayer ISP format - one channel in one surface
    #: with interleaved GRBG ordering and mapped to opaque integer
    #: datatype.
    CU_EGL_COLOR_FORMAT_BAYER_ISP_GRBG = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_ISP_GRBG

    #: Nvidia proprietary Bayer ISP format - one channel in one surface
    #: with interleaved GBRG ordering and mapped to opaque integer
    #: datatype.
    CU_EGL_COLOR_FORMAT_BAYER_ISP_GBRG = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_ISP_GBRG

    #: Bayer format - one channel in one surface with interleaved BCCR
    #: ordering.
    CU_EGL_COLOR_FORMAT_BAYER_BCCR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_BCCR

    #: Bayer format - one channel in one surface with interleaved RCCB
    #: ordering.
    CU_EGL_COLOR_FORMAT_BAYER_RCCB = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_RCCB

    #: Bayer format - one channel in one surface with interleaved CRBC
    #: ordering.
    CU_EGL_COLOR_FORMAT_BAYER_CRBC = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_CRBC

    #: Bayer format - one channel in one surface with interleaved CBRC
    #: ordering.
    CU_EGL_COLOR_FORMAT_BAYER_CBRC = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_CBRC

    #: Bayer10 format - one channel in one surface with interleaved CCCC
    #: ordering. Out of 16 bits, 10 bits used 6 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER10_CCCC = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER10_CCCC

    #: Bayer12 format - one channel in one surface with interleaved BCCR
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER12_BCCR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER12_BCCR

    #: Bayer12 format - one channel in one surface with interleaved RCCB
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER12_RCCB = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER12_RCCB

    #: Bayer12 format - one channel in one surface with interleaved CRBC
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER12_CRBC = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER12_CRBC

    #: Bayer12 format - one channel in one surface with interleaved CBRC
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER12_CBRC = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER12_CBRC

    #: Bayer12 format - one channel in one surface with interleaved CCCC
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER12_CCCC = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER12_CCCC

    #: Color format for single Y plane.
    CU_EGL_COLOR_FORMAT_Y = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y

    #: Y, UV in two surfaces (UV as one surface) U/V width = 1/2 Y width,
    #: U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YUV420_SEMIPLANAR_2020 = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV420_SEMIPLANAR_2020

    #: Y, VU in two surfaces (VU as one surface) U/V width = 1/2 Y width,
    #: U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YVU420_SEMIPLANAR_2020 = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU420_SEMIPLANAR_2020

    #: Y, U, V each in a separate surface, U/V width = 1/2 Y width, U/V
    #: height= 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YUV420_PLANAR_2020 = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV420_PLANAR_2020

    #: Y, V, U each in a separate surface, U/V width = 1/2 Y width, U/V
    #: height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YVU420_PLANAR_2020 = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU420_PLANAR_2020

    #: Y, UV in two surfaces (UV as one surface) U/V width = 1/2 Y width,
    #: U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YUV420_SEMIPLANAR_709 = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV420_SEMIPLANAR_709

    #: Y, VU in two surfaces (VU as one surface) U/V width = 1/2 Y width,
    #: U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YVU420_SEMIPLANAR_709 = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU420_SEMIPLANAR_709

    #: Y, U, V each in a separate surface, U/V width = 1/2 Y width, U/V
    #: height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YUV420_PLANAR_709 = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV420_PLANAR_709

    #: Y, V, U each in a separate surface, U/V width = 1/2 Y width, U/V
    #: height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YVU420_PLANAR_709 = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU420_PLANAR_709

    #: Y10, V10U10 in two surfaces (VU as one surface), U/V width = 1/2 Y
    #: width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_420_SEMIPLANAR_709 = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_420_SEMIPLANAR_709

    #: Y10, V10U10 in two surfaces (VU as one surface), U/V width = 1/2 Y
    #: width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_420_SEMIPLANAR_2020 = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_420_SEMIPLANAR_2020

    #: Y10, V10U10 in two surfaces(VU as one surface) U/V width = 1/2 Y
    #: width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_422_SEMIPLANAR_2020 = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_422_SEMIPLANAR_2020

    #: Y10, V10U10 in two surfaces(VU as one surface) U/V width = 1/2 Y
    #: width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_422_SEMIPLANAR = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_422_SEMIPLANAR

    #: Y10, V10U10 in two surfaces(VU as one surface) U/V width = 1/2 Y
    #: width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_422_SEMIPLANAR_709 = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_422_SEMIPLANAR_709

    #: Extended Range Color format for single Y plane.
    CU_EGL_COLOR_FORMAT_Y_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y_ER

    #: Extended Range Color format for single Y plane.
    CU_EGL_COLOR_FORMAT_Y_709_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y_709_ER

    #: Extended Range Color format for single Y10 plane.
    CU_EGL_COLOR_FORMAT_Y10_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10_ER

    #: Extended Range Color format for single Y10 plane.
    CU_EGL_COLOR_FORMAT_Y10_709_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10_709_ER

    #: Extended Range Color format for single Y12 plane.
    CU_EGL_COLOR_FORMAT_Y12_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y12_ER

    #: Extended Range Color format for single Y12 plane.
    CU_EGL_COLOR_FORMAT_Y12_709_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y12_709_ER

    #: Y, U, V, A four channels in one surface, interleaved as AVUY.
    CU_EGL_COLOR_FORMAT_YUVA = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUVA

    #: Y, U, V three channels in one surface, interleaved as VUY. Only
    #: pitch linear format supported.
    CU_EGL_COLOR_FORMAT_YUV = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV

    #: Y, U, V in one surface, interleaved as YVYU in one channel.
    CU_EGL_COLOR_FORMAT_YVYU = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVYU

    #: Y, U, V in one surface, interleaved as VYUY in one channel.
    CU_EGL_COLOR_FORMAT_VYUY = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_VYUY

    #: Extended Range Y10, V10U10 in two surfaces(VU as one surface) U/V
    #: width = 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_420_SEMIPLANAR_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_420_SEMIPLANAR_ER

    #: Extended Range Y10, V10U10 in two surfaces(VU as one surface) U/V
    #: width = 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_420_SEMIPLANAR_709_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_420_SEMIPLANAR_709_ER

    #: Extended Range Y10, V10U10 in two surfaces (VU as one surface) U/V
    #: width = Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_444_SEMIPLANAR_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_444_SEMIPLANAR_ER

    #: Extended Range Y10, V10U10 in two surfaces (VU as one surface) U/V
    #: width = Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_444_SEMIPLANAR_709_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_444_SEMIPLANAR_709_ER

    #: Extended Range Y12, V12U12 in two surfaces (VU as one surface) U/V
    #: width = 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_Y12V12U12_420_SEMIPLANAR_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y12V12U12_420_SEMIPLANAR_ER

    #: Extended Range Y12, V12U12 in two surfaces (VU as one surface) U/V
    #: width = 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_Y12V12U12_420_SEMIPLANAR_709_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y12V12U12_420_SEMIPLANAR_709_ER

    #: Extended Range Y12, V12U12 in two surfaces (VU as one surface) U/V
    #: width = Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_Y12V12U12_444_SEMIPLANAR_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y12V12U12_444_SEMIPLANAR_ER

    #: Extended Range Y12, V12U12 in two surfaces (VU as one surface) U/V
    #: width = Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_Y12V12U12_444_SEMIPLANAR_709_ER = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y12V12U12_444_SEMIPLANAR_709_ER
    CU_EGL_COLOR_FORMAT_MAX = ccuda.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_MAX

class CUGLDeviceList(IntEnum):
    """
    CUDA devices corresponding to an OpenGL device
    """

    #: The CUDA devices for all GPUs used by the current OpenGL context
    CU_GL_DEVICE_LIST_ALL = ccuda.CUGLDeviceList_enum.CU_GL_DEVICE_LIST_ALL

    #: The CUDA devices for the GPUs used by the current OpenGL context in
    #: its currently rendering frame
    CU_GL_DEVICE_LIST_CURRENT_FRAME = ccuda.CUGLDeviceList_enum.CU_GL_DEVICE_LIST_CURRENT_FRAME

    #: The CUDA devices for the GPUs to be used by the current OpenGL
    #: context in the next frame
    CU_GL_DEVICE_LIST_NEXT_FRAME = ccuda.CUGLDeviceList_enum.CU_GL_DEVICE_LIST_NEXT_FRAME

class CUGLmap_flags(IntEnum):
    """
    Flags to map or unmap a resource
    """
    CU_GL_MAP_RESOURCE_FLAGS_NONE = ccuda.CUGLmap_flags_enum.CU_GL_MAP_RESOURCE_FLAGS_NONE
    CU_GL_MAP_RESOURCE_FLAGS_READ_ONLY = ccuda.CUGLmap_flags_enum.CU_GL_MAP_RESOURCE_FLAGS_READ_ONLY
    CU_GL_MAP_RESOURCE_FLAGS_WRITE_DISCARD = ccuda.CUGLmap_flags_enum.CU_GL_MAP_RESOURCE_FLAGS_WRITE_DISCARD

cdef class CUdeviceptr:
    """

    CUDA device pointer CUdeviceptr is defined as an unsigned integer type whose size matches the size of a pointer on the target platform.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned long long init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUdeviceptr *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<CUdeviceptr ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned long long>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUdevice:
    """

    CUDA device

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, int init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUdevice *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<CUdevice ' + str(self.__int__()) + '>'
    def __int__(self):
        return <int>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUtexObject:
    """

    An opaque value that represents a CUDA texture object

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned long long init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUtexObject *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<CUtexObject ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned long long>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUsurfObject:
    """

    An opaque value that represents a CUDA surface object

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned long long init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUsurfObject *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<CUsurfObject ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned long long>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUgraphConditionalHandle:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, uint64_t init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUgraphConditionalHandle *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<CUgraphConditionalHandle ' + str(self.__int__()) + '>'
    def __int__(self):
        return <uint64_t>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

class CUkernelNodeAttrID(IntEnum):
    """
    Launch attributes enum; used as id field of
    :py:obj:`~.CUlaunchAttribute`
    """

    #: Ignored entry, for convenient composition
    CU_LAUNCH_ATTRIBUTE_IGNORE = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_IGNORE

    #: Valid for streams, graph nodes, launches. See
    #: :py:obj:`~.CUlaunchAttributeValue.accessPolicyWindow`.
    CU_LAUNCH_ATTRIBUTE_ACCESS_POLICY_WINDOW = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_ACCESS_POLICY_WINDOW

    #: Valid for graph nodes, launches. See
    #: :py:obj:`~.CUlaunchAttributeValue.cooperative`.
    CU_LAUNCH_ATTRIBUTE_COOPERATIVE = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_COOPERATIVE

    #: Valid for streams. See
    #: :py:obj:`~.CUlaunchAttributeValue.syncPolicy`.
    CU_LAUNCH_ATTRIBUTE_SYNCHRONIZATION_POLICY = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_SYNCHRONIZATION_POLICY

    #: Valid for graph nodes, launches. See
    #: :py:obj:`~.CUlaunchAttributeValue.clusterDim`.
    CU_LAUNCH_ATTRIBUTE_CLUSTER_DIMENSION = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_CLUSTER_DIMENSION

    #: Valid for graph nodes, launches. See
    #: :py:obj:`~.CUlaunchAttributeValue.clusterSchedulingPolicyPreference`.
    CU_LAUNCH_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE

    #: Valid for launches. Setting
    #: :py:obj:`~.CUlaunchAttributeValue.programmaticStreamSerializationAllowed`
    #: to non-0 signals that the kernel will use programmatic means to
    #: resolve its stream dependency, so that the CUDA runtime should
    #: opportunistically allow the grid's execution to overlap with the
    #: previous kernel in the stream, if that kernel requests the overlap.
    #: The dependent launches can choose to wait on the dependency using
    #: the programmatic sync (cudaGridDependencySynchronize() or equivalent
    #: PTX instructions).
    CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_STREAM_SERIALIZATION = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_STREAM_SERIALIZATION

    #: Valid for launches. Set
    #: :py:obj:`~.CUlaunchAttributeValue.programmaticEvent` to record the
    #: event. Event recorded through this launch attribute is guaranteed to
    #: only trigger after all block in the associated kernel trigger the
    #: event. A block can trigger the event through PTX launchdep.release
    #: or CUDA builtin function cudaTriggerProgrammaticLaunchCompletion().
    #: A trigger can also be inserted at the beginning of each block's
    #: execution if triggerAtBlockStart is set to non-0. The dependent
    #: launches can choose to wait on the dependency using the programmatic
    #: sync (cudaGridDependencySynchronize() or equivalent PTX
    #: instructions). Note that dependents (including the CPU thread
    #: calling :py:obj:`~.cuEventSynchronize()`) are not guaranteed to
    #: observe the release precisely when it is released. For example,
    #: :py:obj:`~.cuEventSynchronize()` may only observe the event trigger
    #: long after the associated kernel has completed. This recording type
    #: is primarily meant for establishing programmatic dependency between
    #: device tasks. Note also this type of dependency allows, but does not
    #: guarantee, concurrent execution of tasks.
    #:  The event supplied must not be an interprocess or interop event.
    #: The event must disable timing (i.e. must be created with the
    #: :py:obj:`~.CU_EVENT_DISABLE_TIMING` flag set).
    CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_EVENT = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_EVENT

    #: Valid for streams, graph nodes, launches. See
    #: :py:obj:`~.CUlaunchAttributeValue.priority`.
    CU_LAUNCH_ATTRIBUTE_PRIORITY = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_PRIORITY

    #: Valid for streams, graph nodes, launches. See
    #: :py:obj:`~.CUlaunchAttributeValue.memSyncDomainMap`.
    CU_LAUNCH_ATTRIBUTE_MEM_SYNC_DOMAIN_MAP = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_MEM_SYNC_DOMAIN_MAP

    #: Valid for streams, graph nodes, launches. See
    #: :py:obj:`~.CUlaunchAttributeValue.memSyncDomain`.
    CU_LAUNCH_ATTRIBUTE_MEM_SYNC_DOMAIN = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_MEM_SYNC_DOMAIN

    #: Valid for launches. Set
    #: :py:obj:`~.CUlaunchAttributeValue.launchCompletionEvent` to record
    #: the event.
    #:  Nominally, the event is triggered once all blocks of the kernel
    #: have begun execution. Currently this is a best effort. If a kernel B
    #: has a launch completion dependency on a kernel A, B may wait until A
    #: is complete. Alternatively, blocks of B may begin before all blocks
    #: of A have begun, for example if B can claim execution resources
    #: unavailable to A (e.g. they run on different GPUs) or if B is a
    #: higher priority than A. Exercise caution if such an ordering
    #: inversion could lead to deadlock.
    #:  A launch completion event is nominally similar to a programmatic
    #: event with `triggerAtBlockStart` set except that it is not visible
    #: to `cudaGridDependencySynchronize()` and can be used with compute
    #: capability less than 9.0.
    #:  The event supplied must not be an interprocess or interop event.
    #: The event must disable timing (i.e. must be created with the
    #: :py:obj:`~.CU_EVENT_DISABLE_TIMING` flag set).
    CU_LAUNCH_ATTRIBUTE_LAUNCH_COMPLETION_EVENT = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_LAUNCH_COMPLETION_EVENT

    #: Valid for graph nodes, launches. This attribute is graphs-only, and
    #: passing it to a launch in a non-capturing stream will result in an
    #: error.
    #: :py:obj:`~.CUlaunchAttributeValue`::deviceUpdatableKernelNode::deviceUpdatable
    #: can only be set to 0 or 1. Setting the field to 1 indicates that the
    #: corresponding kernel node should be device-updatable. On success, a
    #: handle will be returned via
    #: :py:obj:`~.CUlaunchAttributeValue`::deviceUpdatableKernelNode::devNode
    #: which can be passed to the various device-side update functions to
    #: update the node's kernel parameters from within another kernel. For
    #: more information on the types of device updates that can be made, as
    #: well as the relevant limitations thereof, see
    #: :py:obj:`~.cudaGraphKernelNodeUpdatesApply`.
    #:  Nodes which are device-updatable have additional restrictions
    #: compared to regular kernel nodes. Firstly, device-updatable nodes
    #: cannot be removed from their graph via
    #: :py:obj:`~.cuGraphDestroyNode`. Additionally, once opted-in to this
    #: functionality, a node cannot opt out, and any attempt to set the
    #: deviceUpdatable attribute to 0 will result in an error. Device-
    #: updatable kernel nodes also cannot have their attributes copied
    #: to/from another kernel node via
    #: :py:obj:`~.cuGraphKernelNodeCopyAttributes`. Graphs containing one
    #: or more device-updatable nodes also do not allow multiple
    #: instantiation, and neither the graph nor its instantiated version
    #: can be passed to :py:obj:`~.cuGraphExecUpdate`.
    #:  If a graph contains device-updatable nodes and updates those nodes
    #: from the device from within the graph, the graph must be uploaded
    #: with :py:obj:`~.cuGraphUpload` before it is launched. For such a
    #: graph, if host-side executable graph updates are made to the device-
    #: updatable nodes, the graph must be uploaded before it is launched
    #: again.
    CU_LAUNCH_ATTRIBUTE_DEVICE_UPDATABLE_KERNEL_NODE = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_DEVICE_UPDATABLE_KERNEL_NODE

class CUstreamAttrID(IntEnum):
    """
    Launch attributes enum; used as id field of
    :py:obj:`~.CUlaunchAttribute`
    """

    #: Ignored entry, for convenient composition
    CU_LAUNCH_ATTRIBUTE_IGNORE = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_IGNORE

    #: Valid for streams, graph nodes, launches. See
    #: :py:obj:`~.CUlaunchAttributeValue.accessPolicyWindow`.
    CU_LAUNCH_ATTRIBUTE_ACCESS_POLICY_WINDOW = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_ACCESS_POLICY_WINDOW

    #: Valid for graph nodes, launches. See
    #: :py:obj:`~.CUlaunchAttributeValue.cooperative`.
    CU_LAUNCH_ATTRIBUTE_COOPERATIVE = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_COOPERATIVE

    #: Valid for streams. See
    #: :py:obj:`~.CUlaunchAttributeValue.syncPolicy`.
    CU_LAUNCH_ATTRIBUTE_SYNCHRONIZATION_POLICY = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_SYNCHRONIZATION_POLICY

    #: Valid for graph nodes, launches. See
    #: :py:obj:`~.CUlaunchAttributeValue.clusterDim`.
    CU_LAUNCH_ATTRIBUTE_CLUSTER_DIMENSION = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_CLUSTER_DIMENSION

    #: Valid for graph nodes, launches. See
    #: :py:obj:`~.CUlaunchAttributeValue.clusterSchedulingPolicyPreference`.
    CU_LAUNCH_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE

    #: Valid for launches. Setting
    #: :py:obj:`~.CUlaunchAttributeValue.programmaticStreamSerializationAllowed`
    #: to non-0 signals that the kernel will use programmatic means to
    #: resolve its stream dependency, so that the CUDA runtime should
    #: opportunistically allow the grid's execution to overlap with the
    #: previous kernel in the stream, if that kernel requests the overlap.
    #: The dependent launches can choose to wait on the dependency using
    #: the programmatic sync (cudaGridDependencySynchronize() or equivalent
    #: PTX instructions).
    CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_STREAM_SERIALIZATION = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_STREAM_SERIALIZATION

    #: Valid for launches. Set
    #: :py:obj:`~.CUlaunchAttributeValue.programmaticEvent` to record the
    #: event. Event recorded through this launch attribute is guaranteed to
    #: only trigger after all block in the associated kernel trigger the
    #: event. A block can trigger the event through PTX launchdep.release
    #: or CUDA builtin function cudaTriggerProgrammaticLaunchCompletion().
    #: A trigger can also be inserted at the beginning of each block's
    #: execution if triggerAtBlockStart is set to non-0. The dependent
    #: launches can choose to wait on the dependency using the programmatic
    #: sync (cudaGridDependencySynchronize() or equivalent PTX
    #: instructions). Note that dependents (including the CPU thread
    #: calling :py:obj:`~.cuEventSynchronize()`) are not guaranteed to
    #: observe the release precisely when it is released. For example,
    #: :py:obj:`~.cuEventSynchronize()` may only observe the event trigger
    #: long after the associated kernel has completed. This recording type
    #: is primarily meant for establishing programmatic dependency between
    #: device tasks. Note also this type of dependency allows, but does not
    #: guarantee, concurrent execution of tasks.
    #:  The event supplied must not be an interprocess or interop event.
    #: The event must disable timing (i.e. must be created with the
    #: :py:obj:`~.CU_EVENT_DISABLE_TIMING` flag set).
    CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_EVENT = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_EVENT

    #: Valid for streams, graph nodes, launches. See
    #: :py:obj:`~.CUlaunchAttributeValue.priority`.
    CU_LAUNCH_ATTRIBUTE_PRIORITY = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_PRIORITY

    #: Valid for streams, graph nodes, launches. See
    #: :py:obj:`~.CUlaunchAttributeValue.memSyncDomainMap`.
    CU_LAUNCH_ATTRIBUTE_MEM_SYNC_DOMAIN_MAP = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_MEM_SYNC_DOMAIN_MAP

    #: Valid for streams, graph nodes, launches. See
    #: :py:obj:`~.CUlaunchAttributeValue.memSyncDomain`.
    CU_LAUNCH_ATTRIBUTE_MEM_SYNC_DOMAIN = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_MEM_SYNC_DOMAIN

    #: Valid for launches. Set
    #: :py:obj:`~.CUlaunchAttributeValue.launchCompletionEvent` to record
    #: the event.
    #:  Nominally, the event is triggered once all blocks of the kernel
    #: have begun execution. Currently this is a best effort. If a kernel B
    #: has a launch completion dependency on a kernel A, B may wait until A
    #: is complete. Alternatively, blocks of B may begin before all blocks
    #: of A have begun, for example if B can claim execution resources
    #: unavailable to A (e.g. they run on different GPUs) or if B is a
    #: higher priority than A. Exercise caution if such an ordering
    #: inversion could lead to deadlock.
    #:  A launch completion event is nominally similar to a programmatic
    #: event with `triggerAtBlockStart` set except that it is not visible
    #: to `cudaGridDependencySynchronize()` and can be used with compute
    #: capability less than 9.0.
    #:  The event supplied must not be an interprocess or interop event.
    #: The event must disable timing (i.e. must be created with the
    #: :py:obj:`~.CU_EVENT_DISABLE_TIMING` flag set).
    CU_LAUNCH_ATTRIBUTE_LAUNCH_COMPLETION_EVENT = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_LAUNCH_COMPLETION_EVENT

    #: Valid for graph nodes, launches. This attribute is graphs-only, and
    #: passing it to a launch in a non-capturing stream will result in an
    #: error.
    #: :py:obj:`~.CUlaunchAttributeValue`::deviceUpdatableKernelNode::deviceUpdatable
    #: can only be set to 0 or 1. Setting the field to 1 indicates that the
    #: corresponding kernel node should be device-updatable. On success, a
    #: handle will be returned via
    #: :py:obj:`~.CUlaunchAttributeValue`::deviceUpdatableKernelNode::devNode
    #: which can be passed to the various device-side update functions to
    #: update the node's kernel parameters from within another kernel. For
    #: more information on the types of device updates that can be made, as
    #: well as the relevant limitations thereof, see
    #: :py:obj:`~.cudaGraphKernelNodeUpdatesApply`.
    #:  Nodes which are device-updatable have additional restrictions
    #: compared to regular kernel nodes. Firstly, device-updatable nodes
    #: cannot be removed from their graph via
    #: :py:obj:`~.cuGraphDestroyNode`. Additionally, once opted-in to this
    #: functionality, a node cannot opt out, and any attempt to set the
    #: deviceUpdatable attribute to 0 will result in an error. Device-
    #: updatable kernel nodes also cannot have their attributes copied
    #: to/from another kernel node via
    #: :py:obj:`~.cuGraphKernelNodeCopyAttributes`. Graphs containing one
    #: or more device-updatable nodes also do not allow multiple
    #: instantiation, and neither the graph nor its instantiated version
    #: can be passed to :py:obj:`~.cuGraphExecUpdate`.
    #:  If a graph contains device-updatable nodes and updates those nodes
    #: from the device from within the graph, the graph must be uploaded
    #: with :py:obj:`~.cuGraphUpload` before it is launched. For such a
    #: graph, if host-side executable graph updates are made to the device-
    #: updatable nodes, the graph must be uploaded before it is launched
    #: again.
    CU_LAUNCH_ATTRIBUTE_DEVICE_UPDATABLE_KERNEL_NODE = ccuda.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_DEVICE_UPDATABLE_KERNEL_NODE

cdef class CUmemGenericAllocationHandle:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned long long init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUmemGenericAllocationHandle *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<CUmemGenericAllocationHandle ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned long long>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUcontext:
    """

    CUDA context

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUcontext>init_value
        else:
            self._ptr = <ccuda.CUcontext *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUcontext ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUmodule:
    """

    CUDA module

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUmodule>init_value
        else:
            self._ptr = <ccuda.CUmodule *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUmodule ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUfunction:
    """

    CUDA function

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUfunction>init_value
        else:
            self._ptr = <ccuda.CUfunction *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUfunction ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUlibrary:
    """

    CUDA library

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUlibrary>init_value
        else:
            self._ptr = <ccuda.CUlibrary *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUlibrary ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUkernel:
    """

    CUDA kernel

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUkernel>init_value
        else:
            self._ptr = <ccuda.CUkernel *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUkernel ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUarray:
    """

    CUDA array

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUarray>init_value
        else:
            self._ptr = <ccuda.CUarray *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUarray ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUmipmappedArray:
    """

    CUDA mipmapped array

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUmipmappedArray>init_value
        else:
            self._ptr = <ccuda.CUmipmappedArray *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUmipmappedArray ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUtexref:
    """

    CUDA texture reference

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUtexref>init_value
        else:
            self._ptr = <ccuda.CUtexref *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUtexref ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUsurfref:
    """

    CUDA surface reference

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUsurfref>init_value
        else:
            self._ptr = <ccuda.CUsurfref *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUsurfref ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUevent:
    """

    CUDA event

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUevent>init_value
        else:
            self._ptr = <ccuda.CUevent *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUevent ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUstream:
    """

    CUDA stream

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUstream>init_value
        else:
            self._ptr = <ccuda.CUstream *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUstream ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUgraphicsResource:
    """

    CUDA graphics interop resource

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUgraphicsResource>init_value
        else:
            self._ptr = <ccuda.CUgraphicsResource *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUgraphicsResource ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUexternalMemory:
    """

    CUDA external memory

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUexternalMemory>init_value
        else:
            self._ptr = <ccuda.CUexternalMemory *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUexternalMemory ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUexternalSemaphore:
    """

    CUDA external semaphore

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUexternalSemaphore>init_value
        else:
            self._ptr = <ccuda.CUexternalSemaphore *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUexternalSemaphore ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUgraph:
    """

    CUDA graph

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUgraph>init_value
        else:
            self._ptr = <ccuda.CUgraph *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUgraph ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUgraphNode:
    """

    CUDA graph node

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUgraphNode>init_value
        else:
            self._ptr = <ccuda.CUgraphNode *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUgraphNode ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUgraphExec:
    """

    CUDA executable graph

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUgraphExec>init_value
        else:
            self._ptr = <ccuda.CUgraphExec *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUgraphExec ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUmemoryPool:
    """

    CUDA memory pool

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUmemoryPool>init_value
        else:
            self._ptr = <ccuda.CUmemoryPool *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUmemoryPool ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUuserObject:
    """

    CUDA user object for graphs

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUuserObject>init_value
        else:
            self._ptr = <ccuda.CUuserObject *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUuserObject ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUgraphDeviceNode:
    """

    CUDA graph device node handle

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUgraphDeviceNode>init_value
        else:
            self._ptr = <ccuda.CUgraphDeviceNode *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUgraphDeviceNode ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUasyncCallbackHandle:
    """

    CUDA async notification callback handle

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUasyncCallbackHandle>init_value
        else:
            self._ptr = <ccuda.CUasyncCallbackHandle *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUasyncCallbackHandle ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUlinkState:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUlinkState>init_value
        else:
            self._ptr = <ccuda.CUlinkState *>_ptr
    def __init__(self, *args, **kwargs):
        self._keepalive = []
    def __repr__(self):
        return '<CUlinkState ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUgreenCtx:
    """

    A green context handle. This handle can be used safely from only one CPU thread at a time. Created via cuGreenCtxCreate

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUgreenCtx>init_value
        else:
            self._ptr = <ccuda.CUgreenCtx *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUgreenCtx ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUdevResourceDesc:
    """

    An opaque descriptor handle. The descriptor encapsulates multiple created and configured resources. Created via cuDevResourceGenerateDesc

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUdevResourceDesc>init_value
        else:
            self._ptr = <ccuda.CUdevResourceDesc *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUdevResourceDesc ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUeglStreamConnection:
    """

    CUDA EGLSream Connection

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUeglStreamConnection>init_value
        else:
            self._ptr = <ccuda.CUeglStreamConnection *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUeglStreamConnection ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class EGLImageKHR:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.EGLImageKHR>init_value
        else:
            self._ptr = <ccuda.EGLImageKHR *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<EGLImageKHR ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class EGLStreamKHR:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.EGLStreamKHR>init_value
        else:
            self._ptr = <ccuda.EGLStreamKHR *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<EGLStreamKHR ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class EGLSyncKHR:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.EGLSyncKHR>init_value
        else:
            self._ptr = <ccuda.EGLSyncKHR *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<EGLSyncKHR ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUasyncCallback:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUasyncCallback>init_value
        else:
            self._ptr = <ccuda.CUasyncCallback *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUasyncCallback ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUhostFn:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUhostFn>init_value
        else:
            self._ptr = <ccuda.CUhostFn *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUhostFn ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUstreamCallback:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUstreamCallback>init_value
        else:
            self._ptr = <ccuda.CUstreamCallback *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUstreamCallback ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUoccupancyB2DSize:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccuda.CUoccupancyB2DSize>init_value
        else:
            self._ptr = <ccuda.CUoccupancyB2DSize *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUoccupancyB2DSize ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUuuid_st:
    """
    Attributes
    ----------
    bytes : bytes
        < CUDA definition of UUID

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUuuid_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['bytes : ' + str(self.bytes.hex())]
            except ValueError:
                str_list += ['bytes : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def bytes(self):
        return PyBytes_FromStringAndSize(self._ptr[0].bytes, 16)

cdef class CUmemFabricHandle_st:
    """
    Fabric handle - An opaque handle representing a memory allocation
    that can be exported to processes in same or different nodes. For
    IPC between processes on different nodes they must be connected via
    the NVSwitch fabric.

    Attributes
    ----------
    data : bytes


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUmemFabricHandle_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['data : ' + str(self.data)]
            except ValueError:
                str_list += ['data : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def data(self):
        return PyBytes_FromStringAndSize(<char*>self._ptr[0].data, 64)
    @data.setter
    def data(self, data):
        if len(data) != 64:
            raise ValueError("data length must be 64, is " + str(len(data)))
        for i, b in enumerate(data):
            self._ptr[0].data[i] = b

cdef class CUipcEventHandle_st:
    """
    CUDA IPC event handle

    Attributes
    ----------
    reserved : bytes


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUipcEventHandle_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def reserved(self):
        return PyBytes_FromStringAndSize(self._ptr[0].reserved, 64)
    @reserved.setter
    def reserved(self, reserved):
        if len(reserved) != 64:
            raise ValueError("reserved length must be 64, is " + str(len(reserved)))
        if CHAR_MIN == 0:
            for i, b in enumerate(reserved):
                if b < 0 and b > -129:
                    b = b + 256
                self._ptr[0].reserved[i] = b
        else:
            for i, b in enumerate(reserved):
                if b > 127 and b < 256:
                    b = b - 256
                self._ptr[0].reserved[i] = b

cdef class CUipcMemHandle_st:
    """
    CUDA IPC mem handle

    Attributes
    ----------
    reserved : bytes


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUipcMemHandle_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def reserved(self):
        return PyBytes_FromStringAndSize(self._ptr[0].reserved, 64)
    @reserved.setter
    def reserved(self, reserved):
        if len(reserved) != 64:
            raise ValueError("reserved length must be 64, is " + str(len(reserved)))
        if CHAR_MIN == 0:
            for i, b in enumerate(reserved):
                if b < 0 and b > -129:
                    b = b + 256
                self._ptr[0].reserved[i] = b
        else:
            for i, b in enumerate(reserved):
                if b > 127 and b < 256:
                    b = b - 256
                self._ptr[0].reserved[i] = b

cdef class CUstreamMemOpWaitValueParams_st:
    """
    Attributes
    ----------
    operation : CUstreamBatchMemOpType

    address : CUdeviceptr

    value : cuuint32_t

    value64 : cuuint64_t

    flags : unsigned int

    alias : CUdeviceptr
        For driver internal use. Initial value is unimportant.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUstreamBatchMemOpParams_union *>_ptr

    def __init__(self, void_ptr _ptr):
        self._address = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].waitValue.address)
        self._value = cuuint32_t(_ptr=<void_ptr>&self._ptr[0].waitValue.value)
        self._value64 = cuuint64_t(_ptr=<void_ptr>&self._ptr[0].waitValue.value64)
        self._alias = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].waitValue.alias)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].waitValue
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['operation : ' + str(self.operation)]
            except ValueError:
                str_list += ['operation : <ValueError>']
            try:
                str_list += ['address : ' + str(self.address)]
            except ValueError:
                str_list += ['address : <ValueError>']
            try:
                str_list += ['value : ' + str(self.value)]
            except ValueError:
                str_list += ['value : <ValueError>']
            try:
                str_list += ['value64 : ' + str(self.value64)]
            except ValueError:
                str_list += ['value64 : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['alias : ' + str(self.alias)]
            except ValueError:
                str_list += ['alias : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def operation(self):
        return CUstreamBatchMemOpType(self._ptr[0].waitValue.operation)
    @operation.setter
    def operation(self, operation not None : CUstreamBatchMemOpType):
        self._ptr[0].waitValue.operation = operation.value
    @property
    def address(self):
        return self._address
    @address.setter
    def address(self, address):
        cdef ccuda.CUdeviceptr caddress
        if address is None:
            caddress = <ccuda.CUdeviceptr><void_ptr>0
        elif isinstance(address, (CUdeviceptr)):
            paddress = int(address)
            caddress = <ccuda.CUdeviceptr><void_ptr>paddress
        else:
            paddress = int(CUdeviceptr(address))
            caddress = <ccuda.CUdeviceptr><void_ptr>paddress
        self._address._ptr[0] = caddress

    @property
    def value(self):
        return self._value
    @value.setter
    def value(self, value):
        cdef ccuda.cuuint32_t cvalue
        if value is None:
            cvalue = <ccuda.cuuint32_t><void_ptr>0
        elif isinstance(value, (cuuint32_t)):
            pvalue = int(value)
            cvalue = <ccuda.cuuint32_t><void_ptr>pvalue
        else:
            pvalue = int(cuuint32_t(value))
            cvalue = <ccuda.cuuint32_t><void_ptr>pvalue
        self._value._ptr[0] = cvalue

    @property
    def value64(self):
        return self._value64
    @value64.setter
    def value64(self, value64):
        cdef ccuda.cuuint64_t cvalue64
        if value64 is None:
            cvalue64 = <ccuda.cuuint64_t><void_ptr>0
        elif isinstance(value64, (cuuint64_t)):
            pvalue64 = int(value64)
            cvalue64 = <ccuda.cuuint64_t><void_ptr>pvalue64
        else:
            pvalue64 = int(cuuint64_t(value64))
            cvalue64 = <ccuda.cuuint64_t><void_ptr>pvalue64
        self._value64._ptr[0] = cvalue64

    @property
    def flags(self):
        return self._ptr[0].waitValue.flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].waitValue.flags = flags
    @property
    def alias(self):
        return self._alias
    @alias.setter
    def alias(self, alias):
        cdef ccuda.CUdeviceptr calias
        if alias is None:
            calias = <ccuda.CUdeviceptr><void_ptr>0
        elif isinstance(alias, (CUdeviceptr)):
            palias = int(alias)
            calias = <ccuda.CUdeviceptr><void_ptr>palias
        else:
            palias = int(CUdeviceptr(alias))
            calias = <ccuda.CUdeviceptr><void_ptr>palias
        self._alias._ptr[0] = calias

cdef class CUstreamMemOpWriteValueParams_st:
    """
    Attributes
    ----------
    operation : CUstreamBatchMemOpType

    address : CUdeviceptr

    value : cuuint32_t

    value64 : cuuint64_t

    flags : unsigned int

    alias : CUdeviceptr
        For driver internal use. Initial value is unimportant.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUstreamBatchMemOpParams_union *>_ptr

    def __init__(self, void_ptr _ptr):
        self._address = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].writeValue.address)
        self._value = cuuint32_t(_ptr=<void_ptr>&self._ptr[0].writeValue.value)
        self._value64 = cuuint64_t(_ptr=<void_ptr>&self._ptr[0].writeValue.value64)
        self._alias = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].writeValue.alias)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].writeValue
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['operation : ' + str(self.operation)]
            except ValueError:
                str_list += ['operation : <ValueError>']
            try:
                str_list += ['address : ' + str(self.address)]
            except ValueError:
                str_list += ['address : <ValueError>']
            try:
                str_list += ['value : ' + str(self.value)]
            except ValueError:
                str_list += ['value : <ValueError>']
            try:
                str_list += ['value64 : ' + str(self.value64)]
            except ValueError:
                str_list += ['value64 : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['alias : ' + str(self.alias)]
            except ValueError:
                str_list += ['alias : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def operation(self):
        return CUstreamBatchMemOpType(self._ptr[0].writeValue.operation)
    @operation.setter
    def operation(self, operation not None : CUstreamBatchMemOpType):
        self._ptr[0].writeValue.operation = operation.value
    @property
    def address(self):
        return self._address
    @address.setter
    def address(self, address):
        cdef ccuda.CUdeviceptr caddress
        if address is None:
            caddress = <ccuda.CUdeviceptr><void_ptr>0
        elif isinstance(address, (CUdeviceptr)):
            paddress = int(address)
            caddress = <ccuda.CUdeviceptr><void_ptr>paddress
        else:
            paddress = int(CUdeviceptr(address))
            caddress = <ccuda.CUdeviceptr><void_ptr>paddress
        self._address._ptr[0] = caddress

    @property
    def value(self):
        return self._value
    @value.setter
    def value(self, value):
        cdef ccuda.cuuint32_t cvalue
        if value is None:
            cvalue = <ccuda.cuuint32_t><void_ptr>0
        elif isinstance(value, (cuuint32_t)):
            pvalue = int(value)
            cvalue = <ccuda.cuuint32_t><void_ptr>pvalue
        else:
            pvalue = int(cuuint32_t(value))
            cvalue = <ccuda.cuuint32_t><void_ptr>pvalue
        self._value._ptr[0] = cvalue

    @property
    def value64(self):
        return self._value64
    @value64.setter
    def value64(self, value64):
        cdef ccuda.cuuint64_t cvalue64
        if value64 is None:
            cvalue64 = <ccuda.cuuint64_t><void_ptr>0
        elif isinstance(value64, (cuuint64_t)):
            pvalue64 = int(value64)
            cvalue64 = <ccuda.cuuint64_t><void_ptr>pvalue64
        else:
            pvalue64 = int(cuuint64_t(value64))
            cvalue64 = <ccuda.cuuint64_t><void_ptr>pvalue64
        self._value64._ptr[0] = cvalue64

    @property
    def flags(self):
        return self._ptr[0].writeValue.flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].writeValue.flags = flags
    @property
    def alias(self):
        return self._alias
    @alias.setter
    def alias(self, alias):
        cdef ccuda.CUdeviceptr calias
        if alias is None:
            calias = <ccuda.CUdeviceptr><void_ptr>0
        elif isinstance(alias, (CUdeviceptr)):
            palias = int(alias)
            calias = <ccuda.CUdeviceptr><void_ptr>palias
        else:
            palias = int(CUdeviceptr(alias))
            calias = <ccuda.CUdeviceptr><void_ptr>palias
        self._alias._ptr[0] = calias

cdef class CUstreamMemOpFlushRemoteWritesParams_st:
    """
    Attributes
    ----------
    operation : CUstreamBatchMemOpType

    flags : unsigned int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUstreamBatchMemOpParams_union *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].flushRemoteWrites
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['operation : ' + str(self.operation)]
            except ValueError:
                str_list += ['operation : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def operation(self):
        return CUstreamBatchMemOpType(self._ptr[0].flushRemoteWrites.operation)
    @operation.setter
    def operation(self, operation not None : CUstreamBatchMemOpType):
        self._ptr[0].flushRemoteWrites.operation = operation.value
    @property
    def flags(self):
        return self._ptr[0].flushRemoteWrites.flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flushRemoteWrites.flags = flags

cdef class CUstreamMemOpMemoryBarrierParams_st:
    """
    Attributes
    ----------
    operation : CUstreamBatchMemOpType
        < Only supported in the _v2 API
    flags : unsigned int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUstreamBatchMemOpParams_union *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].memoryBarrier
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['operation : ' + str(self.operation)]
            except ValueError:
                str_list += ['operation : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def operation(self):
        return CUstreamBatchMemOpType(self._ptr[0].memoryBarrier.operation)
    @operation.setter
    def operation(self, operation not None : CUstreamBatchMemOpType):
        self._ptr[0].memoryBarrier.operation = operation.value
    @property
    def flags(self):
        return self._ptr[0].memoryBarrier.flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].memoryBarrier.flags = flags

cdef class CUstreamBatchMemOpParams_union:
    """
    Per-operation parameters for cuStreamBatchMemOp

    Attributes
    ----------
    operation : CUstreamBatchMemOpType

    waitValue : CUstreamMemOpWaitValueParams_st

    writeValue : CUstreamMemOpWriteValueParams_st

    flushRemoteWrites : CUstreamMemOpFlushRemoteWritesParams_st

    memoryBarrier : CUstreamMemOpMemoryBarrierParams_st

    pad : List[cuuint64_t]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUstreamBatchMemOpParams_union *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._waitValue = CUstreamMemOpWaitValueParams_st(_ptr=<void_ptr>self._ptr)
        self._writeValue = CUstreamMemOpWriteValueParams_st(_ptr=<void_ptr>self._ptr)
        self._flushRemoteWrites = CUstreamMemOpFlushRemoteWritesParams_st(_ptr=<void_ptr>self._ptr)
        self._memoryBarrier = CUstreamMemOpMemoryBarrierParams_st(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['operation : ' + str(self.operation)]
            except ValueError:
                str_list += ['operation : <ValueError>']
            try:
                str_list += ['waitValue :\n' + '\n'.join(['    ' + line for line in str(self.waitValue).splitlines()])]
            except ValueError:
                str_list += ['waitValue : <ValueError>']
            try:
                str_list += ['writeValue :\n' + '\n'.join(['    ' + line for line in str(self.writeValue).splitlines()])]
            except ValueError:
                str_list += ['writeValue : <ValueError>']
            try:
                str_list += ['flushRemoteWrites :\n' + '\n'.join(['    ' + line for line in str(self.flushRemoteWrites).splitlines()])]
            except ValueError:
                str_list += ['flushRemoteWrites : <ValueError>']
            try:
                str_list += ['memoryBarrier :\n' + '\n'.join(['    ' + line for line in str(self.memoryBarrier).splitlines()])]
            except ValueError:
                str_list += ['memoryBarrier : <ValueError>']
            try:
                str_list += ['pad : ' + str(self.pad)]
            except ValueError:
                str_list += ['pad : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def operation(self):
        return CUstreamBatchMemOpType(self._ptr[0].operation)
    @operation.setter
    def operation(self, operation not None : CUstreamBatchMemOpType):
        self._ptr[0].operation = operation.value
    @property
    def waitValue(self):
        return self._waitValue
    @waitValue.setter
    def waitValue(self, waitValue not None : CUstreamMemOpWaitValueParams_st):
        string.memcpy(&self._ptr[0].waitValue, <ccuda.CUstreamMemOpWaitValueParams_st*><void_ptr>waitValue.getPtr(), sizeof(self._ptr[0].waitValue))
    @property
    def writeValue(self):
        return self._writeValue
    @writeValue.setter
    def writeValue(self, writeValue not None : CUstreamMemOpWriteValueParams_st):
        string.memcpy(&self._ptr[0].writeValue, <ccuda.CUstreamMemOpWriteValueParams_st*><void_ptr>writeValue.getPtr(), sizeof(self._ptr[0].writeValue))
    @property
    def flushRemoteWrites(self):
        return self._flushRemoteWrites
    @flushRemoteWrites.setter
    def flushRemoteWrites(self, flushRemoteWrites not None : CUstreamMemOpFlushRemoteWritesParams_st):
        string.memcpy(&self._ptr[0].flushRemoteWrites, <ccuda.CUstreamMemOpFlushRemoteWritesParams_st*><void_ptr>flushRemoteWrites.getPtr(), sizeof(self._ptr[0].flushRemoteWrites))
    @property
    def memoryBarrier(self):
        return self._memoryBarrier
    @memoryBarrier.setter
    def memoryBarrier(self, memoryBarrier not None : CUstreamMemOpMemoryBarrierParams_st):
        string.memcpy(&self._ptr[0].memoryBarrier, <ccuda.CUstreamMemOpMemoryBarrierParams_st*><void_ptr>memoryBarrier.getPtr(), sizeof(self._ptr[0].memoryBarrier))
    @property
    def pad(self):
        return [cuuint64_t(init_value=_pad) for _pad in self._ptr[0].pad]
    @pad.setter
    def pad(self, pad):
        self._ptr[0].pad = pad

cdef class CUDA_BATCH_MEM_OP_NODE_PARAMS_v1_st:
    """
    Attributes
    ----------
    ctx : CUcontext

    count : unsigned int

    paramArray : CUstreamBatchMemOpParams

    flags : unsigned int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_BATCH_MEM_OP_NODE_PARAMS_v1_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._ctx = CUcontext(_ptr=<void_ptr>&self._ptr[0].ctx)
    def __dealloc__(self):
        if self._paramArray is not NULL:
            free(self._paramArray)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['ctx : ' + str(self.ctx)]
            except ValueError:
                str_list += ['ctx : <ValueError>']
            try:
                str_list += ['count : ' + str(self.count)]
            except ValueError:
                str_list += ['count : <ValueError>']
            try:
                str_list += ['paramArray : ' + str(self.paramArray)]
            except ValueError:
                str_list += ['paramArray : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def ctx(self):
        return self._ctx
    @ctx.setter
    def ctx(self, ctx):
        cdef ccuda.CUcontext cctx
        if ctx is None:
            cctx = <ccuda.CUcontext><void_ptr>0
        elif isinstance(ctx, (CUcontext,)):
            pctx = int(ctx)
            cctx = <ccuda.CUcontext><void_ptr>pctx
        else:
            pctx = int(CUcontext(ctx))
            cctx = <ccuda.CUcontext><void_ptr>pctx
        self._ctx._ptr[0] = cctx
    @property
    def count(self):
        return self._ptr[0].count
    @count.setter
    def count(self, unsigned int count):
        self._ptr[0].count = count
    @property
    def paramArray(self):
        arrs = [<void_ptr>self._ptr[0].paramArray + x*sizeof(ccuda.CUstreamBatchMemOpParams) for x in range(self._paramArray_length)]
        return [CUstreamBatchMemOpParams(_ptr=arr) for arr in arrs]
    @paramArray.setter
    def paramArray(self, val):
        if len(val) == 0:
            free(self._paramArray)
            self._paramArray_length = 0
            self._ptr[0].paramArray = NULL
        else:
            if self._paramArray_length != <size_t>len(val):
                free(self._paramArray)
                self._paramArray = <ccuda.CUstreamBatchMemOpParams*> calloc(len(val), sizeof(ccuda.CUstreamBatchMemOpParams))
                if self._paramArray is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(ccuda.CUstreamBatchMemOpParams)))
                self._paramArray_length = <size_t>len(val)
                self._ptr[0].paramArray = self._paramArray
            for idx in range(len(val)):
                string.memcpy(&self._paramArray[idx], (<CUstreamBatchMemOpParams>val[idx])._ptr, sizeof(ccuda.CUstreamBatchMemOpParams))

    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flags = flags

cdef class CUDA_BATCH_MEM_OP_NODE_PARAMS_v2_st:
    """
    Batch memory operation node parameters

    Attributes
    ----------
    ctx : CUcontext
        Context to use for the operations.
    count : unsigned int
        Number of operations in paramArray.
    paramArray : CUstreamBatchMemOpParams
        Array of batch memory operations.
    flags : unsigned int
        Flags to control the node.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_BATCH_MEM_OP_NODE_PARAMS_v2_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._ctx = CUcontext(_ptr=<void_ptr>&self._ptr[0].ctx)
    def __dealloc__(self):
        if self._paramArray is not NULL:
            free(self._paramArray)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['ctx : ' + str(self.ctx)]
            except ValueError:
                str_list += ['ctx : <ValueError>']
            try:
                str_list += ['count : ' + str(self.count)]
            except ValueError:
                str_list += ['count : <ValueError>']
            try:
                str_list += ['paramArray : ' + str(self.paramArray)]
            except ValueError:
                str_list += ['paramArray : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def ctx(self):
        return self._ctx
    @ctx.setter
    def ctx(self, ctx):
        cdef ccuda.CUcontext cctx
        if ctx is None:
            cctx = <ccuda.CUcontext><void_ptr>0
        elif isinstance(ctx, (CUcontext,)):
            pctx = int(ctx)
            cctx = <ccuda.CUcontext><void_ptr>pctx
        else:
            pctx = int(CUcontext(ctx))
            cctx = <ccuda.CUcontext><void_ptr>pctx
        self._ctx._ptr[0] = cctx
    @property
    def count(self):
        return self._ptr[0].count
    @count.setter
    def count(self, unsigned int count):
        self._ptr[0].count = count
    @property
    def paramArray(self):
        arrs = [<void_ptr>self._ptr[0].paramArray + x*sizeof(ccuda.CUstreamBatchMemOpParams) for x in range(self._paramArray_length)]
        return [CUstreamBatchMemOpParams(_ptr=arr) for arr in arrs]
    @paramArray.setter
    def paramArray(self, val):
        if len(val) == 0:
            free(self._paramArray)
            self._paramArray_length = 0
            self._ptr[0].paramArray = NULL
        else:
            if self._paramArray_length != <size_t>len(val):
                free(self._paramArray)
                self._paramArray = <ccuda.CUstreamBatchMemOpParams*> calloc(len(val), sizeof(ccuda.CUstreamBatchMemOpParams))
                if self._paramArray is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(ccuda.CUstreamBatchMemOpParams)))
                self._paramArray_length = <size_t>len(val)
                self._ptr[0].paramArray = self._paramArray
            for idx in range(len(val)):
                string.memcpy(&self._paramArray[idx], (<CUstreamBatchMemOpParams>val[idx])._ptr, sizeof(ccuda.CUstreamBatchMemOpParams))

    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flags = flags

cdef class anon_struct0:
    """
    Attributes
    ----------
    bytesOverBudget : unsigned long long


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUasyncNotificationInfo_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].info.overBudget
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['bytesOverBudget : ' + str(self.bytesOverBudget)]
            except ValueError:
                str_list += ['bytesOverBudget : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def bytesOverBudget(self):
        return self._ptr[0].info.overBudget.bytesOverBudget
    @bytesOverBudget.setter
    def bytesOverBudget(self, unsigned long long bytesOverBudget):
        self._ptr[0].info.overBudget.bytesOverBudget = bytesOverBudget

cdef class anon_union2:
    """
    Attributes
    ----------
    overBudget : anon_struct0


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUasyncNotificationInfo_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._overBudget = anon_struct0(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].info
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['overBudget :\n' + '\n'.join(['    ' + line for line in str(self.overBudget).splitlines()])]
            except ValueError:
                str_list += ['overBudget : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def overBudget(self):
        return self._overBudget
    @overBudget.setter
    def overBudget(self, overBudget not None : anon_struct0):
        string.memcpy(&self._ptr[0].info.overBudget, <ccuda.anon_struct0*><void_ptr>overBudget.getPtr(), sizeof(self._ptr[0].info.overBudget))

cdef class CUasyncNotificationInfo_st:
    """
    Information passed to the user via the async notification callback

    Attributes
    ----------
    type : CUasyncNotificationType

    info : anon_union2


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._val_ptr = <ccuda.CUasyncNotificationInfo_st *>calloc(1, sizeof(ccuda.CUasyncNotificationInfo_st))
            self._ptr = self._val_ptr
        else:
            self._ptr = <ccuda.CUasyncNotificationInfo_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._info = anon_union2(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        if self._val_ptr is not NULL:
            free(self._val_ptr)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['type : ' + str(self.type)]
            except ValueError:
                str_list += ['type : <ValueError>']
            try:
                str_list += ['info :\n' + '\n'.join(['    ' + line for line in str(self.info).splitlines()])]
            except ValueError:
                str_list += ['info : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def type(self):
        return CUasyncNotificationType(self._ptr[0].type)
    @type.setter
    def type(self, type not None : CUasyncNotificationType):
        self._ptr[0].type = type.value
    @property
    def info(self):
        return self._info
    @info.setter
    def info(self, info not None : anon_union2):
        string.memcpy(&self._ptr[0].info, <ccuda.anon_union2*><void_ptr>info.getPtr(), sizeof(self._ptr[0].info))

cdef class CUdevprop_st:
    """
    Legacy device properties

    Attributes
    ----------
    maxThreadsPerBlock : int
        Maximum number of threads per block
    maxThreadsDim : List[int]
        Maximum size of each dimension of a block
    maxGridSize : List[int]
        Maximum size of each dimension of a grid
    sharedMemPerBlock : int
        Shared memory available per block in bytes
    totalConstantMemory : int
        Constant memory available on device in bytes
    SIMDWidth : int
        Warp size in threads
    memPitch : int
        Maximum pitch in bytes allowed by memory copies
    regsPerBlock : int
        32-bit registers available per block
    clockRate : int
        Clock frequency in kilohertz
    textureAlign : int
        Alignment requirement for textures

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUdevprop_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['maxThreadsPerBlock : ' + str(self.maxThreadsPerBlock)]
            except ValueError:
                str_list += ['maxThreadsPerBlock : <ValueError>']
            try:
                str_list += ['maxThreadsDim : ' + str(self.maxThreadsDim)]
            except ValueError:
                str_list += ['maxThreadsDim : <ValueError>']
            try:
                str_list += ['maxGridSize : ' + str(self.maxGridSize)]
            except ValueError:
                str_list += ['maxGridSize : <ValueError>']
            try:
                str_list += ['sharedMemPerBlock : ' + str(self.sharedMemPerBlock)]
            except ValueError:
                str_list += ['sharedMemPerBlock : <ValueError>']
            try:
                str_list += ['totalConstantMemory : ' + str(self.totalConstantMemory)]
            except ValueError:
                str_list += ['totalConstantMemory : <ValueError>']
            try:
                str_list += ['SIMDWidth : ' + str(self.SIMDWidth)]
            except ValueError:
                str_list += ['SIMDWidth : <ValueError>']
            try:
                str_list += ['memPitch : ' + str(self.memPitch)]
            except ValueError:
                str_list += ['memPitch : <ValueError>']
            try:
                str_list += ['regsPerBlock : ' + str(self.regsPerBlock)]
            except ValueError:
                str_list += ['regsPerBlock : <ValueError>']
            try:
                str_list += ['clockRate : ' + str(self.clockRate)]
            except ValueError:
                str_list += ['clockRate : <ValueError>']
            try:
                str_list += ['textureAlign : ' + str(self.textureAlign)]
            except ValueError:
                str_list += ['textureAlign : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def maxThreadsPerBlock(self):
        return self._ptr[0].maxThreadsPerBlock
    @maxThreadsPerBlock.setter
    def maxThreadsPerBlock(self, int maxThreadsPerBlock):
        self._ptr[0].maxThreadsPerBlock = maxThreadsPerBlock
    @property
    def maxThreadsDim(self):
        return self._ptr[0].maxThreadsDim
    @maxThreadsDim.setter
    def maxThreadsDim(self, maxThreadsDim):
        self._ptr[0].maxThreadsDim = maxThreadsDim
    @property
    def maxGridSize(self):
        return self._ptr[0].maxGridSize
    @maxGridSize.setter
    def maxGridSize(self, maxGridSize):
        self._ptr[0].maxGridSize = maxGridSize
    @property
    def sharedMemPerBlock(self):
        return self._ptr[0].sharedMemPerBlock
    @sharedMemPerBlock.setter
    def sharedMemPerBlock(self, int sharedMemPerBlock):
        self._ptr[0].sharedMemPerBlock = sharedMemPerBlock
    @property
    def totalConstantMemory(self):
        return self._ptr[0].totalConstantMemory
    @totalConstantMemory.setter
    def totalConstantMemory(self, int totalConstantMemory):
        self._ptr[0].totalConstantMemory = totalConstantMemory
    @property
    def SIMDWidth(self):
        return self._ptr[0].SIMDWidth
    @SIMDWidth.setter
    def SIMDWidth(self, int SIMDWidth):
        self._ptr[0].SIMDWidth = SIMDWidth
    @property
    def memPitch(self):
        return self._ptr[0].memPitch
    @memPitch.setter
    def memPitch(self, int memPitch):
        self._ptr[0].memPitch = memPitch
    @property
    def regsPerBlock(self):
        return self._ptr[0].regsPerBlock
    @regsPerBlock.setter
    def regsPerBlock(self, int regsPerBlock):
        self._ptr[0].regsPerBlock = regsPerBlock
    @property
    def clockRate(self):
        return self._ptr[0].clockRate
    @clockRate.setter
    def clockRate(self, int clockRate):
        self._ptr[0].clockRate = clockRate
    @property
    def textureAlign(self):
        return self._ptr[0].textureAlign
    @textureAlign.setter
    def textureAlign(self, int textureAlign):
        self._ptr[0].textureAlign = textureAlign

cdef class CUaccessPolicyWindow_st:
    """
    Specifies an access policy for a window, a contiguous extent of
    memory beginning at base_ptr and ending at base_ptr + num_bytes.
    num_bytes is limited by
    CU_DEVICE_ATTRIBUTE_MAX_ACCESS_POLICY_WINDOW_SIZE. Partition into
    many segments and assign segments such that: sum of "hit segments"
    / window == approx. ratio. sum of "miss segments" / window ==
    approx 1-ratio. Segments and ratio specifications are fitted to the
    capabilities of the architecture. Accesses in a hit segment apply
    the hitProp access policy. Accesses in a miss segment apply the
    missProp access policy.

    Attributes
    ----------
    base_ptr : Any
        Starting address of the access policy window. CUDA driver may align
        it.
    num_bytes : size_t
        Size in bytes of the window policy. CUDA driver may restrict the
        maximum size and alignment.
    hitRatio : float
        hitRatio specifies percentage of lines assigned hitProp, rest are
        assigned missProp.
    hitProp : CUaccessProperty
        CUaccessProperty set for hit.
    missProp : CUaccessProperty
        CUaccessProperty set for miss. Must be either NORMAL or STREAMING

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUaccessPolicyWindow_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['base_ptr : ' + hex(self.base_ptr)]
            except ValueError:
                str_list += ['base_ptr : <ValueError>']
            try:
                str_list += ['num_bytes : ' + str(self.num_bytes)]
            except ValueError:
                str_list += ['num_bytes : <ValueError>']
            try:
                str_list += ['hitRatio : ' + str(self.hitRatio)]
            except ValueError:
                str_list += ['hitRatio : <ValueError>']
            try:
                str_list += ['hitProp : ' + str(self.hitProp)]
            except ValueError:
                str_list += ['hitProp : <ValueError>']
            try:
                str_list += ['missProp : ' + str(self.missProp)]
            except ValueError:
                str_list += ['missProp : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def base_ptr(self):
        return <void_ptr>self._ptr[0].base_ptr
    @base_ptr.setter
    def base_ptr(self, base_ptr):
        _cbase_ptr = utils.HelperInputVoidPtr(base_ptr)
        self._ptr[0].base_ptr = <void*><void_ptr>_cbase_ptr.cptr
    @property
    def num_bytes(self):
        return self._ptr[0].num_bytes
    @num_bytes.setter
    def num_bytes(self, size_t num_bytes):
        self._ptr[0].num_bytes = num_bytes
    @property
    def hitRatio(self):
        return self._ptr[0].hitRatio
    @hitRatio.setter
    def hitRatio(self, float hitRatio):
        self._ptr[0].hitRatio = hitRatio
    @property
    def hitProp(self):
        return CUaccessProperty(self._ptr[0].hitProp)
    @hitProp.setter
    def hitProp(self, hitProp not None : CUaccessProperty):
        self._ptr[0].hitProp = hitProp.value
    @property
    def missProp(self):
        return CUaccessProperty(self._ptr[0].missProp)
    @missProp.setter
    def missProp(self, missProp not None : CUaccessProperty):
        self._ptr[0].missProp = missProp.value

cdef class CUDA_KERNEL_NODE_PARAMS_st:
    """
    GPU kernel node parameters

    Attributes
    ----------
    func : CUfunction
        Kernel to launch
    gridDimX : unsigned int
        Width of grid in blocks
    gridDimY : unsigned int
        Height of grid in blocks
    gridDimZ : unsigned int
        Depth of grid in blocks
    blockDimX : unsigned int
        X dimension of each thread block
    blockDimY : unsigned int
        Y dimension of each thread block
    blockDimZ : unsigned int
        Z dimension of each thread block
    sharedMemBytes : unsigned int
        Dynamic shared-memory size per thread block in bytes
    kernelParams : Any
        Array of pointers to kernel parameters
    extra : Any
        Extra options

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_KERNEL_NODE_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._func = CUfunction(_ptr=<void_ptr>&self._ptr[0].func)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['func : ' + str(self.func)]
            except ValueError:
                str_list += ['func : <ValueError>']
            try:
                str_list += ['gridDimX : ' + str(self.gridDimX)]
            except ValueError:
                str_list += ['gridDimX : <ValueError>']
            try:
                str_list += ['gridDimY : ' + str(self.gridDimY)]
            except ValueError:
                str_list += ['gridDimY : <ValueError>']
            try:
                str_list += ['gridDimZ : ' + str(self.gridDimZ)]
            except ValueError:
                str_list += ['gridDimZ : <ValueError>']
            try:
                str_list += ['blockDimX : ' + str(self.blockDimX)]
            except ValueError:
                str_list += ['blockDimX : <ValueError>']
            try:
                str_list += ['blockDimY : ' + str(self.blockDimY)]
            except ValueError:
                str_list += ['blockDimY : <ValueError>']
            try:
                str_list += ['blockDimZ : ' + str(self.blockDimZ)]
            except ValueError:
                str_list += ['blockDimZ : <ValueError>']
            try:
                str_list += ['sharedMemBytes : ' + str(self.sharedMemBytes)]
            except ValueError:
                str_list += ['sharedMemBytes : <ValueError>']
            try:
                str_list += ['kernelParams : ' + str(self.kernelParams)]
            except ValueError:
                str_list += ['kernelParams : <ValueError>']
            try:
                str_list += ['extra : ' + str(self.extra)]
            except ValueError:
                str_list += ['extra : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def func(self):
        return self._func
    @func.setter
    def func(self, func):
        cdef ccuda.CUfunction cfunc
        if func is None:
            cfunc = <ccuda.CUfunction><void_ptr>0
        elif isinstance(func, (CUfunction,)):
            pfunc = int(func)
            cfunc = <ccuda.CUfunction><void_ptr>pfunc
        else:
            pfunc = int(CUfunction(func))
            cfunc = <ccuda.CUfunction><void_ptr>pfunc
        self._func._ptr[0] = cfunc
    @property
    def gridDimX(self):
        return self._ptr[0].gridDimX
    @gridDimX.setter
    def gridDimX(self, unsigned int gridDimX):
        self._ptr[0].gridDimX = gridDimX
    @property
    def gridDimY(self):
        return self._ptr[0].gridDimY
    @gridDimY.setter
    def gridDimY(self, unsigned int gridDimY):
        self._ptr[0].gridDimY = gridDimY
    @property
    def gridDimZ(self):
        return self._ptr[0].gridDimZ
    @gridDimZ.setter
    def gridDimZ(self, unsigned int gridDimZ):
        self._ptr[0].gridDimZ = gridDimZ
    @property
    def blockDimX(self):
        return self._ptr[0].blockDimX
    @blockDimX.setter
    def blockDimX(self, unsigned int blockDimX):
        self._ptr[0].blockDimX = blockDimX
    @property
    def blockDimY(self):
        return self._ptr[0].blockDimY
    @blockDimY.setter
    def blockDimY(self, unsigned int blockDimY):
        self._ptr[0].blockDimY = blockDimY
    @property
    def blockDimZ(self):
        return self._ptr[0].blockDimZ
    @blockDimZ.setter
    def blockDimZ(self, unsigned int blockDimZ):
        self._ptr[0].blockDimZ = blockDimZ
    @property
    def sharedMemBytes(self):
        return self._ptr[0].sharedMemBytes
    @sharedMemBytes.setter
    def sharedMemBytes(self, unsigned int sharedMemBytes):
        self._ptr[0].sharedMemBytes = sharedMemBytes
    @property
    def kernelParams(self):
        return <void_ptr>self._ptr[0].kernelParams
    @kernelParams.setter
    def kernelParams(self, kernelParams):
        self._ckernelParams = utils.HelperKernelParams(kernelParams)
        self._ptr[0].kernelParams = <void**><void_ptr>self._ckernelParams.ckernelParams
    @property
    def extra(self):
        return <void_ptr>self._ptr[0].extra
    @extra.setter
    def extra(self, void_ptr extra):
        self._ptr[0].extra = <void**>extra

cdef class CUDA_KERNEL_NODE_PARAMS_v2_st:
    """
    GPU kernel node parameters

    Attributes
    ----------
    func : CUfunction
        Kernel to launch
    gridDimX : unsigned int
        Width of grid in blocks
    gridDimY : unsigned int
        Height of grid in blocks
    gridDimZ : unsigned int
        Depth of grid in blocks
    blockDimX : unsigned int
        X dimension of each thread block
    blockDimY : unsigned int
        Y dimension of each thread block
    blockDimZ : unsigned int
        Z dimension of each thread block
    sharedMemBytes : unsigned int
        Dynamic shared-memory size per thread block in bytes
    kernelParams : Any
        Array of pointers to kernel parameters
    extra : Any
        Extra options
    kern : CUkernel
        Kernel to launch, will only be referenced if func is NULL
    ctx : CUcontext
        Context for the kernel task to run in. The value NULL will indicate
        the current context should be used by the api. This field is
        ignored if func is set.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_KERNEL_NODE_PARAMS_v2_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._func = CUfunction(_ptr=<void_ptr>&self._ptr[0].func)
        self._kern = CUkernel(_ptr=<void_ptr>&self._ptr[0].kern)
        self._ctx = CUcontext(_ptr=<void_ptr>&self._ptr[0].ctx)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['func : ' + str(self.func)]
            except ValueError:
                str_list += ['func : <ValueError>']
            try:
                str_list += ['gridDimX : ' + str(self.gridDimX)]
            except ValueError:
                str_list += ['gridDimX : <ValueError>']
            try:
                str_list += ['gridDimY : ' + str(self.gridDimY)]
            except ValueError:
                str_list += ['gridDimY : <ValueError>']
            try:
                str_list += ['gridDimZ : ' + str(self.gridDimZ)]
            except ValueError:
                str_list += ['gridDimZ : <ValueError>']
            try:
                str_list += ['blockDimX : ' + str(self.blockDimX)]
            except ValueError:
                str_list += ['blockDimX : <ValueError>']
            try:
                str_list += ['blockDimY : ' + str(self.blockDimY)]
            except ValueError:
                str_list += ['blockDimY : <ValueError>']
            try:
                str_list += ['blockDimZ : ' + str(self.blockDimZ)]
            except ValueError:
                str_list += ['blockDimZ : <ValueError>']
            try:
                str_list += ['sharedMemBytes : ' + str(self.sharedMemBytes)]
            except ValueError:
                str_list += ['sharedMemBytes : <ValueError>']
            try:
                str_list += ['kernelParams : ' + str(self.kernelParams)]
            except ValueError:
                str_list += ['kernelParams : <ValueError>']
            try:
                str_list += ['extra : ' + str(self.extra)]
            except ValueError:
                str_list += ['extra : <ValueError>']
            try:
                str_list += ['kern : ' + str(self.kern)]
            except ValueError:
                str_list += ['kern : <ValueError>']
            try:
                str_list += ['ctx : ' + str(self.ctx)]
            except ValueError:
                str_list += ['ctx : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def func(self):
        return self._func
    @func.setter
    def func(self, func):
        cdef ccuda.CUfunction cfunc
        if func is None:
            cfunc = <ccuda.CUfunction><void_ptr>0
        elif isinstance(func, (CUfunction,)):
            pfunc = int(func)
            cfunc = <ccuda.CUfunction><void_ptr>pfunc
        else:
            pfunc = int(CUfunction(func))
            cfunc = <ccuda.CUfunction><void_ptr>pfunc
        self._func._ptr[0] = cfunc
    @property
    def gridDimX(self):
        return self._ptr[0].gridDimX
    @gridDimX.setter
    def gridDimX(self, unsigned int gridDimX):
        self._ptr[0].gridDimX = gridDimX
    @property
    def gridDimY(self):
        return self._ptr[0].gridDimY
    @gridDimY.setter
    def gridDimY(self, unsigned int gridDimY):
        self._ptr[0].gridDimY = gridDimY
    @property
    def gridDimZ(self):
        return self._ptr[0].gridDimZ
    @gridDimZ.setter
    def gridDimZ(self, unsigned int gridDimZ):
        self._ptr[0].gridDimZ = gridDimZ
    @property
    def blockDimX(self):
        return self._ptr[0].blockDimX
    @blockDimX.setter
    def blockDimX(self, unsigned int blockDimX):
        self._ptr[0].blockDimX = blockDimX
    @property
    def blockDimY(self):
        return self._ptr[0].blockDimY
    @blockDimY.setter
    def blockDimY(self, unsigned int blockDimY):
        self._ptr[0].blockDimY = blockDimY
    @property
    def blockDimZ(self):
        return self._ptr[0].blockDimZ
    @blockDimZ.setter
    def blockDimZ(self, unsigned int blockDimZ):
        self._ptr[0].blockDimZ = blockDimZ
    @property
    def sharedMemBytes(self):
        return self._ptr[0].sharedMemBytes
    @sharedMemBytes.setter
    def sharedMemBytes(self, unsigned int sharedMemBytes):
        self._ptr[0].sharedMemBytes = sharedMemBytes
    @property
    def kernelParams(self):
        return <void_ptr>self._ptr[0].kernelParams
    @kernelParams.setter
    def kernelParams(self, kernelParams):
        self._ckernelParams = utils.HelperKernelParams(kernelParams)
        self._ptr[0].kernelParams = <void**><void_ptr>self._ckernelParams.ckernelParams
    @property
    def extra(self):
        return <void_ptr>self._ptr[0].extra
    @extra.setter
    def extra(self, void_ptr extra):
        self._ptr[0].extra = <void**>extra
    @property
    def kern(self):
        return self._kern
    @kern.setter
    def kern(self, kern):
        cdef ccuda.CUkernel ckern
        if kern is None:
            ckern = <ccuda.CUkernel><void_ptr>0
        elif isinstance(kern, (CUkernel,)):
            pkern = int(kern)
            ckern = <ccuda.CUkernel><void_ptr>pkern
        else:
            pkern = int(CUkernel(kern))
            ckern = <ccuda.CUkernel><void_ptr>pkern
        self._kern._ptr[0] = ckern
    @property
    def ctx(self):
        return self._ctx
    @ctx.setter
    def ctx(self, ctx):
        cdef ccuda.CUcontext cctx
        if ctx is None:
            cctx = <ccuda.CUcontext><void_ptr>0
        elif isinstance(ctx, (CUcontext,)):
            pctx = int(ctx)
            cctx = <ccuda.CUcontext><void_ptr>pctx
        else:
            pctx = int(CUcontext(ctx))
            cctx = <ccuda.CUcontext><void_ptr>pctx
        self._ctx._ptr[0] = cctx

cdef class CUDA_KERNEL_NODE_PARAMS_v3_st:
    """
    GPU kernel node parameters

    Attributes
    ----------
    func : CUfunction
        Kernel to launch
    gridDimX : unsigned int
        Width of grid in blocks
    gridDimY : unsigned int
        Height of grid in blocks
    gridDimZ : unsigned int
        Depth of grid in blocks
    blockDimX : unsigned int
        X dimension of each thread block
    blockDimY : unsigned int
        Y dimension of each thread block
    blockDimZ : unsigned int
        Z dimension of each thread block
    sharedMemBytes : unsigned int
        Dynamic shared-memory size per thread block in bytes
    kernelParams : Any
        Array of pointers to kernel parameters
    extra : Any
        Extra options
    kern : CUkernel
        Kernel to launch, will only be referenced if func is NULL
    ctx : CUcontext
        Context for the kernel task to run in. The value NULL will indicate
        the current context should be used by the api. This field is
        ignored if func is set.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_KERNEL_NODE_PARAMS_v3_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._func = CUfunction(_ptr=<void_ptr>&self._ptr[0].func)
        self._kern = CUkernel(_ptr=<void_ptr>&self._ptr[0].kern)
        self._ctx = CUcontext(_ptr=<void_ptr>&self._ptr[0].ctx)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['func : ' + str(self.func)]
            except ValueError:
                str_list += ['func : <ValueError>']
            try:
                str_list += ['gridDimX : ' + str(self.gridDimX)]
            except ValueError:
                str_list += ['gridDimX : <ValueError>']
            try:
                str_list += ['gridDimY : ' + str(self.gridDimY)]
            except ValueError:
                str_list += ['gridDimY : <ValueError>']
            try:
                str_list += ['gridDimZ : ' + str(self.gridDimZ)]
            except ValueError:
                str_list += ['gridDimZ : <ValueError>']
            try:
                str_list += ['blockDimX : ' + str(self.blockDimX)]
            except ValueError:
                str_list += ['blockDimX : <ValueError>']
            try:
                str_list += ['blockDimY : ' + str(self.blockDimY)]
            except ValueError:
                str_list += ['blockDimY : <ValueError>']
            try:
                str_list += ['blockDimZ : ' + str(self.blockDimZ)]
            except ValueError:
                str_list += ['blockDimZ : <ValueError>']
            try:
                str_list += ['sharedMemBytes : ' + str(self.sharedMemBytes)]
            except ValueError:
                str_list += ['sharedMemBytes : <ValueError>']
            try:
                str_list += ['kernelParams : ' + str(self.kernelParams)]
            except ValueError:
                str_list += ['kernelParams : <ValueError>']
            try:
                str_list += ['extra : ' + str(self.extra)]
            except ValueError:
                str_list += ['extra : <ValueError>']
            try:
                str_list += ['kern : ' + str(self.kern)]
            except ValueError:
                str_list += ['kern : <ValueError>']
            try:
                str_list += ['ctx : ' + str(self.ctx)]
            except ValueError:
                str_list += ['ctx : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def func(self):
        return self._func
    @func.setter
    def func(self, func):
        cdef ccuda.CUfunction cfunc
        if func is None:
            cfunc = <ccuda.CUfunction><void_ptr>0
        elif isinstance(func, (CUfunction,)):
            pfunc = int(func)
            cfunc = <ccuda.CUfunction><void_ptr>pfunc
        else:
            pfunc = int(CUfunction(func))
            cfunc = <ccuda.CUfunction><void_ptr>pfunc
        self._func._ptr[0] = cfunc
    @property
    def gridDimX(self):
        return self._ptr[0].gridDimX
    @gridDimX.setter
    def gridDimX(self, unsigned int gridDimX):
        self._ptr[0].gridDimX = gridDimX
    @property
    def gridDimY(self):
        return self._ptr[0].gridDimY
    @gridDimY.setter
    def gridDimY(self, unsigned int gridDimY):
        self._ptr[0].gridDimY = gridDimY
    @property
    def gridDimZ(self):
        return self._ptr[0].gridDimZ
    @gridDimZ.setter
    def gridDimZ(self, unsigned int gridDimZ):
        self._ptr[0].gridDimZ = gridDimZ
    @property
    def blockDimX(self):
        return self._ptr[0].blockDimX
    @blockDimX.setter
    def blockDimX(self, unsigned int blockDimX):
        self._ptr[0].blockDimX = blockDimX
    @property
    def blockDimY(self):
        return self._ptr[0].blockDimY
    @blockDimY.setter
    def blockDimY(self, unsigned int blockDimY):
        self._ptr[0].blockDimY = blockDimY
    @property
    def blockDimZ(self):
        return self._ptr[0].blockDimZ
    @blockDimZ.setter
    def blockDimZ(self, unsigned int blockDimZ):
        self._ptr[0].blockDimZ = blockDimZ
    @property
    def sharedMemBytes(self):
        return self._ptr[0].sharedMemBytes
    @sharedMemBytes.setter
    def sharedMemBytes(self, unsigned int sharedMemBytes):
        self._ptr[0].sharedMemBytes = sharedMemBytes
    @property
    def kernelParams(self):
        return <void_ptr>self._ptr[0].kernelParams
    @kernelParams.setter
    def kernelParams(self, kernelParams):
        self._ckernelParams = utils.HelperKernelParams(kernelParams)
        self._ptr[0].kernelParams = <void**><void_ptr>self._ckernelParams.ckernelParams
    @property
    def extra(self):
        return <void_ptr>self._ptr[0].extra
    @extra.setter
    def extra(self, void_ptr extra):
        self._ptr[0].extra = <void**>extra
    @property
    def kern(self):
        return self._kern
    @kern.setter
    def kern(self, kern):
        cdef ccuda.CUkernel ckern
        if kern is None:
            ckern = <ccuda.CUkernel><void_ptr>0
        elif isinstance(kern, (CUkernel,)):
            pkern = int(kern)
            ckern = <ccuda.CUkernel><void_ptr>pkern
        else:
            pkern = int(CUkernel(kern))
            ckern = <ccuda.CUkernel><void_ptr>pkern
        self._kern._ptr[0] = ckern
    @property
    def ctx(self):
        return self._ctx
    @ctx.setter
    def ctx(self, ctx):
        cdef ccuda.CUcontext cctx
        if ctx is None:
            cctx = <ccuda.CUcontext><void_ptr>0
        elif isinstance(ctx, (CUcontext,)):
            pctx = int(ctx)
            cctx = <ccuda.CUcontext><void_ptr>pctx
        else:
            pctx = int(CUcontext(ctx))
            cctx = <ccuda.CUcontext><void_ptr>pctx
        self._ctx._ptr[0] = cctx

cdef class CUDA_MEMSET_NODE_PARAMS_st:
    """
    Memset node parameters

    Attributes
    ----------
    dst : CUdeviceptr
        Destination device pointer
    pitch : size_t
        Pitch of destination device pointer. Unused if height is 1
    value : unsigned int
        Value to be set
    elementSize : unsigned int
        Size of each element in bytes. Must be 1, 2, or 4.
    width : size_t
        Width of the row in elements
    height : size_t
        Number of rows

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_MEMSET_NODE_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._dst = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].dst)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['dst : ' + str(self.dst)]
            except ValueError:
                str_list += ['dst : <ValueError>']
            try:
                str_list += ['pitch : ' + str(self.pitch)]
            except ValueError:
                str_list += ['pitch : <ValueError>']
            try:
                str_list += ['value : ' + str(self.value)]
            except ValueError:
                str_list += ['value : <ValueError>']
            try:
                str_list += ['elementSize : ' + str(self.elementSize)]
            except ValueError:
                str_list += ['elementSize : <ValueError>']
            try:
                str_list += ['width : ' + str(self.width)]
            except ValueError:
                str_list += ['width : <ValueError>']
            try:
                str_list += ['height : ' + str(self.height)]
            except ValueError:
                str_list += ['height : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def dst(self):
        return self._dst
    @dst.setter
    def dst(self, dst):
        cdef ccuda.CUdeviceptr cdst
        if dst is None:
            cdst = <ccuda.CUdeviceptr><void_ptr>0
        elif isinstance(dst, (CUdeviceptr)):
            pdst = int(dst)
            cdst = <ccuda.CUdeviceptr><void_ptr>pdst
        else:
            pdst = int(CUdeviceptr(dst))
            cdst = <ccuda.CUdeviceptr><void_ptr>pdst
        self._dst._ptr[0] = cdst

    @property
    def pitch(self):
        return self._ptr[0].pitch
    @pitch.setter
    def pitch(self, size_t pitch):
        self._ptr[0].pitch = pitch
    @property
    def value(self):
        return self._ptr[0].value
    @value.setter
    def value(self, unsigned int value):
        self._ptr[0].value = value
    @property
    def elementSize(self):
        return self._ptr[0].elementSize
    @elementSize.setter
    def elementSize(self, unsigned int elementSize):
        self._ptr[0].elementSize = elementSize
    @property
    def width(self):
        return self._ptr[0].width
    @width.setter
    def width(self, size_t width):
        self._ptr[0].width = width
    @property
    def height(self):
        return self._ptr[0].height
    @height.setter
    def height(self, size_t height):
        self._ptr[0].height = height

cdef class CUDA_MEMSET_NODE_PARAMS_v2_st:
    """
    Memset node parameters

    Attributes
    ----------
    dst : CUdeviceptr
        Destination device pointer
    pitch : size_t
        Pitch of destination device pointer. Unused if height is 1
    value : unsigned int
        Value to be set
    elementSize : unsigned int
        Size of each element in bytes. Must be 1, 2, or 4.
    width : size_t
        Width of the row in elements
    height : size_t
        Number of rows
    ctx : CUcontext
        Context on which to run the node

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_MEMSET_NODE_PARAMS_v2_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._dst = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].dst)
        self._ctx = CUcontext(_ptr=<void_ptr>&self._ptr[0].ctx)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['dst : ' + str(self.dst)]
            except ValueError:
                str_list += ['dst : <ValueError>']
            try:
                str_list += ['pitch : ' + str(self.pitch)]
            except ValueError:
                str_list += ['pitch : <ValueError>']
            try:
                str_list += ['value : ' + str(self.value)]
            except ValueError:
                str_list += ['value : <ValueError>']
            try:
                str_list += ['elementSize : ' + str(self.elementSize)]
            except ValueError:
                str_list += ['elementSize : <ValueError>']
            try:
                str_list += ['width : ' + str(self.width)]
            except ValueError:
                str_list += ['width : <ValueError>']
            try:
                str_list += ['height : ' + str(self.height)]
            except ValueError:
                str_list += ['height : <ValueError>']
            try:
                str_list += ['ctx : ' + str(self.ctx)]
            except ValueError:
                str_list += ['ctx : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def dst(self):
        return self._dst
    @dst.setter
    def dst(self, dst):
        cdef ccuda.CUdeviceptr cdst
        if dst is None:
            cdst = <ccuda.CUdeviceptr><void_ptr>0
        elif isinstance(dst, (CUdeviceptr)):
            pdst = int(dst)
            cdst = <ccuda.CUdeviceptr><void_ptr>pdst
        else:
            pdst = int(CUdeviceptr(dst))
            cdst = <ccuda.CUdeviceptr><void_ptr>pdst
        self._dst._ptr[0] = cdst

    @property
    def pitch(self):
        return self._ptr[0].pitch
    @pitch.setter
    def pitch(self, size_t pitch):
        self._ptr[0].pitch = pitch
    @property
    def value(self):
        return self._ptr[0].value
    @value.setter
    def value(self, unsigned int value):
        self._ptr[0].value = value
    @property
    def elementSize(self):
        return self._ptr[0].elementSize
    @elementSize.setter
    def elementSize(self, unsigned int elementSize):
        self._ptr[0].elementSize = elementSize
    @property
    def width(self):
        return self._ptr[0].width
    @width.setter
    def width(self, size_t width):
        self._ptr[0].width = width
    @property
    def height(self):
        return self._ptr[0].height
    @height.setter
    def height(self, size_t height):
        self._ptr[0].height = height
    @property
    def ctx(self):
        return self._ctx
    @ctx.setter
    def ctx(self, ctx):
        cdef ccuda.CUcontext cctx
        if ctx is None:
            cctx = <ccuda.CUcontext><void_ptr>0
        elif isinstance(ctx, (CUcontext,)):
            pctx = int(ctx)
            cctx = <ccuda.CUcontext><void_ptr>pctx
        else:
            pctx = int(CUcontext(ctx))
            cctx = <ccuda.CUcontext><void_ptr>pctx
        self._ctx._ptr[0] = cctx

cdef class CUDA_HOST_NODE_PARAMS_st:
    """
    Host node parameters

    Attributes
    ----------
    fn : CUhostFn
        The function to call when the node executes
    userData : Any
        Argument to pass to the function

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_HOST_NODE_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._fn = CUhostFn(_ptr=<void_ptr>&self._ptr[0].fn)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['fn : ' + str(self.fn)]
            except ValueError:
                str_list += ['fn : <ValueError>']
            try:
                str_list += ['userData : ' + hex(self.userData)]
            except ValueError:
                str_list += ['userData : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def fn(self):
        return self._fn
    @fn.setter
    def fn(self, fn):
        cdef ccuda.CUhostFn cfn
        if fn is None:
            cfn = <ccuda.CUhostFn><void_ptr>0
        elif isinstance(fn, (CUhostFn)):
            pfn = int(fn)
            cfn = <ccuda.CUhostFn><void_ptr>pfn
        else:
            pfn = int(CUhostFn(fn))
            cfn = <ccuda.CUhostFn><void_ptr>pfn
        self._fn._ptr[0] = cfn
    @property
    def userData(self):
        return <void_ptr>self._ptr[0].userData
    @userData.setter
    def userData(self, userData):
        _cuserData = utils.HelperInputVoidPtr(userData)
        self._ptr[0].userData = <void*><void_ptr>_cuserData.cptr

cdef class CUDA_HOST_NODE_PARAMS_v2_st:
    """
    Host node parameters

    Attributes
    ----------
    fn : CUhostFn
        The function to call when the node executes
    userData : Any
        Argument to pass to the function

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_HOST_NODE_PARAMS_v2_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._fn = CUhostFn(_ptr=<void_ptr>&self._ptr[0].fn)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['fn : ' + str(self.fn)]
            except ValueError:
                str_list += ['fn : <ValueError>']
            try:
                str_list += ['userData : ' + hex(self.userData)]
            except ValueError:
                str_list += ['userData : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def fn(self):
        return self._fn
    @fn.setter
    def fn(self, fn):
        cdef ccuda.CUhostFn cfn
        if fn is None:
            cfn = <ccuda.CUhostFn><void_ptr>0
        elif isinstance(fn, (CUhostFn)):
            pfn = int(fn)
            cfn = <ccuda.CUhostFn><void_ptr>pfn
        else:
            pfn = int(CUhostFn(fn))
            cfn = <ccuda.CUhostFn><void_ptr>pfn
        self._fn._ptr[0] = cfn
    @property
    def userData(self):
        return <void_ptr>self._ptr[0].userData
    @userData.setter
    def userData(self, userData):
        _cuserData = utils.HelperInputVoidPtr(userData)
        self._ptr[0].userData = <void*><void_ptr>_cuserData.cptr

cdef class CUDA_CONDITIONAL_NODE_PARAMS:
    """
    Conditional node parameters

    Attributes
    ----------
    handle : CUgraphConditionalHandle
        Conditional node handle. Handles must be created in advance of
        creating the node using cuGraphConditionalHandleCreate.
    type : CUgraphConditionalNodeType
        Type of conditional node.
    size : unsigned int
        Size of graph output array. Must be 1.
    phGraph_out : CUgraph
        CUDA-owned array populated with conditional node child graphs
        during creation of the node. Valid for the lifetime of the
        conditional node. The contents of the graph(s) are subject to the
        following constraints:   - Allowed node types are kernel nodes,
        empty nodes, child graphs, memsets, memcopies, and conditionals.
        This applies recursively to child graphs and conditional bodies.
        - All kernels, including kernels in nested conditionals or child
        graphs at any level, must belong to the same CUDA context.
        These graphs may be populated using graph node creation APIs or
        cuStreamBeginCaptureToGraph.
    ctx : CUcontext
        Context on which to run the node. Must match context used to create
        the handle and all body nodes.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_CONDITIONAL_NODE_PARAMS *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._handle = CUgraphConditionalHandle(_ptr=<void_ptr>&self._ptr[0].handle)
        self._ctx = CUcontext(_ptr=<void_ptr>&self._ptr[0].ctx)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['handle : ' + str(self.handle)]
            except ValueError:
                str_list += ['handle : <ValueError>']
            try:
                str_list += ['type : ' + str(self.type)]
            except ValueError:
                str_list += ['type : <ValueError>']
            try:
                str_list += ['size : ' + str(self.size)]
            except ValueError:
                str_list += ['size : <ValueError>']
            try:
                str_list += ['phGraph_out : ' + str(self.phGraph_out)]
            except ValueError:
                str_list += ['phGraph_out : <ValueError>']
            try:
                str_list += ['ctx : ' + str(self.ctx)]
            except ValueError:
                str_list += ['ctx : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def handle(self):
        return self._handle
    @handle.setter
    def handle(self, handle):
        cdef ccuda.CUgraphConditionalHandle chandle
        if handle is None:
            chandle = <ccuda.CUgraphConditionalHandle><void_ptr>0
        elif isinstance(handle, (CUgraphConditionalHandle)):
            phandle = int(handle)
            chandle = <ccuda.CUgraphConditionalHandle><void_ptr>phandle
        else:
            phandle = int(CUgraphConditionalHandle(handle))
            chandle = <ccuda.CUgraphConditionalHandle><void_ptr>phandle
        self._handle._ptr[0] = chandle

    @property
    def type(self):
        return CUgraphConditionalNodeType(self._ptr[0].type)
    @type.setter
    def type(self, type not None : CUgraphConditionalNodeType):
        self._ptr[0].type = type.value
    @property
    def size(self):
        return self._ptr[0].size
    @size.setter
    def size(self, unsigned int size):
        self._ptr[0].size = size
    @property
    def phGraph_out(self):
        arrs = [<void_ptr>self._ptr[0].phGraph_out + x*sizeof(ccuda.CUgraph) for x in range(self.size)]
        return [CUgraph(_ptr=arr) for arr in arrs]
    @property
    def ctx(self):
        return self._ctx
    @ctx.setter
    def ctx(self, ctx):
        cdef ccuda.CUcontext cctx
        if ctx is None:
            cctx = <ccuda.CUcontext><void_ptr>0
        elif isinstance(ctx, (CUcontext,)):
            pctx = int(ctx)
            cctx = <ccuda.CUcontext><void_ptr>pctx
        else:
            pctx = int(CUcontext(ctx))
            cctx = <ccuda.CUcontext><void_ptr>pctx
        self._ctx._ptr[0] = cctx

cdef class CUgraphEdgeData_st:
    """
    Optional annotation for edges in a CUDA graph. Note, all edges
    implicitly have annotations and default to a zero-initialized value
    if not specified. A zero-initialized struct indicates a standard
    full serialization of two nodes with memory visibility.

    Attributes
    ----------
    from_port : bytes
        This indicates when the dependency is triggered from the upstream
        node on the edge. The meaning is specfic to the node type. A value
        of 0 in all cases means full completion of the upstream node, with
        memory visibility to the downstream node or portion thereof
        (indicated by `to_port`).   Only kernel nodes define non-zero
        ports. A kernel node can use the following output port types:
        CU_GRAPH_KERNEL_NODE_PORT_DEFAULT,
        CU_GRAPH_KERNEL_NODE_PORT_PROGRAMMATIC, or
        CU_GRAPH_KERNEL_NODE_PORT_LAUNCH_ORDER.
    to_port : bytes
        This indicates what portion of the downstream node is dependent on
        the upstream node or portion thereof (indicated by `from_port`).
        The meaning is specific to the node type. A value of 0 in all cases
        means the entirety of the downstream node is dependent on the
        upstream work.   Currently no node types define non-zero ports.
        Accordingly, this field must be set to zero.
    type : bytes
        This should be populated with a value from CUgraphDependencyType.
        (It is typed as char due to compiler-specific layout of bitfields.)
        See CUgraphDependencyType.
    reserved : bytes
        These bytes are unused and must be zeroed. This ensures
        compatibility if additional fields are added in the future.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUgraphEdgeData_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['from_port : ' + str(self.from_port)]
            except ValueError:
                str_list += ['from_port : <ValueError>']
            try:
                str_list += ['to_port : ' + str(self.to_port)]
            except ValueError:
                str_list += ['to_port : <ValueError>']
            try:
                str_list += ['type : ' + str(self.type)]
            except ValueError:
                str_list += ['type : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def from_port(self):
        return self._ptr[0].from_port
    @from_port.setter
    def from_port(self, unsigned char from_port):
        self._ptr[0].from_port = from_port
    @property
    def to_port(self):
        return self._ptr[0].to_port
    @to_port.setter
    def to_port(self, unsigned char to_port):
        self._ptr[0].to_port = to_port
    @property
    def type(self):
        return self._ptr[0].type
    @type.setter
    def type(self, unsigned char type):
        self._ptr[0].type = type
    @property
    def reserved(self):
        return PyBytes_FromStringAndSize(<char*>self._ptr[0].reserved, 5)
    @reserved.setter
    def reserved(self, reserved):
        if len(reserved) != 5:
            raise ValueError("reserved length must be 5, is " + str(len(reserved)))
        for i, b in enumerate(reserved):
            self._ptr[0].reserved[i] = b

cdef class CUDA_GRAPH_INSTANTIATE_PARAMS_st:
    """
    Graph instantiation parameters

    Attributes
    ----------
    flags : cuuint64_t
        Instantiation flags
    hUploadStream : CUstream
        Upload stream
    hErrNode_out : CUgraphNode
        The node which caused instantiation to fail, if any
    result_out : CUgraphInstantiateResult
        Whether instantiation was successful. If it failed, the reason why

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_GRAPH_INSTANTIATE_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._flags = cuuint64_t(_ptr=<void_ptr>&self._ptr[0].flags)
        self._hUploadStream = CUstream(_ptr=<void_ptr>&self._ptr[0].hUploadStream)
        self._hErrNode_out = CUgraphNode(_ptr=<void_ptr>&self._ptr[0].hErrNode_out)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['hUploadStream : ' + str(self.hUploadStream)]
            except ValueError:
                str_list += ['hUploadStream : <ValueError>']
            try:
                str_list += ['hErrNode_out : ' + str(self.hErrNode_out)]
            except ValueError:
                str_list += ['hErrNode_out : <ValueError>']
            try:
                str_list += ['result_out : ' + str(self.result_out)]
            except ValueError:
                str_list += ['result_out : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def flags(self):
        return self._flags
    @flags.setter
    def flags(self, flags):
        cdef ccuda.cuuint64_t cflags
        if flags is None:
            cflags = <ccuda.cuuint64_t><void_ptr>0
        elif isinstance(flags, (cuuint64_t)):
            pflags = int(flags)
            cflags = <ccuda.cuuint64_t><void_ptr>pflags
        else:
            pflags = int(cuuint64_t(flags))
            cflags = <ccuda.cuuint64_t><void_ptr>pflags
        self._flags._ptr[0] = cflags

    @property
    def hUploadStream(self):
        return self._hUploadStream
    @hUploadStream.setter
    def hUploadStream(self, hUploadStream):
        cdef ccuda.CUstream chUploadStream
        if hUploadStream is None:
            chUploadStream = <ccuda.CUstream><void_ptr>0
        elif isinstance(hUploadStream, (CUstream,)):
            phUploadStream = int(hUploadStream)
            chUploadStream = <ccuda.CUstream><void_ptr>phUploadStream
        else:
            phUploadStream = int(CUstream(hUploadStream))
            chUploadStream = <ccuda.CUstream><void_ptr>phUploadStream
        self._hUploadStream._ptr[0] = chUploadStream
    @property
    def hErrNode_out(self):
        return self._hErrNode_out
    @hErrNode_out.setter
    def hErrNode_out(self, hErrNode_out):
        cdef ccuda.CUgraphNode chErrNode_out
        if hErrNode_out is None:
            chErrNode_out = <ccuda.CUgraphNode><void_ptr>0
        elif isinstance(hErrNode_out, (CUgraphNode,)):
            phErrNode_out = int(hErrNode_out)
            chErrNode_out = <ccuda.CUgraphNode><void_ptr>phErrNode_out
        else:
            phErrNode_out = int(CUgraphNode(hErrNode_out))
            chErrNode_out = <ccuda.CUgraphNode><void_ptr>phErrNode_out
        self._hErrNode_out._ptr[0] = chErrNode_out
    @property
    def result_out(self):
        return CUgraphInstantiateResult(self._ptr[0].result_out)
    @result_out.setter
    def result_out(self, result_out not None : CUgraphInstantiateResult):
        self._ptr[0].result_out = result_out.value

cdef class CUlaunchMemSyncDomainMap_st:
    """
    Memory Synchronization Domain map  See ::cudaLaunchMemSyncDomain.
    By default, kernels are launched in domain 0. Kernel launched with
    CU_LAUNCH_MEM_SYNC_DOMAIN_REMOTE will have a different domain ID.
    User may also alter the domain ID with CUlaunchMemSyncDomainMap for
    a specific stream / graph node / kernel launch. See
    CU_LAUNCH_ATTRIBUTE_MEM_SYNC_DOMAIN_MAP.  Domain ID range is
    available through CU_DEVICE_ATTRIBUTE_MEM_SYNC_DOMAIN_COUNT.

    Attributes
    ----------
    default_ : bytes
        The default domain ID to use for designated kernels
    remote : bytes
        The remote domain ID to use for designated kernels

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUlaunchMemSyncDomainMap_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['default_ : ' + str(self.default_)]
            except ValueError:
                str_list += ['default_ : <ValueError>']
            try:
                str_list += ['remote : ' + str(self.remote)]
            except ValueError:
                str_list += ['remote : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def default_(self):
        return self._ptr[0].default_
    @default_.setter
    def default_(self, unsigned char default_):
        self._ptr[0].default_ = default_
    @property
    def remote(self):
        return self._ptr[0].remote
    @remote.setter
    def remote(self, unsigned char remote):
        self._ptr[0].remote = remote

cdef class anon_struct1:
    """
    Attributes
    ----------
    x : unsigned int

    y : unsigned int

    z : unsigned int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUlaunchAttributeValue_union *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].clusterDim
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['x : ' + str(self.x)]
            except ValueError:
                str_list += ['x : <ValueError>']
            try:
                str_list += ['y : ' + str(self.y)]
            except ValueError:
                str_list += ['y : <ValueError>']
            try:
                str_list += ['z : ' + str(self.z)]
            except ValueError:
                str_list += ['z : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def x(self):
        return self._ptr[0].clusterDim.x
    @x.setter
    def x(self, unsigned int x):
        self._ptr[0].clusterDim.x = x
    @property
    def y(self):
        return self._ptr[0].clusterDim.y
    @y.setter
    def y(self, unsigned int y):
        self._ptr[0].clusterDim.y = y
    @property
    def z(self):
        return self._ptr[0].clusterDim.z
    @z.setter
    def z(self, unsigned int z):
        self._ptr[0].clusterDim.z = z

cdef class anon_struct2:
    """
    Attributes
    ----------
    event : CUevent

    flags : int

    triggerAtBlockStart : int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUlaunchAttributeValue_union *>_ptr

    def __init__(self, void_ptr _ptr):
        self._event = CUevent(_ptr=<void_ptr>&self._ptr[0].programmaticEvent.event)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].programmaticEvent
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['event : ' + str(self.event)]
            except ValueError:
                str_list += ['event : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['triggerAtBlockStart : ' + str(self.triggerAtBlockStart)]
            except ValueError:
                str_list += ['triggerAtBlockStart : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def event(self):
        return self._event
    @event.setter
    def event(self, event):
        cdef ccuda.CUevent cevent
        if event is None:
            cevent = <ccuda.CUevent><void_ptr>0
        elif isinstance(event, (CUevent,)):
            pevent = int(event)
            cevent = <ccuda.CUevent><void_ptr>pevent
        else:
            pevent = int(CUevent(event))
            cevent = <ccuda.CUevent><void_ptr>pevent
        self._event._ptr[0] = cevent
    @property
    def flags(self):
        return self._ptr[0].programmaticEvent.flags
    @flags.setter
    def flags(self, int flags):
        self._ptr[0].programmaticEvent.flags = flags
    @property
    def triggerAtBlockStart(self):
        return self._ptr[0].programmaticEvent.triggerAtBlockStart
    @triggerAtBlockStart.setter
    def triggerAtBlockStart(self, int triggerAtBlockStart):
        self._ptr[0].programmaticEvent.triggerAtBlockStart = triggerAtBlockStart

cdef class anon_struct3:
    """
    Attributes
    ----------
    event : CUevent

    flags : int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUlaunchAttributeValue_union *>_ptr

    def __init__(self, void_ptr _ptr):
        self._event = CUevent(_ptr=<void_ptr>&self._ptr[0].launchCompletionEvent.event)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].launchCompletionEvent
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['event : ' + str(self.event)]
            except ValueError:
                str_list += ['event : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def event(self):
        return self._event
    @event.setter
    def event(self, event):
        cdef ccuda.CUevent cevent
        if event is None:
            cevent = <ccuda.CUevent><void_ptr>0
        elif isinstance(event, (CUevent,)):
            pevent = int(event)
            cevent = <ccuda.CUevent><void_ptr>pevent
        else:
            pevent = int(CUevent(event))
            cevent = <ccuda.CUevent><void_ptr>pevent
        self._event._ptr[0] = cevent
    @property
    def flags(self):
        return self._ptr[0].launchCompletionEvent.flags
    @flags.setter
    def flags(self, int flags):
        self._ptr[0].launchCompletionEvent.flags = flags

cdef class anon_struct4:
    """
    Attributes
    ----------
    deviceUpdatable : int

    devNode : CUgraphDeviceNode


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUlaunchAttributeValue_union *>_ptr

    def __init__(self, void_ptr _ptr):
        self._devNode = CUgraphDeviceNode(_ptr=<void_ptr>&self._ptr[0].deviceUpdatableKernelNode.devNode)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].deviceUpdatableKernelNode
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['deviceUpdatable : ' + str(self.deviceUpdatable)]
            except ValueError:
                str_list += ['deviceUpdatable : <ValueError>']
            try:
                str_list += ['devNode : ' + str(self.devNode)]
            except ValueError:
                str_list += ['devNode : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def deviceUpdatable(self):
        return self._ptr[0].deviceUpdatableKernelNode.deviceUpdatable
    @deviceUpdatable.setter
    def deviceUpdatable(self, int deviceUpdatable):
        self._ptr[0].deviceUpdatableKernelNode.deviceUpdatable = deviceUpdatable
    @property
    def devNode(self):
        return self._devNode
    @devNode.setter
    def devNode(self, devNode):
        cdef ccuda.CUgraphDeviceNode cdevNode
        if devNode is None:
            cdevNode = <ccuda.CUgraphDeviceNode><void_ptr>0
        elif isinstance(devNode, (CUgraphDeviceNode,)):
            pdevNode = int(devNode)
            cdevNode = <ccuda.CUgraphDeviceNode><void_ptr>pdevNode
        else:
            pdevNode = int(CUgraphDeviceNode(devNode))
            cdevNode = <ccuda.CUgraphDeviceNode><void_ptr>pdevNode
        self._devNode._ptr[0] = cdevNode

cdef class CUlaunchAttributeValue_union:
    """
    Launch attributes union; used as value field of CUlaunchAttribute

    Attributes
    ----------
    pad : bytes

    accessPolicyWindow : CUaccessPolicyWindow
        Value of launch attribute CU_LAUNCH_ATTRIBUTE_ACCESS_POLICY_WINDOW.
    cooperative : int
        Value of launch attribute CU_LAUNCH_ATTRIBUTE_COOPERATIVE. Nonzero
        indicates a cooperative kernel (see cuLaunchCooperativeKernel).
    syncPolicy : CUsynchronizationPolicy
        Value of launch attribute
        CU_LAUNCH_ATTRIBUTE_SYNCHRONIZATION_POLICY.
        ::CUsynchronizationPolicy for work queued up in this stream
    clusterDim : anon_struct1
        Value of launch attribute CU_LAUNCH_ATTRIBUTE_CLUSTER_DIMENSION
        that represents the desired cluster dimensions for the kernel.
        Opaque type with the following fields: - `x` - The X dimension of
        the cluster, in blocks. Must be a divisor of the grid X dimension.
        - `y` - The Y dimension of the cluster, in blocks. Must be a
        divisor of the grid Y dimension.    - `z` - The Z dimension of the
        cluster, in blocks. Must be a divisor of the grid Z dimension.
    clusterSchedulingPolicyPreference : CUclusterSchedulingPolicy
        Value of launch attribute
        CU_LAUNCH_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE. Cluster
        scheduling policy preference for the kernel.
    programmaticStreamSerializationAllowed : int
        Value of launch attribute
        CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_STREAM_SERIALIZATION.
    programmaticEvent : anon_struct2
        Value of launch attribute CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_EVENT.
    launchCompletionEvent : anon_struct3
        Value of launch attribute
        CU_LAUNCH_ATTRIBUTE_LAUNCH_COMPLETION_EVENT.
    priority : int
        Value of launch attribute CU_LAUNCH_ATTRIBUTE_PRIORITY. Execution
        priority of the kernel.
    memSyncDomainMap : CUlaunchMemSyncDomainMap
        Value of launch attribute CU_LAUNCH_ATTRIBUTE_MEM_SYNC_DOMAIN_MAP.
        See CUlaunchMemSyncDomainMap.
    memSyncDomain : CUlaunchMemSyncDomain
        Value of launch attribute CU_LAUNCH_ATTRIBUTE_MEM_SYNC_DOMAIN.
        See::CUlaunchMemSyncDomain
    deviceUpdatableKernelNode : anon_struct4
        Value of launch attribute
        CU_LAUNCH_ATTRIBUTE_DEVICE_UPDATABLE_KERNEL_NODE.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUlaunchAttributeValue_union *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._accessPolicyWindow = CUaccessPolicyWindow(_ptr=<void_ptr>&self._ptr[0].accessPolicyWindow)
        self._clusterDim = anon_struct1(_ptr=<void_ptr>self._ptr)
        self._programmaticEvent = anon_struct2(_ptr=<void_ptr>self._ptr)
        self._launchCompletionEvent = anon_struct3(_ptr=<void_ptr>self._ptr)
        self._memSyncDomainMap = CUlaunchMemSyncDomainMap(_ptr=<void_ptr>&self._ptr[0].memSyncDomainMap)
        self._deviceUpdatableKernelNode = anon_struct4(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['pad : ' + str(self.pad)]
            except ValueError:
                str_list += ['pad : <ValueError>']
            try:
                str_list += ['accessPolicyWindow :\n' + '\n'.join(['    ' + line for line in str(self.accessPolicyWindow).splitlines()])]
            except ValueError:
                str_list += ['accessPolicyWindow : <ValueError>']
            try:
                str_list += ['cooperative : ' + str(self.cooperative)]
            except ValueError:
                str_list += ['cooperative : <ValueError>']
            try:
                str_list += ['syncPolicy : ' + str(self.syncPolicy)]
            except ValueError:
                str_list += ['syncPolicy : <ValueError>']
            try:
                str_list += ['clusterDim :\n' + '\n'.join(['    ' + line for line in str(self.clusterDim).splitlines()])]
            except ValueError:
                str_list += ['clusterDim : <ValueError>']
            try:
                str_list += ['clusterSchedulingPolicyPreference : ' + str(self.clusterSchedulingPolicyPreference)]
            except ValueError:
                str_list += ['clusterSchedulingPolicyPreference : <ValueError>']
            try:
                str_list += ['programmaticStreamSerializationAllowed : ' + str(self.programmaticStreamSerializationAllowed)]
            except ValueError:
                str_list += ['programmaticStreamSerializationAllowed : <ValueError>']
            try:
                str_list += ['programmaticEvent :\n' + '\n'.join(['    ' + line for line in str(self.programmaticEvent).splitlines()])]
            except ValueError:
                str_list += ['programmaticEvent : <ValueError>']
            try:
                str_list += ['launchCompletionEvent :\n' + '\n'.join(['    ' + line for line in str(self.launchCompletionEvent).splitlines()])]
            except ValueError:
                str_list += ['launchCompletionEvent : <ValueError>']
            try:
                str_list += ['priority : ' + str(self.priority)]
            except ValueError:
                str_list += ['priority : <ValueError>']
            try:
                str_list += ['memSyncDomainMap :\n' + '\n'.join(['    ' + line for line in str(self.memSyncDomainMap).splitlines()])]
            except ValueError:
                str_list += ['memSyncDomainMap : <ValueError>']
            try:
                str_list += ['memSyncDomain : ' + str(self.memSyncDomain)]
            except ValueError:
                str_list += ['memSyncDomain : <ValueError>']
            try:
                str_list += ['deviceUpdatableKernelNode :\n' + '\n'.join(['    ' + line for line in str(self.deviceUpdatableKernelNode).splitlines()])]
            except ValueError:
                str_list += ['deviceUpdatableKernelNode : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def pad(self):
        return PyBytes_FromStringAndSize(self._ptr[0].pad, 64)
    @pad.setter
    def pad(self, pad):
        if len(pad) != 64:
            raise ValueError("pad length must be 64, is " + str(len(pad)))
        if CHAR_MIN == 0:
            for i, b in enumerate(pad):
                if b < 0 and b > -129:
                    b = b + 256
                self._ptr[0].pad[i] = b
        else:
            for i, b in enumerate(pad):
                if b > 127 and b < 256:
                    b = b - 256
                self._ptr[0].pad[i] = b
    @property
    def accessPolicyWindow(self):
        return self._accessPolicyWindow
    @accessPolicyWindow.setter
    def accessPolicyWindow(self, accessPolicyWindow not None : CUaccessPolicyWindow):
        string.memcpy(&self._ptr[0].accessPolicyWindow, <ccuda.CUaccessPolicyWindow*><void_ptr>accessPolicyWindow.getPtr(), sizeof(self._ptr[0].accessPolicyWindow))
    @property
    def cooperative(self):
        return self._ptr[0].cooperative
    @cooperative.setter
    def cooperative(self, int cooperative):
        self._ptr[0].cooperative = cooperative
    @property
    def syncPolicy(self):
        return CUsynchronizationPolicy(self._ptr[0].syncPolicy)
    @syncPolicy.setter
    def syncPolicy(self, syncPolicy not None : CUsynchronizationPolicy):
        self._ptr[0].syncPolicy = syncPolicy.value
    @property
    def clusterDim(self):
        return self._clusterDim
    @clusterDim.setter
    def clusterDim(self, clusterDim not None : anon_struct1):
        string.memcpy(&self._ptr[0].clusterDim, <ccuda.anon_struct1*><void_ptr>clusterDim.getPtr(), sizeof(self._ptr[0].clusterDim))
    @property
    def clusterSchedulingPolicyPreference(self):
        return CUclusterSchedulingPolicy(self._ptr[0].clusterSchedulingPolicyPreference)
    @clusterSchedulingPolicyPreference.setter
    def clusterSchedulingPolicyPreference(self, clusterSchedulingPolicyPreference not None : CUclusterSchedulingPolicy):
        self._ptr[0].clusterSchedulingPolicyPreference = clusterSchedulingPolicyPreference.value
    @property
    def programmaticStreamSerializationAllowed(self):
        return self._ptr[0].programmaticStreamSerializationAllowed
    @programmaticStreamSerializationAllowed.setter
    def programmaticStreamSerializationAllowed(self, int programmaticStreamSerializationAllowed):
        self._ptr[0].programmaticStreamSerializationAllowed = programmaticStreamSerializationAllowed
    @property
    def programmaticEvent(self):
        return self._programmaticEvent
    @programmaticEvent.setter
    def programmaticEvent(self, programmaticEvent not None : anon_struct2):
        string.memcpy(&self._ptr[0].programmaticEvent, <ccuda.anon_struct2*><void_ptr>programmaticEvent.getPtr(), sizeof(self._ptr[0].programmaticEvent))
    @property
    def launchCompletionEvent(self):
        return self._launchCompletionEvent
    @launchCompletionEvent.setter
    def launchCompletionEvent(self, launchCompletionEvent not None : anon_struct3):
        string.memcpy(&self._ptr[0].launchCompletionEvent, <ccuda.anon_struct3*><void_ptr>launchCompletionEvent.getPtr(), sizeof(self._ptr[0].launchCompletionEvent))
    @property
    def priority(self):
        return self._ptr[0].priority
    @priority.setter
    def priority(self, int priority):
        self._ptr[0].priority = priority
    @property
    def memSyncDomainMap(self):
        return self._memSyncDomainMap
    @memSyncDomainMap.setter
    def memSyncDomainMap(self, memSyncDomainMap not None : CUlaunchMemSyncDomainMap):
        string.memcpy(&self._ptr[0].memSyncDomainMap, <ccuda.CUlaunchMemSyncDomainMap*><void_ptr>memSyncDomainMap.getPtr(), sizeof(self._ptr[0].memSyncDomainMap))
    @property
    def memSyncDomain(self):
        return CUlaunchMemSyncDomain(self._ptr[0].memSyncDomain)
    @memSyncDomain.setter
    def memSyncDomain(self, memSyncDomain not None : CUlaunchMemSyncDomain):
        self._ptr[0].memSyncDomain = memSyncDomain.value
    @property
    def deviceUpdatableKernelNode(self):
        return self._deviceUpdatableKernelNode
    @deviceUpdatableKernelNode.setter
    def deviceUpdatableKernelNode(self, deviceUpdatableKernelNode not None : anon_struct4):
        string.memcpy(&self._ptr[0].deviceUpdatableKernelNode, <ccuda.anon_struct4*><void_ptr>deviceUpdatableKernelNode.getPtr(), sizeof(self._ptr[0].deviceUpdatableKernelNode))

cdef class CUlaunchAttribute_st:
    """
    Launch attribute

    Attributes
    ----------
    id : CUlaunchAttributeID
        Attribute to set
    value : CUlaunchAttributeValue
        Value of the attribute

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUlaunchAttribute_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._value = CUlaunchAttributeValue(_ptr=<void_ptr>&self._ptr[0].value)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['id : ' + str(self.id)]
            except ValueError:
                str_list += ['id : <ValueError>']
            try:
                str_list += ['value :\n' + '\n'.join(['    ' + line for line in str(self.value).splitlines()])]
            except ValueError:
                str_list += ['value : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def id(self):
        return CUlaunchAttributeID(self._ptr[0].id)
    @id.setter
    def id(self, id not None : CUlaunchAttributeID):
        self._ptr[0].id = id.value
    @property
    def value(self):
        return self._value
    @value.setter
    def value(self, value not None : CUlaunchAttributeValue):
        string.memcpy(&self._ptr[0].value, <ccuda.CUlaunchAttributeValue*><void_ptr>value.getPtr(), sizeof(self._ptr[0].value))

cdef class CUlaunchConfig_st:
    """
    CUDA extensible launch configuration

    Attributes
    ----------
    gridDimX : unsigned int
        Width of grid in blocks
    gridDimY : unsigned int
        Height of grid in blocks
    gridDimZ : unsigned int
        Depth of grid in blocks
    blockDimX : unsigned int
        X dimension of each thread block
    blockDimY : unsigned int
        Y dimension of each thread block
    blockDimZ : unsigned int
        Z dimension of each thread block
    sharedMemBytes : unsigned int
        Dynamic shared-memory size per thread block in bytes
    hStream : CUstream
        Stream identifier
    attrs : CUlaunchAttribute
        List of attributes; nullable if CUlaunchConfig::numAttrs == 0
    numAttrs : unsigned int
        Number of attributes populated in CUlaunchConfig::attrs

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUlaunchConfig_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._hStream = CUstream(_ptr=<void_ptr>&self._ptr[0].hStream)
    def __dealloc__(self):
        if self._attrs is not NULL:
            free(self._attrs)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['gridDimX : ' + str(self.gridDimX)]
            except ValueError:
                str_list += ['gridDimX : <ValueError>']
            try:
                str_list += ['gridDimY : ' + str(self.gridDimY)]
            except ValueError:
                str_list += ['gridDimY : <ValueError>']
            try:
                str_list += ['gridDimZ : ' + str(self.gridDimZ)]
            except ValueError:
                str_list += ['gridDimZ : <ValueError>']
            try:
                str_list += ['blockDimX : ' + str(self.blockDimX)]
            except ValueError:
                str_list += ['blockDimX : <ValueError>']
            try:
                str_list += ['blockDimY : ' + str(self.blockDimY)]
            except ValueError:
                str_list += ['blockDimY : <ValueError>']
            try:
                str_list += ['blockDimZ : ' + str(self.blockDimZ)]
            except ValueError:
                str_list += ['blockDimZ : <ValueError>']
            try:
                str_list += ['sharedMemBytes : ' + str(self.sharedMemBytes)]
            except ValueError:
                str_list += ['sharedMemBytes : <ValueError>']
            try:
                str_list += ['hStream : ' + str(self.hStream)]
            except ValueError:
                str_list += ['hStream : <ValueError>']
            try:
                str_list += ['attrs : ' + str(self.attrs)]
            except ValueError:
                str_list += ['attrs : <ValueError>']
            try:
                str_list += ['numAttrs : ' + str(self.numAttrs)]
            except ValueError:
                str_list += ['numAttrs : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def gridDimX(self):
        return self._ptr[0].gridDimX
    @gridDimX.setter
    def gridDimX(self, unsigned int gridDimX):
        self._ptr[0].gridDimX = gridDimX
    @property
    def gridDimY(self):
        return self._ptr[0].gridDimY
    @gridDimY.setter
    def gridDimY(self, unsigned int gridDimY):
        self._ptr[0].gridDimY = gridDimY
    @property
    def gridDimZ(self):
        return self._ptr[0].gridDimZ
    @gridDimZ.setter
    def gridDimZ(self, unsigned int gridDimZ):
        self._ptr[0].gridDimZ = gridDimZ
    @property
    def blockDimX(self):
        return self._ptr[0].blockDimX
    @blockDimX.setter
    def blockDimX(self, unsigned int blockDimX):
        self._ptr[0].blockDimX = blockDimX
    @property
    def blockDimY(self):
        return self._ptr[0].blockDimY
    @blockDimY.setter
    def blockDimY(self, unsigned int blockDimY):
        self._ptr[0].blockDimY = blockDimY
    @property
    def blockDimZ(self):
        return self._ptr[0].blockDimZ
    @blockDimZ.setter
    def blockDimZ(self, unsigned int blockDimZ):
        self._ptr[0].blockDimZ = blockDimZ
    @property
    def sharedMemBytes(self):
        return self._ptr[0].sharedMemBytes
    @sharedMemBytes.setter
    def sharedMemBytes(self, unsigned int sharedMemBytes):
        self._ptr[0].sharedMemBytes = sharedMemBytes
    @property
    def hStream(self):
        return self._hStream
    @hStream.setter
    def hStream(self, hStream):
        cdef ccuda.CUstream chStream
        if hStream is None:
            chStream = <ccuda.CUstream><void_ptr>0
        elif isinstance(hStream, (CUstream,)):
            phStream = int(hStream)
            chStream = <ccuda.CUstream><void_ptr>phStream
        else:
            phStream = int(CUstream(hStream))
            chStream = <ccuda.CUstream><void_ptr>phStream
        self._hStream._ptr[0] = chStream
    @property
    def attrs(self):
        arrs = [<void_ptr>self._ptr[0].attrs + x*sizeof(ccuda.CUlaunchAttribute) for x in range(self._attrs_length)]
        return [CUlaunchAttribute(_ptr=arr) for arr in arrs]
    @attrs.setter
    def attrs(self, val):
        if len(val) == 0:
            free(self._attrs)
            self._attrs_length = 0
            self._ptr[0].attrs = NULL
        else:
            if self._attrs_length != <size_t>len(val):
                free(self._attrs)
                self._attrs = <ccuda.CUlaunchAttribute*> calloc(len(val), sizeof(ccuda.CUlaunchAttribute))
                if self._attrs is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(ccuda.CUlaunchAttribute)))
                self._attrs_length = <size_t>len(val)
                self._ptr[0].attrs = self._attrs
            for idx in range(len(val)):
                string.memcpy(&self._attrs[idx], (<CUlaunchAttribute>val[idx])._ptr, sizeof(ccuda.CUlaunchAttribute))

    @property
    def numAttrs(self):
        return self._ptr[0].numAttrs
    @numAttrs.setter
    def numAttrs(self, unsigned int numAttrs):
        self._ptr[0].numAttrs = numAttrs

cdef class CUexecAffinitySmCount_st:
    """
    Value for CU_EXEC_AFFINITY_TYPE_SM_COUNT

    Attributes
    ----------
    val : unsigned int
        The number of SMs the context is limited to use.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUexecAffinitySmCount_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['val : ' + str(self.val)]
            except ValueError:
                str_list += ['val : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def val(self):
        return self._ptr[0].val
    @val.setter
    def val(self, unsigned int val):
        self._ptr[0].val = val

cdef class anon_union3:
    """
    Attributes
    ----------
    smCount : CUexecAffinitySmCount


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUexecAffinityParam_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._smCount = CUexecAffinitySmCount(_ptr=<void_ptr>&self._ptr[0].param.smCount)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].param
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['smCount :\n' + '\n'.join(['    ' + line for line in str(self.smCount).splitlines()])]
            except ValueError:
                str_list += ['smCount : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def smCount(self):
        return self._smCount
    @smCount.setter
    def smCount(self, smCount not None : CUexecAffinitySmCount):
        string.memcpy(&self._ptr[0].param.smCount, <ccuda.CUexecAffinitySmCount*><void_ptr>smCount.getPtr(), sizeof(self._ptr[0].param.smCount))

cdef class CUexecAffinityParam_st:
    """
    Execution Affinity Parameters

    Attributes
    ----------
    type : CUexecAffinityType

    param : anon_union3


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._val_ptr = <ccuda.CUexecAffinityParam_st *>calloc(1, sizeof(ccuda.CUexecAffinityParam_st))
            self._ptr = self._val_ptr
        else:
            self._ptr = <ccuda.CUexecAffinityParam_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._param = anon_union3(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        if self._val_ptr is not NULL:
            free(self._val_ptr)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['type : ' + str(self.type)]
            except ValueError:
                str_list += ['type : <ValueError>']
            try:
                str_list += ['param :\n' + '\n'.join(['    ' + line for line in str(self.param).splitlines()])]
            except ValueError:
                str_list += ['param : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def type(self):
        return CUexecAffinityType(self._ptr[0].type)
    @type.setter
    def type(self, type not None : CUexecAffinityType):
        self._ptr[0].type = type.value
    @property
    def param(self):
        return self._param
    @param.setter
    def param(self, param not None : anon_union3):
        string.memcpy(&self._ptr[0].param, <ccuda.anon_union3*><void_ptr>param.getPtr(), sizeof(self._ptr[0].param))

cdef class CUlibraryHostUniversalFunctionAndDataTable_st:
    """
    Attributes
    ----------
    functionTable : Any

    functionWindowSize : size_t

    dataTable : Any

    dataWindowSize : size_t


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUlibraryHostUniversalFunctionAndDataTable_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['functionTable : ' + hex(self.functionTable)]
            except ValueError:
                str_list += ['functionTable : <ValueError>']
            try:
                str_list += ['functionWindowSize : ' + str(self.functionWindowSize)]
            except ValueError:
                str_list += ['functionWindowSize : <ValueError>']
            try:
                str_list += ['dataTable : ' + hex(self.dataTable)]
            except ValueError:
                str_list += ['dataTable : <ValueError>']
            try:
                str_list += ['dataWindowSize : ' + str(self.dataWindowSize)]
            except ValueError:
                str_list += ['dataWindowSize : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def functionTable(self):
        return <void_ptr>self._ptr[0].functionTable
    @functionTable.setter
    def functionTable(self, functionTable):
        _cfunctionTable = utils.HelperInputVoidPtr(functionTable)
        self._ptr[0].functionTable = <void*><void_ptr>_cfunctionTable.cptr
    @property
    def functionWindowSize(self):
        return self._ptr[0].functionWindowSize
    @functionWindowSize.setter
    def functionWindowSize(self, size_t functionWindowSize):
        self._ptr[0].functionWindowSize = functionWindowSize
    @property
    def dataTable(self):
        return <void_ptr>self._ptr[0].dataTable
    @dataTable.setter
    def dataTable(self, dataTable):
        _cdataTable = utils.HelperInputVoidPtr(dataTable)
        self._ptr[0].dataTable = <void*><void_ptr>_cdataTable.cptr
    @property
    def dataWindowSize(self):
        return self._ptr[0].dataWindowSize
    @dataWindowSize.setter
    def dataWindowSize(self, size_t dataWindowSize):
        self._ptr[0].dataWindowSize = dataWindowSize

cdef class CUDA_MEMCPY2D_st:
    """
    2D memory copy parameters

    Attributes
    ----------
    srcXInBytes : size_t
        Source X in bytes
    srcY : size_t
        Source Y
    srcMemoryType : CUmemorytype
        Source memory type (host, device, array)
    srcHost : Any
        Source host pointer
    srcDevice : CUdeviceptr
        Source device pointer
    srcArray : CUarray
        Source array reference
    srcPitch : size_t
        Source pitch (ignored when src is array)
    dstXInBytes : size_t
        Destination X in bytes
    dstY : size_t
        Destination Y
    dstMemoryType : CUmemorytype
        Destination memory type (host, device, array)
    dstHost : Any
        Destination host pointer
    dstDevice : CUdeviceptr
        Destination device pointer
    dstArray : CUarray
        Destination array reference
    dstPitch : size_t
        Destination pitch (ignored when dst is array)
    WidthInBytes : size_t
        Width of 2D memory copy in bytes
    Height : size_t
        Height of 2D memory copy

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_MEMCPY2D_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._srcDevice = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].srcDevice)
        self._srcArray = CUarray(_ptr=<void_ptr>&self._ptr[0].srcArray)
        self._dstDevice = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].dstDevice)
        self._dstArray = CUarray(_ptr=<void_ptr>&self._ptr[0].dstArray)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['srcXInBytes : ' + str(self.srcXInBytes)]
            except ValueError:
                str_list += ['srcXInBytes : <ValueError>']
            try:
                str_list += ['srcY : ' + str(self.srcY)]
            except ValueError:
                str_list += ['srcY : <ValueError>']
            try:
                str_list += ['srcMemoryType : ' + str(self.srcMemoryType)]
            except ValueError:
                str_list += ['srcMemoryType : <ValueError>']
            try:
                str_list += ['srcHost : ' + hex(self.srcHost)]
            except ValueError:
                str_list += ['srcHost : <ValueError>']
            try:
                str_list += ['srcDevice : ' + str(self.srcDevice)]
            except ValueError:
                str_list += ['srcDevice : <ValueError>']
            try:
                str_list += ['srcArray : ' + str(self.srcArray)]
            except ValueError:
                str_list += ['srcArray : <ValueError>']
            try:
                str_list += ['srcPitch : ' + str(self.srcPitch)]
            except ValueError:
                str_list += ['srcPitch : <ValueError>']
            try:
                str_list += ['dstXInBytes : ' + str(self.dstXInBytes)]
            except ValueError:
                str_list += ['dstXInBytes : <ValueError>']
            try:
                str_list += ['dstY : ' + str(self.dstY)]
            except ValueError:
                str_list += ['dstY : <ValueError>']
            try:
                str_list += ['dstMemoryType : ' + str(self.dstMemoryType)]
            except ValueError:
                str_list += ['dstMemoryType : <ValueError>']
            try:
                str_list += ['dstHost : ' + hex(self.dstHost)]
            except ValueError:
                str_list += ['dstHost : <ValueError>']
            try:
                str_list += ['dstDevice : ' + str(self.dstDevice)]
            except ValueError:
                str_list += ['dstDevice : <ValueError>']
            try:
                str_list += ['dstArray : ' + str(self.dstArray)]
            except ValueError:
                str_list += ['dstArray : <ValueError>']
            try:
                str_list += ['dstPitch : ' + str(self.dstPitch)]
            except ValueError:
                str_list += ['dstPitch : <ValueError>']
            try:
                str_list += ['WidthInBytes : ' + str(self.WidthInBytes)]
            except ValueError:
                str_list += ['WidthInBytes : <ValueError>']
            try:
                str_list += ['Height : ' + str(self.Height)]
            except ValueError:
                str_list += ['Height : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def srcXInBytes(self):
        return self._ptr[0].srcXInBytes
    @srcXInBytes.setter
    def srcXInBytes(self, size_t srcXInBytes):
        self._ptr[0].srcXInBytes = srcXInBytes
    @property
    def srcY(self):
        return self._ptr[0].srcY
    @srcY.setter
    def srcY(self, size_t srcY):
        self._ptr[0].srcY = srcY
    @property
    def srcMemoryType(self):
        return CUmemorytype(self._ptr[0].srcMemoryType)
    @srcMemoryType.setter
    def srcMemoryType(self, srcMemoryType not None : CUmemorytype):
        self._ptr[0].srcMemoryType = srcMemoryType.value
    @property
    def srcHost(self):
        return <void_ptr>self._ptr[0].srcHost
    @srcHost.setter
    def srcHost(self, srcHost):
        _csrcHost = utils.HelperInputVoidPtr(srcHost)
        self._ptr[0].srcHost = <void*><void_ptr>_csrcHost.cptr
    @property
    def srcDevice(self):
        return self._srcDevice
    @srcDevice.setter
    def srcDevice(self, srcDevice):
        cdef ccuda.CUdeviceptr csrcDevice
        if srcDevice is None:
            csrcDevice = <ccuda.CUdeviceptr><void_ptr>0
        elif isinstance(srcDevice, (CUdeviceptr)):
            psrcDevice = int(srcDevice)
            csrcDevice = <ccuda.CUdeviceptr><void_ptr>psrcDevice
        else:
            psrcDevice = int(CUdeviceptr(srcDevice))
            csrcDevice = <ccuda.CUdeviceptr><void_ptr>psrcDevice
        self._srcDevice._ptr[0] = csrcDevice

    @property
    def srcArray(self):
        return self._srcArray
    @srcArray.setter
    def srcArray(self, srcArray):
        cdef ccuda.CUarray csrcArray
        if srcArray is None:
            csrcArray = <ccuda.CUarray><void_ptr>0
        elif isinstance(srcArray, (CUarray,)):
            psrcArray = int(srcArray)
            csrcArray = <ccuda.CUarray><void_ptr>psrcArray
        else:
            psrcArray = int(CUarray(srcArray))
            csrcArray = <ccuda.CUarray><void_ptr>psrcArray
        self._srcArray._ptr[0] = csrcArray
    @property
    def srcPitch(self):
        return self._ptr[0].srcPitch
    @srcPitch.setter
    def srcPitch(self, size_t srcPitch):
        self._ptr[0].srcPitch = srcPitch
    @property
    def dstXInBytes(self):
        return self._ptr[0].dstXInBytes
    @dstXInBytes.setter
    def dstXInBytes(self, size_t dstXInBytes):
        self._ptr[0].dstXInBytes = dstXInBytes
    @property
    def dstY(self):
        return self._ptr[0].dstY
    @dstY.setter
    def dstY(self, size_t dstY):
        self._ptr[0].dstY = dstY
    @property
    def dstMemoryType(self):
        return CUmemorytype(self._ptr[0].dstMemoryType)
    @dstMemoryType.setter
    def dstMemoryType(self, dstMemoryType not None : CUmemorytype):
        self._ptr[0].dstMemoryType = dstMemoryType.value
    @property
    def dstHost(self):
        return <void_ptr>self._ptr[0].dstHost
    @dstHost.setter
    def dstHost(self, dstHost):
        _cdstHost = utils.HelperInputVoidPtr(dstHost)
        self._ptr[0].dstHost = <void*><void_ptr>_cdstHost.cptr
    @property
    def dstDevice(self):
        return self._dstDevice
    @dstDevice.setter
    def dstDevice(self, dstDevice):
        cdef ccuda.CUdeviceptr cdstDevice
        if dstDevice is None:
            cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
        elif isinstance(dstDevice, (CUdeviceptr)):
            pdstDevice = int(dstDevice)
            cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
        else:
            pdstDevice = int(CUdeviceptr(dstDevice))
            cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
        self._dstDevice._ptr[0] = cdstDevice

    @property
    def dstArray(self):
        return self._dstArray
    @dstArray.setter
    def dstArray(self, dstArray):
        cdef ccuda.CUarray cdstArray
        if dstArray is None:
            cdstArray = <ccuda.CUarray><void_ptr>0
        elif isinstance(dstArray, (CUarray,)):
            pdstArray = int(dstArray)
            cdstArray = <ccuda.CUarray><void_ptr>pdstArray
        else:
            pdstArray = int(CUarray(dstArray))
            cdstArray = <ccuda.CUarray><void_ptr>pdstArray
        self._dstArray._ptr[0] = cdstArray
    @property
    def dstPitch(self):
        return self._ptr[0].dstPitch
    @dstPitch.setter
    def dstPitch(self, size_t dstPitch):
        self._ptr[0].dstPitch = dstPitch
    @property
    def WidthInBytes(self):
        return self._ptr[0].WidthInBytes
    @WidthInBytes.setter
    def WidthInBytes(self, size_t WidthInBytes):
        self._ptr[0].WidthInBytes = WidthInBytes
    @property
    def Height(self):
        return self._ptr[0].Height
    @Height.setter
    def Height(self, size_t Height):
        self._ptr[0].Height = Height

cdef class CUDA_MEMCPY3D_st:
    """
    3D memory copy parameters

    Attributes
    ----------
    srcXInBytes : size_t
        Source X in bytes
    srcY : size_t
        Source Y
    srcZ : size_t
        Source Z
    srcLOD : size_t
        Source LOD
    srcMemoryType : CUmemorytype
        Source memory type (host, device, array)
    srcHost : Any
        Source host pointer
    srcDevice : CUdeviceptr
        Source device pointer
    srcArray : CUarray
        Source array reference
    reserved0 : Any
        Must be NULL
    srcPitch : size_t
        Source pitch (ignored when src is array)
    srcHeight : size_t
        Source height (ignored when src is array; may be 0 if Depth==1)
    dstXInBytes : size_t
        Destination X in bytes
    dstY : size_t
        Destination Y
    dstZ : size_t
        Destination Z
    dstLOD : size_t
        Destination LOD
    dstMemoryType : CUmemorytype
        Destination memory type (host, device, array)
    dstHost : Any
        Destination host pointer
    dstDevice : CUdeviceptr
        Destination device pointer
    dstArray : CUarray
        Destination array reference
    reserved1 : Any
        Must be NULL
    dstPitch : size_t
        Destination pitch (ignored when dst is array)
    dstHeight : size_t
        Destination height (ignored when dst is array; may be 0 if
        Depth==1)
    WidthInBytes : size_t
        Width of 3D memory copy in bytes
    Height : size_t
        Height of 3D memory copy
    Depth : size_t
        Depth of 3D memory copy

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_MEMCPY3D_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._srcDevice = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].srcDevice)
        self._srcArray = CUarray(_ptr=<void_ptr>&self._ptr[0].srcArray)
        self._dstDevice = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].dstDevice)
        self._dstArray = CUarray(_ptr=<void_ptr>&self._ptr[0].dstArray)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['srcXInBytes : ' + str(self.srcXInBytes)]
            except ValueError:
                str_list += ['srcXInBytes : <ValueError>']
            try:
                str_list += ['srcY : ' + str(self.srcY)]
            except ValueError:
                str_list += ['srcY : <ValueError>']
            try:
                str_list += ['srcZ : ' + str(self.srcZ)]
            except ValueError:
                str_list += ['srcZ : <ValueError>']
            try:
                str_list += ['srcLOD : ' + str(self.srcLOD)]
            except ValueError:
                str_list += ['srcLOD : <ValueError>']
            try:
                str_list += ['srcMemoryType : ' + str(self.srcMemoryType)]
            except ValueError:
                str_list += ['srcMemoryType : <ValueError>']
            try:
                str_list += ['srcHost : ' + hex(self.srcHost)]
            except ValueError:
                str_list += ['srcHost : <ValueError>']
            try:
                str_list += ['srcDevice : ' + str(self.srcDevice)]
            except ValueError:
                str_list += ['srcDevice : <ValueError>']
            try:
                str_list += ['srcArray : ' + str(self.srcArray)]
            except ValueError:
                str_list += ['srcArray : <ValueError>']
            try:
                str_list += ['reserved0 : ' + hex(self.reserved0)]
            except ValueError:
                str_list += ['reserved0 : <ValueError>']
            try:
                str_list += ['srcPitch : ' + str(self.srcPitch)]
            except ValueError:
                str_list += ['srcPitch : <ValueError>']
            try:
                str_list += ['srcHeight : ' + str(self.srcHeight)]
            except ValueError:
                str_list += ['srcHeight : <ValueError>']
            try:
                str_list += ['dstXInBytes : ' + str(self.dstXInBytes)]
            except ValueError:
                str_list += ['dstXInBytes : <ValueError>']
            try:
                str_list += ['dstY : ' + str(self.dstY)]
            except ValueError:
                str_list += ['dstY : <ValueError>']
            try:
                str_list += ['dstZ : ' + str(self.dstZ)]
            except ValueError:
                str_list += ['dstZ : <ValueError>']
            try:
                str_list += ['dstLOD : ' + str(self.dstLOD)]
            except ValueError:
                str_list += ['dstLOD : <ValueError>']
            try:
                str_list += ['dstMemoryType : ' + str(self.dstMemoryType)]
            except ValueError:
                str_list += ['dstMemoryType : <ValueError>']
            try:
                str_list += ['dstHost : ' + hex(self.dstHost)]
            except ValueError:
                str_list += ['dstHost : <ValueError>']
            try:
                str_list += ['dstDevice : ' + str(self.dstDevice)]
            except ValueError:
                str_list += ['dstDevice : <ValueError>']
            try:
                str_list += ['dstArray : ' + str(self.dstArray)]
            except ValueError:
                str_list += ['dstArray : <ValueError>']
            try:
                str_list += ['reserved1 : ' + hex(self.reserved1)]
            except ValueError:
                str_list += ['reserved1 : <ValueError>']
            try:
                str_list += ['dstPitch : ' + str(self.dstPitch)]
            except ValueError:
                str_list += ['dstPitch : <ValueError>']
            try:
                str_list += ['dstHeight : ' + str(self.dstHeight)]
            except ValueError:
                str_list += ['dstHeight : <ValueError>']
            try:
                str_list += ['WidthInBytes : ' + str(self.WidthInBytes)]
            except ValueError:
                str_list += ['WidthInBytes : <ValueError>']
            try:
                str_list += ['Height : ' + str(self.Height)]
            except ValueError:
                str_list += ['Height : <ValueError>']
            try:
                str_list += ['Depth : ' + str(self.Depth)]
            except ValueError:
                str_list += ['Depth : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def srcXInBytes(self):
        return self._ptr[0].srcXInBytes
    @srcXInBytes.setter
    def srcXInBytes(self, size_t srcXInBytes):
        self._ptr[0].srcXInBytes = srcXInBytes
    @property
    def srcY(self):
        return self._ptr[0].srcY
    @srcY.setter
    def srcY(self, size_t srcY):
        self._ptr[0].srcY = srcY
    @property
    def srcZ(self):
        return self._ptr[0].srcZ
    @srcZ.setter
    def srcZ(self, size_t srcZ):
        self._ptr[0].srcZ = srcZ
    @property
    def srcLOD(self):
        return self._ptr[0].srcLOD
    @srcLOD.setter
    def srcLOD(self, size_t srcLOD):
        self._ptr[0].srcLOD = srcLOD
    @property
    def srcMemoryType(self):
        return CUmemorytype(self._ptr[0].srcMemoryType)
    @srcMemoryType.setter
    def srcMemoryType(self, srcMemoryType not None : CUmemorytype):
        self._ptr[0].srcMemoryType = srcMemoryType.value
    @property
    def srcHost(self):
        return <void_ptr>self._ptr[0].srcHost
    @srcHost.setter
    def srcHost(self, srcHost):
        _csrcHost = utils.HelperInputVoidPtr(srcHost)
        self._ptr[0].srcHost = <void*><void_ptr>_csrcHost.cptr
    @property
    def srcDevice(self):
        return self._srcDevice
    @srcDevice.setter
    def srcDevice(self, srcDevice):
        cdef ccuda.CUdeviceptr csrcDevice
        if srcDevice is None:
            csrcDevice = <ccuda.CUdeviceptr><void_ptr>0
        elif isinstance(srcDevice, (CUdeviceptr)):
            psrcDevice = int(srcDevice)
            csrcDevice = <ccuda.CUdeviceptr><void_ptr>psrcDevice
        else:
            psrcDevice = int(CUdeviceptr(srcDevice))
            csrcDevice = <ccuda.CUdeviceptr><void_ptr>psrcDevice
        self._srcDevice._ptr[0] = csrcDevice

    @property
    def srcArray(self):
        return self._srcArray
    @srcArray.setter
    def srcArray(self, srcArray):
        cdef ccuda.CUarray csrcArray
        if srcArray is None:
            csrcArray = <ccuda.CUarray><void_ptr>0
        elif isinstance(srcArray, (CUarray,)):
            psrcArray = int(srcArray)
            csrcArray = <ccuda.CUarray><void_ptr>psrcArray
        else:
            psrcArray = int(CUarray(srcArray))
            csrcArray = <ccuda.CUarray><void_ptr>psrcArray
        self._srcArray._ptr[0] = csrcArray
    @property
    def reserved0(self):
        return <void_ptr>self._ptr[0].reserved0
    @reserved0.setter
    def reserved0(self, reserved0):
        _creserved0 = utils.HelperInputVoidPtr(reserved0)
        self._ptr[0].reserved0 = <void*><void_ptr>_creserved0.cptr
    @property
    def srcPitch(self):
        return self._ptr[0].srcPitch
    @srcPitch.setter
    def srcPitch(self, size_t srcPitch):
        self._ptr[0].srcPitch = srcPitch
    @property
    def srcHeight(self):
        return self._ptr[0].srcHeight
    @srcHeight.setter
    def srcHeight(self, size_t srcHeight):
        self._ptr[0].srcHeight = srcHeight
    @property
    def dstXInBytes(self):
        return self._ptr[0].dstXInBytes
    @dstXInBytes.setter
    def dstXInBytes(self, size_t dstXInBytes):
        self._ptr[0].dstXInBytes = dstXInBytes
    @property
    def dstY(self):
        return self._ptr[0].dstY
    @dstY.setter
    def dstY(self, size_t dstY):
        self._ptr[0].dstY = dstY
    @property
    def dstZ(self):
        return self._ptr[0].dstZ
    @dstZ.setter
    def dstZ(self, size_t dstZ):
        self._ptr[0].dstZ = dstZ
    @property
    def dstLOD(self):
        return self._ptr[0].dstLOD
    @dstLOD.setter
    def dstLOD(self, size_t dstLOD):
        self._ptr[0].dstLOD = dstLOD
    @property
    def dstMemoryType(self):
        return CUmemorytype(self._ptr[0].dstMemoryType)
    @dstMemoryType.setter
    def dstMemoryType(self, dstMemoryType not None : CUmemorytype):
        self._ptr[0].dstMemoryType = dstMemoryType.value
    @property
    def dstHost(self):
        return <void_ptr>self._ptr[0].dstHost
    @dstHost.setter
    def dstHost(self, dstHost):
        _cdstHost = utils.HelperInputVoidPtr(dstHost)
        self._ptr[0].dstHost = <void*><void_ptr>_cdstHost.cptr
    @property
    def dstDevice(self):
        return self._dstDevice
    @dstDevice.setter
    def dstDevice(self, dstDevice):
        cdef ccuda.CUdeviceptr cdstDevice
        if dstDevice is None:
            cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
        elif isinstance(dstDevice, (CUdeviceptr)):
            pdstDevice = int(dstDevice)
            cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
        else:
            pdstDevice = int(CUdeviceptr(dstDevice))
            cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
        self._dstDevice._ptr[0] = cdstDevice

    @property
    def dstArray(self):
        return self._dstArray
    @dstArray.setter
    def dstArray(self, dstArray):
        cdef ccuda.CUarray cdstArray
        if dstArray is None:
            cdstArray = <ccuda.CUarray><void_ptr>0
        elif isinstance(dstArray, (CUarray,)):
            pdstArray = int(dstArray)
            cdstArray = <ccuda.CUarray><void_ptr>pdstArray
        else:
            pdstArray = int(CUarray(dstArray))
            cdstArray = <ccuda.CUarray><void_ptr>pdstArray
        self._dstArray._ptr[0] = cdstArray
    @property
    def reserved1(self):
        return <void_ptr>self._ptr[0].reserved1
    @reserved1.setter
    def reserved1(self, reserved1):
        _creserved1 = utils.HelperInputVoidPtr(reserved1)
        self._ptr[0].reserved1 = <void*><void_ptr>_creserved1.cptr
    @property
    def dstPitch(self):
        return self._ptr[0].dstPitch
    @dstPitch.setter
    def dstPitch(self, size_t dstPitch):
        self._ptr[0].dstPitch = dstPitch
    @property
    def dstHeight(self):
        return self._ptr[0].dstHeight
    @dstHeight.setter
    def dstHeight(self, size_t dstHeight):
        self._ptr[0].dstHeight = dstHeight
    @property
    def WidthInBytes(self):
        return self._ptr[0].WidthInBytes
    @WidthInBytes.setter
    def WidthInBytes(self, size_t WidthInBytes):
        self._ptr[0].WidthInBytes = WidthInBytes
    @property
    def Height(self):
        return self._ptr[0].Height
    @Height.setter
    def Height(self, size_t Height):
        self._ptr[0].Height = Height
    @property
    def Depth(self):
        return self._ptr[0].Depth
    @Depth.setter
    def Depth(self, size_t Depth):
        self._ptr[0].Depth = Depth

cdef class CUDA_MEMCPY3D_PEER_st:
    """
    3D memory cross-context copy parameters

    Attributes
    ----------
    srcXInBytes : size_t
        Source X in bytes
    srcY : size_t
        Source Y
    srcZ : size_t
        Source Z
    srcLOD : size_t
        Source LOD
    srcMemoryType : CUmemorytype
        Source memory type (host, device, array)
    srcHost : Any
        Source host pointer
    srcDevice : CUdeviceptr
        Source device pointer
    srcArray : CUarray
        Source array reference
    srcContext : CUcontext
        Source context (ignored with srcMemoryType is CU_MEMORYTYPE_ARRAY)
    srcPitch : size_t
        Source pitch (ignored when src is array)
    srcHeight : size_t
        Source height (ignored when src is array; may be 0 if Depth==1)
    dstXInBytes : size_t
        Destination X in bytes
    dstY : size_t
        Destination Y
    dstZ : size_t
        Destination Z
    dstLOD : size_t
        Destination LOD
    dstMemoryType : CUmemorytype
        Destination memory type (host, device, array)
    dstHost : Any
        Destination host pointer
    dstDevice : CUdeviceptr
        Destination device pointer
    dstArray : CUarray
        Destination array reference
    dstContext : CUcontext
        Destination context (ignored with dstMemoryType is
        CU_MEMORYTYPE_ARRAY)
    dstPitch : size_t
        Destination pitch (ignored when dst is array)
    dstHeight : size_t
        Destination height (ignored when dst is array; may be 0 if
        Depth==1)
    WidthInBytes : size_t
        Width of 3D memory copy in bytes
    Height : size_t
        Height of 3D memory copy
    Depth : size_t
        Depth of 3D memory copy

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_MEMCPY3D_PEER_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._srcDevice = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].srcDevice)
        self._srcArray = CUarray(_ptr=<void_ptr>&self._ptr[0].srcArray)
        self._srcContext = CUcontext(_ptr=<void_ptr>&self._ptr[0].srcContext)
        self._dstDevice = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].dstDevice)
        self._dstArray = CUarray(_ptr=<void_ptr>&self._ptr[0].dstArray)
        self._dstContext = CUcontext(_ptr=<void_ptr>&self._ptr[0].dstContext)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['srcXInBytes : ' + str(self.srcXInBytes)]
            except ValueError:
                str_list += ['srcXInBytes : <ValueError>']
            try:
                str_list += ['srcY : ' + str(self.srcY)]
            except ValueError:
                str_list += ['srcY : <ValueError>']
            try:
                str_list += ['srcZ : ' + str(self.srcZ)]
            except ValueError:
                str_list += ['srcZ : <ValueError>']
            try:
                str_list += ['srcLOD : ' + str(self.srcLOD)]
            except ValueError:
                str_list += ['srcLOD : <ValueError>']
            try:
                str_list += ['srcMemoryType : ' + str(self.srcMemoryType)]
            except ValueError:
                str_list += ['srcMemoryType : <ValueError>']
            try:
                str_list += ['srcHost : ' + hex(self.srcHost)]
            except ValueError:
                str_list += ['srcHost : <ValueError>']
            try:
                str_list += ['srcDevice : ' + str(self.srcDevice)]
            except ValueError:
                str_list += ['srcDevice : <ValueError>']
            try:
                str_list += ['srcArray : ' + str(self.srcArray)]
            except ValueError:
                str_list += ['srcArray : <ValueError>']
            try:
                str_list += ['srcContext : ' + str(self.srcContext)]
            except ValueError:
                str_list += ['srcContext : <ValueError>']
            try:
                str_list += ['srcPitch : ' + str(self.srcPitch)]
            except ValueError:
                str_list += ['srcPitch : <ValueError>']
            try:
                str_list += ['srcHeight : ' + str(self.srcHeight)]
            except ValueError:
                str_list += ['srcHeight : <ValueError>']
            try:
                str_list += ['dstXInBytes : ' + str(self.dstXInBytes)]
            except ValueError:
                str_list += ['dstXInBytes : <ValueError>']
            try:
                str_list += ['dstY : ' + str(self.dstY)]
            except ValueError:
                str_list += ['dstY : <ValueError>']
            try:
                str_list += ['dstZ : ' + str(self.dstZ)]
            except ValueError:
                str_list += ['dstZ : <ValueError>']
            try:
                str_list += ['dstLOD : ' + str(self.dstLOD)]
            except ValueError:
                str_list += ['dstLOD : <ValueError>']
            try:
                str_list += ['dstMemoryType : ' + str(self.dstMemoryType)]
            except ValueError:
                str_list += ['dstMemoryType : <ValueError>']
            try:
                str_list += ['dstHost : ' + hex(self.dstHost)]
            except ValueError:
                str_list += ['dstHost : <ValueError>']
            try:
                str_list += ['dstDevice : ' + str(self.dstDevice)]
            except ValueError:
                str_list += ['dstDevice : <ValueError>']
            try:
                str_list += ['dstArray : ' + str(self.dstArray)]
            except ValueError:
                str_list += ['dstArray : <ValueError>']
            try:
                str_list += ['dstContext : ' + str(self.dstContext)]
            except ValueError:
                str_list += ['dstContext : <ValueError>']
            try:
                str_list += ['dstPitch : ' + str(self.dstPitch)]
            except ValueError:
                str_list += ['dstPitch : <ValueError>']
            try:
                str_list += ['dstHeight : ' + str(self.dstHeight)]
            except ValueError:
                str_list += ['dstHeight : <ValueError>']
            try:
                str_list += ['WidthInBytes : ' + str(self.WidthInBytes)]
            except ValueError:
                str_list += ['WidthInBytes : <ValueError>']
            try:
                str_list += ['Height : ' + str(self.Height)]
            except ValueError:
                str_list += ['Height : <ValueError>']
            try:
                str_list += ['Depth : ' + str(self.Depth)]
            except ValueError:
                str_list += ['Depth : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def srcXInBytes(self):
        return self._ptr[0].srcXInBytes
    @srcXInBytes.setter
    def srcXInBytes(self, size_t srcXInBytes):
        self._ptr[0].srcXInBytes = srcXInBytes
    @property
    def srcY(self):
        return self._ptr[0].srcY
    @srcY.setter
    def srcY(self, size_t srcY):
        self._ptr[0].srcY = srcY
    @property
    def srcZ(self):
        return self._ptr[0].srcZ
    @srcZ.setter
    def srcZ(self, size_t srcZ):
        self._ptr[0].srcZ = srcZ
    @property
    def srcLOD(self):
        return self._ptr[0].srcLOD
    @srcLOD.setter
    def srcLOD(self, size_t srcLOD):
        self._ptr[0].srcLOD = srcLOD
    @property
    def srcMemoryType(self):
        return CUmemorytype(self._ptr[0].srcMemoryType)
    @srcMemoryType.setter
    def srcMemoryType(self, srcMemoryType not None : CUmemorytype):
        self._ptr[0].srcMemoryType = srcMemoryType.value
    @property
    def srcHost(self):
        return <void_ptr>self._ptr[0].srcHost
    @srcHost.setter
    def srcHost(self, srcHost):
        _csrcHost = utils.HelperInputVoidPtr(srcHost)
        self._ptr[0].srcHost = <void*><void_ptr>_csrcHost.cptr
    @property
    def srcDevice(self):
        return self._srcDevice
    @srcDevice.setter
    def srcDevice(self, srcDevice):
        cdef ccuda.CUdeviceptr csrcDevice
        if srcDevice is None:
            csrcDevice = <ccuda.CUdeviceptr><void_ptr>0
        elif isinstance(srcDevice, (CUdeviceptr)):
            psrcDevice = int(srcDevice)
            csrcDevice = <ccuda.CUdeviceptr><void_ptr>psrcDevice
        else:
            psrcDevice = int(CUdeviceptr(srcDevice))
            csrcDevice = <ccuda.CUdeviceptr><void_ptr>psrcDevice
        self._srcDevice._ptr[0] = csrcDevice

    @property
    def srcArray(self):
        return self._srcArray
    @srcArray.setter
    def srcArray(self, srcArray):
        cdef ccuda.CUarray csrcArray
        if srcArray is None:
            csrcArray = <ccuda.CUarray><void_ptr>0
        elif isinstance(srcArray, (CUarray,)):
            psrcArray = int(srcArray)
            csrcArray = <ccuda.CUarray><void_ptr>psrcArray
        else:
            psrcArray = int(CUarray(srcArray))
            csrcArray = <ccuda.CUarray><void_ptr>psrcArray
        self._srcArray._ptr[0] = csrcArray
    @property
    def srcContext(self):
        return self._srcContext
    @srcContext.setter
    def srcContext(self, srcContext):
        cdef ccuda.CUcontext csrcContext
        if srcContext is None:
            csrcContext = <ccuda.CUcontext><void_ptr>0
        elif isinstance(srcContext, (CUcontext,)):
            psrcContext = int(srcContext)
            csrcContext = <ccuda.CUcontext><void_ptr>psrcContext
        else:
            psrcContext = int(CUcontext(srcContext))
            csrcContext = <ccuda.CUcontext><void_ptr>psrcContext
        self._srcContext._ptr[0] = csrcContext
    @property
    def srcPitch(self):
        return self._ptr[0].srcPitch
    @srcPitch.setter
    def srcPitch(self, size_t srcPitch):
        self._ptr[0].srcPitch = srcPitch
    @property
    def srcHeight(self):
        return self._ptr[0].srcHeight
    @srcHeight.setter
    def srcHeight(self, size_t srcHeight):
        self._ptr[0].srcHeight = srcHeight
    @property
    def dstXInBytes(self):
        return self._ptr[0].dstXInBytes
    @dstXInBytes.setter
    def dstXInBytes(self, size_t dstXInBytes):
        self._ptr[0].dstXInBytes = dstXInBytes
    @property
    def dstY(self):
        return self._ptr[0].dstY
    @dstY.setter
    def dstY(self, size_t dstY):
        self._ptr[0].dstY = dstY
    @property
    def dstZ(self):
        return self._ptr[0].dstZ
    @dstZ.setter
    def dstZ(self, size_t dstZ):
        self._ptr[0].dstZ = dstZ
    @property
    def dstLOD(self):
        return self._ptr[0].dstLOD
    @dstLOD.setter
    def dstLOD(self, size_t dstLOD):
        self._ptr[0].dstLOD = dstLOD
    @property
    def dstMemoryType(self):
        return CUmemorytype(self._ptr[0].dstMemoryType)
    @dstMemoryType.setter
    def dstMemoryType(self, dstMemoryType not None : CUmemorytype):
        self._ptr[0].dstMemoryType = dstMemoryType.value
    @property
    def dstHost(self):
        return <void_ptr>self._ptr[0].dstHost
    @dstHost.setter
    def dstHost(self, dstHost):
        _cdstHost = utils.HelperInputVoidPtr(dstHost)
        self._ptr[0].dstHost = <void*><void_ptr>_cdstHost.cptr
    @property
    def dstDevice(self):
        return self._dstDevice
    @dstDevice.setter
    def dstDevice(self, dstDevice):
        cdef ccuda.CUdeviceptr cdstDevice
        if dstDevice is None:
            cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
        elif isinstance(dstDevice, (CUdeviceptr)):
            pdstDevice = int(dstDevice)
            cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
        else:
            pdstDevice = int(CUdeviceptr(dstDevice))
            cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
        self._dstDevice._ptr[0] = cdstDevice

    @property
    def dstArray(self):
        return self._dstArray
    @dstArray.setter
    def dstArray(self, dstArray):
        cdef ccuda.CUarray cdstArray
        if dstArray is None:
            cdstArray = <ccuda.CUarray><void_ptr>0
        elif isinstance(dstArray, (CUarray,)):
            pdstArray = int(dstArray)
            cdstArray = <ccuda.CUarray><void_ptr>pdstArray
        else:
            pdstArray = int(CUarray(dstArray))
            cdstArray = <ccuda.CUarray><void_ptr>pdstArray
        self._dstArray._ptr[0] = cdstArray
    @property
    def dstContext(self):
        return self._dstContext
    @dstContext.setter
    def dstContext(self, dstContext):
        cdef ccuda.CUcontext cdstContext
        if dstContext is None:
            cdstContext = <ccuda.CUcontext><void_ptr>0
        elif isinstance(dstContext, (CUcontext,)):
            pdstContext = int(dstContext)
            cdstContext = <ccuda.CUcontext><void_ptr>pdstContext
        else:
            pdstContext = int(CUcontext(dstContext))
            cdstContext = <ccuda.CUcontext><void_ptr>pdstContext
        self._dstContext._ptr[0] = cdstContext
    @property
    def dstPitch(self):
        return self._ptr[0].dstPitch
    @dstPitch.setter
    def dstPitch(self, size_t dstPitch):
        self._ptr[0].dstPitch = dstPitch
    @property
    def dstHeight(self):
        return self._ptr[0].dstHeight
    @dstHeight.setter
    def dstHeight(self, size_t dstHeight):
        self._ptr[0].dstHeight = dstHeight
    @property
    def WidthInBytes(self):
        return self._ptr[0].WidthInBytes
    @WidthInBytes.setter
    def WidthInBytes(self, size_t WidthInBytes):
        self._ptr[0].WidthInBytes = WidthInBytes
    @property
    def Height(self):
        return self._ptr[0].Height
    @Height.setter
    def Height(self, size_t Height):
        self._ptr[0].Height = Height
    @property
    def Depth(self):
        return self._ptr[0].Depth
    @Depth.setter
    def Depth(self, size_t Depth):
        self._ptr[0].Depth = Depth

cdef class CUDA_MEMCPY_NODE_PARAMS_st:
    """
    Memcpy node parameters

    Attributes
    ----------
    flags : int
        Must be zero
    reserved : int
        Must be zero
    copyCtx : CUcontext
        Context on which to run the node
    copyParams : CUDA_MEMCPY3D
        Parameters for the memory copy

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_MEMCPY_NODE_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._copyCtx = CUcontext(_ptr=<void_ptr>&self._ptr[0].copyCtx)
        self._copyParams = CUDA_MEMCPY3D(_ptr=<void_ptr>&self._ptr[0].copyParams)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            try:
                str_list += ['copyCtx : ' + str(self.copyCtx)]
            except ValueError:
                str_list += ['copyCtx : <ValueError>']
            try:
                str_list += ['copyParams :\n' + '\n'.join(['    ' + line for line in str(self.copyParams).splitlines()])]
            except ValueError:
                str_list += ['copyParams : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, int flags):
        self._ptr[0].flags = flags
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, int reserved):
        self._ptr[0].reserved = reserved
    @property
    def copyCtx(self):
        return self._copyCtx
    @copyCtx.setter
    def copyCtx(self, copyCtx):
        cdef ccuda.CUcontext ccopyCtx
        if copyCtx is None:
            ccopyCtx = <ccuda.CUcontext><void_ptr>0
        elif isinstance(copyCtx, (CUcontext,)):
            pcopyCtx = int(copyCtx)
            ccopyCtx = <ccuda.CUcontext><void_ptr>pcopyCtx
        else:
            pcopyCtx = int(CUcontext(copyCtx))
            ccopyCtx = <ccuda.CUcontext><void_ptr>pcopyCtx
        self._copyCtx._ptr[0] = ccopyCtx
    @property
    def copyParams(self):
        return self._copyParams
    @copyParams.setter
    def copyParams(self, copyParams not None : CUDA_MEMCPY3D):
        string.memcpy(&self._ptr[0].copyParams, <ccuda.CUDA_MEMCPY3D*><void_ptr>copyParams.getPtr(), sizeof(self._ptr[0].copyParams))

cdef class CUDA_ARRAY_DESCRIPTOR_st:
    """
    Array descriptor

    Attributes
    ----------
    Width : size_t
        Width of array
    Height : size_t
        Height of array
    Format : CUarray_format
        Array format
    NumChannels : unsigned int
        Channels per array element

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_ARRAY_DESCRIPTOR_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['Width : ' + str(self.Width)]
            except ValueError:
                str_list += ['Width : <ValueError>']
            try:
                str_list += ['Height : ' + str(self.Height)]
            except ValueError:
                str_list += ['Height : <ValueError>']
            try:
                str_list += ['Format : ' + str(self.Format)]
            except ValueError:
                str_list += ['Format : <ValueError>']
            try:
                str_list += ['NumChannels : ' + str(self.NumChannels)]
            except ValueError:
                str_list += ['NumChannels : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def Width(self):
        return self._ptr[0].Width
    @Width.setter
    def Width(self, size_t Width):
        self._ptr[0].Width = Width
    @property
    def Height(self):
        return self._ptr[0].Height
    @Height.setter
    def Height(self, size_t Height):
        self._ptr[0].Height = Height
    @property
    def Format(self):
        return CUarray_format(self._ptr[0].Format)
    @Format.setter
    def Format(self, Format not None : CUarray_format):
        self._ptr[0].Format = Format.value
    @property
    def NumChannels(self):
        return self._ptr[0].NumChannels
    @NumChannels.setter
    def NumChannels(self, unsigned int NumChannels):
        self._ptr[0].NumChannels = NumChannels

cdef class CUDA_ARRAY3D_DESCRIPTOR_st:
    """
    3D array descriptor

    Attributes
    ----------
    Width : size_t
        Width of 3D array
    Height : size_t
        Height of 3D array
    Depth : size_t
        Depth of 3D array
    Format : CUarray_format
        Array format
    NumChannels : unsigned int
        Channels per array element
    Flags : unsigned int
        Flags

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_ARRAY3D_DESCRIPTOR_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['Width : ' + str(self.Width)]
            except ValueError:
                str_list += ['Width : <ValueError>']
            try:
                str_list += ['Height : ' + str(self.Height)]
            except ValueError:
                str_list += ['Height : <ValueError>']
            try:
                str_list += ['Depth : ' + str(self.Depth)]
            except ValueError:
                str_list += ['Depth : <ValueError>']
            try:
                str_list += ['Format : ' + str(self.Format)]
            except ValueError:
                str_list += ['Format : <ValueError>']
            try:
                str_list += ['NumChannels : ' + str(self.NumChannels)]
            except ValueError:
                str_list += ['NumChannels : <ValueError>']
            try:
                str_list += ['Flags : ' + str(self.Flags)]
            except ValueError:
                str_list += ['Flags : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def Width(self):
        return self._ptr[0].Width
    @Width.setter
    def Width(self, size_t Width):
        self._ptr[0].Width = Width
    @property
    def Height(self):
        return self._ptr[0].Height
    @Height.setter
    def Height(self, size_t Height):
        self._ptr[0].Height = Height
    @property
    def Depth(self):
        return self._ptr[0].Depth
    @Depth.setter
    def Depth(self, size_t Depth):
        self._ptr[0].Depth = Depth
    @property
    def Format(self):
        return CUarray_format(self._ptr[0].Format)
    @Format.setter
    def Format(self, Format not None : CUarray_format):
        self._ptr[0].Format = Format.value
    @property
    def NumChannels(self):
        return self._ptr[0].NumChannels
    @NumChannels.setter
    def NumChannels(self, unsigned int NumChannels):
        self._ptr[0].NumChannels = NumChannels
    @property
    def Flags(self):
        return self._ptr[0].Flags
    @Flags.setter
    def Flags(self, unsigned int Flags):
        self._ptr[0].Flags = Flags

cdef class anon_struct5:
    """
    Attributes
    ----------
    width : unsigned int

    height : unsigned int

    depth : unsigned int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUDA_ARRAY_SPARSE_PROPERTIES_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].tileExtent
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['width : ' + str(self.width)]
            except ValueError:
                str_list += ['width : <ValueError>']
            try:
                str_list += ['height : ' + str(self.height)]
            except ValueError:
                str_list += ['height : <ValueError>']
            try:
                str_list += ['depth : ' + str(self.depth)]
            except ValueError:
                str_list += ['depth : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def width(self):
        return self._ptr[0].tileExtent.width
    @width.setter
    def width(self, unsigned int width):
        self._ptr[0].tileExtent.width = width
    @property
    def height(self):
        return self._ptr[0].tileExtent.height
    @height.setter
    def height(self, unsigned int height):
        self._ptr[0].tileExtent.height = height
    @property
    def depth(self):
        return self._ptr[0].tileExtent.depth
    @depth.setter
    def depth(self, unsigned int depth):
        self._ptr[0].tileExtent.depth = depth

cdef class CUDA_ARRAY_SPARSE_PROPERTIES_st:
    """
    CUDA array sparse properties

    Attributes
    ----------
    tileExtent : anon_struct5

    miptailFirstLevel : unsigned int
        First mip level at which the mip tail begins.
    miptailSize : unsigned long long
        Total size of the mip tail.
    flags : unsigned int
        Flags will either be zero or
        CU_ARRAY_SPARSE_PROPERTIES_SINGLE_MIPTAIL
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_ARRAY_SPARSE_PROPERTIES_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._tileExtent = anon_struct5(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['tileExtent :\n' + '\n'.join(['    ' + line for line in str(self.tileExtent).splitlines()])]
            except ValueError:
                str_list += ['tileExtent : <ValueError>']
            try:
                str_list += ['miptailFirstLevel : ' + str(self.miptailFirstLevel)]
            except ValueError:
                str_list += ['miptailFirstLevel : <ValueError>']
            try:
                str_list += ['miptailSize : ' + str(self.miptailSize)]
            except ValueError:
                str_list += ['miptailSize : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def tileExtent(self):
        return self._tileExtent
    @tileExtent.setter
    def tileExtent(self, tileExtent not None : anon_struct5):
        string.memcpy(&self._ptr[0].tileExtent, <ccuda.anon_struct5*><void_ptr>tileExtent.getPtr(), sizeof(self._ptr[0].tileExtent))
    @property
    def miptailFirstLevel(self):
        return self._ptr[0].miptailFirstLevel
    @miptailFirstLevel.setter
    def miptailFirstLevel(self, unsigned int miptailFirstLevel):
        self._ptr[0].miptailFirstLevel = miptailFirstLevel
    @property
    def miptailSize(self):
        return self._ptr[0].miptailSize
    @miptailSize.setter
    def miptailSize(self, unsigned long long miptailSize):
        self._ptr[0].miptailSize = miptailSize
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flags = flags
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class CUDA_ARRAY_MEMORY_REQUIREMENTS_st:
    """
    CUDA array memory requirements

    Attributes
    ----------
    size : size_t
        Total required memory size
    alignment : size_t
        alignment requirement
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_ARRAY_MEMORY_REQUIREMENTS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['size : ' + str(self.size)]
            except ValueError:
                str_list += ['size : <ValueError>']
            try:
                str_list += ['alignment : ' + str(self.alignment)]
            except ValueError:
                str_list += ['alignment : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def size(self):
        return self._ptr[0].size
    @size.setter
    def size(self, size_t size):
        self._ptr[0].size = size
    @property
    def alignment(self):
        return self._ptr[0].alignment
    @alignment.setter
    def alignment(self, size_t alignment):
        self._ptr[0].alignment = alignment
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class anon_struct6:
    """
    Attributes
    ----------
    hArray : CUarray


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUDA_RESOURCE_DESC_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._hArray = CUarray(_ptr=<void_ptr>&self._ptr[0].res.array.hArray)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].res.array
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['hArray : ' + str(self.hArray)]
            except ValueError:
                str_list += ['hArray : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def hArray(self):
        return self._hArray
    @hArray.setter
    def hArray(self, hArray):
        cdef ccuda.CUarray chArray
        if hArray is None:
            chArray = <ccuda.CUarray><void_ptr>0
        elif isinstance(hArray, (CUarray,)):
            phArray = int(hArray)
            chArray = <ccuda.CUarray><void_ptr>phArray
        else:
            phArray = int(CUarray(hArray))
            chArray = <ccuda.CUarray><void_ptr>phArray
        self._hArray._ptr[0] = chArray

cdef class anon_struct7:
    """
    Attributes
    ----------
    hMipmappedArray : CUmipmappedArray


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUDA_RESOURCE_DESC_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._hMipmappedArray = CUmipmappedArray(_ptr=<void_ptr>&self._ptr[0].res.mipmap.hMipmappedArray)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].res.mipmap
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['hMipmappedArray : ' + str(self.hMipmappedArray)]
            except ValueError:
                str_list += ['hMipmappedArray : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def hMipmappedArray(self):
        return self._hMipmappedArray
    @hMipmappedArray.setter
    def hMipmappedArray(self, hMipmappedArray):
        cdef ccuda.CUmipmappedArray chMipmappedArray
        if hMipmappedArray is None:
            chMipmappedArray = <ccuda.CUmipmappedArray><void_ptr>0
        elif isinstance(hMipmappedArray, (CUmipmappedArray,)):
            phMipmappedArray = int(hMipmappedArray)
            chMipmappedArray = <ccuda.CUmipmappedArray><void_ptr>phMipmappedArray
        else:
            phMipmappedArray = int(CUmipmappedArray(hMipmappedArray))
            chMipmappedArray = <ccuda.CUmipmappedArray><void_ptr>phMipmappedArray
        self._hMipmappedArray._ptr[0] = chMipmappedArray

cdef class anon_struct8:
    """
    Attributes
    ----------
    devPtr : CUdeviceptr

    format : CUarray_format

    numChannels : unsigned int

    sizeInBytes : size_t


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUDA_RESOURCE_DESC_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._devPtr = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].res.linear.devPtr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].res.linear
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['devPtr : ' + str(self.devPtr)]
            except ValueError:
                str_list += ['devPtr : <ValueError>']
            try:
                str_list += ['format : ' + str(self.format)]
            except ValueError:
                str_list += ['format : <ValueError>']
            try:
                str_list += ['numChannels : ' + str(self.numChannels)]
            except ValueError:
                str_list += ['numChannels : <ValueError>']
            try:
                str_list += ['sizeInBytes : ' + str(self.sizeInBytes)]
            except ValueError:
                str_list += ['sizeInBytes : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def devPtr(self):
        return self._devPtr
    @devPtr.setter
    def devPtr(self, devPtr):
        cdef ccuda.CUdeviceptr cdevPtr
        if devPtr is None:
            cdevPtr = <ccuda.CUdeviceptr><void_ptr>0
        elif isinstance(devPtr, (CUdeviceptr)):
            pdevPtr = int(devPtr)
            cdevPtr = <ccuda.CUdeviceptr><void_ptr>pdevPtr
        else:
            pdevPtr = int(CUdeviceptr(devPtr))
            cdevPtr = <ccuda.CUdeviceptr><void_ptr>pdevPtr
        self._devPtr._ptr[0] = cdevPtr

    @property
    def format(self):
        return CUarray_format(self._ptr[0].res.linear.format)
    @format.setter
    def format(self, format not None : CUarray_format):
        self._ptr[0].res.linear.format = format.value
    @property
    def numChannels(self):
        return self._ptr[0].res.linear.numChannels
    @numChannels.setter
    def numChannels(self, unsigned int numChannels):
        self._ptr[0].res.linear.numChannels = numChannels
    @property
    def sizeInBytes(self):
        return self._ptr[0].res.linear.sizeInBytes
    @sizeInBytes.setter
    def sizeInBytes(self, size_t sizeInBytes):
        self._ptr[0].res.linear.sizeInBytes = sizeInBytes

cdef class anon_struct9:
    """
    Attributes
    ----------
    devPtr : CUdeviceptr

    format : CUarray_format

    numChannels : unsigned int

    width : size_t

    height : size_t

    pitchInBytes : size_t


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUDA_RESOURCE_DESC_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._devPtr = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].res.pitch2D.devPtr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].res.pitch2D
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['devPtr : ' + str(self.devPtr)]
            except ValueError:
                str_list += ['devPtr : <ValueError>']
            try:
                str_list += ['format : ' + str(self.format)]
            except ValueError:
                str_list += ['format : <ValueError>']
            try:
                str_list += ['numChannels : ' + str(self.numChannels)]
            except ValueError:
                str_list += ['numChannels : <ValueError>']
            try:
                str_list += ['width : ' + str(self.width)]
            except ValueError:
                str_list += ['width : <ValueError>']
            try:
                str_list += ['height : ' + str(self.height)]
            except ValueError:
                str_list += ['height : <ValueError>']
            try:
                str_list += ['pitchInBytes : ' + str(self.pitchInBytes)]
            except ValueError:
                str_list += ['pitchInBytes : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def devPtr(self):
        return self._devPtr
    @devPtr.setter
    def devPtr(self, devPtr):
        cdef ccuda.CUdeviceptr cdevPtr
        if devPtr is None:
            cdevPtr = <ccuda.CUdeviceptr><void_ptr>0
        elif isinstance(devPtr, (CUdeviceptr)):
            pdevPtr = int(devPtr)
            cdevPtr = <ccuda.CUdeviceptr><void_ptr>pdevPtr
        else:
            pdevPtr = int(CUdeviceptr(devPtr))
            cdevPtr = <ccuda.CUdeviceptr><void_ptr>pdevPtr
        self._devPtr._ptr[0] = cdevPtr

    @property
    def format(self):
        return CUarray_format(self._ptr[0].res.pitch2D.format)
    @format.setter
    def format(self, format not None : CUarray_format):
        self._ptr[0].res.pitch2D.format = format.value
    @property
    def numChannels(self):
        return self._ptr[0].res.pitch2D.numChannels
    @numChannels.setter
    def numChannels(self, unsigned int numChannels):
        self._ptr[0].res.pitch2D.numChannels = numChannels
    @property
    def width(self):
        return self._ptr[0].res.pitch2D.width
    @width.setter
    def width(self, size_t width):
        self._ptr[0].res.pitch2D.width = width
    @property
    def height(self):
        return self._ptr[0].res.pitch2D.height
    @height.setter
    def height(self, size_t height):
        self._ptr[0].res.pitch2D.height = height
    @property
    def pitchInBytes(self):
        return self._ptr[0].res.pitch2D.pitchInBytes
    @pitchInBytes.setter
    def pitchInBytes(self, size_t pitchInBytes):
        self._ptr[0].res.pitch2D.pitchInBytes = pitchInBytes

cdef class anon_struct10:
    """
    Attributes
    ----------
    reserved : List[int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUDA_RESOURCE_DESC_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].res.reserved
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def reserved(self):
        return self._ptr[0].res.reserved.reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].res.reserved.reserved = reserved

cdef class anon_union4:
    """
    Attributes
    ----------
    array : anon_struct6

    mipmap : anon_struct7

    linear : anon_struct8

    pitch2D : anon_struct9

    reserved : anon_struct10


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUDA_RESOURCE_DESC_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._array = anon_struct6(_ptr=<void_ptr>self._ptr)
        self._mipmap = anon_struct7(_ptr=<void_ptr>self._ptr)
        self._linear = anon_struct8(_ptr=<void_ptr>self._ptr)
        self._pitch2D = anon_struct9(_ptr=<void_ptr>self._ptr)
        self._reserved = anon_struct10(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].res
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['array :\n' + '\n'.join(['    ' + line for line in str(self.array).splitlines()])]
            except ValueError:
                str_list += ['array : <ValueError>']
            try:
                str_list += ['mipmap :\n' + '\n'.join(['    ' + line for line in str(self.mipmap).splitlines()])]
            except ValueError:
                str_list += ['mipmap : <ValueError>']
            try:
                str_list += ['linear :\n' + '\n'.join(['    ' + line for line in str(self.linear).splitlines()])]
            except ValueError:
                str_list += ['linear : <ValueError>']
            try:
                str_list += ['pitch2D :\n' + '\n'.join(['    ' + line for line in str(self.pitch2D).splitlines()])]
            except ValueError:
                str_list += ['pitch2D : <ValueError>']
            try:
                str_list += ['reserved :\n' + '\n'.join(['    ' + line for line in str(self.reserved).splitlines()])]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def array(self):
        return self._array
    @array.setter
    def array(self, array not None : anon_struct6):
        string.memcpy(&self._ptr[0].res.array, <ccuda.anon_struct6*><void_ptr>array.getPtr(), sizeof(self._ptr[0].res.array))
    @property
    def mipmap(self):
        return self._mipmap
    @mipmap.setter
    def mipmap(self, mipmap not None : anon_struct7):
        string.memcpy(&self._ptr[0].res.mipmap, <ccuda.anon_struct7*><void_ptr>mipmap.getPtr(), sizeof(self._ptr[0].res.mipmap))
    @property
    def linear(self):
        return self._linear
    @linear.setter
    def linear(self, linear not None : anon_struct8):
        string.memcpy(&self._ptr[0].res.linear, <ccuda.anon_struct8*><void_ptr>linear.getPtr(), sizeof(self._ptr[0].res.linear))
    @property
    def pitch2D(self):
        return self._pitch2D
    @pitch2D.setter
    def pitch2D(self, pitch2D not None : anon_struct9):
        string.memcpy(&self._ptr[0].res.pitch2D, <ccuda.anon_struct9*><void_ptr>pitch2D.getPtr(), sizeof(self._ptr[0].res.pitch2D))
    @property
    def reserved(self):
        return self._reserved
    @reserved.setter
    def reserved(self, reserved not None : anon_struct10):
        string.memcpy(&self._ptr[0].res.reserved, <ccuda.anon_struct10*><void_ptr>reserved.getPtr(), sizeof(self._ptr[0].res.reserved))

cdef class CUDA_RESOURCE_DESC_st:
    """
    CUDA Resource descriptor

    Attributes
    ----------
    resType : CUresourcetype
        Resource type
    res : anon_union4

    flags : unsigned int
        Flags (must be zero)

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._val_ptr = <ccuda.CUDA_RESOURCE_DESC_st *>calloc(1, sizeof(ccuda.CUDA_RESOURCE_DESC_st))
            self._ptr = self._val_ptr
        else:
            self._ptr = <ccuda.CUDA_RESOURCE_DESC_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._res = anon_union4(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        if self._val_ptr is not NULL:
            free(self._val_ptr)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['resType : ' + str(self.resType)]
            except ValueError:
                str_list += ['resType : <ValueError>']
            try:
                str_list += ['res :\n' + '\n'.join(['    ' + line for line in str(self.res).splitlines()])]
            except ValueError:
                str_list += ['res : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def resType(self):
        return CUresourcetype(self._ptr[0].resType)
    @resType.setter
    def resType(self, resType not None : CUresourcetype):
        self._ptr[0].resType = resType.value
    @property
    def res(self):
        return self._res
    @res.setter
    def res(self, res not None : anon_union4):
        string.memcpy(&self._ptr[0].res, <ccuda.anon_union4*><void_ptr>res.getPtr(), sizeof(self._ptr[0].res))
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flags = flags

cdef class CUDA_TEXTURE_DESC_st:
    """
    Texture descriptor

    Attributes
    ----------
    addressMode : List[CUaddress_mode]
        Address modes
    filterMode : CUfilter_mode
        Filter mode
    flags : unsigned int
        Flags
    maxAnisotropy : unsigned int
        Maximum anisotropy ratio
    mipmapFilterMode : CUfilter_mode
        Mipmap filter mode
    mipmapLevelBias : float
        Mipmap level bias
    minMipmapLevelClamp : float
        Mipmap minimum level clamp
    maxMipmapLevelClamp : float
        Mipmap maximum level clamp
    borderColor : List[float]
        Border Color
    reserved : List[int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_TEXTURE_DESC_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['addressMode : ' + str(self.addressMode)]
            except ValueError:
                str_list += ['addressMode : <ValueError>']
            try:
                str_list += ['filterMode : ' + str(self.filterMode)]
            except ValueError:
                str_list += ['filterMode : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['maxAnisotropy : ' + str(self.maxAnisotropy)]
            except ValueError:
                str_list += ['maxAnisotropy : <ValueError>']
            try:
                str_list += ['mipmapFilterMode : ' + str(self.mipmapFilterMode)]
            except ValueError:
                str_list += ['mipmapFilterMode : <ValueError>']
            try:
                str_list += ['mipmapLevelBias : ' + str(self.mipmapLevelBias)]
            except ValueError:
                str_list += ['mipmapLevelBias : <ValueError>']
            try:
                str_list += ['minMipmapLevelClamp : ' + str(self.minMipmapLevelClamp)]
            except ValueError:
                str_list += ['minMipmapLevelClamp : <ValueError>']
            try:
                str_list += ['maxMipmapLevelClamp : ' + str(self.maxMipmapLevelClamp)]
            except ValueError:
                str_list += ['maxMipmapLevelClamp : <ValueError>']
            try:
                str_list += ['borderColor : ' + str(self.borderColor)]
            except ValueError:
                str_list += ['borderColor : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def addressMode(self):
        return [CUaddress_mode(_x) for _x in list(self._ptr[0].addressMode)]
    @addressMode.setter
    def addressMode(self, addressMode):
        self._ptr[0].addressMode = [_x.value for _x in addressMode]
    @property
    def filterMode(self):
        return CUfilter_mode(self._ptr[0].filterMode)
    @filterMode.setter
    def filterMode(self, filterMode not None : CUfilter_mode):
        self._ptr[0].filterMode = filterMode.value
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flags = flags
    @property
    def maxAnisotropy(self):
        return self._ptr[0].maxAnisotropy
    @maxAnisotropy.setter
    def maxAnisotropy(self, unsigned int maxAnisotropy):
        self._ptr[0].maxAnisotropy = maxAnisotropy
    @property
    def mipmapFilterMode(self):
        return CUfilter_mode(self._ptr[0].mipmapFilterMode)
    @mipmapFilterMode.setter
    def mipmapFilterMode(self, mipmapFilterMode not None : CUfilter_mode):
        self._ptr[0].mipmapFilterMode = mipmapFilterMode.value
    @property
    def mipmapLevelBias(self):
        return self._ptr[0].mipmapLevelBias
    @mipmapLevelBias.setter
    def mipmapLevelBias(self, float mipmapLevelBias):
        self._ptr[0].mipmapLevelBias = mipmapLevelBias
    @property
    def minMipmapLevelClamp(self):
        return self._ptr[0].minMipmapLevelClamp
    @minMipmapLevelClamp.setter
    def minMipmapLevelClamp(self, float minMipmapLevelClamp):
        self._ptr[0].minMipmapLevelClamp = minMipmapLevelClamp
    @property
    def maxMipmapLevelClamp(self):
        return self._ptr[0].maxMipmapLevelClamp
    @maxMipmapLevelClamp.setter
    def maxMipmapLevelClamp(self, float maxMipmapLevelClamp):
        self._ptr[0].maxMipmapLevelClamp = maxMipmapLevelClamp
    @property
    def borderColor(self):
        return self._ptr[0].borderColor
    @borderColor.setter
    def borderColor(self, borderColor):
        self._ptr[0].borderColor = borderColor
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class CUDA_RESOURCE_VIEW_DESC_st:
    """
    Resource view descriptor

    Attributes
    ----------
    format : CUresourceViewFormat
        Resource view format
    width : size_t
        Width of the resource view
    height : size_t
        Height of the resource view
    depth : size_t
        Depth of the resource view
    firstMipmapLevel : unsigned int
        First defined mipmap level
    lastMipmapLevel : unsigned int
        Last defined mipmap level
    firstLayer : unsigned int
        First layer index
    lastLayer : unsigned int
        Last layer index
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_RESOURCE_VIEW_DESC_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['format : ' + str(self.format)]
            except ValueError:
                str_list += ['format : <ValueError>']
            try:
                str_list += ['width : ' + str(self.width)]
            except ValueError:
                str_list += ['width : <ValueError>']
            try:
                str_list += ['height : ' + str(self.height)]
            except ValueError:
                str_list += ['height : <ValueError>']
            try:
                str_list += ['depth : ' + str(self.depth)]
            except ValueError:
                str_list += ['depth : <ValueError>']
            try:
                str_list += ['firstMipmapLevel : ' + str(self.firstMipmapLevel)]
            except ValueError:
                str_list += ['firstMipmapLevel : <ValueError>']
            try:
                str_list += ['lastMipmapLevel : ' + str(self.lastMipmapLevel)]
            except ValueError:
                str_list += ['lastMipmapLevel : <ValueError>']
            try:
                str_list += ['firstLayer : ' + str(self.firstLayer)]
            except ValueError:
                str_list += ['firstLayer : <ValueError>']
            try:
                str_list += ['lastLayer : ' + str(self.lastLayer)]
            except ValueError:
                str_list += ['lastLayer : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def format(self):
        return CUresourceViewFormat(self._ptr[0].format)
    @format.setter
    def format(self, format not None : CUresourceViewFormat):
        self._ptr[0].format = format.value
    @property
    def width(self):
        return self._ptr[0].width
    @width.setter
    def width(self, size_t width):
        self._ptr[0].width = width
    @property
    def height(self):
        return self._ptr[0].height
    @height.setter
    def height(self, size_t height):
        self._ptr[0].height = height
    @property
    def depth(self):
        return self._ptr[0].depth
    @depth.setter
    def depth(self, size_t depth):
        self._ptr[0].depth = depth
    @property
    def firstMipmapLevel(self):
        return self._ptr[0].firstMipmapLevel
    @firstMipmapLevel.setter
    def firstMipmapLevel(self, unsigned int firstMipmapLevel):
        self._ptr[0].firstMipmapLevel = firstMipmapLevel
    @property
    def lastMipmapLevel(self):
        return self._ptr[0].lastMipmapLevel
    @lastMipmapLevel.setter
    def lastMipmapLevel(self, unsigned int lastMipmapLevel):
        self._ptr[0].lastMipmapLevel = lastMipmapLevel
    @property
    def firstLayer(self):
        return self._ptr[0].firstLayer
    @firstLayer.setter
    def firstLayer(self, unsigned int firstLayer):
        self._ptr[0].firstLayer = firstLayer
    @property
    def lastLayer(self):
        return self._ptr[0].lastLayer
    @lastLayer.setter
    def lastLayer(self, unsigned int lastLayer):
        self._ptr[0].lastLayer = lastLayer
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class CUtensorMap_st:
    """
    Tensor map descriptor. Requires compiler support for aligning to 64
    bytes.

    Attributes
    ----------
    opaque : List[cuuint64_t]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUtensorMap_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['opaque : ' + str(self.opaque)]
            except ValueError:
                str_list += ['opaque : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def opaque(self):
        return [cuuint64_t(init_value=_opaque) for _opaque in self._ptr[0].opaque]
    @opaque.setter
    def opaque(self, opaque):
        self._ptr[0].opaque = opaque

cdef class CUDA_POINTER_ATTRIBUTE_P2P_TOKENS_st:
    """
    GPU Direct v3 tokens

    Attributes
    ----------
    p2pToken : unsigned long long

    vaSpaceToken : unsigned int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_POINTER_ATTRIBUTE_P2P_TOKENS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['p2pToken : ' + str(self.p2pToken)]
            except ValueError:
                str_list += ['p2pToken : <ValueError>']
            try:
                str_list += ['vaSpaceToken : ' + str(self.vaSpaceToken)]
            except ValueError:
                str_list += ['vaSpaceToken : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def p2pToken(self):
        return self._ptr[0].p2pToken
    @p2pToken.setter
    def p2pToken(self, unsigned long long p2pToken):
        self._ptr[0].p2pToken = p2pToken
    @property
    def vaSpaceToken(self):
        return self._ptr[0].vaSpaceToken
    @vaSpaceToken.setter
    def vaSpaceToken(self, unsigned int vaSpaceToken):
        self._ptr[0].vaSpaceToken = vaSpaceToken

cdef class CUDA_LAUNCH_PARAMS_st:
    """
    Kernel launch parameters

    Attributes
    ----------
    function : CUfunction
        Kernel to launch
    gridDimX : unsigned int
        Width of grid in blocks
    gridDimY : unsigned int
        Height of grid in blocks
    gridDimZ : unsigned int
        Depth of grid in blocks
    blockDimX : unsigned int
        X dimension of each thread block
    blockDimY : unsigned int
        Y dimension of each thread block
    blockDimZ : unsigned int
        Z dimension of each thread block
    sharedMemBytes : unsigned int
        Dynamic shared-memory size per thread block in bytes
    hStream : CUstream
        Stream identifier
    kernelParams : Any
        Array of pointers to kernel parameters

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_LAUNCH_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._function = CUfunction(_ptr=<void_ptr>&self._ptr[0].function)
        self._hStream = CUstream(_ptr=<void_ptr>&self._ptr[0].hStream)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['function : ' + str(self.function)]
            except ValueError:
                str_list += ['function : <ValueError>']
            try:
                str_list += ['gridDimX : ' + str(self.gridDimX)]
            except ValueError:
                str_list += ['gridDimX : <ValueError>']
            try:
                str_list += ['gridDimY : ' + str(self.gridDimY)]
            except ValueError:
                str_list += ['gridDimY : <ValueError>']
            try:
                str_list += ['gridDimZ : ' + str(self.gridDimZ)]
            except ValueError:
                str_list += ['gridDimZ : <ValueError>']
            try:
                str_list += ['blockDimX : ' + str(self.blockDimX)]
            except ValueError:
                str_list += ['blockDimX : <ValueError>']
            try:
                str_list += ['blockDimY : ' + str(self.blockDimY)]
            except ValueError:
                str_list += ['blockDimY : <ValueError>']
            try:
                str_list += ['blockDimZ : ' + str(self.blockDimZ)]
            except ValueError:
                str_list += ['blockDimZ : <ValueError>']
            try:
                str_list += ['sharedMemBytes : ' + str(self.sharedMemBytes)]
            except ValueError:
                str_list += ['sharedMemBytes : <ValueError>']
            try:
                str_list += ['hStream : ' + str(self.hStream)]
            except ValueError:
                str_list += ['hStream : <ValueError>']
            try:
                str_list += ['kernelParams : ' + str(self.kernelParams)]
            except ValueError:
                str_list += ['kernelParams : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def function(self):
        return self._function
    @function.setter
    def function(self, function):
        cdef ccuda.CUfunction cfunction
        if function is None:
            cfunction = <ccuda.CUfunction><void_ptr>0
        elif isinstance(function, (CUfunction,)):
            pfunction = int(function)
            cfunction = <ccuda.CUfunction><void_ptr>pfunction
        else:
            pfunction = int(CUfunction(function))
            cfunction = <ccuda.CUfunction><void_ptr>pfunction
        self._function._ptr[0] = cfunction
    @property
    def gridDimX(self):
        return self._ptr[0].gridDimX
    @gridDimX.setter
    def gridDimX(self, unsigned int gridDimX):
        self._ptr[0].gridDimX = gridDimX
    @property
    def gridDimY(self):
        return self._ptr[0].gridDimY
    @gridDimY.setter
    def gridDimY(self, unsigned int gridDimY):
        self._ptr[0].gridDimY = gridDimY
    @property
    def gridDimZ(self):
        return self._ptr[0].gridDimZ
    @gridDimZ.setter
    def gridDimZ(self, unsigned int gridDimZ):
        self._ptr[0].gridDimZ = gridDimZ
    @property
    def blockDimX(self):
        return self._ptr[0].blockDimX
    @blockDimX.setter
    def blockDimX(self, unsigned int blockDimX):
        self._ptr[0].blockDimX = blockDimX
    @property
    def blockDimY(self):
        return self._ptr[0].blockDimY
    @blockDimY.setter
    def blockDimY(self, unsigned int blockDimY):
        self._ptr[0].blockDimY = blockDimY
    @property
    def blockDimZ(self):
        return self._ptr[0].blockDimZ
    @blockDimZ.setter
    def blockDimZ(self, unsigned int blockDimZ):
        self._ptr[0].blockDimZ = blockDimZ
    @property
    def sharedMemBytes(self):
        return self._ptr[0].sharedMemBytes
    @sharedMemBytes.setter
    def sharedMemBytes(self, unsigned int sharedMemBytes):
        self._ptr[0].sharedMemBytes = sharedMemBytes
    @property
    def hStream(self):
        return self._hStream
    @hStream.setter
    def hStream(self, hStream):
        cdef ccuda.CUstream chStream
        if hStream is None:
            chStream = <ccuda.CUstream><void_ptr>0
        elif isinstance(hStream, (CUstream,)):
            phStream = int(hStream)
            chStream = <ccuda.CUstream><void_ptr>phStream
        else:
            phStream = int(CUstream(hStream))
            chStream = <ccuda.CUstream><void_ptr>phStream
        self._hStream._ptr[0] = chStream
    @property
    def kernelParams(self):
        return <void_ptr>self._ptr[0].kernelParams
    @kernelParams.setter
    def kernelParams(self, kernelParams):
        self._ckernelParams = utils.HelperKernelParams(kernelParams)
        self._ptr[0].kernelParams = <void**><void_ptr>self._ckernelParams.ckernelParams

cdef class anon_struct11:
    """
    Attributes
    ----------
    handle : Any

    name : Any


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUDA_EXTERNAL_MEMORY_HANDLE_DESC_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].handle.win32
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['handle : ' + hex(self.handle)]
            except ValueError:
                str_list += ['handle : <ValueError>']
            try:
                str_list += ['name : ' + hex(self.name)]
            except ValueError:
                str_list += ['name : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def handle(self):
        return <void_ptr>self._ptr[0].handle.win32.handle
    @handle.setter
    def handle(self, handle):
        _chandle = utils.HelperInputVoidPtr(handle)
        self._ptr[0].handle.win32.handle = <void*><void_ptr>_chandle.cptr
    @property
    def name(self):
        return <void_ptr>self._ptr[0].handle.win32.name
    @name.setter
    def name(self, name):
        _cname = utils.HelperInputVoidPtr(name)
        self._ptr[0].handle.win32.name = <void*><void_ptr>_cname.cptr

cdef class anon_union5:
    """
    Attributes
    ----------
    fd : int

    win32 : anon_struct11

    nvSciBufObject : Any


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUDA_EXTERNAL_MEMORY_HANDLE_DESC_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._win32 = anon_struct11(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].handle
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['fd : ' + str(self.fd)]
            except ValueError:
                str_list += ['fd : <ValueError>']
            try:
                str_list += ['win32 :\n' + '\n'.join(['    ' + line for line in str(self.win32).splitlines()])]
            except ValueError:
                str_list += ['win32 : <ValueError>']
            try:
                str_list += ['nvSciBufObject : ' + hex(self.nvSciBufObject)]
            except ValueError:
                str_list += ['nvSciBufObject : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def fd(self):
        return self._ptr[0].handle.fd
    @fd.setter
    def fd(self, int fd):
        self._ptr[0].handle.fd = fd
    @property
    def win32(self):
        return self._win32
    @win32.setter
    def win32(self, win32 not None : anon_struct11):
        string.memcpy(&self._ptr[0].handle.win32, <ccuda.anon_struct11*><void_ptr>win32.getPtr(), sizeof(self._ptr[0].handle.win32))
    @property
    def nvSciBufObject(self):
        return <void_ptr>self._ptr[0].handle.nvSciBufObject
    @nvSciBufObject.setter
    def nvSciBufObject(self, nvSciBufObject):
        _cnvSciBufObject = utils.HelperInputVoidPtr(nvSciBufObject)
        self._ptr[0].handle.nvSciBufObject = <void*><void_ptr>_cnvSciBufObject.cptr

cdef class CUDA_EXTERNAL_MEMORY_HANDLE_DESC_st:
    """
    External memory handle descriptor

    Attributes
    ----------
    type : CUexternalMemoryHandleType
        Type of the handle
    handle : anon_union5

    size : unsigned long long
        Size of the memory allocation
    flags : unsigned int
        Flags must either be zero or CUDA_EXTERNAL_MEMORY_DEDICATED
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._val_ptr = <ccuda.CUDA_EXTERNAL_MEMORY_HANDLE_DESC_st *>calloc(1, sizeof(ccuda.CUDA_EXTERNAL_MEMORY_HANDLE_DESC_st))
            self._ptr = self._val_ptr
        else:
            self._ptr = <ccuda.CUDA_EXTERNAL_MEMORY_HANDLE_DESC_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._handle = anon_union5(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        if self._val_ptr is not NULL:
            free(self._val_ptr)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['type : ' + str(self.type)]
            except ValueError:
                str_list += ['type : <ValueError>']
            try:
                str_list += ['handle :\n' + '\n'.join(['    ' + line for line in str(self.handle).splitlines()])]
            except ValueError:
                str_list += ['handle : <ValueError>']
            try:
                str_list += ['size : ' + str(self.size)]
            except ValueError:
                str_list += ['size : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def type(self):
        return CUexternalMemoryHandleType(self._ptr[0].type)
    @type.setter
    def type(self, type not None : CUexternalMemoryHandleType):
        self._ptr[0].type = type.value
    @property
    def handle(self):
        return self._handle
    @handle.setter
    def handle(self, handle not None : anon_union5):
        string.memcpy(&self._ptr[0].handle, <ccuda.anon_union5*><void_ptr>handle.getPtr(), sizeof(self._ptr[0].handle))
    @property
    def size(self):
        return self._ptr[0].size
    @size.setter
    def size(self, unsigned long long size):
        self._ptr[0].size = size
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flags = flags
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class CUDA_EXTERNAL_MEMORY_BUFFER_DESC_st:
    """
    External memory buffer descriptor

    Attributes
    ----------
    offset : unsigned long long
        Offset into the memory object where the buffer's base is
    size : unsigned long long
        Size of the buffer
    flags : unsigned int
        Flags reserved for future use. Must be zero.
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_EXTERNAL_MEMORY_BUFFER_DESC_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['offset : ' + str(self.offset)]
            except ValueError:
                str_list += ['offset : <ValueError>']
            try:
                str_list += ['size : ' + str(self.size)]
            except ValueError:
                str_list += ['size : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def offset(self):
        return self._ptr[0].offset
    @offset.setter
    def offset(self, unsigned long long offset):
        self._ptr[0].offset = offset
    @property
    def size(self):
        return self._ptr[0].size
    @size.setter
    def size(self, unsigned long long size):
        self._ptr[0].size = size
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flags = flags
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class CUDA_EXTERNAL_MEMORY_MIPMAPPED_ARRAY_DESC_st:
    """
    External memory mipmap descriptor

    Attributes
    ----------
    offset : unsigned long long
        Offset into the memory object where the base level of the mipmap
        chain is.
    arrayDesc : CUDA_ARRAY3D_DESCRIPTOR
        Format, dimension and type of base level of the mipmap chain
    numLevels : unsigned int
        Total number of levels in the mipmap chain
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_EXTERNAL_MEMORY_MIPMAPPED_ARRAY_DESC_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._arrayDesc = CUDA_ARRAY3D_DESCRIPTOR(_ptr=<void_ptr>&self._ptr[0].arrayDesc)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['offset : ' + str(self.offset)]
            except ValueError:
                str_list += ['offset : <ValueError>']
            try:
                str_list += ['arrayDesc :\n' + '\n'.join(['    ' + line for line in str(self.arrayDesc).splitlines()])]
            except ValueError:
                str_list += ['arrayDesc : <ValueError>']
            try:
                str_list += ['numLevels : ' + str(self.numLevels)]
            except ValueError:
                str_list += ['numLevels : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def offset(self):
        return self._ptr[0].offset
    @offset.setter
    def offset(self, unsigned long long offset):
        self._ptr[0].offset = offset
    @property
    def arrayDesc(self):
        return self._arrayDesc
    @arrayDesc.setter
    def arrayDesc(self, arrayDesc not None : CUDA_ARRAY3D_DESCRIPTOR):
        string.memcpy(&self._ptr[0].arrayDesc, <ccuda.CUDA_ARRAY3D_DESCRIPTOR*><void_ptr>arrayDesc.getPtr(), sizeof(self._ptr[0].arrayDesc))
    @property
    def numLevels(self):
        return self._ptr[0].numLevels
    @numLevels.setter
    def numLevels(self, unsigned int numLevels):
        self._ptr[0].numLevels = numLevels
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class anon_struct12:
    """
    Attributes
    ----------
    handle : Any

    name : Any


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].handle.win32
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['handle : ' + hex(self.handle)]
            except ValueError:
                str_list += ['handle : <ValueError>']
            try:
                str_list += ['name : ' + hex(self.name)]
            except ValueError:
                str_list += ['name : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def handle(self):
        return <void_ptr>self._ptr[0].handle.win32.handle
    @handle.setter
    def handle(self, handle):
        _chandle = utils.HelperInputVoidPtr(handle)
        self._ptr[0].handle.win32.handle = <void*><void_ptr>_chandle.cptr
    @property
    def name(self):
        return <void_ptr>self._ptr[0].handle.win32.name
    @name.setter
    def name(self, name):
        _cname = utils.HelperInputVoidPtr(name)
        self._ptr[0].handle.win32.name = <void*><void_ptr>_cname.cptr

cdef class anon_union6:
    """
    Attributes
    ----------
    fd : int

    win32 : anon_struct12

    nvSciSyncObj : Any


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._win32 = anon_struct12(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].handle
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['fd : ' + str(self.fd)]
            except ValueError:
                str_list += ['fd : <ValueError>']
            try:
                str_list += ['win32 :\n' + '\n'.join(['    ' + line for line in str(self.win32).splitlines()])]
            except ValueError:
                str_list += ['win32 : <ValueError>']
            try:
                str_list += ['nvSciSyncObj : ' + hex(self.nvSciSyncObj)]
            except ValueError:
                str_list += ['nvSciSyncObj : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def fd(self):
        return self._ptr[0].handle.fd
    @fd.setter
    def fd(self, int fd):
        self._ptr[0].handle.fd = fd
    @property
    def win32(self):
        return self._win32
    @win32.setter
    def win32(self, win32 not None : anon_struct12):
        string.memcpy(&self._ptr[0].handle.win32, <ccuda.anon_struct12*><void_ptr>win32.getPtr(), sizeof(self._ptr[0].handle.win32))
    @property
    def nvSciSyncObj(self):
        return <void_ptr>self._ptr[0].handle.nvSciSyncObj
    @nvSciSyncObj.setter
    def nvSciSyncObj(self, nvSciSyncObj):
        _cnvSciSyncObj = utils.HelperInputVoidPtr(nvSciSyncObj)
        self._ptr[0].handle.nvSciSyncObj = <void*><void_ptr>_cnvSciSyncObj.cptr

cdef class CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC_st:
    """
    External semaphore handle descriptor

    Attributes
    ----------
    type : CUexternalSemaphoreHandleType
        Type of the handle
    handle : anon_union6

    flags : unsigned int
        Flags reserved for the future. Must be zero.
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._val_ptr = <ccuda.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC_st *>calloc(1, sizeof(ccuda.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC_st))
            self._ptr = self._val_ptr
        else:
            self._ptr = <ccuda.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._handle = anon_union6(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        if self._val_ptr is not NULL:
            free(self._val_ptr)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['type : ' + str(self.type)]
            except ValueError:
                str_list += ['type : <ValueError>']
            try:
                str_list += ['handle :\n' + '\n'.join(['    ' + line for line in str(self.handle).splitlines()])]
            except ValueError:
                str_list += ['handle : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def type(self):
        return CUexternalSemaphoreHandleType(self._ptr[0].type)
    @type.setter
    def type(self, type not None : CUexternalSemaphoreHandleType):
        self._ptr[0].type = type.value
    @property
    def handle(self):
        return self._handle
    @handle.setter
    def handle(self, handle not None : anon_union6):
        string.memcpy(&self._ptr[0].handle, <ccuda.anon_union6*><void_ptr>handle.getPtr(), sizeof(self._ptr[0].handle))
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flags = flags
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class anon_struct13:
    """
    Attributes
    ----------
    value : unsigned long long


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].params.fence
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['value : ' + str(self.value)]
            except ValueError:
                str_list += ['value : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def value(self):
        return self._ptr[0].params.fence.value
    @value.setter
    def value(self, unsigned long long value):
        self._ptr[0].params.fence.value = value

cdef class anon_union7:
    """
    Attributes
    ----------
    fence : Any

    reserved : unsigned long long


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].params.nvSciSync
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['fence : ' + hex(self.fence)]
            except ValueError:
                str_list += ['fence : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def fence(self):
        return <void_ptr>self._ptr[0].params.nvSciSync.fence
    @fence.setter
    def fence(self, fence):
        _cfence = utils.HelperInputVoidPtr(fence)
        self._ptr[0].params.nvSciSync.fence = <void*><void_ptr>_cfence.cptr
    @property
    def reserved(self):
        return self._ptr[0].params.nvSciSync.reserved
    @reserved.setter
    def reserved(self, unsigned long long reserved):
        self._ptr[0].params.nvSciSync.reserved = reserved

cdef class anon_struct14:
    """
    Attributes
    ----------
    key : unsigned long long


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].params.keyedMutex
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['key : ' + str(self.key)]
            except ValueError:
                str_list += ['key : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def key(self):
        return self._ptr[0].params.keyedMutex.key
    @key.setter
    def key(self, unsigned long long key):
        self._ptr[0].params.keyedMutex.key = key

cdef class anon_struct15:
    """
    Attributes
    ----------
    fence : anon_struct13

    nvSciSync : anon_union7

    keyedMutex : anon_struct14

    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._fence = anon_struct13(_ptr=<void_ptr>self._ptr)
        self._nvSciSync = anon_union7(_ptr=<void_ptr>self._ptr)
        self._keyedMutex = anon_struct14(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].params
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['fence :\n' + '\n'.join(['    ' + line for line in str(self.fence).splitlines()])]
            except ValueError:
                str_list += ['fence : <ValueError>']
            try:
                str_list += ['nvSciSync :\n' + '\n'.join(['    ' + line for line in str(self.nvSciSync).splitlines()])]
            except ValueError:
                str_list += ['nvSciSync : <ValueError>']
            try:
                str_list += ['keyedMutex :\n' + '\n'.join(['    ' + line for line in str(self.keyedMutex).splitlines()])]
            except ValueError:
                str_list += ['keyedMutex : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def fence(self):
        return self._fence
    @fence.setter
    def fence(self, fence not None : anon_struct13):
        string.memcpy(&self._ptr[0].params.fence, <ccuda.anon_struct13*><void_ptr>fence.getPtr(), sizeof(self._ptr[0].params.fence))
    @property
    def nvSciSync(self):
        return self._nvSciSync
    @nvSciSync.setter
    def nvSciSync(self, nvSciSync not None : anon_union7):
        string.memcpy(&self._ptr[0].params.nvSciSync, <ccuda.anon_union7*><void_ptr>nvSciSync.getPtr(), sizeof(self._ptr[0].params.nvSciSync))
    @property
    def keyedMutex(self):
        return self._keyedMutex
    @keyedMutex.setter
    def keyedMutex(self, keyedMutex not None : anon_struct14):
        string.memcpy(&self._ptr[0].params.keyedMutex, <ccuda.anon_struct14*><void_ptr>keyedMutex.getPtr(), sizeof(self._ptr[0].params.keyedMutex))
    @property
    def reserved(self):
        return self._ptr[0].params.reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].params.reserved = reserved

cdef class CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS_st:
    """
    External semaphore signal parameters

    Attributes
    ----------
    params : anon_struct15

    flags : unsigned int
        Only when ::CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS is used to signal
        a CUexternalSemaphore of type
        CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_NVSCISYNC, the valid flag is
        CUDA_EXTERNAL_SEMAPHORE_SIGNAL_SKIP_NVSCIBUF_MEMSYNC which
        indicates that while signaling the CUexternalSemaphore, no memory
        synchronization operations should be performed for any external
        memory object imported as CU_EXTERNAL_MEMORY_HANDLE_TYPE_NVSCIBUF.
        For all other types of CUexternalSemaphore, flags must be zero.
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._params = anon_struct15(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['params :\n' + '\n'.join(['    ' + line for line in str(self.params).splitlines()])]
            except ValueError:
                str_list += ['params : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def params(self):
        return self._params
    @params.setter
    def params(self, params not None : anon_struct15):
        string.memcpy(&self._ptr[0].params, <ccuda.anon_struct15*><void_ptr>params.getPtr(), sizeof(self._ptr[0].params))
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flags = flags
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class anon_struct16:
    """
    Attributes
    ----------
    value : unsigned long long


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].params.fence
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['value : ' + str(self.value)]
            except ValueError:
                str_list += ['value : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def value(self):
        return self._ptr[0].params.fence.value
    @value.setter
    def value(self, unsigned long long value):
        self._ptr[0].params.fence.value = value

cdef class anon_union8:
    """
    Attributes
    ----------
    fence : Any

    reserved : unsigned long long


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].params.nvSciSync
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['fence : ' + hex(self.fence)]
            except ValueError:
                str_list += ['fence : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def fence(self):
        return <void_ptr>self._ptr[0].params.nvSciSync.fence
    @fence.setter
    def fence(self, fence):
        _cfence = utils.HelperInputVoidPtr(fence)
        self._ptr[0].params.nvSciSync.fence = <void*><void_ptr>_cfence.cptr
    @property
    def reserved(self):
        return self._ptr[0].params.nvSciSync.reserved
    @reserved.setter
    def reserved(self, unsigned long long reserved):
        self._ptr[0].params.nvSciSync.reserved = reserved

cdef class anon_struct17:
    """
    Attributes
    ----------
    key : unsigned long long

    timeoutMs : unsigned int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].params.keyedMutex
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['key : ' + str(self.key)]
            except ValueError:
                str_list += ['key : <ValueError>']
            try:
                str_list += ['timeoutMs : ' + str(self.timeoutMs)]
            except ValueError:
                str_list += ['timeoutMs : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def key(self):
        return self._ptr[0].params.keyedMutex.key
    @key.setter
    def key(self, unsigned long long key):
        self._ptr[0].params.keyedMutex.key = key
    @property
    def timeoutMs(self):
        return self._ptr[0].params.keyedMutex.timeoutMs
    @timeoutMs.setter
    def timeoutMs(self, unsigned int timeoutMs):
        self._ptr[0].params.keyedMutex.timeoutMs = timeoutMs

cdef class anon_struct18:
    """
    Attributes
    ----------
    fence : anon_struct16

    nvSciSync : anon_union8

    keyedMutex : anon_struct17

    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._fence = anon_struct16(_ptr=<void_ptr>self._ptr)
        self._nvSciSync = anon_union8(_ptr=<void_ptr>self._ptr)
        self._keyedMutex = anon_struct17(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].params
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['fence :\n' + '\n'.join(['    ' + line for line in str(self.fence).splitlines()])]
            except ValueError:
                str_list += ['fence : <ValueError>']
            try:
                str_list += ['nvSciSync :\n' + '\n'.join(['    ' + line for line in str(self.nvSciSync).splitlines()])]
            except ValueError:
                str_list += ['nvSciSync : <ValueError>']
            try:
                str_list += ['keyedMutex :\n' + '\n'.join(['    ' + line for line in str(self.keyedMutex).splitlines()])]
            except ValueError:
                str_list += ['keyedMutex : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def fence(self):
        return self._fence
    @fence.setter
    def fence(self, fence not None : anon_struct16):
        string.memcpy(&self._ptr[0].params.fence, <ccuda.anon_struct16*><void_ptr>fence.getPtr(), sizeof(self._ptr[0].params.fence))
    @property
    def nvSciSync(self):
        return self._nvSciSync
    @nvSciSync.setter
    def nvSciSync(self, nvSciSync not None : anon_union8):
        string.memcpy(&self._ptr[0].params.nvSciSync, <ccuda.anon_union8*><void_ptr>nvSciSync.getPtr(), sizeof(self._ptr[0].params.nvSciSync))
    @property
    def keyedMutex(self):
        return self._keyedMutex
    @keyedMutex.setter
    def keyedMutex(self, keyedMutex not None : anon_struct17):
        string.memcpy(&self._ptr[0].params.keyedMutex, <ccuda.anon_struct17*><void_ptr>keyedMutex.getPtr(), sizeof(self._ptr[0].params.keyedMutex))
    @property
    def reserved(self):
        return self._ptr[0].params.reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].params.reserved = reserved

cdef class CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS_st:
    """
    External semaphore wait parameters

    Attributes
    ----------
    params : anon_struct18

    flags : unsigned int
        Only when ::CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS is used to wait on
        a CUexternalSemaphore of type
        CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_NVSCISYNC, the valid flag is
        CUDA_EXTERNAL_SEMAPHORE_WAIT_SKIP_NVSCIBUF_MEMSYNC which indicates
        that while waiting for the CUexternalSemaphore, no memory
        synchronization operations should be performed for any external
        memory object imported as CU_EXTERNAL_MEMORY_HANDLE_TYPE_NVSCIBUF.
        For all other types of CUexternalSemaphore, flags must be zero.
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._params = anon_struct18(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['params :\n' + '\n'.join(['    ' + line for line in str(self.params).splitlines()])]
            except ValueError:
                str_list += ['params : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def params(self):
        return self._params
    @params.setter
    def params(self, params not None : anon_struct18):
        string.memcpy(&self._ptr[0].params, <ccuda.anon_struct18*><void_ptr>params.getPtr(), sizeof(self._ptr[0].params))
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flags = flags
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class CUDA_EXT_SEM_SIGNAL_NODE_PARAMS_st:
    """
    Semaphore signal node parameters

    Attributes
    ----------
    extSemArray : CUexternalSemaphore
        Array of external semaphore handles.
    paramsArray : CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS
        Array of external semaphore signal parameters.
    numExtSems : unsigned int
        Number of handles and parameters supplied in extSemArray and
        paramsArray.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_EXT_SEM_SIGNAL_NODE_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        if self._extSemArray is not NULL:
            free(self._extSemArray)
        if self._paramsArray is not NULL:
            free(self._paramsArray)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['extSemArray : ' + str(self.extSemArray)]
            except ValueError:
                str_list += ['extSemArray : <ValueError>']
            try:
                str_list += ['paramsArray : ' + str(self.paramsArray)]
            except ValueError:
                str_list += ['paramsArray : <ValueError>']
            try:
                str_list += ['numExtSems : ' + str(self.numExtSems)]
            except ValueError:
                str_list += ['numExtSems : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def extSemArray(self):
        arrs = [<void_ptr>self._ptr[0].extSemArray + x*sizeof(ccuda.CUexternalSemaphore) for x in range(self._extSemArray_length)]
        return [CUexternalSemaphore(_ptr=arr) for arr in arrs]
    @extSemArray.setter
    def extSemArray(self, val):
        if len(val) == 0:
            free(self._extSemArray)
            self._extSemArray_length = 0
            self._ptr[0].extSemArray = NULL
        else:
            if self._extSemArray_length != <size_t>len(val):
                free(self._extSemArray)
                self._extSemArray = <ccuda.CUexternalSemaphore*> calloc(len(val), sizeof(ccuda.CUexternalSemaphore))
                if self._extSemArray is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(ccuda.CUexternalSemaphore)))
                self._extSemArray_length = <size_t>len(val)
                self._ptr[0].extSemArray = self._extSemArray
            for idx in range(len(val)):
                self._extSemArray[idx] = (<CUexternalSemaphore>val[idx])._ptr[0]

    @property
    def paramsArray(self):
        arrs = [<void_ptr>self._ptr[0].paramsArray + x*sizeof(ccuda.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS) for x in range(self._paramsArray_length)]
        return [CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS(_ptr=arr) for arr in arrs]
    @paramsArray.setter
    def paramsArray(self, val):
        if len(val) == 0:
            free(self._paramsArray)
            self._paramsArray_length = 0
            self._ptr[0].paramsArray = NULL
        else:
            if self._paramsArray_length != <size_t>len(val):
                free(self._paramsArray)
                self._paramsArray = <ccuda.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS*> calloc(len(val), sizeof(ccuda.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS))
                if self._paramsArray is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(ccuda.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS)))
                self._paramsArray_length = <size_t>len(val)
                self._ptr[0].paramsArray = self._paramsArray
            for idx in range(len(val)):
                string.memcpy(&self._paramsArray[idx], (<CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS>val[idx])._ptr, sizeof(ccuda.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS))

    @property
    def numExtSems(self):
        return self._ptr[0].numExtSems
    @numExtSems.setter
    def numExtSems(self, unsigned int numExtSems):
        self._ptr[0].numExtSems = numExtSems

cdef class CUDA_EXT_SEM_SIGNAL_NODE_PARAMS_v2_st:
    """
    Semaphore signal node parameters

    Attributes
    ----------
    extSemArray : CUexternalSemaphore
        Array of external semaphore handles.
    paramsArray : CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS
        Array of external semaphore signal parameters.
    numExtSems : unsigned int
        Number of handles and parameters supplied in extSemArray and
        paramsArray.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_EXT_SEM_SIGNAL_NODE_PARAMS_v2_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        if self._extSemArray is not NULL:
            free(self._extSemArray)
        if self._paramsArray is not NULL:
            free(self._paramsArray)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['extSemArray : ' + str(self.extSemArray)]
            except ValueError:
                str_list += ['extSemArray : <ValueError>']
            try:
                str_list += ['paramsArray : ' + str(self.paramsArray)]
            except ValueError:
                str_list += ['paramsArray : <ValueError>']
            try:
                str_list += ['numExtSems : ' + str(self.numExtSems)]
            except ValueError:
                str_list += ['numExtSems : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def extSemArray(self):
        arrs = [<void_ptr>self._ptr[0].extSemArray + x*sizeof(ccuda.CUexternalSemaphore) for x in range(self._extSemArray_length)]
        return [CUexternalSemaphore(_ptr=arr) for arr in arrs]
    @extSemArray.setter
    def extSemArray(self, val):
        if len(val) == 0:
            free(self._extSemArray)
            self._extSemArray_length = 0
            self._ptr[0].extSemArray = NULL
        else:
            if self._extSemArray_length != <size_t>len(val):
                free(self._extSemArray)
                self._extSemArray = <ccuda.CUexternalSemaphore*> calloc(len(val), sizeof(ccuda.CUexternalSemaphore))
                if self._extSemArray is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(ccuda.CUexternalSemaphore)))
                self._extSemArray_length = <size_t>len(val)
                self._ptr[0].extSemArray = self._extSemArray
            for idx in range(len(val)):
                self._extSemArray[idx] = (<CUexternalSemaphore>val[idx])._ptr[0]

    @property
    def paramsArray(self):
        arrs = [<void_ptr>self._ptr[0].paramsArray + x*sizeof(ccuda.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS) for x in range(self._paramsArray_length)]
        return [CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS(_ptr=arr) for arr in arrs]
    @paramsArray.setter
    def paramsArray(self, val):
        if len(val) == 0:
            free(self._paramsArray)
            self._paramsArray_length = 0
            self._ptr[0].paramsArray = NULL
        else:
            if self._paramsArray_length != <size_t>len(val):
                free(self._paramsArray)
                self._paramsArray = <ccuda.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS*> calloc(len(val), sizeof(ccuda.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS))
                if self._paramsArray is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(ccuda.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS)))
                self._paramsArray_length = <size_t>len(val)
                self._ptr[0].paramsArray = self._paramsArray
            for idx in range(len(val)):
                string.memcpy(&self._paramsArray[idx], (<CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS>val[idx])._ptr, sizeof(ccuda.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS))

    @property
    def numExtSems(self):
        return self._ptr[0].numExtSems
    @numExtSems.setter
    def numExtSems(self, unsigned int numExtSems):
        self._ptr[0].numExtSems = numExtSems

cdef class CUDA_EXT_SEM_WAIT_NODE_PARAMS_st:
    """
    Semaphore wait node parameters

    Attributes
    ----------
    extSemArray : CUexternalSemaphore
        Array of external semaphore handles.
    paramsArray : CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS
        Array of external semaphore wait parameters.
    numExtSems : unsigned int
        Number of handles and parameters supplied in extSemArray and
        paramsArray.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_EXT_SEM_WAIT_NODE_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        if self._extSemArray is not NULL:
            free(self._extSemArray)
        if self._paramsArray is not NULL:
            free(self._paramsArray)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['extSemArray : ' + str(self.extSemArray)]
            except ValueError:
                str_list += ['extSemArray : <ValueError>']
            try:
                str_list += ['paramsArray : ' + str(self.paramsArray)]
            except ValueError:
                str_list += ['paramsArray : <ValueError>']
            try:
                str_list += ['numExtSems : ' + str(self.numExtSems)]
            except ValueError:
                str_list += ['numExtSems : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def extSemArray(self):
        arrs = [<void_ptr>self._ptr[0].extSemArray + x*sizeof(ccuda.CUexternalSemaphore) for x in range(self._extSemArray_length)]
        return [CUexternalSemaphore(_ptr=arr) for arr in arrs]
    @extSemArray.setter
    def extSemArray(self, val):
        if len(val) == 0:
            free(self._extSemArray)
            self._extSemArray_length = 0
            self._ptr[0].extSemArray = NULL
        else:
            if self._extSemArray_length != <size_t>len(val):
                free(self._extSemArray)
                self._extSemArray = <ccuda.CUexternalSemaphore*> calloc(len(val), sizeof(ccuda.CUexternalSemaphore))
                if self._extSemArray is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(ccuda.CUexternalSemaphore)))
                self._extSemArray_length = <size_t>len(val)
                self._ptr[0].extSemArray = self._extSemArray
            for idx in range(len(val)):
                self._extSemArray[idx] = (<CUexternalSemaphore>val[idx])._ptr[0]

    @property
    def paramsArray(self):
        arrs = [<void_ptr>self._ptr[0].paramsArray + x*sizeof(ccuda.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS) for x in range(self._paramsArray_length)]
        return [CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS(_ptr=arr) for arr in arrs]
    @paramsArray.setter
    def paramsArray(self, val):
        if len(val) == 0:
            free(self._paramsArray)
            self._paramsArray_length = 0
            self._ptr[0].paramsArray = NULL
        else:
            if self._paramsArray_length != <size_t>len(val):
                free(self._paramsArray)
                self._paramsArray = <ccuda.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS*> calloc(len(val), sizeof(ccuda.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS))
                if self._paramsArray is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(ccuda.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS)))
                self._paramsArray_length = <size_t>len(val)
                self._ptr[0].paramsArray = self._paramsArray
            for idx in range(len(val)):
                string.memcpy(&self._paramsArray[idx], (<CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS>val[idx])._ptr, sizeof(ccuda.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS))

    @property
    def numExtSems(self):
        return self._ptr[0].numExtSems
    @numExtSems.setter
    def numExtSems(self, unsigned int numExtSems):
        self._ptr[0].numExtSems = numExtSems

cdef class CUDA_EXT_SEM_WAIT_NODE_PARAMS_v2_st:
    """
    Semaphore wait node parameters

    Attributes
    ----------
    extSemArray : CUexternalSemaphore
        Array of external semaphore handles.
    paramsArray : CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS
        Array of external semaphore wait parameters.
    numExtSems : unsigned int
        Number of handles and parameters supplied in extSemArray and
        paramsArray.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_EXT_SEM_WAIT_NODE_PARAMS_v2_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        if self._extSemArray is not NULL:
            free(self._extSemArray)
        if self._paramsArray is not NULL:
            free(self._paramsArray)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['extSemArray : ' + str(self.extSemArray)]
            except ValueError:
                str_list += ['extSemArray : <ValueError>']
            try:
                str_list += ['paramsArray : ' + str(self.paramsArray)]
            except ValueError:
                str_list += ['paramsArray : <ValueError>']
            try:
                str_list += ['numExtSems : ' + str(self.numExtSems)]
            except ValueError:
                str_list += ['numExtSems : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def extSemArray(self):
        arrs = [<void_ptr>self._ptr[0].extSemArray + x*sizeof(ccuda.CUexternalSemaphore) for x in range(self._extSemArray_length)]
        return [CUexternalSemaphore(_ptr=arr) for arr in arrs]
    @extSemArray.setter
    def extSemArray(self, val):
        if len(val) == 0:
            free(self._extSemArray)
            self._extSemArray_length = 0
            self._ptr[0].extSemArray = NULL
        else:
            if self._extSemArray_length != <size_t>len(val):
                free(self._extSemArray)
                self._extSemArray = <ccuda.CUexternalSemaphore*> calloc(len(val), sizeof(ccuda.CUexternalSemaphore))
                if self._extSemArray is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(ccuda.CUexternalSemaphore)))
                self._extSemArray_length = <size_t>len(val)
                self._ptr[0].extSemArray = self._extSemArray
            for idx in range(len(val)):
                self._extSemArray[idx] = (<CUexternalSemaphore>val[idx])._ptr[0]

    @property
    def paramsArray(self):
        arrs = [<void_ptr>self._ptr[0].paramsArray + x*sizeof(ccuda.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS) for x in range(self._paramsArray_length)]
        return [CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS(_ptr=arr) for arr in arrs]
    @paramsArray.setter
    def paramsArray(self, val):
        if len(val) == 0:
            free(self._paramsArray)
            self._paramsArray_length = 0
            self._ptr[0].paramsArray = NULL
        else:
            if self._paramsArray_length != <size_t>len(val):
                free(self._paramsArray)
                self._paramsArray = <ccuda.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS*> calloc(len(val), sizeof(ccuda.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS))
                if self._paramsArray is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(ccuda.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS)))
                self._paramsArray_length = <size_t>len(val)
                self._ptr[0].paramsArray = self._paramsArray
            for idx in range(len(val)):
                string.memcpy(&self._paramsArray[idx], (<CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS>val[idx])._ptr, sizeof(ccuda.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS))

    @property
    def numExtSems(self):
        return self._ptr[0].numExtSems
    @numExtSems.setter
    def numExtSems(self, unsigned int numExtSems):
        self._ptr[0].numExtSems = numExtSems

cdef class anon_union9:
    """
    Attributes
    ----------
    mipmap : CUmipmappedArray

    array : CUarray


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUarrayMapInfo_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._mipmap = CUmipmappedArray(_ptr=<void_ptr>&self._ptr[0].resource.mipmap)
        self._array = CUarray(_ptr=<void_ptr>&self._ptr[0].resource.array)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].resource
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['mipmap : ' + str(self.mipmap)]
            except ValueError:
                str_list += ['mipmap : <ValueError>']
            try:
                str_list += ['array : ' + str(self.array)]
            except ValueError:
                str_list += ['array : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def mipmap(self):
        return self._mipmap
    @mipmap.setter
    def mipmap(self, mipmap):
        cdef ccuda.CUmipmappedArray cmipmap
        if mipmap is None:
            cmipmap = <ccuda.CUmipmappedArray><void_ptr>0
        elif isinstance(mipmap, (CUmipmappedArray,)):
            pmipmap = int(mipmap)
            cmipmap = <ccuda.CUmipmappedArray><void_ptr>pmipmap
        else:
            pmipmap = int(CUmipmappedArray(mipmap))
            cmipmap = <ccuda.CUmipmappedArray><void_ptr>pmipmap
        self._mipmap._ptr[0] = cmipmap
    @property
    def array(self):
        return self._array
    @array.setter
    def array(self, array):
        cdef ccuda.CUarray carray
        if array is None:
            carray = <ccuda.CUarray><void_ptr>0
        elif isinstance(array, (CUarray,)):
            parray = int(array)
            carray = <ccuda.CUarray><void_ptr>parray
        else:
            parray = int(CUarray(array))
            carray = <ccuda.CUarray><void_ptr>parray
        self._array._ptr[0] = carray

cdef class anon_struct19:
    """
    Attributes
    ----------
    level : unsigned int

    layer : unsigned int

    offsetX : unsigned int

    offsetY : unsigned int

    offsetZ : unsigned int

    extentWidth : unsigned int

    extentHeight : unsigned int

    extentDepth : unsigned int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUarrayMapInfo_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].subresource.sparseLevel
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['level : ' + str(self.level)]
            except ValueError:
                str_list += ['level : <ValueError>']
            try:
                str_list += ['layer : ' + str(self.layer)]
            except ValueError:
                str_list += ['layer : <ValueError>']
            try:
                str_list += ['offsetX : ' + str(self.offsetX)]
            except ValueError:
                str_list += ['offsetX : <ValueError>']
            try:
                str_list += ['offsetY : ' + str(self.offsetY)]
            except ValueError:
                str_list += ['offsetY : <ValueError>']
            try:
                str_list += ['offsetZ : ' + str(self.offsetZ)]
            except ValueError:
                str_list += ['offsetZ : <ValueError>']
            try:
                str_list += ['extentWidth : ' + str(self.extentWidth)]
            except ValueError:
                str_list += ['extentWidth : <ValueError>']
            try:
                str_list += ['extentHeight : ' + str(self.extentHeight)]
            except ValueError:
                str_list += ['extentHeight : <ValueError>']
            try:
                str_list += ['extentDepth : ' + str(self.extentDepth)]
            except ValueError:
                str_list += ['extentDepth : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def level(self):
        return self._ptr[0].subresource.sparseLevel.level
    @level.setter
    def level(self, unsigned int level):
        self._ptr[0].subresource.sparseLevel.level = level
    @property
    def layer(self):
        return self._ptr[0].subresource.sparseLevel.layer
    @layer.setter
    def layer(self, unsigned int layer):
        self._ptr[0].subresource.sparseLevel.layer = layer
    @property
    def offsetX(self):
        return self._ptr[0].subresource.sparseLevel.offsetX
    @offsetX.setter
    def offsetX(self, unsigned int offsetX):
        self._ptr[0].subresource.sparseLevel.offsetX = offsetX
    @property
    def offsetY(self):
        return self._ptr[0].subresource.sparseLevel.offsetY
    @offsetY.setter
    def offsetY(self, unsigned int offsetY):
        self._ptr[0].subresource.sparseLevel.offsetY = offsetY
    @property
    def offsetZ(self):
        return self._ptr[0].subresource.sparseLevel.offsetZ
    @offsetZ.setter
    def offsetZ(self, unsigned int offsetZ):
        self._ptr[0].subresource.sparseLevel.offsetZ = offsetZ
    @property
    def extentWidth(self):
        return self._ptr[0].subresource.sparseLevel.extentWidth
    @extentWidth.setter
    def extentWidth(self, unsigned int extentWidth):
        self._ptr[0].subresource.sparseLevel.extentWidth = extentWidth
    @property
    def extentHeight(self):
        return self._ptr[0].subresource.sparseLevel.extentHeight
    @extentHeight.setter
    def extentHeight(self, unsigned int extentHeight):
        self._ptr[0].subresource.sparseLevel.extentHeight = extentHeight
    @property
    def extentDepth(self):
        return self._ptr[0].subresource.sparseLevel.extentDepth
    @extentDepth.setter
    def extentDepth(self, unsigned int extentDepth):
        self._ptr[0].subresource.sparseLevel.extentDepth = extentDepth

cdef class anon_struct20:
    """
    Attributes
    ----------
    layer : unsigned int

    offset : unsigned long long

    size : unsigned long long


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUarrayMapInfo_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].subresource.miptail
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['layer : ' + str(self.layer)]
            except ValueError:
                str_list += ['layer : <ValueError>']
            try:
                str_list += ['offset : ' + str(self.offset)]
            except ValueError:
                str_list += ['offset : <ValueError>']
            try:
                str_list += ['size : ' + str(self.size)]
            except ValueError:
                str_list += ['size : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def layer(self):
        return self._ptr[0].subresource.miptail.layer
    @layer.setter
    def layer(self, unsigned int layer):
        self._ptr[0].subresource.miptail.layer = layer
    @property
    def offset(self):
        return self._ptr[0].subresource.miptail.offset
    @offset.setter
    def offset(self, unsigned long long offset):
        self._ptr[0].subresource.miptail.offset = offset
    @property
    def size(self):
        return self._ptr[0].subresource.miptail.size
    @size.setter
    def size(self, unsigned long long size):
        self._ptr[0].subresource.miptail.size = size

cdef class anon_union10:
    """
    Attributes
    ----------
    sparseLevel : anon_struct19

    miptail : anon_struct20


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUarrayMapInfo_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._sparseLevel = anon_struct19(_ptr=<void_ptr>self._ptr)
        self._miptail = anon_struct20(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].subresource
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['sparseLevel :\n' + '\n'.join(['    ' + line for line in str(self.sparseLevel).splitlines()])]
            except ValueError:
                str_list += ['sparseLevel : <ValueError>']
            try:
                str_list += ['miptail :\n' + '\n'.join(['    ' + line for line in str(self.miptail).splitlines()])]
            except ValueError:
                str_list += ['miptail : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def sparseLevel(self):
        return self._sparseLevel
    @sparseLevel.setter
    def sparseLevel(self, sparseLevel not None : anon_struct19):
        string.memcpy(&self._ptr[0].subresource.sparseLevel, <ccuda.anon_struct19*><void_ptr>sparseLevel.getPtr(), sizeof(self._ptr[0].subresource.sparseLevel))
    @property
    def miptail(self):
        return self._miptail
    @miptail.setter
    def miptail(self, miptail not None : anon_struct20):
        string.memcpy(&self._ptr[0].subresource.miptail, <ccuda.anon_struct20*><void_ptr>miptail.getPtr(), sizeof(self._ptr[0].subresource.miptail))

cdef class anon_union11:
    """
    Attributes
    ----------
    memHandle : CUmemGenericAllocationHandle


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUarrayMapInfo_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._memHandle = CUmemGenericAllocationHandle(_ptr=<void_ptr>&self._ptr[0].memHandle.memHandle)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].memHandle
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['memHandle : ' + str(self.memHandle)]
            except ValueError:
                str_list += ['memHandle : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def memHandle(self):
        return self._memHandle
    @memHandle.setter
    def memHandle(self, memHandle):
        cdef ccuda.CUmemGenericAllocationHandle cmemHandle
        if memHandle is None:
            cmemHandle = <ccuda.CUmemGenericAllocationHandle><void_ptr>0
        elif isinstance(memHandle, (CUmemGenericAllocationHandle)):
            pmemHandle = int(memHandle)
            cmemHandle = <ccuda.CUmemGenericAllocationHandle><void_ptr>pmemHandle
        else:
            pmemHandle = int(CUmemGenericAllocationHandle(memHandle))
            cmemHandle = <ccuda.CUmemGenericAllocationHandle><void_ptr>pmemHandle
        self._memHandle._ptr[0] = cmemHandle

cdef class CUarrayMapInfo_st:
    """
    Specifies the CUDA array or CUDA mipmapped array memory mapping
    information

    Attributes
    ----------
    resourceType : CUresourcetype
        Resource type
    resource : anon_union9

    subresourceType : CUarraySparseSubresourceType
        Sparse subresource type
    subresource : anon_union10

    memOperationType : CUmemOperationType
        Memory operation type
    memHandleType : CUmemHandleType
        Memory handle type
    memHandle : anon_union11

    offset : unsigned long long
        Offset within mip tail  Offset within the memory
    deviceBitMask : unsigned int
        Device ordinal bit mask
    flags : unsigned int
        flags for future use, must be zero now.
    reserved : List[unsigned int]
        Reserved for future use, must be zero now.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._val_ptr = <ccuda.CUarrayMapInfo_st *>calloc(1, sizeof(ccuda.CUarrayMapInfo_st))
            self._ptr = self._val_ptr
        else:
            self._ptr = <ccuda.CUarrayMapInfo_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._resource = anon_union9(_ptr=<void_ptr>self._ptr)
        self._subresource = anon_union10(_ptr=<void_ptr>self._ptr)
        self._memHandle = anon_union11(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        if self._val_ptr is not NULL:
            free(self._val_ptr)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['resourceType : ' + str(self.resourceType)]
            except ValueError:
                str_list += ['resourceType : <ValueError>']
            try:
                str_list += ['resource :\n' + '\n'.join(['    ' + line for line in str(self.resource).splitlines()])]
            except ValueError:
                str_list += ['resource : <ValueError>']
            try:
                str_list += ['subresourceType : ' + str(self.subresourceType)]
            except ValueError:
                str_list += ['subresourceType : <ValueError>']
            try:
                str_list += ['subresource :\n' + '\n'.join(['    ' + line for line in str(self.subresource).splitlines()])]
            except ValueError:
                str_list += ['subresource : <ValueError>']
            try:
                str_list += ['memOperationType : ' + str(self.memOperationType)]
            except ValueError:
                str_list += ['memOperationType : <ValueError>']
            try:
                str_list += ['memHandleType : ' + str(self.memHandleType)]
            except ValueError:
                str_list += ['memHandleType : <ValueError>']
            try:
                str_list += ['memHandle :\n' + '\n'.join(['    ' + line for line in str(self.memHandle).splitlines()])]
            except ValueError:
                str_list += ['memHandle : <ValueError>']
            try:
                str_list += ['offset : ' + str(self.offset)]
            except ValueError:
                str_list += ['offset : <ValueError>']
            try:
                str_list += ['deviceBitMask : ' + str(self.deviceBitMask)]
            except ValueError:
                str_list += ['deviceBitMask : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def resourceType(self):
        return CUresourcetype(self._ptr[0].resourceType)
    @resourceType.setter
    def resourceType(self, resourceType not None : CUresourcetype):
        self._ptr[0].resourceType = resourceType.value
    @property
    def resource(self):
        return self._resource
    @resource.setter
    def resource(self, resource not None : anon_union9):
        string.memcpy(&self._ptr[0].resource, <ccuda.anon_union9*><void_ptr>resource.getPtr(), sizeof(self._ptr[0].resource))
    @property
    def subresourceType(self):
        return CUarraySparseSubresourceType(self._ptr[0].subresourceType)
    @subresourceType.setter
    def subresourceType(self, subresourceType not None : CUarraySparseSubresourceType):
        self._ptr[0].subresourceType = subresourceType.value
    @property
    def subresource(self):
        return self._subresource
    @subresource.setter
    def subresource(self, subresource not None : anon_union10):
        string.memcpy(&self._ptr[0].subresource, <ccuda.anon_union10*><void_ptr>subresource.getPtr(), sizeof(self._ptr[0].subresource))
    @property
    def memOperationType(self):
        return CUmemOperationType(self._ptr[0].memOperationType)
    @memOperationType.setter
    def memOperationType(self, memOperationType not None : CUmemOperationType):
        self._ptr[0].memOperationType = memOperationType.value
    @property
    def memHandleType(self):
        return CUmemHandleType(self._ptr[0].memHandleType)
    @memHandleType.setter
    def memHandleType(self, memHandleType not None : CUmemHandleType):
        self._ptr[0].memHandleType = memHandleType.value
    @property
    def memHandle(self):
        return self._memHandle
    @memHandle.setter
    def memHandle(self, memHandle not None : anon_union11):
        string.memcpy(&self._ptr[0].memHandle, <ccuda.anon_union11*><void_ptr>memHandle.getPtr(), sizeof(self._ptr[0].memHandle))
    @property
    def offset(self):
        return self._ptr[0].offset
    @offset.setter
    def offset(self, unsigned long long offset):
        self._ptr[0].offset = offset
    @property
    def deviceBitMask(self):
        return self._ptr[0].deviceBitMask
    @deviceBitMask.setter
    def deviceBitMask(self, unsigned int deviceBitMask):
        self._ptr[0].deviceBitMask = deviceBitMask
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flags = flags
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class CUmemLocation_st:
    """
    Specifies a memory location.

    Attributes
    ----------
    type : CUmemLocationType
        Specifies the location type, which modifies the meaning of id.
    id : int
        identifier for a given this location's CUmemLocationType.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUmemLocation_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['type : ' + str(self.type)]
            except ValueError:
                str_list += ['type : <ValueError>']
            try:
                str_list += ['id : ' + str(self.id)]
            except ValueError:
                str_list += ['id : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def type(self):
        return CUmemLocationType(self._ptr[0].type)
    @type.setter
    def type(self, type not None : CUmemLocationType):
        self._ptr[0].type = type.value
    @property
    def id(self):
        return self._ptr[0].id
    @id.setter
    def id(self, int id):
        self._ptr[0].id = id

cdef class anon_struct21:
    """
    Attributes
    ----------
    compressionType : bytes

    gpuDirectRDMACapable : bytes

    usage : unsigned short

    reserved : bytes


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUmemAllocationProp_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].allocFlags
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['compressionType : ' + str(self.compressionType)]
            except ValueError:
                str_list += ['compressionType : <ValueError>']
            try:
                str_list += ['gpuDirectRDMACapable : ' + str(self.gpuDirectRDMACapable)]
            except ValueError:
                str_list += ['gpuDirectRDMACapable : <ValueError>']
            try:
                str_list += ['usage : ' + str(self.usage)]
            except ValueError:
                str_list += ['usage : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def compressionType(self):
        return self._ptr[0].allocFlags.compressionType
    @compressionType.setter
    def compressionType(self, unsigned char compressionType):
        self._ptr[0].allocFlags.compressionType = compressionType
    @property
    def gpuDirectRDMACapable(self):
        return self._ptr[0].allocFlags.gpuDirectRDMACapable
    @gpuDirectRDMACapable.setter
    def gpuDirectRDMACapable(self, unsigned char gpuDirectRDMACapable):
        self._ptr[0].allocFlags.gpuDirectRDMACapable = gpuDirectRDMACapable
    @property
    def usage(self):
        return self._ptr[0].allocFlags.usage
    @usage.setter
    def usage(self, unsigned short usage):
        self._ptr[0].allocFlags.usage = usage
    @property
    def reserved(self):
        return PyBytes_FromStringAndSize(<char*>self._ptr[0].allocFlags.reserved, 4)
    @reserved.setter
    def reserved(self, reserved):
        if len(reserved) != 4:
            raise ValueError("reserved length must be 4, is " + str(len(reserved)))
        for i, b in enumerate(reserved):
            self._ptr[0].allocFlags.reserved[i] = b

cdef class CUmemAllocationProp_st:
    """
    Specifies the allocation properties for a allocation.

    Attributes
    ----------
    type : CUmemAllocationType
        Allocation type
    requestedHandleTypes : CUmemAllocationHandleType
        requested CUmemAllocationHandleType
    location : CUmemLocation
        Location of allocation
    win32HandleMetaData : Any
        Windows-specific POBJECT_ATTRIBUTES required when
        CU_MEM_HANDLE_TYPE_WIN32 is specified. This object attributes
        structure includes security attributes that define the scope of
        which exported allocations may be transferred to other processes.
        In all other cases, this field is required to be zero.
    allocFlags : anon_struct21


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUmemAllocationProp_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._location = CUmemLocation(_ptr=<void_ptr>&self._ptr[0].location)
        self._allocFlags = anon_struct21(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['type : ' + str(self.type)]
            except ValueError:
                str_list += ['type : <ValueError>']
            try:
                str_list += ['requestedHandleTypes : ' + str(self.requestedHandleTypes)]
            except ValueError:
                str_list += ['requestedHandleTypes : <ValueError>']
            try:
                str_list += ['location :\n' + '\n'.join(['    ' + line for line in str(self.location).splitlines()])]
            except ValueError:
                str_list += ['location : <ValueError>']
            try:
                str_list += ['win32HandleMetaData : ' + hex(self.win32HandleMetaData)]
            except ValueError:
                str_list += ['win32HandleMetaData : <ValueError>']
            try:
                str_list += ['allocFlags :\n' + '\n'.join(['    ' + line for line in str(self.allocFlags).splitlines()])]
            except ValueError:
                str_list += ['allocFlags : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def type(self):
        return CUmemAllocationType(self._ptr[0].type)
    @type.setter
    def type(self, type not None : CUmemAllocationType):
        self._ptr[0].type = type.value
    @property
    def requestedHandleTypes(self):
        return CUmemAllocationHandleType(self._ptr[0].requestedHandleTypes)
    @requestedHandleTypes.setter
    def requestedHandleTypes(self, requestedHandleTypes not None : CUmemAllocationHandleType):
        self._ptr[0].requestedHandleTypes = requestedHandleTypes.value
    @property
    def location(self):
        return self._location
    @location.setter
    def location(self, location not None : CUmemLocation):
        string.memcpy(&self._ptr[0].location, <ccuda.CUmemLocation*><void_ptr>location.getPtr(), sizeof(self._ptr[0].location))
    @property
    def win32HandleMetaData(self):
        return <void_ptr>self._ptr[0].win32HandleMetaData
    @win32HandleMetaData.setter
    def win32HandleMetaData(self, win32HandleMetaData):
        _cwin32HandleMetaData = utils.HelperInputVoidPtr(win32HandleMetaData)
        self._ptr[0].win32HandleMetaData = <void*><void_ptr>_cwin32HandleMetaData.cptr
    @property
    def allocFlags(self):
        return self._allocFlags
    @allocFlags.setter
    def allocFlags(self, allocFlags not None : anon_struct21):
        string.memcpy(&self._ptr[0].allocFlags, <ccuda.anon_struct21*><void_ptr>allocFlags.getPtr(), sizeof(self._ptr[0].allocFlags))

cdef class CUmulticastObjectProp_st:
    """
    Specifies the properties for a multicast object.

    Attributes
    ----------
    numDevices : unsigned int
        The number of devices in the multicast team that will bind memory
        to this object
    size : size_t
        The maximum amount of memory that can be bound to this multicast
        object per device
    handleTypes : unsigned long long
        Bitmask of exportable handle types (see CUmemAllocationHandleType)
        for this object
    flags : unsigned long long
        Flags for future use, must be zero now

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUmulticastObjectProp_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['numDevices : ' + str(self.numDevices)]
            except ValueError:
                str_list += ['numDevices : <ValueError>']
            try:
                str_list += ['size : ' + str(self.size)]
            except ValueError:
                str_list += ['size : <ValueError>']
            try:
                str_list += ['handleTypes : ' + str(self.handleTypes)]
            except ValueError:
                str_list += ['handleTypes : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def numDevices(self):
        return self._ptr[0].numDevices
    @numDevices.setter
    def numDevices(self, unsigned int numDevices):
        self._ptr[0].numDevices = numDevices
    @property
    def size(self):
        return self._ptr[0].size
    @size.setter
    def size(self, size_t size):
        self._ptr[0].size = size
    @property
    def handleTypes(self):
        return self._ptr[0].handleTypes
    @handleTypes.setter
    def handleTypes(self, unsigned long long handleTypes):
        self._ptr[0].handleTypes = handleTypes
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned long long flags):
        self._ptr[0].flags = flags

cdef class CUmemAccessDesc_st:
    """
    Memory access descriptor

    Attributes
    ----------
    location : CUmemLocation
        Location on which the request is to change it's accessibility
    flags : CUmemAccess_flags
        ::CUmemProt accessibility flags to set on the request

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUmemAccessDesc_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._location = CUmemLocation(_ptr=<void_ptr>&self._ptr[0].location)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['location :\n' + '\n'.join(['    ' + line for line in str(self.location).splitlines()])]
            except ValueError:
                str_list += ['location : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def location(self):
        return self._location
    @location.setter
    def location(self, location not None : CUmemLocation):
        string.memcpy(&self._ptr[0].location, <ccuda.CUmemLocation*><void_ptr>location.getPtr(), sizeof(self._ptr[0].location))
    @property
    def flags(self):
        return CUmemAccess_flags(self._ptr[0].flags)
    @flags.setter
    def flags(self, flags not None : CUmemAccess_flags):
        self._ptr[0].flags = flags.value

cdef class CUgraphExecUpdateResultInfo_st:
    """
    Result information returned by cuGraphExecUpdate

    Attributes
    ----------
    result : CUgraphExecUpdateResult
        Gives more specific detail when a cuda graph update fails.
    errorNode : CUgraphNode
        The "to node" of the error edge when the topologies do not match.
        The error node when the error is associated with a specific node.
        NULL when the error is generic.
    errorFromNode : CUgraphNode
        The from node of error edge when the topologies do not match.
        Otherwise NULL.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUgraphExecUpdateResultInfo_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._errorNode = CUgraphNode(_ptr=<void_ptr>&self._ptr[0].errorNode)
        self._errorFromNode = CUgraphNode(_ptr=<void_ptr>&self._ptr[0].errorFromNode)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['result : ' + str(self.result)]
            except ValueError:
                str_list += ['result : <ValueError>']
            try:
                str_list += ['errorNode : ' + str(self.errorNode)]
            except ValueError:
                str_list += ['errorNode : <ValueError>']
            try:
                str_list += ['errorFromNode : ' + str(self.errorFromNode)]
            except ValueError:
                str_list += ['errorFromNode : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def result(self):
        return CUgraphExecUpdateResult(self._ptr[0].result)
    @result.setter
    def result(self, result not None : CUgraphExecUpdateResult):
        self._ptr[0].result = result.value
    @property
    def errorNode(self):
        return self._errorNode
    @errorNode.setter
    def errorNode(self, errorNode):
        cdef ccuda.CUgraphNode cerrorNode
        if errorNode is None:
            cerrorNode = <ccuda.CUgraphNode><void_ptr>0
        elif isinstance(errorNode, (CUgraphNode,)):
            perrorNode = int(errorNode)
            cerrorNode = <ccuda.CUgraphNode><void_ptr>perrorNode
        else:
            perrorNode = int(CUgraphNode(errorNode))
            cerrorNode = <ccuda.CUgraphNode><void_ptr>perrorNode
        self._errorNode._ptr[0] = cerrorNode
    @property
    def errorFromNode(self):
        return self._errorFromNode
    @errorFromNode.setter
    def errorFromNode(self, errorFromNode):
        cdef ccuda.CUgraphNode cerrorFromNode
        if errorFromNode is None:
            cerrorFromNode = <ccuda.CUgraphNode><void_ptr>0
        elif isinstance(errorFromNode, (CUgraphNode,)):
            perrorFromNode = int(errorFromNode)
            cerrorFromNode = <ccuda.CUgraphNode><void_ptr>perrorFromNode
        else:
            perrorFromNode = int(CUgraphNode(errorFromNode))
            cerrorFromNode = <ccuda.CUgraphNode><void_ptr>perrorFromNode
        self._errorFromNode._ptr[0] = cerrorFromNode

cdef class CUmemPoolProps_st:
    """
    Specifies the properties of allocations made from the pool.

    Attributes
    ----------
    allocType : CUmemAllocationType
        Allocation type. Currently must be specified as
        CU_MEM_ALLOCATION_TYPE_PINNED
    handleTypes : CUmemAllocationHandleType
        Handle types that will be supported by allocations from the pool.
    location : CUmemLocation
        Location where allocations should reside.
    win32SecurityAttributes : Any
        Windows-specific LPSECURITYATTRIBUTES required when
        CU_MEM_HANDLE_TYPE_WIN32 is specified. This security attribute
        defines the scope of which exported allocations may be transferred
        to other processes. In all other cases, this field is required to
        be zero.
    maxSize : size_t
        Maximum pool size. When set to 0, defaults to a system dependent
        value.
    reserved : bytes
        reserved for future use, must be 0

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUmemPoolProps_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._location = CUmemLocation(_ptr=<void_ptr>&self._ptr[0].location)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['allocType : ' + str(self.allocType)]
            except ValueError:
                str_list += ['allocType : <ValueError>']
            try:
                str_list += ['handleTypes : ' + str(self.handleTypes)]
            except ValueError:
                str_list += ['handleTypes : <ValueError>']
            try:
                str_list += ['location :\n' + '\n'.join(['    ' + line for line in str(self.location).splitlines()])]
            except ValueError:
                str_list += ['location : <ValueError>']
            try:
                str_list += ['win32SecurityAttributes : ' + hex(self.win32SecurityAttributes)]
            except ValueError:
                str_list += ['win32SecurityAttributes : <ValueError>']
            try:
                str_list += ['maxSize : ' + str(self.maxSize)]
            except ValueError:
                str_list += ['maxSize : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def allocType(self):
        return CUmemAllocationType(self._ptr[0].allocType)
    @allocType.setter
    def allocType(self, allocType not None : CUmemAllocationType):
        self._ptr[0].allocType = allocType.value
    @property
    def handleTypes(self):
        return CUmemAllocationHandleType(self._ptr[0].handleTypes)
    @handleTypes.setter
    def handleTypes(self, handleTypes not None : CUmemAllocationHandleType):
        self._ptr[0].handleTypes = handleTypes.value
    @property
    def location(self):
        return self._location
    @location.setter
    def location(self, location not None : CUmemLocation):
        string.memcpy(&self._ptr[0].location, <ccuda.CUmemLocation*><void_ptr>location.getPtr(), sizeof(self._ptr[0].location))
    @property
    def win32SecurityAttributes(self):
        return <void_ptr>self._ptr[0].win32SecurityAttributes
    @win32SecurityAttributes.setter
    def win32SecurityAttributes(self, win32SecurityAttributes):
        _cwin32SecurityAttributes = utils.HelperInputVoidPtr(win32SecurityAttributes)
        self._ptr[0].win32SecurityAttributes = <void*><void_ptr>_cwin32SecurityAttributes.cptr
    @property
    def maxSize(self):
        return self._ptr[0].maxSize
    @maxSize.setter
    def maxSize(self, size_t maxSize):
        self._ptr[0].maxSize = maxSize
    @property
    def reserved(self):
        return PyBytes_FromStringAndSize(<char*>self._ptr[0].reserved, 56)
    @reserved.setter
    def reserved(self, reserved):
        if len(reserved) != 56:
            raise ValueError("reserved length must be 56, is " + str(len(reserved)))
        for i, b in enumerate(reserved):
            self._ptr[0].reserved[i] = b

cdef class CUmemPoolPtrExportData_st:
    """
    Opaque data for exporting a pool allocation

    Attributes
    ----------
    reserved : bytes


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUmemPoolPtrExportData_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def reserved(self):
        return PyBytes_FromStringAndSize(<char*>self._ptr[0].reserved, 64)
    @reserved.setter
    def reserved(self, reserved):
        if len(reserved) != 64:
            raise ValueError("reserved length must be 64, is " + str(len(reserved)))
        for i, b in enumerate(reserved):
            self._ptr[0].reserved[i] = b

cdef class CUDA_MEM_ALLOC_NODE_PARAMS_v1_st:
    """
    Memory allocation node parameters

    Attributes
    ----------
    poolProps : CUmemPoolProps
        in: location where the allocation should reside (specified in
        ::location). ::handleTypes must be CU_MEM_HANDLE_TYPE_NONE. IPC is
        not supported.
    accessDescs : CUmemAccessDesc
        in: array of memory access descriptors. Used to describe peer GPU
        access
    accessDescCount : size_t
        in: number of memory access descriptors. Must not exceed the number
        of GPUs.
    bytesize : size_t
        in: size in bytes of the requested allocation
    dptr : CUdeviceptr
        out: address of the allocation returned by CUDA

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_MEM_ALLOC_NODE_PARAMS_v1_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._poolProps = CUmemPoolProps(_ptr=<void_ptr>&self._ptr[0].poolProps)
        self._dptr = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].dptr)
    def __dealloc__(self):
        if self._accessDescs is not NULL:
            free(self._accessDescs)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['poolProps :\n' + '\n'.join(['    ' + line for line in str(self.poolProps).splitlines()])]
            except ValueError:
                str_list += ['poolProps : <ValueError>']
            try:
                str_list += ['accessDescs : ' + str(self.accessDescs)]
            except ValueError:
                str_list += ['accessDescs : <ValueError>']
            try:
                str_list += ['accessDescCount : ' + str(self.accessDescCount)]
            except ValueError:
                str_list += ['accessDescCount : <ValueError>']
            try:
                str_list += ['bytesize : ' + str(self.bytesize)]
            except ValueError:
                str_list += ['bytesize : <ValueError>']
            try:
                str_list += ['dptr : ' + str(self.dptr)]
            except ValueError:
                str_list += ['dptr : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def poolProps(self):
        return self._poolProps
    @poolProps.setter
    def poolProps(self, poolProps not None : CUmemPoolProps):
        string.memcpy(&self._ptr[0].poolProps, <ccuda.CUmemPoolProps*><void_ptr>poolProps.getPtr(), sizeof(self._ptr[0].poolProps))
    @property
    def accessDescs(self):
        arrs = [<void_ptr>self._ptr[0].accessDescs + x*sizeof(ccuda.CUmemAccessDesc) for x in range(self._accessDescs_length)]
        return [CUmemAccessDesc(_ptr=arr) for arr in arrs]
    @accessDescs.setter
    def accessDescs(self, val):
        if len(val) == 0:
            free(self._accessDescs)
            self._accessDescs_length = 0
            self._ptr[0].accessDescs = NULL
        else:
            if self._accessDescs_length != <size_t>len(val):
                free(self._accessDescs)
                self._accessDescs = <ccuda.CUmemAccessDesc*> calloc(len(val), sizeof(ccuda.CUmemAccessDesc))
                if self._accessDescs is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(ccuda.CUmemAccessDesc)))
                self._accessDescs_length = <size_t>len(val)
                self._ptr[0].accessDescs = self._accessDescs
            for idx in range(len(val)):
                string.memcpy(&self._accessDescs[idx], (<CUmemAccessDesc>val[idx])._ptr, sizeof(ccuda.CUmemAccessDesc))

    @property
    def accessDescCount(self):
        return self._ptr[0].accessDescCount
    @accessDescCount.setter
    def accessDescCount(self, size_t accessDescCount):
        self._ptr[0].accessDescCount = accessDescCount
    @property
    def bytesize(self):
        return self._ptr[0].bytesize
    @bytesize.setter
    def bytesize(self, size_t bytesize):
        self._ptr[0].bytesize = bytesize
    @property
    def dptr(self):
        return self._dptr
    @dptr.setter
    def dptr(self, dptr):
        cdef ccuda.CUdeviceptr cdptr
        if dptr is None:
            cdptr = <ccuda.CUdeviceptr><void_ptr>0
        elif isinstance(dptr, (CUdeviceptr)):
            pdptr = int(dptr)
            cdptr = <ccuda.CUdeviceptr><void_ptr>pdptr
        else:
            pdptr = int(CUdeviceptr(dptr))
            cdptr = <ccuda.CUdeviceptr><void_ptr>pdptr
        self._dptr._ptr[0] = cdptr

cdef class CUDA_MEM_ALLOC_NODE_PARAMS_v2_st:
    """
    Memory allocation node parameters

    Attributes
    ----------
    poolProps : CUmemPoolProps
        in: location where the allocation should reside (specified in
        ::location). ::handleTypes must be CU_MEM_HANDLE_TYPE_NONE. IPC is
        not supported.
    accessDescs : CUmemAccessDesc
        in: array of memory access descriptors. Used to describe peer GPU
        access
    accessDescCount : size_t
        in: number of memory access descriptors. Must not exceed the number
        of GPUs.
    bytesize : size_t
        in: size in bytes of the requested allocation
    dptr : CUdeviceptr
        out: address of the allocation returned by CUDA

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_MEM_ALLOC_NODE_PARAMS_v2_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._poolProps = CUmemPoolProps(_ptr=<void_ptr>&self._ptr[0].poolProps)
        self._dptr = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].dptr)
    def __dealloc__(self):
        if self._accessDescs is not NULL:
            free(self._accessDescs)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['poolProps :\n' + '\n'.join(['    ' + line for line in str(self.poolProps).splitlines()])]
            except ValueError:
                str_list += ['poolProps : <ValueError>']
            try:
                str_list += ['accessDescs : ' + str(self.accessDescs)]
            except ValueError:
                str_list += ['accessDescs : <ValueError>']
            try:
                str_list += ['accessDescCount : ' + str(self.accessDescCount)]
            except ValueError:
                str_list += ['accessDescCount : <ValueError>']
            try:
                str_list += ['bytesize : ' + str(self.bytesize)]
            except ValueError:
                str_list += ['bytesize : <ValueError>']
            try:
                str_list += ['dptr : ' + str(self.dptr)]
            except ValueError:
                str_list += ['dptr : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def poolProps(self):
        return self._poolProps
    @poolProps.setter
    def poolProps(self, poolProps not None : CUmemPoolProps):
        string.memcpy(&self._ptr[0].poolProps, <ccuda.CUmemPoolProps*><void_ptr>poolProps.getPtr(), sizeof(self._ptr[0].poolProps))
    @property
    def accessDescs(self):
        arrs = [<void_ptr>self._ptr[0].accessDescs + x*sizeof(ccuda.CUmemAccessDesc) for x in range(self._accessDescs_length)]
        return [CUmemAccessDesc(_ptr=arr) for arr in arrs]
    @accessDescs.setter
    def accessDescs(self, val):
        if len(val) == 0:
            free(self._accessDescs)
            self._accessDescs_length = 0
            self._ptr[0].accessDescs = NULL
        else:
            if self._accessDescs_length != <size_t>len(val):
                free(self._accessDescs)
                self._accessDescs = <ccuda.CUmemAccessDesc*> calloc(len(val), sizeof(ccuda.CUmemAccessDesc))
                if self._accessDescs is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(ccuda.CUmemAccessDesc)))
                self._accessDescs_length = <size_t>len(val)
                self._ptr[0].accessDescs = self._accessDescs
            for idx in range(len(val)):
                string.memcpy(&self._accessDescs[idx], (<CUmemAccessDesc>val[idx])._ptr, sizeof(ccuda.CUmemAccessDesc))

    @property
    def accessDescCount(self):
        return self._ptr[0].accessDescCount
    @accessDescCount.setter
    def accessDescCount(self, size_t accessDescCount):
        self._ptr[0].accessDescCount = accessDescCount
    @property
    def bytesize(self):
        return self._ptr[0].bytesize
    @bytesize.setter
    def bytesize(self, size_t bytesize):
        self._ptr[0].bytesize = bytesize
    @property
    def dptr(self):
        return self._dptr
    @dptr.setter
    def dptr(self, dptr):
        cdef ccuda.CUdeviceptr cdptr
        if dptr is None:
            cdptr = <ccuda.CUdeviceptr><void_ptr>0
        elif isinstance(dptr, (CUdeviceptr)):
            pdptr = int(dptr)
            cdptr = <ccuda.CUdeviceptr><void_ptr>pdptr
        else:
            pdptr = int(CUdeviceptr(dptr))
            cdptr = <ccuda.CUdeviceptr><void_ptr>pdptr
        self._dptr._ptr[0] = cdptr

cdef class CUDA_MEM_FREE_NODE_PARAMS_st:
    """
    Memory free node parameters

    Attributes
    ----------
    dptr : CUdeviceptr
        in: the pointer to free

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_MEM_FREE_NODE_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._dptr = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].dptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['dptr : ' + str(self.dptr)]
            except ValueError:
                str_list += ['dptr : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def dptr(self):
        return self._dptr
    @dptr.setter
    def dptr(self, dptr):
        cdef ccuda.CUdeviceptr cdptr
        if dptr is None:
            cdptr = <ccuda.CUdeviceptr><void_ptr>0
        elif isinstance(dptr, (CUdeviceptr)):
            pdptr = int(dptr)
            cdptr = <ccuda.CUdeviceptr><void_ptr>pdptr
        else:
            pdptr = int(CUdeviceptr(dptr))
            cdptr = <ccuda.CUdeviceptr><void_ptr>pdptr
        self._dptr._ptr[0] = cdptr

cdef class CUDA_CHILD_GRAPH_NODE_PARAMS_st:
    """
    Child graph node parameters

    Attributes
    ----------
    graph : CUgraph
        The child graph to clone into the node for node creation, or a
        handle to the graph owned by the node for node query

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_CHILD_GRAPH_NODE_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._graph = CUgraph(_ptr=<void_ptr>&self._ptr[0].graph)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['graph : ' + str(self.graph)]
            except ValueError:
                str_list += ['graph : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def graph(self):
        return self._graph
    @graph.setter
    def graph(self, graph):
        cdef ccuda.CUgraph cgraph
        if graph is None:
            cgraph = <ccuda.CUgraph><void_ptr>0
        elif isinstance(graph, (CUgraph,)):
            pgraph = int(graph)
            cgraph = <ccuda.CUgraph><void_ptr>pgraph
        else:
            pgraph = int(CUgraph(graph))
            cgraph = <ccuda.CUgraph><void_ptr>pgraph
        self._graph._ptr[0] = cgraph

cdef class CUDA_EVENT_RECORD_NODE_PARAMS_st:
    """
    Event record node parameters

    Attributes
    ----------
    event : CUevent
        The event to record when the node executes

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_EVENT_RECORD_NODE_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._event = CUevent(_ptr=<void_ptr>&self._ptr[0].event)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['event : ' + str(self.event)]
            except ValueError:
                str_list += ['event : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def event(self):
        return self._event
    @event.setter
    def event(self, event):
        cdef ccuda.CUevent cevent
        if event is None:
            cevent = <ccuda.CUevent><void_ptr>0
        elif isinstance(event, (CUevent,)):
            pevent = int(event)
            cevent = <ccuda.CUevent><void_ptr>pevent
        else:
            pevent = int(CUevent(event))
            cevent = <ccuda.CUevent><void_ptr>pevent
        self._event._ptr[0] = cevent

cdef class CUDA_EVENT_WAIT_NODE_PARAMS_st:
    """
    Event wait node parameters

    Attributes
    ----------
    event : CUevent
        The event to wait on from the node

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUDA_EVENT_WAIT_NODE_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._event = CUevent(_ptr=<void_ptr>&self._ptr[0].event)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['event : ' + str(self.event)]
            except ValueError:
                str_list += ['event : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def event(self):
        return self._event
    @event.setter
    def event(self, event):
        cdef ccuda.CUevent cevent
        if event is None:
            cevent = <ccuda.CUevent><void_ptr>0
        elif isinstance(event, (CUevent,)):
            pevent = int(event)
            cevent = <ccuda.CUevent><void_ptr>pevent
        else:
            pevent = int(CUevent(event))
            cevent = <ccuda.CUevent><void_ptr>pevent
        self._event._ptr[0] = cevent

cdef class CUgraphNodeParams_st:
    """
    Graph node parameters. See cuGraphAddNode.

    Attributes
    ----------
    type : CUgraphNodeType
        Type of the node
    reserved0 : List[int]
        Reserved. Must be zero.
    reserved1 : List[long long]
        Padding. Unused bytes must be zero.
    kernel : CUDA_KERNEL_NODE_PARAMS_v3
        Kernel node parameters.
    memcpy : CUDA_MEMCPY_NODE_PARAMS
        Memcpy node parameters.
    memset : CUDA_MEMSET_NODE_PARAMS_v2
        Memset node parameters.
    host : CUDA_HOST_NODE_PARAMS_v2
        Host node parameters.
    graph : CUDA_CHILD_GRAPH_NODE_PARAMS
        Child graph node parameters.
    eventWait : CUDA_EVENT_WAIT_NODE_PARAMS
        Event wait node parameters.
    eventRecord : CUDA_EVENT_RECORD_NODE_PARAMS
        Event record node parameters.
    extSemSignal : CUDA_EXT_SEM_SIGNAL_NODE_PARAMS_v2
        External semaphore signal node parameters.
    extSemWait : CUDA_EXT_SEM_WAIT_NODE_PARAMS_v2
        External semaphore wait node parameters.
    alloc : CUDA_MEM_ALLOC_NODE_PARAMS_v2
        Memory allocation node parameters.
    free : CUDA_MEM_FREE_NODE_PARAMS
        Memory free node parameters.
    memOp : CUDA_BATCH_MEM_OP_NODE_PARAMS_v2
        MemOp node parameters.
    conditional : CUDA_CONDITIONAL_NODE_PARAMS
        Conditional node parameters.
    reserved2 : long long
        Reserved bytes. Must be zero.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._val_ptr = <ccuda.CUgraphNodeParams_st *>calloc(1, sizeof(ccuda.CUgraphNodeParams_st))
            self._ptr = self._val_ptr
        else:
            self._ptr = <ccuda.CUgraphNodeParams_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._kernel = CUDA_KERNEL_NODE_PARAMS_v3(_ptr=<void_ptr>&self._ptr[0].kernel)
        self._memcpy = CUDA_MEMCPY_NODE_PARAMS(_ptr=<void_ptr>&self._ptr[0].memcpy)
        self._memset = CUDA_MEMSET_NODE_PARAMS_v2(_ptr=<void_ptr>&self._ptr[0].memset)
        self._host = CUDA_HOST_NODE_PARAMS_v2(_ptr=<void_ptr>&self._ptr[0].host)
        self._graph = CUDA_CHILD_GRAPH_NODE_PARAMS(_ptr=<void_ptr>&self._ptr[0].graph)
        self._eventWait = CUDA_EVENT_WAIT_NODE_PARAMS(_ptr=<void_ptr>&self._ptr[0].eventWait)
        self._eventRecord = CUDA_EVENT_RECORD_NODE_PARAMS(_ptr=<void_ptr>&self._ptr[0].eventRecord)
        self._extSemSignal = CUDA_EXT_SEM_SIGNAL_NODE_PARAMS_v2(_ptr=<void_ptr>&self._ptr[0].extSemSignal)
        self._extSemWait = CUDA_EXT_SEM_WAIT_NODE_PARAMS_v2(_ptr=<void_ptr>&self._ptr[0].extSemWait)
        self._alloc = CUDA_MEM_ALLOC_NODE_PARAMS_v2(_ptr=<void_ptr>&self._ptr[0].alloc)
        self._free = CUDA_MEM_FREE_NODE_PARAMS(_ptr=<void_ptr>&self._ptr[0].free)
        self._memOp = CUDA_BATCH_MEM_OP_NODE_PARAMS_v2(_ptr=<void_ptr>&self._ptr[0].memOp)
        self._conditional = CUDA_CONDITIONAL_NODE_PARAMS(_ptr=<void_ptr>&self._ptr[0].conditional)
    def __dealloc__(self):
        if self._val_ptr is not NULL:
            free(self._val_ptr)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['type : ' + str(self.type)]
            except ValueError:
                str_list += ['type : <ValueError>']
            try:
                str_list += ['reserved0 : ' + str(self.reserved0)]
            except ValueError:
                str_list += ['reserved0 : <ValueError>']
            try:
                str_list += ['reserved1 : ' + str(self.reserved1)]
            except ValueError:
                str_list += ['reserved1 : <ValueError>']
            try:
                str_list += ['kernel :\n' + '\n'.join(['    ' + line for line in str(self.kernel).splitlines()])]
            except ValueError:
                str_list += ['kernel : <ValueError>']
            try:
                str_list += ['memcpy :\n' + '\n'.join(['    ' + line for line in str(self.memcpy).splitlines()])]
            except ValueError:
                str_list += ['memcpy : <ValueError>']
            try:
                str_list += ['memset :\n' + '\n'.join(['    ' + line for line in str(self.memset).splitlines()])]
            except ValueError:
                str_list += ['memset : <ValueError>']
            try:
                str_list += ['host :\n' + '\n'.join(['    ' + line for line in str(self.host).splitlines()])]
            except ValueError:
                str_list += ['host : <ValueError>']
            try:
                str_list += ['graph :\n' + '\n'.join(['    ' + line for line in str(self.graph).splitlines()])]
            except ValueError:
                str_list += ['graph : <ValueError>']
            try:
                str_list += ['eventWait :\n' + '\n'.join(['    ' + line for line in str(self.eventWait).splitlines()])]
            except ValueError:
                str_list += ['eventWait : <ValueError>']
            try:
                str_list += ['eventRecord :\n' + '\n'.join(['    ' + line for line in str(self.eventRecord).splitlines()])]
            except ValueError:
                str_list += ['eventRecord : <ValueError>']
            try:
                str_list += ['extSemSignal :\n' + '\n'.join(['    ' + line for line in str(self.extSemSignal).splitlines()])]
            except ValueError:
                str_list += ['extSemSignal : <ValueError>']
            try:
                str_list += ['extSemWait :\n' + '\n'.join(['    ' + line for line in str(self.extSemWait).splitlines()])]
            except ValueError:
                str_list += ['extSemWait : <ValueError>']
            try:
                str_list += ['alloc :\n' + '\n'.join(['    ' + line for line in str(self.alloc).splitlines()])]
            except ValueError:
                str_list += ['alloc : <ValueError>']
            try:
                str_list += ['free :\n' + '\n'.join(['    ' + line for line in str(self.free).splitlines()])]
            except ValueError:
                str_list += ['free : <ValueError>']
            try:
                str_list += ['memOp :\n' + '\n'.join(['    ' + line for line in str(self.memOp).splitlines()])]
            except ValueError:
                str_list += ['memOp : <ValueError>']
            try:
                str_list += ['conditional :\n' + '\n'.join(['    ' + line for line in str(self.conditional).splitlines()])]
            except ValueError:
                str_list += ['conditional : <ValueError>']
            try:
                str_list += ['reserved2 : ' + str(self.reserved2)]
            except ValueError:
                str_list += ['reserved2 : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def type(self):
        return CUgraphNodeType(self._ptr[0].type)
    @type.setter
    def type(self, type not None : CUgraphNodeType):
        self._ptr[0].type = type.value
    @property
    def reserved0(self):
        return self._ptr[0].reserved0
    @reserved0.setter
    def reserved0(self, reserved0):
        self._ptr[0].reserved0 = reserved0
    @property
    def reserved1(self):
        return self._ptr[0].reserved1
    @reserved1.setter
    def reserved1(self, reserved1):
        self._ptr[0].reserved1 = reserved1
    @property
    def kernel(self):
        return self._kernel
    @kernel.setter
    def kernel(self, kernel not None : CUDA_KERNEL_NODE_PARAMS_v3):
        string.memcpy(&self._ptr[0].kernel, <ccuda.CUDA_KERNEL_NODE_PARAMS_v3*><void_ptr>kernel.getPtr(), sizeof(self._ptr[0].kernel))
    @property
    def memcpy(self):
        return self._memcpy
    @memcpy.setter
    def memcpy(self, memcpy not None : CUDA_MEMCPY_NODE_PARAMS):
        string.memcpy(&self._ptr[0].memcpy, <ccuda.CUDA_MEMCPY_NODE_PARAMS*><void_ptr>memcpy.getPtr(), sizeof(self._ptr[0].memcpy))
    @property
    def memset(self):
        return self._memset
    @memset.setter
    def memset(self, memset not None : CUDA_MEMSET_NODE_PARAMS_v2):
        string.memcpy(&self._ptr[0].memset, <ccuda.CUDA_MEMSET_NODE_PARAMS_v2*><void_ptr>memset.getPtr(), sizeof(self._ptr[0].memset))
    @property
    def host(self):
        return self._host
    @host.setter
    def host(self, host not None : CUDA_HOST_NODE_PARAMS_v2):
        string.memcpy(&self._ptr[0].host, <ccuda.CUDA_HOST_NODE_PARAMS_v2*><void_ptr>host.getPtr(), sizeof(self._ptr[0].host))
    @property
    def graph(self):
        return self._graph
    @graph.setter
    def graph(self, graph not None : CUDA_CHILD_GRAPH_NODE_PARAMS):
        string.memcpy(&self._ptr[0].graph, <ccuda.CUDA_CHILD_GRAPH_NODE_PARAMS*><void_ptr>graph.getPtr(), sizeof(self._ptr[0].graph))
    @property
    def eventWait(self):
        return self._eventWait
    @eventWait.setter
    def eventWait(self, eventWait not None : CUDA_EVENT_WAIT_NODE_PARAMS):
        string.memcpy(&self._ptr[0].eventWait, <ccuda.CUDA_EVENT_WAIT_NODE_PARAMS*><void_ptr>eventWait.getPtr(), sizeof(self._ptr[0].eventWait))
    @property
    def eventRecord(self):
        return self._eventRecord
    @eventRecord.setter
    def eventRecord(self, eventRecord not None : CUDA_EVENT_RECORD_NODE_PARAMS):
        string.memcpy(&self._ptr[0].eventRecord, <ccuda.CUDA_EVENT_RECORD_NODE_PARAMS*><void_ptr>eventRecord.getPtr(), sizeof(self._ptr[0].eventRecord))
    @property
    def extSemSignal(self):
        return self._extSemSignal
    @extSemSignal.setter
    def extSemSignal(self, extSemSignal not None : CUDA_EXT_SEM_SIGNAL_NODE_PARAMS_v2):
        string.memcpy(&self._ptr[0].extSemSignal, <ccuda.CUDA_EXT_SEM_SIGNAL_NODE_PARAMS_v2*><void_ptr>extSemSignal.getPtr(), sizeof(self._ptr[0].extSemSignal))
    @property
    def extSemWait(self):
        return self._extSemWait
    @extSemWait.setter
    def extSemWait(self, extSemWait not None : CUDA_EXT_SEM_WAIT_NODE_PARAMS_v2):
        string.memcpy(&self._ptr[0].extSemWait, <ccuda.CUDA_EXT_SEM_WAIT_NODE_PARAMS_v2*><void_ptr>extSemWait.getPtr(), sizeof(self._ptr[0].extSemWait))
    @property
    def alloc(self):
        return self._alloc
    @alloc.setter
    def alloc(self, alloc not None : CUDA_MEM_ALLOC_NODE_PARAMS_v2):
        string.memcpy(&self._ptr[0].alloc, <ccuda.CUDA_MEM_ALLOC_NODE_PARAMS_v2*><void_ptr>alloc.getPtr(), sizeof(self._ptr[0].alloc))
    @property
    def free(self):
        return self._free
    @free.setter
    def free(self, free not None : CUDA_MEM_FREE_NODE_PARAMS):
        string.memcpy(&self._ptr[0].free, <ccuda.CUDA_MEM_FREE_NODE_PARAMS*><void_ptr>free.getPtr(), sizeof(self._ptr[0].free))
    @property
    def memOp(self):
        return self._memOp
    @memOp.setter
    def memOp(self, memOp not None : CUDA_BATCH_MEM_OP_NODE_PARAMS_v2):
        string.memcpy(&self._ptr[0].memOp, <ccuda.CUDA_BATCH_MEM_OP_NODE_PARAMS_v2*><void_ptr>memOp.getPtr(), sizeof(self._ptr[0].memOp))
    @property
    def conditional(self):
        return self._conditional
    @conditional.setter
    def conditional(self, conditional not None : CUDA_CONDITIONAL_NODE_PARAMS):
        string.memcpy(&self._ptr[0].conditional, <ccuda.CUDA_CONDITIONAL_NODE_PARAMS*><void_ptr>conditional.getPtr(), sizeof(self._ptr[0].conditional))
    @property
    def reserved2(self):
        return self._ptr[0].reserved2
    @reserved2.setter
    def reserved2(self, long long reserved2):
        self._ptr[0].reserved2 = reserved2

cdef class CUdevSmResource_st:
    """
    Attributes
    ----------
    smCount : unsigned int
        The amount of streaming multiprocessors available in this resource.
        This is an output parameter only, do not write to this field.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUdevSmResource_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['smCount : ' + str(self.smCount)]
            except ValueError:
                str_list += ['smCount : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def smCount(self):
        return self._ptr[0].smCount
    @smCount.setter
    def smCount(self, unsigned int smCount):
        self._ptr[0].smCount = smCount

cdef class CUdevResource_st:
    """
    Attributes
    ----------
    type : CUdevResourceType
        Type of resource, dictates which union field was last set
    _internal_padding : bytes

    sm : CUdevSmResource
        Resource corresponding to CU_DEV_RESOURCE_TYPE_SM ``. type.
    _oversize : bytes


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._val_ptr = <ccuda.CUdevResource_st *>calloc(1, sizeof(ccuda.CUdevResource_st))
            self._ptr = self._val_ptr
        else:
            self._ptr = <ccuda.CUdevResource_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._sm = CUdevSmResource(_ptr=<void_ptr>&self._ptr[0].sm)
    def __dealloc__(self):
        if self._val_ptr is not NULL:
            free(self._val_ptr)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['type : ' + str(self.type)]
            except ValueError:
                str_list += ['type : <ValueError>']
            try:
                str_list += ['_internal_padding : ' + str(self._internal_padding)]
            except ValueError:
                str_list += ['_internal_padding : <ValueError>']
            try:
                str_list += ['sm :\n' + '\n'.join(['    ' + line for line in str(self.sm).splitlines()])]
            except ValueError:
                str_list += ['sm : <ValueError>']
            try:
                str_list += ['_oversize : ' + str(self._oversize)]
            except ValueError:
                str_list += ['_oversize : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def type(self):
        return CUdevResourceType(self._ptr[0].type)
    @type.setter
    def type(self, type not None : CUdevResourceType):
        self._ptr[0].type = type.value
    @property
    def _internal_padding(self):
        return PyBytes_FromStringAndSize(<char*>self._ptr[0]._internal_padding, 92)
    @_internal_padding.setter
    def _internal_padding(self, _internal_padding):
        if len(_internal_padding) != 92:
            raise ValueError("_internal_padding length must be 92, is " + str(len(_internal_padding)))
        for i, b in enumerate(_internal_padding):
            self._ptr[0]._internal_padding[i] = b
    @property
    def sm(self):
        return self._sm
    @sm.setter
    def sm(self, sm not None : CUdevSmResource):
        string.memcpy(&self._ptr[0].sm, <ccuda.CUdevSmResource*><void_ptr>sm.getPtr(), sizeof(self._ptr[0].sm))
    @property
    def _oversize(self):
        return PyBytes_FromStringAndSize(<char*>self._ptr[0]._oversize, 48)
    @_oversize.setter
    def _oversize(self, _oversize):
        if len(_oversize) != 48:
            raise ValueError("_oversize length must be 48, is " + str(len(_oversize)))
        for i, b in enumerate(_oversize):
            self._ptr[0]._oversize[i] = b

cdef class anon_union14:
    """
    Attributes
    ----------
    pArray : List[CUarray]

    pPitch : List[Any]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccuda.CUeglFrame_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].frame
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['pArray : ' + str(self.pArray)]
            except ValueError:
                str_list += ['pArray : <ValueError>']
            try:
                str_list += ['pPitch : ' + hex(self.pPitch)]
            except ValueError:
                str_list += ['pPitch : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def pArray(self):
        return [CUarray(init_value=<void_ptr>_pArray) for _pArray in self._ptr[0].frame.pArray]
    @pArray.setter
    def pArray(self, pArray : List[CUarray]):
        if len(pArray) != 3:
            raise IndexError('not enough values found during array assignment, expected 3, got', len(pArray))
        pArray = [int(_pArray) for _pArray in pArray]
        for _idx, _pArray in enumerate(pArray):
            self._ptr[0].frame.pArray[_idx] = <ccuda.CUarray><void_ptr>_pArray

    @property
    def pPitch(self):
        return [<void_ptr>_pPitch for _pPitch in self._ptr[0].frame.pPitch]
    @pPitch.setter
    def pPitch(self, pPitch : List[int]):
        if len(pPitch) != 3:
            raise IndexError('not enough values found during array assignment, expected 3, got', len(pPitch))
        pPitch = [<void_ptr>_pPitch for _pPitch in pPitch]
        for _idx, _pPitch in enumerate(pPitch):
            self._ptr[0].frame.pPitch[_idx] = <void*><void_ptr>_pPitch

cdef class CUeglFrame_st:
    """
    CUDA EGLFrame structure Descriptor - structure defining one frame
    of EGL.  Each frame may contain one or more planes depending on
    whether the surface * is Multiplanar or not.

    Attributes
    ----------
    frame : anon_union14

    width : unsigned int
        Width of first plane
    height : unsigned int
        Height of first plane
    depth : unsigned int
        Depth of first plane
    pitch : unsigned int
        Pitch of first plane
    planeCount : unsigned int
        Number of planes
    numChannels : unsigned int
        Number of channels for the plane
    frameType : CUeglFrameType
        Array or Pitch
    eglColorFormat : CUeglColorFormat
        CUDA EGL Color Format
    cuFormat : CUarray_format
        CUDA Array Format

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._val_ptr = <ccuda.CUeglFrame_st *>calloc(1, sizeof(ccuda.CUeglFrame_st))
            self._ptr = self._val_ptr
        else:
            self._ptr = <ccuda.CUeglFrame_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._frame = anon_union14(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        if self._val_ptr is not NULL:
            free(self._val_ptr)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['frame :\n' + '\n'.join(['    ' + line for line in str(self.frame).splitlines()])]
            except ValueError:
                str_list += ['frame : <ValueError>']
            try:
                str_list += ['width : ' + str(self.width)]
            except ValueError:
                str_list += ['width : <ValueError>']
            try:
                str_list += ['height : ' + str(self.height)]
            except ValueError:
                str_list += ['height : <ValueError>']
            try:
                str_list += ['depth : ' + str(self.depth)]
            except ValueError:
                str_list += ['depth : <ValueError>']
            try:
                str_list += ['pitch : ' + str(self.pitch)]
            except ValueError:
                str_list += ['pitch : <ValueError>']
            try:
                str_list += ['planeCount : ' + str(self.planeCount)]
            except ValueError:
                str_list += ['planeCount : <ValueError>']
            try:
                str_list += ['numChannels : ' + str(self.numChannels)]
            except ValueError:
                str_list += ['numChannels : <ValueError>']
            try:
                str_list += ['frameType : ' + str(self.frameType)]
            except ValueError:
                str_list += ['frameType : <ValueError>']
            try:
                str_list += ['eglColorFormat : ' + str(self.eglColorFormat)]
            except ValueError:
                str_list += ['eglColorFormat : <ValueError>']
            try:
                str_list += ['cuFormat : ' + str(self.cuFormat)]
            except ValueError:
                str_list += ['cuFormat : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def frame(self):
        return self._frame
    @frame.setter
    def frame(self, frame not None : anon_union14):
        string.memcpy(&self._ptr[0].frame, <ccuda.anon_union14*><void_ptr>frame.getPtr(), sizeof(self._ptr[0].frame))
    @property
    def width(self):
        return self._ptr[0].width
    @width.setter
    def width(self, unsigned int width):
        self._ptr[0].width = width
    @property
    def height(self):
        return self._ptr[0].height
    @height.setter
    def height(self, unsigned int height):
        self._ptr[0].height = height
    @property
    def depth(self):
        return self._ptr[0].depth
    @depth.setter
    def depth(self, unsigned int depth):
        self._ptr[0].depth = depth
    @property
    def pitch(self):
        return self._ptr[0].pitch
    @pitch.setter
    def pitch(self, unsigned int pitch):
        self._ptr[0].pitch = pitch
    @property
    def planeCount(self):
        return self._ptr[0].planeCount
    @planeCount.setter
    def planeCount(self, unsigned int planeCount):
        self._ptr[0].planeCount = planeCount
    @property
    def numChannels(self):
        return self._ptr[0].numChannels
    @numChannels.setter
    def numChannels(self, unsigned int numChannels):
        self._ptr[0].numChannels = numChannels
    @property
    def frameType(self):
        return CUeglFrameType(self._ptr[0].frameType)
    @frameType.setter
    def frameType(self, frameType not None : CUeglFrameType):
        self._ptr[0].frameType = frameType.value
    @property
    def eglColorFormat(self):
        return CUeglColorFormat(self._ptr[0].eglColorFormat)
    @eglColorFormat.setter
    def eglColorFormat(self, eglColorFormat not None : CUeglColorFormat):
        self._ptr[0].eglColorFormat = eglColorFormat.value
    @property
    def cuFormat(self):
        return CUarray_format(self._ptr[0].cuFormat)
    @cuFormat.setter
    def cuFormat(self, cuFormat not None : CUarray_format):
        self._ptr[0].cuFormat = cuFormat.value

cdef class cuuint32_t:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, uint32_t init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.cuuint32_t *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<cuuint32_t ' + str(self.__int__()) + '>'
    def __int__(self):
        return <uint32_t>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class cuuint64_t:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, uint64_t init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.cuuint64_t *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<cuuint64_t ' + str(self.__int__()) + '>'
    def __int__(self):
        return <uint64_t>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUdeviceptr_v2:
    """

    CUDA device pointer CUdeviceptr is defined as an unsigned integer type whose size matches the size of a pointer on the target platform.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned long long init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUdeviceptr_v2 *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<CUdeviceptr_v2 ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned long long>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUdevice_v1:
    """

    CUDA device

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, int init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUdevice_v1 *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<CUdevice_v1 ' + str(self.__int__()) + '>'
    def __int__(self):
        return <int>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUtexObject_v1:
    """

    An opaque value that represents a CUDA texture object

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned long long init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUtexObject_v1 *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<CUtexObject_v1 ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned long long>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUsurfObject_v1:
    """

    An opaque value that represents a CUDA surface object

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned long long init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUsurfObject_v1 *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<CUsurfObject_v1 ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned long long>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUmemGenericAllocationHandle_v1:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned long long init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.CUmemGenericAllocationHandle_v1 *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<CUmemGenericAllocationHandle_v1 ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned long long>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class GLenum:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned int init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.GLenum *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<GLenum ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned int>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class GLuint:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned int init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.GLuint *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<GLuint ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned int>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class EGLint:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned int init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.EGLint *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<EGLint ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned int>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class VdpDevice:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, uint32_t init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.VdpDevice *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<VdpDevice ' + str(self.__int__()) + '>'
    def __int__(self):
        return <uint32_t>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class VdpGetProcAddress:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned long long init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.VdpGetProcAddress *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<VdpGetProcAddress ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned long long>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class VdpVideoSurface:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, uint32_t init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.VdpVideoSurface *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<VdpVideoSurface ' + str(self.__int__()) + '>'
    def __int__(self):
        return <uint32_t>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class VdpOutputSurface:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, uint32_t init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccuda.VdpOutputSurface *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<VdpOutputSurface ' + str(self.__int__()) + '>'
    def __int__(self):
        return <uint32_t>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

@cython.embedsignature(True)
def cuGetErrorString(error not None : CUresult):
    """ Gets the string description of an error code.

    Sets `*pStr` to the address of a NULL-terminated string description of
    the error code `error`. If the error code is not recognized,
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE` will be returned and `*pStr` will
    be set to the NULL address.

    Parameters
    ----------
    error : :py:obj:`~.CUresult`
        Error code to convert to string

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pStr : bytes
        Address of the string pointer.

    See Also
    --------
    :py:obj:`~.CUresult`, :py:obj:`~.cudaGetErrorString`
    """
    cdef ccuda.CUresult cerror = error.value
    cdef const char* pStr = NULL
    err = ccuda.cuGetErrorString(cerror, &pStr)
    return (CUresult(err), <bytes>pStr)

@cython.embedsignature(True)
def cuGetErrorName(error not None : CUresult):
    """ Gets the string representation of an error code enum name.

    Sets `*pStr` to the address of a NULL-terminated string representation
    of the name of the enum error code `error`. If the error code is not
    recognized, :py:obj:`~.CUDA_ERROR_INVALID_VALUE` will be returned and
    `*pStr` will be set to the NULL address.

    Parameters
    ----------
    error : :py:obj:`~.CUresult`
        Error code to convert to string

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pStr : bytes
        Address of the string pointer.

    See Also
    --------
    :py:obj:`~.CUresult`, :py:obj:`~.cudaGetErrorName`
    """
    cdef ccuda.CUresult cerror = error.value
    cdef const char* pStr = NULL
    err = ccuda.cuGetErrorName(cerror, &pStr)
    return (CUresult(err), <bytes>pStr)

@cython.embedsignature(True)
def cuInit(unsigned int Flags):
    """ Initialize the CUDA driver API Initializes the driver API and must be called before any other function from the driver API in the current process. Currently, the `Flags` parameter must be 0. If :py:obj:`~.cuInit()` has not been called, any function from the driver API will return :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`.

    Parameters
    ----------
    Flags : unsigned int
        Initialization flag for CUDA.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_SYSTEM_DRIVER_MISMATCH`, :py:obj:`~.CUDA_ERROR_COMPAT_NOT_SUPPORTED_ON_DEVICE`
    """
    err = ccuda.cuInit(Flags)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuDriverGetVersion():
    """ Returns the latest CUDA version supported by driver.

    Returns in `*driverVersion` the version of CUDA supported by the
    driver. The version is returned as (1000 * major + 10 * minor). For
    example, CUDA 9.2 would be represented by 9020.

    This function automatically returns
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE` if `driverVersion` is NULL.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    driverVersion : int
        Returns the CUDA driver version

    See Also
    --------
    :py:obj:`~.cudaDriverGetVersion`, :py:obj:`~.cudaRuntimeGetVersion`
    """
    cdef int driverVersion = 0
    err = ccuda.cuDriverGetVersion(&driverVersion)
    return (CUresult(err), driverVersion)

@cython.embedsignature(True)
def cuDeviceGet(int ordinal):
    """ Returns a handle to a compute device.

    Returns in `*device` a device handle given an ordinal in the range [0,
    :py:obj:`~.cuDeviceGetCount()`-1].

    Parameters
    ----------
    ordinal : int
        Device number to get handle for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    device : :py:obj:`~.CUdevice`
        Returned device handle

    See Also
    --------
    :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGetUuid`, :py:obj:`~.cuDeviceGetLuid`, :py:obj:`~.cuDeviceTotalMem`, :py:obj:`~.cuDeviceGetExecAffinitySupport`
    """
    cdef CUdevice device = CUdevice()
    err = ccuda.cuDeviceGet(<ccuda.CUdevice*>device._ptr, ordinal)
    return (CUresult(err), device)

@cython.embedsignature(True)
def cuDeviceGetCount():
    """ Returns the number of compute-capable devices.

    Returns in `*count` the number of devices with compute capability
    greater than or equal to 2.0 that are available for execution. If there
    is no such device, :py:obj:`~.cuDeviceGetCount()` returns 0.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    count : int
        Returned number of compute-capable devices

    See Also
    --------
    :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGetUuid`, :py:obj:`~.cuDeviceGetLuid`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceTotalMem`, :py:obj:`~.cuDeviceGetExecAffinitySupport`, :py:obj:`~.cudaGetDeviceCount`
    """
    cdef int count = 0
    err = ccuda.cuDeviceGetCount(&count)
    return (CUresult(err), count)

@cython.embedsignature(True)
def cuDeviceGetName(int length, dev):
    """ Returns an identifier string for the device.

    Returns an ASCII string identifying the device `dev` in the NULL-
    terminated string pointed to by `name`. `length` specifies the maximum
    length of the string that may be returned.

    Parameters
    ----------
    length : int
        Maximum length of string to store in `name`
    dev : :py:obj:`~.CUdevice`
        Device to get identifier string for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    name : bytes
        Returned identifier string for the device

    See Also
    --------
    :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetUuid`, :py:obj:`~.cuDeviceGetLuid`, :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceTotalMem`, :py:obj:`~.cuDeviceGetExecAffinitySupport`, :py:obj:`~.cudaGetDeviceProperties`
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    pyname = b" " * length
    cdef char* name = pyname
    err = ccuda.cuDeviceGetName(name, length, cdev)
    return (CUresult(err), pyname)

@cython.embedsignature(True)
def cuDeviceGetUuid(dev):
    """ Return an UUID for the device.

    Note there is a later version of this API,
    :py:obj:`~.cuDeviceGetUuid_v2`. It will supplant this version in 12.0,
    which is retained for minor version compatibility.

    Returns 16-octets identifying the device `dev` in the structure pointed
    by the `uuid`.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device to get identifier string for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    uuid : :py:obj:`~.CUuuid`
        Returned UUID

    See Also
    --------
    :py:obj:`~.cuDeviceGetUuid_v2` :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGetLuid`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceTotalMem`, :py:obj:`~.cuDeviceGetExecAffinitySupport`, :py:obj:`~.cudaGetDeviceProperties`
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    cdef CUuuid uuid = CUuuid()
    err = ccuda.cuDeviceGetUuid(<ccuda.CUuuid*>uuid._ptr, cdev)
    return (CUresult(err), uuid)

@cython.embedsignature(True)
def cuDeviceGetUuid_v2(dev):
    """ Return an UUID for the device (11.4+)

    Returns 16-octets identifying the device `dev` in the structure pointed
    by the `uuid`. If the device is in MIG mode, returns its MIG UUID which
    uniquely identifies the subscribed MIG compute instance.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device to get identifier string for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    uuid : :py:obj:`~.CUuuid`
        Returned UUID

    See Also
    --------
    :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGetLuid`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceTotalMem`, :py:obj:`~.cudaGetDeviceProperties`
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    cdef CUuuid uuid = CUuuid()
    err = ccuda.cuDeviceGetUuid_v2(<ccuda.CUuuid*>uuid._ptr, cdev)
    return (CUresult(err), uuid)

@cython.embedsignature(True)
def cuDeviceGetLuid(dev):
    """ Return an LUID and device node mask for the device.

    Return identifying information (`luid` and `deviceNodeMask`) to allow
    matching device with graphics APIs.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device to get identifier string for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    luid : bytes
        Returned LUID
    deviceNodeMask : unsigned int
        Returned device node mask

    See Also
    --------
    :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceTotalMem`, :py:obj:`~.cuDeviceGetExecAffinitySupport`, :py:obj:`~.cudaGetDeviceProperties`
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    cdef char luid[8]
    cdef unsigned int deviceNodeMask = 0
    err = ccuda.cuDeviceGetLuid(luid, &deviceNodeMask, cdev)
    return (CUresult(err), <bytes>luid, deviceNodeMask)

@cython.embedsignature(True)
def cuDeviceTotalMem(dev):
    """ Returns the total amount of memory on the device.

    Returns in `*bytes` the total amount of memory available on the device
    `dev` in bytes.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device handle

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    numbytes : int
        Returned memory available on device in bytes

    See Also
    --------
    :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGetUuid`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceGetExecAffinitySupport`, :py:obj:`~.cudaMemGetInfo`
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    cdef size_t numbytes = 0
    err = ccuda.cuDeviceTotalMem(&numbytes, cdev)
    return (CUresult(err), numbytes)

@cython.embedsignature(True)
def cuDeviceGetTexture1DLinearMaxWidth(pformat not None : CUarray_format, unsigned numChannels, dev):
    """ Returns the maximum number of elements allocatable in a 1D linear texture for a given texture element size.

    Returns in `maxWidthInElements` the maximum number of texture elements
    allocatable in a 1D linear texture for given `pformat` and
    `numChannels`.

    Parameters
    ----------
    pformat : :py:obj:`~.CUarray_format`
        Texture format.
    numChannels : unsigned
        Number of channels per texture element.
    dev : :py:obj:`~.CUdevice`
        Device handle.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    maxWidthInElements : int
        Returned maximum number of texture elements allocatable for given
        `pformat` and `numChannels`.

    See Also
    --------
    :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGetUuid`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cudaMemGetInfo`, :py:obj:`~.cuDeviceTotalMem`
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    cdef size_t maxWidthInElements = 0
    cdef ccuda.CUarray_format cpformat = pformat.value
    err = ccuda.cuDeviceGetTexture1DLinearMaxWidth(&maxWidthInElements, cpformat, numChannels, cdev)
    return (CUresult(err), maxWidthInElements)

@cython.embedsignature(True)
def cuDeviceGetAttribute(attrib not None : CUdevice_attribute, dev):
    """ Returns information about the device.

    Returns in `*pi` the integer value of the attribute `attrib` on device
    `dev`. The supported attributes are:

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_THREADS_PER_BLOCK`: Maximum number
      of threads per block;

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_BLOCK_DIM_X`: Maximum x-dimension
      of a block

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_BLOCK_DIM_Y`: Maximum y-dimension
      of a block

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_BLOCK_DIM_Z`: Maximum z-dimension
      of a block

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_X`: Maximum x-dimension
      of a grid

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_Y`: Maximum y-dimension
      of a grid

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_Z`: Maximum z-dimension
      of a grid

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_BLOCK`: Maximum
      amount of shared memory available to a thread block in bytes

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_TOTAL_CONSTANT_MEMORY`: Memory
      available on device for constant variables in a CUDA C kernel in
      bytes

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_WARP_SIZE`: Warp size in threads

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_PITCH`: Maximum pitch in bytes
      allowed by the memory copy functions that involve memory regions
      allocated through :py:obj:`~.cuMemAllocPitch()`

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_WIDTH`: Maximum 1D
      texture width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_LINEAR_WIDTH`:
      Maximum width for a 1D texture bound to linear memory

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_MIPMAPPED_WIDTH`:
      Maximum mipmapped 1D texture width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_WIDTH`: Maximum 2D
      texture width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_HEIGHT`: Maximum 2D
      texture height

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LINEAR_WIDTH`:
      Maximum width for a 2D texture bound to linear memory

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LINEAR_HEIGHT`:
      Maximum height for a 2D texture bound to linear memory

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LINEAR_PITCH`:
      Maximum pitch in bytes for a 2D texture bound to linear memory

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_MIPMAPPED_WIDTH`:
      Maximum mipmapped 2D texture width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_MIPMAPPED_HEIGHT`:
      Maximum mipmapped 2D texture height

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_WIDTH`: Maximum 3D
      texture width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_HEIGHT`: Maximum 3D
      texture height

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_DEPTH`: Maximum 3D
      texture depth

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_WIDTH_ALTERNATE`:
      Alternate maximum 3D texture width, 0 if no alternate maximum 3D
      texture size is supported

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_HEIGHT_ALTERNATE`:
      Alternate maximum 3D texture height, 0 if no alternate maximum 3D
      texture size is supported

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_DEPTH_ALTERNATE`:
      Alternate maximum 3D texture depth, 0 if no alternate maximum 3D
      texture size is supported

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURECUBEMAP_WIDTH`: Maximum
      cubemap texture width or height

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_LAYERED_WIDTH`:
      Maximum 1D layered texture width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_LAYERED_LAYERS`:
      Maximum layers in a 1D layered texture

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_WIDTH`:
      Maximum 2D layered texture width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_HEIGHT`:
      Maximum 2D layered texture height

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_LAYERS`:
      Maximum layers in a 2D layered texture

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURECUBEMAP_LAYERED_WIDTH`:
      Maximum cubemap layered texture width or height

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURECUBEMAP_LAYERED_LAYERS`:
      Maximum layers in a cubemap layered texture

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE1D_WIDTH`: Maximum 1D
      surface width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_WIDTH`: Maximum 2D
      surface width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_HEIGHT`: Maximum 2D
      surface height

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE3D_WIDTH`: Maximum 3D
      surface width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE3D_HEIGHT`: Maximum 3D
      surface height

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE3D_DEPTH`: Maximum 3D
      surface depth

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE1D_LAYERED_WIDTH`:
      Maximum 1D layered surface width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE1D_LAYERED_LAYERS`:
      Maximum layers in a 1D layered surface

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_LAYERED_WIDTH`:
      Maximum 2D layered surface width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_LAYERED_HEIGHT`:
      Maximum 2D layered surface height

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_LAYERED_LAYERS`:
      Maximum layers in a 2D layered surface

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACECUBEMAP_WIDTH`: Maximum
      cubemap surface width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACECUBEMAP_LAYERED_WIDTH`:
      Maximum cubemap layered surface width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACECUBEMAP_LAYERED_LAYERS`:
      Maximum layers in a cubemap layered surface

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_REGISTERS_PER_BLOCK`: Maximum
      number of 32-bit registers available to a thread block

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_CLOCK_RATE`: The typical clock
      frequency in kilohertz

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_TEXTURE_ALIGNMENT`: Alignment
      requirement; texture base addresses aligned to
      :py:obj:`~.textureAlign` bytes do not need an offset applied to
      texture fetches

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_TEXTURE_PITCH_ALIGNMENT`: Pitch
      alignment requirement for 2D texture references bound to pitched
      memory

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_GPU_OVERLAP`: 1 if the device can
      concurrently copy memory between host and device while executing a
      kernel, or 0 if not

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MULTIPROCESSOR_COUNT`: Number of
      multiprocessors on the device

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_KERNEL_EXEC_TIMEOUT`: 1 if there is a
      run time limit for kernels executed on the device, or 0 if not

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_INTEGRATED`: 1 if the device is
      integrated with the memory subsystem, or 0 if not

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_CAN_MAP_HOST_MEMORY`: 1 if the device
      can map host memory into the CUDA address space, or 0 if not

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_COMPUTE_MODE`: Compute mode that
      device is currently in. Available modes are as follows:

      - :py:obj:`~.CU_COMPUTEMODE_DEFAULT`: Default mode - Device is not
        restricted and can have multiple CUDA contexts present at a single
        time.

      - :py:obj:`~.CU_COMPUTEMODE_PROHIBITED`: Compute-prohibited mode -
        Device is prohibited from creating new CUDA contexts.

      - :py:obj:`~.CU_COMPUTEMODE_EXCLUSIVE_PROCESS`: Compute-exclusive-
        process mode - Device can have only one context used by a single
        process at a time.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_KERNELS`: 1 if the device
      supports executing multiple kernels within the same context
      simultaneously, or 0 if not. It is not guaranteed that multiple
      kernels will be resident on the device concurrently so this feature
      should not be relied upon for correctness.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_ECC_ENABLED`: 1 if error correction is
      enabled on the device, 0 if error correction is disabled or not
      supported by the device

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_PCI_BUS_ID`: PCI bus identifier of the
      device

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_PCI_DEVICE_ID`: PCI device (also known
      as slot) identifier of the device

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_PCI_DOMAIN_ID`: PCI domain identifier
      of the device

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_TCC_DRIVER`: 1 if the device is using
      a TCC driver. TCC is only available on Tesla hardware running Windows
      Vista or later

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MEMORY_CLOCK_RATE`: Peak memory clock
      frequency in kilohertz

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_GLOBAL_MEMORY_BUS_WIDTH`: Global
      memory bus width in bits

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_L2_CACHE_SIZE`: Size of L2 cache in
      bytes. 0 if the device doesn't have L2 cache

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_THREADS_PER_MULTIPROCESSOR`:
      Maximum resident threads per multiprocessor

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_UNIFIED_ADDRESSING`: 1 if the device
      shares a unified address space with the host, or 0 if not

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_COMPUTE_CAPABILITY_MAJOR`: Major
      compute capability version number

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_COMPUTE_CAPABILITY_MINOR`: Minor
      compute capability version number

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_GLOBAL_L1_CACHE_SUPPORTED`: 1 if
      device supports caching globals in L1 cache, 0 if caching globals in
      L1 cache is not supported by the device

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_LOCAL_L1_CACHE_SUPPORTED`: 1 if device
      supports caching locals in L1 cache, 0 if caching locals in L1 cache
      is not supported by the device

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_MULTIPROCESSOR`:
      Maximum amount of shared memory available to a multiprocessor in
      bytes; this amount is shared by all thread blocks simultaneously
      resident on a multiprocessor

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_REGISTERS_PER_MULTIPROCESSOR`:
      Maximum number of 32-bit registers available to a multiprocessor;
      this number is shared by all thread blocks simultaneously resident on
      a multiprocessor

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MANAGED_MEMORY`: 1 if device supports
      allocating managed memory on this system, 0 if allocating managed
      memory is not supported by the device on this system.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MULTI_GPU_BOARD`: 1 if device is on a
      multi-GPU board, 0 if not.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MULTI_GPU_BOARD_GROUP_ID`: Unique
      identifier for a group of devices associated with the same board.
      Devices on the same multi-GPU board will share the same identifier.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_HOST_NATIVE_ATOMIC_SUPPORTED`: 1 if
      Link between the device and the host supports native atomic
      operations.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_SINGLE_TO_DOUBLE_PRECISION_PERF_RATIO`:
      Ratio of single precision performance (in floating-point operations
      per second) to double precision performance.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS`: Device
      supports coherently accessing pageable memory without calling
      cudaHostRegister on it.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS`: Device can
      coherently access managed memory concurrently with the CPU.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_COMPUTE_PREEMPTION_SUPPORTED`: Device
      supports Compute Preemption.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_CAN_USE_HOST_POINTER_FOR_REGISTERED_MEM`:
      Device can access host registered memory at the same virtual address
      as the CPU.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_BLOCK_OPTIN`:
      The maximum per block shared memory size supported on this device.
      This is the maximum value that can be opted into when using the
      :py:obj:`~.cuFuncSetAttribute()` or
      :py:obj:`~.cuKernelSetAttribute()` call. For more details see
      :py:obj:`~.CU_FUNC_ATTRIBUTE_MAX_DYNAMIC_SHARED_SIZE_BYTES`

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS_USES_HOST_PAGE_TABLES`:
      Device accesses pageable memory via the host's page tables.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_DIRECT_MANAGED_MEM_ACCESS_FROM_HOST`:
      The host can directly access managed memory on the device without
      migration.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_VIRTUAL_MEMORY_MANAGEMENT_SUPPORTED`:
      Device supports virtual memory management APIs like
      :py:obj:`~.cuMemAddressReserve`, :py:obj:`~.cuMemCreate`,
      :py:obj:`~.cuMemMap` and related APIs

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_HANDLE_TYPE_POSIX_FILE_DESCRIPTOR_SUPPORTED`:
      Device supports exporting memory to a posix file descriptor with
      :py:obj:`~.cuMemExportToShareableHandle`, if requested via
      :py:obj:`~.cuMemCreate`

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_HANDLE_TYPE_WIN32_HANDLE_SUPPORTED`:
      Device supports exporting memory to a Win32 NT handle with
      :py:obj:`~.cuMemExportToShareableHandle`, if requested via
      :py:obj:`~.cuMemCreate`

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_HANDLE_TYPE_WIN32_KMT_HANDLE_SUPPORTED`:
      Device supports exporting memory to a Win32 KMT handle with
      :py:obj:`~.cuMemExportToShareableHandle`, if requested via
      :py:obj:`~.cuMemCreate`

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_BLOCKS_PER_MULTIPROCESSOR`:
      Maximum number of thread blocks that can reside on a multiprocessor

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_GENERIC_COMPRESSION_SUPPORTED`: Device
      supports compressible memory allocation via :py:obj:`~.cuMemCreate`

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_PERSISTING_L2_CACHE_SIZE`: Maximum
      L2 persisting lines capacity setting in bytes

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_ACCESS_POLICY_WINDOW_SIZE`:
      Maximum value of :py:obj:`~.CUaccessPolicyWindow.num_bytes`

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_WITH_CUDA_VMM_SUPPORTED`:
      Device supports specifying the GPUDirect RDMA flag with
      :py:obj:`~.cuMemCreate`.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_RESERVED_SHARED_MEMORY_PER_BLOCK`:
      Amount of shared memory per block reserved by CUDA driver in bytes

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_SPARSE_CUDA_ARRAY_SUPPORTED`: Device
      supports sparse CUDA arrays and sparse CUDA mipmapped arrays.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_READ_ONLY_HOST_REGISTER_SUPPORTED`:
      Device supports using the :py:obj:`~.cuMemHostRegister` flag
      :py:obj:`~.CU_MEMHOSTERGISTER_READ_ONLY` to register memory that must
      be mapped as read-only to the GPU

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MEMORY_POOLS_SUPPORTED`: Device
      supports using the :py:obj:`~.cuMemAllocAsync` and
      :py:obj:`~.cuMemPool` family of APIs

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_SUPPORTED`: Device
      supports GPUDirect RDMA APIs, like nvidia_p2p_get_pages (see
      https://docs.nvidia.com/cuda/gpudirect-rdma for more information)

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_FLUSH_WRITES_OPTIONS`:
      The returned attribute shall be interpreted as a bitmask, where the
      individual bits are described by the
      :py:obj:`~.CUflushGPUDirectRDMAWritesOptions` enum

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_WRITES_ORDERING`:
      GPUDirect RDMA writes to the device do not need to be flushed for
      consumers within the scope indicated by the returned attribute. See
      :py:obj:`~.CUGPUDirectRDMAWritesOrdering` for the numerical values
      returned here.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MEMPOOL_SUPPORTED_HANDLE_TYPES`:
      Bitmask of handle types supported with mempool based IPC

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_DEFERRED_MAPPING_CUDA_ARRAY_SUPPORTED`:
      Device supports deferred mapping CUDA arrays and CUDA mipmapped
      arrays.

    Parameters
    ----------
    attrib : :py:obj:`~.CUdevice_attribute`
        Device attribute to query
    dev : :py:obj:`~.CUdevice`
        Device handle

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    pi : int
        Returned device attribute value

    See Also
    --------
    :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGetUuid`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceTotalMem`, :py:obj:`~.cuDeviceGetExecAffinitySupport`, :py:obj:`~.cudaDeviceGetAttribute`, :py:obj:`~.cudaGetDeviceProperties`
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    cdef int pi = 0
    cdef ccuda.CUdevice_attribute cattrib = attrib.value
    err = ccuda.cuDeviceGetAttribute(&pi, cattrib, cdev)
    return (CUresult(err), pi)

@cython.embedsignature(True)
def cuDeviceGetNvSciSyncAttributes(nvSciSyncAttrList, dev, int flags):
    """ Return NvSciSync attributes that this device can support.

    Returns in `nvSciSyncAttrList`, the properties of NvSciSync that this
    CUDA device, `dev` can support. The returned `nvSciSyncAttrList` can be
    used to create an NvSciSync object that matches this device's
    capabilities.

    If NvSciSyncAttrKey_RequiredPerm field in `nvSciSyncAttrList` is
    already set this API will return :py:obj:`~.CUDA_ERROR_INVALID_VALUE`.

    The applications should set `nvSciSyncAttrList` to a valid
    NvSciSyncAttrList failing which this API will return
    :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`.

    The `flags` controls how applications intends to use the NvSciSync
    created from the `nvSciSyncAttrList`. The valid flags are:

    - :py:obj:`~.CUDA_NVSCISYNC_ATTR_SIGNAL`, specifies that the
      applications intends to signal an NvSciSync on this CUDA device.

    - :py:obj:`~.CUDA_NVSCISYNC_ATTR_WAIT`, specifies that the applications
      intends to wait on an NvSciSync on this CUDA device.

    At least one of these flags must be set, failing which the API returns
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE`. Both the flags are orthogonal to
    one another: a developer may set both these flags that allows to set
    both wait and signal specific attributes in the same
    `nvSciSyncAttrList`.

    Note that this API updates the input `nvSciSyncAttrList` with values
    equivalent to the following public attribute key-values:
    NvSciSyncAttrKey_RequiredPerm is set to

    - NvSciSyncAccessPerm_SignalOnly if
      :py:obj:`~.CUDA_NVSCISYNC_ATTR_SIGNAL` is set in `flags`.

    - NvSciSyncAccessPerm_WaitOnly if :py:obj:`~.CUDA_NVSCISYNC_ATTR_WAIT`
      is set in `flags`.

    - NvSciSyncAccessPerm_WaitSignal if both
      :py:obj:`~.CUDA_NVSCISYNC_ATTR_WAIT` and
      :py:obj:`~.CUDA_NVSCISYNC_ATTR_SIGNAL` are set in `flags`.
      NvSciSyncAttrKey_PrimitiveInfo is set to

    - NvSciSyncAttrValPrimitiveType_SysmemSemaphore on any valid `device`.

    - NvSciSyncAttrValPrimitiveType_Syncpoint if `device` is a Tegra
      device.

    - NvSciSyncAttrValPrimitiveType_SysmemSemaphorePayload64b if `device`
      is GA10X+. NvSciSyncAttrKey_GpuId is set to the same UUID that is
      returned for this `device` from :py:obj:`~.cuDeviceGetUuid`.

    :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`,
    :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`,
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE`,
    :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`,
    :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`,
    :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`,
    :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`

    Parameters
    ----------
    nvSciSyncAttrList : Any
        Return NvSciSync attributes supported.
    dev : :py:obj:`~.CUdevice`
        Valid Cuda Device to get NvSciSync attributes for.
    flags : int
        flags describing NvSciSync usage.

    Returns
    -------
    CUresult


    See Also
    --------
    :py:obj:`~.cuImportExternalSemaphore`, :py:obj:`~.cuDestroyExternalSemaphore`, :py:obj:`~.cuSignalExternalSemaphoresAsync`, :py:obj:`~.cuWaitExternalSemaphoresAsync`
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    cnvSciSyncAttrList = utils.HelperInputVoidPtr(nvSciSyncAttrList)
    cdef void* cnvSciSyncAttrList_ptr = <void*><void_ptr>cnvSciSyncAttrList.cptr
    err = ccuda.cuDeviceGetNvSciSyncAttributes(cnvSciSyncAttrList_ptr, cdev, flags)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuDeviceSetMemPool(dev, pool):
    """ Sets the current memory pool of a device.

    The memory pool must be local to the specified device.
    :py:obj:`~.cuMemAllocAsync` allocates from the current mempool of the
    provided stream's device. By default, a device's current memory pool is
    its default memory pool.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        None
    pool : :py:obj:`~.CUmemoryPool` or :py:obj:`~.cudaMemPool_t`
        None

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuDeviceGetDefaultMemPool`, :py:obj:`~.cuDeviceGetMemPool`, :py:obj:`~.cuMemPoolCreate`, :py:obj:`~.cuMemPoolDestroy`, :py:obj:`~.cuMemAllocFromPoolAsync`

    Notes
    -----
    Use :py:obj:`~.cuMemAllocFromPoolAsync` to specify asynchronous allocations from a device different than the one the stream runs on.
    """
    cdef ccuda.CUmemoryPool cpool
    if pool is None:
        cpool = <ccuda.CUmemoryPool><void_ptr>0
    elif isinstance(pool, (CUmemoryPool,)):
        ppool = int(pool)
        cpool = <ccuda.CUmemoryPool><void_ptr>ppool
    else:
        ppool = int(CUmemoryPool(pool))
        cpool = <ccuda.CUmemoryPool><void_ptr>ppool
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    err = ccuda.cuDeviceSetMemPool(cdev, cpool)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuDeviceGetMemPool(dev):
    """ Gets the current mempool for a device.

    Returns the last pool provided to :py:obj:`~.cuDeviceSetMemPool` for
    this device or the device's default memory pool if
    :py:obj:`~.cuDeviceSetMemPool` has never been called. By default the
    current mempool is the default mempool for a device. Otherwise the
    returned pool must have been set with :py:obj:`~.cuDeviceSetMemPool`.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        None

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pool : :py:obj:`~.CUmemoryPool`
        None

    See Also
    --------
    :py:obj:`~.cuDeviceGetDefaultMemPool`, :py:obj:`~.cuMemPoolCreate`, :py:obj:`~.cuDeviceSetMemPool`
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    cdef CUmemoryPool pool = CUmemoryPool()
    err = ccuda.cuDeviceGetMemPool(<ccuda.CUmemoryPool*>pool._ptr, cdev)
    return (CUresult(err), pool)

@cython.embedsignature(True)
def cuDeviceGetDefaultMemPool(dev):
    """ Returns the default mempool of a device.

    The default mempool of a device contains device memory from that
    device.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        None

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED` :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`
    pool_out : :py:obj:`~.CUmemoryPool`
        None

    See Also
    --------
    :py:obj:`~.cuMemAllocAsync`, :py:obj:`~.cuMemPoolTrimTo`, :py:obj:`~.cuMemPoolGetAttribute`, :py:obj:`~.cuMemPoolSetAttribute`, :py:obj:`~.cuMemPoolSetAccess`, :py:obj:`~.cuDeviceGetMemPool`, :py:obj:`~.cuMemPoolCreate`
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    cdef CUmemoryPool pool_out = CUmemoryPool()
    err = ccuda.cuDeviceGetDefaultMemPool(<ccuda.CUmemoryPool*>pool_out._ptr, cdev)
    return (CUresult(err), pool_out)

@cython.embedsignature(True)
def cuDeviceGetExecAffinitySupport(typename not None : CUexecAffinityType, dev):
    """ Returns information about the execution affinity support of the device.

    Returns in `*pi` whether execution affinity type `typename` is
    supported by device `dev`. The supported types are:

    - :py:obj:`~.CU_EXEC_AFFINITY_TYPE_SM_COUNT`: 1 if context with limited
      SMs is supported by the device, or 0 if not;

    Parameters
    ----------
    typename : :py:obj:`~.CUexecAffinityType`
        Execution affinity type to query
    dev : :py:obj:`~.CUdevice`
        Device handle

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    pi : int
        1 if the execution affinity type `typename` is supported by the
        device, or 0 if not

    See Also
    --------
    :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGetUuid`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceTotalMem`
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    cdef int pi = 0
    cdef ccuda.CUexecAffinityType ctypename = typename.value
    err = ccuda.cuDeviceGetExecAffinitySupport(&pi, ctypename, cdev)
    return (CUresult(err), pi)

@cython.embedsignature(True)
def cuFlushGPUDirectRDMAWrites(target not None : CUflushGPUDirectRDMAWritesTarget, scope not None : CUflushGPUDirectRDMAWritesScope):
    """ Blocks until remote writes are visible to the specified scope.

    Blocks until GPUDirect RDMA writes to the target context via mappings
    created through APIs like nvidia_p2p_get_pages (see
    https://docs.nvidia.com/cuda/gpudirect-rdma for more information), are
    visible to the specified scope.

    If the scope equals or lies within the scope indicated by
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_WRITES_ORDERING`, the
    call will be a no-op and can be safely omitted for performance. This
    can be determined by comparing the numerical values between the two
    enums, with smaller scopes having smaller values.

    Users may query support for this API via
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_FLUSH_FLUSH_GPU_DIRECT_RDMA_OPTIONS`.

    Parameters
    ----------
    target : :py:obj:`~.CUflushGPUDirectRDMAWritesTarget`
        The target of the operation, see
        :py:obj:`~.CUflushGPUDirectRDMAWritesTarget`
    scope : :py:obj:`~.CUflushGPUDirectRDMAWritesScope`
        The scope of the operation, see
        :py:obj:`~.CUflushGPUDirectRDMAWritesScope`

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`,
    """
    cdef ccuda.CUflushGPUDirectRDMAWritesTarget ctarget = target.value
    cdef ccuda.CUflushGPUDirectRDMAWritesScope cscope = scope.value
    err = ccuda.cuFlushGPUDirectRDMAWrites(ctarget, cscope)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuDeviceGetProperties(dev):
    """ Returns properties for a selected device.

    [Deprecated]

    This function was deprecated as of CUDA 5.0 and replaced by
    :py:obj:`~.cuDeviceGetAttribute()`.

    Returns in `*prop` the properties of device `dev`. The
    :py:obj:`~.CUdevprop` structure is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where:

    - :py:obj:`~.maxThreadsPerBlock` is the maximum number of threads per
      block;

    - :py:obj:`~.maxThreadsDim`[3] is the maximum sizes of each dimension
      of a block;

    - :py:obj:`~.maxGridSize`[3] is the maximum sizes of each dimension of
      a grid;

    - :py:obj:`~.sharedMemPerBlock` is the total amount of shared memory
      available per block in bytes;

    - :py:obj:`~.totalConstantMemory` is the total amount of constant
      memory available on the device in bytes;

    - :py:obj:`~.SIMDWidth` is the warp size;

    - :py:obj:`~.memPitch` is the maximum pitch allowed by the memory copy
      functions that involve memory regions allocated through
      :py:obj:`~.cuMemAllocPitch()`;

    - :py:obj:`~.regsPerBlock` is the total number of registers available
      per block;

    - :py:obj:`~.clockRate` is the clock frequency in kilohertz;

    - :py:obj:`~.textureAlign` is the alignment requirement; texture base
      addresses that are aligned to :py:obj:`~.textureAlign` bytes do not
      need an offset applied to texture fetches.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device to get properties for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    prop : :py:obj:`~.CUdevprop`
        Returned properties of device

    See Also
    --------
    :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGetUuid`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceTotalMem`
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    cdef CUdevprop prop = CUdevprop()
    err = ccuda.cuDeviceGetProperties(<ccuda.CUdevprop*>prop._ptr, cdev)
    return (CUresult(err), prop)

@cython.embedsignature(True)
def cuDeviceComputeCapability(dev):
    """ Returns the compute capability of the device.

    [Deprecated]

    This function was deprecated as of CUDA 5.0 and its functionality
    superseded by :py:obj:`~.cuDeviceGetAttribute()`.

    Returns in `*major` and `*minor` the major and minor revision numbers
    that define the compute capability of the device `dev`.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device handle

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    major : int
        Major revision number
    minor : int
        Minor revision number

    See Also
    --------
    :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGetUuid`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceTotalMem`
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    cdef int major = 0
    cdef int minor = 0
    err = ccuda.cuDeviceComputeCapability(&major, &minor, cdev)
    return (CUresult(err), major, minor)

@cython.embedsignature(True)
def cuDevicePrimaryCtxRetain(dev):
    """ Retain the primary context on the GPU.

    Retains the primary context on the device. Once the user successfully
    retains the primary context, the primary context will be active and
    available to the user until the user releases it with
    :py:obj:`~.cuDevicePrimaryCtxRelease()` or resets it with
    :py:obj:`~.cuDevicePrimaryCtxReset()`. Unlike :py:obj:`~.cuCtxCreate()`
    the newly retained context is not pushed onto the stack.

    Retaining the primary context for the first time will fail with
    :py:obj:`~.CUDA_ERROR_UNKNOWN` if the compute mode of the device is
    :py:obj:`~.CU_COMPUTEMODE_PROHIBITED`. The function
    :py:obj:`~.cuDeviceGetAttribute()` can be used with
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_COMPUTE_MODE` to determine the compute
    mode of the device. The `nvidia-smi` tool can be used to set the
    compute mode for devices. Documentation for `nvidia-smi` can be
    obtained by passing a -h option to it.

    Please note that the primary context always supports pinned
    allocations. Other flags can be specified by
    :py:obj:`~.cuDevicePrimaryCtxSetFlags()`.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device for which primary context is requested

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_UNKNOWN`
    pctx : :py:obj:`~.CUcontext`
        Returned context handle of the new context

    See Also
    --------
    :py:obj:`~.cuDevicePrimaryCtxRelease`, :py:obj:`~.cuDevicePrimaryCtxSetFlags`, :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    cdef CUcontext pctx = CUcontext()
    err = ccuda.cuDevicePrimaryCtxRetain(<ccuda.CUcontext*>pctx._ptr, cdev)
    return (CUresult(err), pctx)

@cython.embedsignature(True)
def cuDevicePrimaryCtxRelease(dev):
    """ Release the primary context on the GPU.

    Releases the primary context interop on the device. A retained context
    should always be released once the user is done using it. The context
    is automatically reset once the last reference to it is released. This
    behavior is different when the primary context was retained by the CUDA
    runtime from CUDA 4.0 and earlier. In this case, the primary context
    remains always active.

    Releasing a primary context that has not been previously retained will
    fail with :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`.

    Please note that unlike :py:obj:`~.cuCtxDestroy()` this method does not
    pop the context from stack in any circumstances.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device which primary context is released

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`

    See Also
    --------
    :py:obj:`~.cuDevicePrimaryCtxRetain`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    err = ccuda.cuDevicePrimaryCtxRelease(cdev)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuDevicePrimaryCtxSetFlags(dev, unsigned int flags):
    """ Set flags for the primary context.

    Sets the flags for the primary context on the device overwriting
    perviously set ones.

    The three LSBs of the `flags` parameter can be used to control how the
    OS thread, which owns the CUDA context at the time of an API call,
    interacts with the OS scheduler when waiting for results from the GPU.
    Only one of the scheduling flags can be set when creating a context.

    - :py:obj:`~.CU_CTX_SCHED_SPIN`: Instruct CUDA to actively spin when
      waiting for results from the GPU. This can decrease latency when
      waiting for the GPU, but may lower the performance of CPU threads if
      they are performing work in parallel with the CUDA thread.

    - :py:obj:`~.CU_CTX_SCHED_YIELD`: Instruct CUDA to yield its thread
      when waiting for results from the GPU. This can increase latency when
      waiting for the GPU, but can increase the performance of CPU threads
      performing work in parallel with the GPU.

    - :py:obj:`~.CU_CTX_SCHED_BLOCKING_SYNC`: Instruct CUDA to block the
      CPU thread on a synchronization primitive when waiting for the GPU to
      finish work.

    - :py:obj:`~.CU_CTX_BLOCKING_SYNC`: Instruct CUDA to block the CPU
      thread on a synchronization primitive when waiting for the GPU to
      finish work.   Deprecated: This flag was deprecated as of CUDA 4.0
      and was replaced with :py:obj:`~.CU_CTX_SCHED_BLOCKING_SYNC`.

    - :py:obj:`~.CU_CTX_SCHED_AUTO`: The default value if the `flags`
      parameter is zero, uses a heuristic based on the number of active
      CUDA contexts in the process `C` and the number of logical processors
      in the system `P`. If `C` > `P`, then CUDA will yield to other OS
      threads when waiting for the GPU (:py:obj:`~.CU_CTX_SCHED_YIELD`),
      otherwise CUDA will not yield while waiting for results and actively
      spin on the processor (:py:obj:`~.CU_CTX_SCHED_SPIN`). Additionally,
      on Tegra devices, :py:obj:`~.CU_CTX_SCHED_AUTO` uses a heuristic
      based on the power profile of the platform and may choose
      :py:obj:`~.CU_CTX_SCHED_BLOCKING_SYNC` for low-powered devices.

    - :py:obj:`~.CU_CTX_LMEM_RESIZE_TO_MAX`: Instruct CUDA to not reduce
      local memory after resizing local memory for a kernel. This can
      prevent thrashing by local memory allocations when launching many
      kernels with high local memory usage at the cost of potentially
      increased memory usage.   Deprecated: This flag is deprecated and the
      behavior enabled by this flag is now the default and cannot be
      disabled.

    - :py:obj:`~.CU_CTX_COREDUMP_ENABLE`: If GPU coredumps have not been
      enabled globally with :py:obj:`~.cuCoredumpSetAttributeGlobal` or
      environment variables, this flag can be set during context creation
      to instruct CUDA to create a coredump if this context raises an
      exception during execution. These environment variables are described
      in the CUDA-GDB user guide under the "GPU core dump support" section.
      The initial settings will be taken from the global settings at the
      time of context creation. The other settings that control coredump
      output can be modified by calling :py:obj:`~.cuCoredumpSetAttribute`
      from the created context after it becomes current.

    - :py:obj:`~.CU_CTX_USER_COREDUMP_ENABLE`: If user-triggered GPU
      coredumps have not been enabled globally with
      :py:obj:`~.cuCoredumpSetAttributeGlobal` or environment variables,
      this flag can be set during context creation to instruct CUDA to
      create a coredump if data is written to a certain pipe that is
      present in the OS space. These environment variables are described in
      the CUDA-GDB user guide under the "GPU core dump support" section. It
      is important to note that the pipe name `must` be set with
      :py:obj:`~.cuCoredumpSetAttributeGlobal` before creating the context
      if this flag is used. Setting this flag implies that
      :py:obj:`~.CU_CTX_COREDUMP_ENABLE` is set. The initial settings will
      be taken from the global settings at the time of context creation.
      The other settings that control coredump output can be modified by
      calling :py:obj:`~.cuCoredumpSetAttribute` from the created context
      after it becomes current.

    - :py:obj:`~.CU_CTX_SYNC_MEMOPS`: Ensures that synchronous memory
      operations initiated on this context will always synchronize. See
      further documentation in the section titled "API Synchronization
      behavior" to learn more about cases when synchronous memory
      operations can exhibit asynchronous behavior.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device for which the primary context flags are set
    flags : unsigned int
        New flags for the device

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`,

    See Also
    --------
    :py:obj:`~.cuDevicePrimaryCtxRetain`, :py:obj:`~.cuDevicePrimaryCtxGetState`, :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxSetFlags`, :py:obj:`~.cudaSetDeviceFlags`
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    err = ccuda.cuDevicePrimaryCtxSetFlags(cdev, flags)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuDevicePrimaryCtxGetState(dev):
    """ Get the state of the primary context.

    Returns in `*flags` the flags for the primary context of `dev`, and in
    `*active` whether it is active. See
    :py:obj:`~.cuDevicePrimaryCtxSetFlags` for flag values.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device to get primary context flags for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`,
    flags : unsigned int
        Pointer to store flags
    active : int
        Pointer to store context state; 0 = inactive, 1 = active

    See Also
    --------
    :py:obj:`~.cuDevicePrimaryCtxSetFlags`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxSetFlags`, :py:obj:`~.cudaGetDeviceFlags`
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    cdef unsigned int flags = 0
    cdef int active = 0
    err = ccuda.cuDevicePrimaryCtxGetState(cdev, &flags, &active)
    return (CUresult(err), flags, active)

@cython.embedsignature(True)
def cuDevicePrimaryCtxReset(dev):
    """ Destroy all allocations and reset all state on the primary context.

    Explicitly destroys and cleans up all resources associated with the
    current device in the current process.

    Note that it is responsibility of the calling function to ensure that
    no other module in the process is using the device any more. For that
    reason it is recommended to use :py:obj:`~.cuDevicePrimaryCtxRelease()`
    in most cases. However it is safe for other modules to call
    :py:obj:`~.cuDevicePrimaryCtxRelease()` even after resetting the
    device. Resetting the primary context does not release it, an
    application that has retained the primary context should explicitly
    release its usage.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device for which primary context is destroyed

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_PRIMARY_CONTEXT_ACTIVE`

    See Also
    --------
    :py:obj:`~.cuDevicePrimaryCtxRetain`, :py:obj:`~.cuDevicePrimaryCtxRelease`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`, :py:obj:`~.cudaDeviceReset`
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    err = ccuda.cuDevicePrimaryCtxReset(cdev)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuCtxCreate(unsigned int flags, dev):
    """ Create a CUDA context.

    Creates a new CUDA context and associates it with the calling thread.
    The `flags` parameter is described below. The context is created with a
    usage count of 1 and the caller of :py:obj:`~.cuCtxCreate()` must call
    :py:obj:`~.cuCtxDestroy()` when done using the context. If a context is
    already current to the thread, it is supplanted by the newly created
    context and may be restored by a subsequent call to
    :py:obj:`~.cuCtxPopCurrent()`.

    The three LSBs of the `flags` parameter can be used to control how the
    OS thread, which owns the CUDA context at the time of an API call,
    interacts with the OS scheduler when waiting for results from the GPU.
    Only one of the scheduling flags can be set when creating a context.

    - :py:obj:`~.CU_CTX_SCHED_SPIN`: Instruct CUDA to actively spin when
      waiting for results from the GPU. This can decrease latency when
      waiting for the GPU, but may lower the performance of CPU threads if
      they are performing work in parallel with the CUDA thread.

    - :py:obj:`~.CU_CTX_SCHED_YIELD`: Instruct CUDA to yield its thread
      when waiting for results from the GPU. This can increase latency when
      waiting for the GPU, but can increase the performance of CPU threads
      performing work in parallel with the GPU.

    - :py:obj:`~.CU_CTX_SCHED_BLOCKING_SYNC`: Instruct CUDA to block the
      CPU thread on a synchronization primitive when waiting for the GPU to
      finish work.

    - :py:obj:`~.CU_CTX_BLOCKING_SYNC`: Instruct CUDA to block the CPU
      thread on a synchronization primitive when waiting for the GPU to
      finish work.   Deprecated: This flag was deprecated as of CUDA 4.0
      and was replaced with :py:obj:`~.CU_CTX_SCHED_BLOCKING_SYNC`.

    - :py:obj:`~.CU_CTX_SCHED_AUTO`: The default value if the `flags`
      parameter is zero, uses a heuristic based on the number of active
      CUDA contexts in the process `C` and the number of logical processors
      in the system `P`. If `C` > `P`, then CUDA will yield to other OS
      threads when waiting for the GPU (:py:obj:`~.CU_CTX_SCHED_YIELD`),
      otherwise CUDA will not yield while waiting for results and actively
      spin on the processor (:py:obj:`~.CU_CTX_SCHED_SPIN`). Additionally,
      on Tegra devices, :py:obj:`~.CU_CTX_SCHED_AUTO` uses a heuristic
      based on the power profile of the platform and may choose
      :py:obj:`~.CU_CTX_SCHED_BLOCKING_SYNC` for low-powered devices.

    - :py:obj:`~.CU_CTX_MAP_HOST`: Instruct CUDA to support mapped pinned
      allocations. This flag must be set in order to allocate pinned host
      memory that is accessible to the GPU.

    - :py:obj:`~.CU_CTX_LMEM_RESIZE_TO_MAX`: Instruct CUDA to not reduce
      local memory after resizing local memory for a kernel. This can
      prevent thrashing by local memory allocations when launching many
      kernels with high local memory usage at the cost of potentially
      increased memory usage.   Deprecated: This flag is deprecated and the
      behavior enabled by this flag is now the default and cannot be
      disabled. Instead, the per-thread stack size can be controlled with
      :py:obj:`~.cuCtxSetLimit()`.

    - :py:obj:`~.CU_CTX_COREDUMP_ENABLE`: If GPU coredumps have not been
      enabled globally with :py:obj:`~.cuCoredumpSetAttributeGlobal` or
      environment variables, this flag can be set during context creation
      to instruct CUDA to create a coredump if this context raises an
      exception during execution. These environment variables are described
      in the CUDA-GDB user guide under the "GPU core dump support" section.
      The initial attributes will be taken from the global attributes at
      the time of context creation. The other attributes that control
      coredump output can be modified by calling
      :py:obj:`~.cuCoredumpSetAttribute` from the created context after it
      becomes current.

    - :py:obj:`~.CU_CTX_USER_COREDUMP_ENABLE`: If user-triggered GPU
      coredumps have not been enabled globally with
      :py:obj:`~.cuCoredumpSetAttributeGlobal` or environment variables,
      this flag can be set during context creation to instruct CUDA to
      create a coredump if data is written to a certain pipe that is
      present in the OS space. These environment variables are described in
      the CUDA-GDB user guide under the "GPU core dump support" section. It
      is important to note that the pipe name `must` be set with
      :py:obj:`~.cuCoredumpSetAttributeGlobal` before creating the context
      if this flag is used. Setting this flag implies that
      :py:obj:`~.CU_CTX_COREDUMP_ENABLE` is set. The initial attributes
      will be taken from the global attributes at the time of context
      creation. The other attributes that control coredump output can be
      modified by calling :py:obj:`~.cuCoredumpSetAttribute` from the
      created context after it becomes current. Setting this flag on any
      context creation is equivalent to setting the
      :py:obj:`~.CU_COREDUMP_ENABLE_USER_TRIGGER` attribute to `true`
      globally.

    - :py:obj:`~.CU_CTX_SYNC_MEMOPS`: Ensures that synchronous memory
      operations initiated on this context will always synchronize. See
      further documentation in the section titled "API Synchronization
      behavior" to learn more about cases when synchronous memory
      operations can exhibit asynchronous behavior.

    Context creation will fail with :py:obj:`~.CUDA_ERROR_UNKNOWN` if the
    compute mode of the device is :py:obj:`~.CU_COMPUTEMODE_PROHIBITED`.
    The function :py:obj:`~.cuDeviceGetAttribute()` can be used with
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_COMPUTE_MODE` to determine the compute
    mode of the device. The `nvidia-smi` tool can be used to set the
    compute mode for * devices. Documentation for `nvidia-smi` can be
    obtained by passing a -h option to it.

    Parameters
    ----------
    flags : unsigned int
        Context creation flags
    dev : :py:obj:`~.CUdevice`
        Device to create context on

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_UNKNOWN`
    pctx : :py:obj:`~.CUcontext`
        Returned context handle of the new context

    See Also
    --------
    :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCoredumpSetAttributeGlobal`, :py:obj:`~.cuCoredumpSetAttribute`, :py:obj:`~.cuCtxSynchronize`

    Notes
    -----
    In most cases it is recommended to use :py:obj:`~.cuDevicePrimaryCtxRetain`.
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    cdef CUcontext pctx = CUcontext()
    err = ccuda.cuCtxCreate(<ccuda.CUcontext*>pctx._ptr, flags, cdev)
    return (CUresult(err), pctx)

@cython.embedsignature(True)
def cuCtxCreate_v3(paramsArray : Optional[List[CUexecAffinityParam]], int numParams, unsigned int flags, dev):
    """ Create a CUDA context with execution affinity.

    Creates a new CUDA context with execution affinity and associates it
    with the calling thread. The `paramsArray` and `flags` parameter are
    described below. The context is created with a usage count of 1 and the
    caller of :py:obj:`~.cuCtxCreate()` must call
    :py:obj:`~.cuCtxDestroy()` when done using the context. If a context is
    already current to the thread, it is supplanted by the newly created
    context and may be restored by a subsequent call to
    :py:obj:`~.cuCtxPopCurrent()`.

    The type and the amount of execution resource the context can use is
    limited by `paramsArray` and `numParams`. The `paramsArray` is an array
    of `CUexecAffinityParam` and the `numParams` describes the size of the
    array. If two `CUexecAffinityParam` in the array have the same type,
    the latter execution affinity parameter overrides the former execution
    affinity parameter. The supported execution affinity types are:

    - :py:obj:`~.CU_EXEC_AFFINITY_TYPE_SM_COUNT` limits the portion of SMs
      that the context can use. The portion of SMs is specified as the
      number of SMs via `CUexecAffinitySmCount`. This limit will be
      internally rounded up to the next hardware-supported amount. Hence,
      it is imperative to query the actual execution affinity of the
      context via `cuCtxGetExecAffinity` after context creation. Currently,
      this attribute is only supported under Volta+ MPS.

    The three LSBs of the `flags` parameter can be used to control how the
    OS thread, which owns the CUDA context at the time of an API call,
    interacts with the OS scheduler when waiting for results from the GPU.
    Only one of the scheduling flags can be set when creating a context.

    - :py:obj:`~.CU_CTX_SCHED_SPIN`: Instruct CUDA to actively spin when
      waiting for results from the GPU. This can decrease latency when
      waiting for the GPU, but may lower the performance of CPU threads if
      they are performing work in parallel with the CUDA thread.

    - :py:obj:`~.CU_CTX_SCHED_YIELD`: Instruct CUDA to yield its thread
      when waiting for results from the GPU. This can increase latency when
      waiting for the GPU, but can increase the performance of CPU threads
      performing work in parallel with the GPU.

    - :py:obj:`~.CU_CTX_SCHED_BLOCKING_SYNC`: Instruct CUDA to block the
      CPU thread on a synchronization primitive when waiting for the GPU to
      finish work.

    - :py:obj:`~.CU_CTX_BLOCKING_SYNC`: Instruct CUDA to block the CPU
      thread on a synchronization primitive when waiting for the GPU to
      finish work.   Deprecated: This flag was deprecated as of CUDA 4.0
      and was replaced with :py:obj:`~.CU_CTX_SCHED_BLOCKING_SYNC`.

    - :py:obj:`~.CU_CTX_SCHED_AUTO`: The default value if the `flags`
      parameter is zero, uses a heuristic based on the number of active
      CUDA contexts in the process `C` and the number of logical processors
      in the system `P`. If `C` > `P`, then CUDA will yield to other OS
      threads when waiting for the GPU (:py:obj:`~.CU_CTX_SCHED_YIELD`),
      otherwise CUDA will not yield while waiting for results and actively
      spin on the processor (:py:obj:`~.CU_CTX_SCHED_SPIN`). Additionally,
      on Tegra devices, :py:obj:`~.CU_CTX_SCHED_AUTO` uses a heuristic
      based on the power profile of the platform and may choose
      :py:obj:`~.CU_CTX_SCHED_BLOCKING_SYNC` for low-powered devices.

    - :py:obj:`~.CU_CTX_MAP_HOST`: Instruct CUDA to support mapped pinned
      allocations. This flag must be set in order to allocate pinned host
      memory that is accessible to the GPU.

    - :py:obj:`~.CU_CTX_LMEM_RESIZE_TO_MAX`: Instruct CUDA to not reduce
      local memory after resizing local memory for a kernel. This can
      prevent thrashing by local memory allocations when launching many
      kernels with high local memory usage at the cost of potentially
      increased memory usage.   Deprecated: This flag is deprecated and the
      behavior enabled by this flag is now the default and cannot be
      disabled. Instead, the per-thread stack size can be controlled with
      :py:obj:`~.cuCtxSetLimit()`.

    - :py:obj:`~.CU_CTX_COREDUMP_ENABLE`: If GPU coredumps have not been
      enabled globally with :py:obj:`~.cuCoredumpSetAttributeGlobal` or
      environment variables, this flag can be set during context creation
      to instruct CUDA to create a coredump if this context raises an
      exception during execution. These environment variables are described
      in the CUDA-GDB user guide under the "GPU core dump support" section.
      The initial attributes will be taken from the global attributes at
      the time of context creation. The other attributes that control
      coredump output can be modified by calling
      :py:obj:`~.cuCoredumpSetAttribute` from the created context after it
      becomes current.

    - :py:obj:`~.CU_CTX_USER_COREDUMP_ENABLE`: If user-triggered GPU
      coredumps have not been enabled globally with
      :py:obj:`~.cuCoredumpSetAttributeGlobal` or environment variables,
      this flag can be set during context creation to instruct CUDA to
      create a coredump if data is written to a certain pipe that is
      present in the OS space. These environment variables are described in
      the CUDA-GDB user guide under the "GPU core dump support" section. It
      is important to note that the pipe name `must` be set with
      :py:obj:`~.cuCoredumpSetAttributeGlobal` before creating the context
      if this flag is used. Setting this flag implies that
      :py:obj:`~.CU_CTX_COREDUMP_ENABLE` is set. The initial attributes
      will be taken from the global attributes at the time of context
      creation. The other attributes that control coredump output can be
      modified by calling :py:obj:`~.cuCoredumpSetAttribute` from the
      created context after it becomes current. Setting this flag on any
      context creation is equivalent to setting the
      :py:obj:`~.CU_COREDUMP_ENABLE_USER_TRIGGER` attribute to `true`
      globally.

    Context creation will fail with :py:obj:`~.CUDA_ERROR_UNKNOWN` if the
    compute mode of the device is :py:obj:`~.CU_COMPUTEMODE_PROHIBITED`.
    The function :py:obj:`~.cuDeviceGetAttribute()` can be used with
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_COMPUTE_MODE` to determine the compute
    mode of the device. The `nvidia-smi` tool can be used to set the
    compute mode for * devices. Documentation for `nvidia-smi` can be
    obtained by passing a -h option to it.

    Parameters
    ----------
    paramsArray : List[:py:obj:`~.CUexecAffinityParam`]
        Execution affinity parameters
    numParams : int
        Number of execution affinity parameters
    flags : unsigned int
        Context creation flags
    dev : :py:obj:`~.CUdevice`
        Device to create context on

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_EXEC_AFFINITY`, :py:obj:`~.CUDA_ERROR_UNKNOWN`
    pctx : :py:obj:`~.CUcontext`
        Returned context handle of the new context

    See Also
    --------
    :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`, :py:obj:`~.cuCoredumpSetAttributeGlobal`, :py:obj:`~.cuCoredumpSetAttribute`, :py:obj:`~.CUexecAffinityParam`
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    paramsArray = [] if paramsArray is None else paramsArray
    if not all(isinstance(_x, (CUexecAffinityParam,)) for _x in paramsArray):
        raise TypeError("Argument 'paramsArray' is not instance of type (expected List[ccuda.CUexecAffinityParam,]")
    cdef CUcontext pctx = CUcontext()
    cdef ccuda.CUexecAffinityParam* cparamsArray = NULL
    if len(paramsArray) > 0:
        cparamsArray = <ccuda.CUexecAffinityParam*> calloc(len(paramsArray), sizeof(ccuda.CUexecAffinityParam))
        if cparamsArray is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(paramsArray)) + 'x' + str(sizeof(ccuda.CUexecAffinityParam)))
        for idx in range(len(paramsArray)):
            string.memcpy(&cparamsArray[idx], (<CUexecAffinityParam>paramsArray[idx])._ptr, sizeof(ccuda.CUexecAffinityParam))
    err = ccuda.cuCtxCreate_v3(<ccuda.CUcontext*>pctx._ptr, (<CUexecAffinityParam>paramsArray[0])._ptr if len(paramsArray) == 1 else cparamsArray, numParams, flags, cdev)
    if cparamsArray is not NULL:
        free(cparamsArray)
    return (CUresult(err), pctx)

@cython.embedsignature(True)
def cuCtxDestroy(ctx):
    """ Destroy a CUDA context.

    Destroys the CUDA context specified by `ctx`. The context `ctx` will be
    destroyed regardless of how many threads it is current to. It is the
    responsibility of the calling function to ensure that no API call
    issues using `ctx` while :py:obj:`~.cuCtxDestroy()` is executing.

    Destroys and cleans up all resources associated with the context. It is
    the caller's responsibility to ensure that the context or its resources
    are not accessed or passed in subsequent API calls and doing so will
    result in undefined behavior. These resources include CUDA types such
    as :py:obj:`~.CUmodule`, :py:obj:`~.CUfunction`, :py:obj:`~.CUstream`,
    :py:obj:`~.CUevent`, :py:obj:`~.CUarray`, :py:obj:`~.CUmipmappedArray`,
    :py:obj:`~.CUtexObject`, :py:obj:`~.CUsurfObject`,
    :py:obj:`~.CUtexref`, :py:obj:`~.CUsurfref`,
    :py:obj:`~.CUgraphicsResource`, :py:obj:`~.CUlinkState`,
    :py:obj:`~.CUexternalMemory` and :py:obj:`~.CUexternalSemaphore`.

    If `ctx` is current to the calling thread then `ctx` will also be
    popped from the current thread's context stack (as though
    :py:obj:`~.cuCtxPopCurrent()` were called). If `ctx` is current to
    other threads, then `ctx` will remain current to those threads, and
    attempting to access `ctx` from those threads will result in the error
    :py:obj:`~.CUDA_ERROR_CONTEXT_IS_DESTROYED`.

    Parameters
    ----------
    ctx : :py:obj:`~.CUcontext`
        Context to destroy

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`
    """
    cdef ccuda.CUcontext cctx
    if ctx is None:
        cctx = <ccuda.CUcontext><void_ptr>0
    elif isinstance(ctx, (CUcontext,)):
        pctx = int(ctx)
        cctx = <ccuda.CUcontext><void_ptr>pctx
    else:
        pctx = int(CUcontext(ctx))
        cctx = <ccuda.CUcontext><void_ptr>pctx
    err = ccuda.cuCtxDestroy(cctx)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuCtxPushCurrent(ctx):
    """ Pushes a context on the current CPU thread.

    Pushes the given context `ctx` onto the CPU thread's stack of current
    contexts. The specified context becomes the CPU thread's current
    context, so all CUDA functions that operate on the current context are
    affected.

    The previous current context may be made current again by calling
    :py:obj:`~.cuCtxDestroy()` or :py:obj:`~.cuCtxPopCurrent()`.

    Parameters
    ----------
    ctx : :py:obj:`~.CUcontext`
        Context to push

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`
    """
    cdef ccuda.CUcontext cctx
    if ctx is None:
        cctx = <ccuda.CUcontext><void_ptr>0
    elif isinstance(ctx, (CUcontext,)):
        pctx = int(ctx)
        cctx = <ccuda.CUcontext><void_ptr>pctx
    else:
        pctx = int(CUcontext(ctx))
        cctx = <ccuda.CUcontext><void_ptr>pctx
    err = ccuda.cuCtxPushCurrent(cctx)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuCtxPopCurrent():
    """ Pops the current CUDA context from the current CPU thread.

    Pops the current CUDA context from the CPU thread and passes back the
    old context handle in `*pctx`. That context may then be made current to
    a different CPU thread by calling :py:obj:`~.cuCtxPushCurrent()`.

    If a context was current to the CPU thread before
    :py:obj:`~.cuCtxCreate()` or :py:obj:`~.cuCtxPushCurrent()` was called,
    this function makes that context current to the CPU thread again.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`
    pctx : :py:obj:`~.CUcontext`
        Returned popped context handle

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`
    """
    cdef CUcontext pctx = CUcontext()
    err = ccuda.cuCtxPopCurrent(<ccuda.CUcontext*>pctx._ptr)
    return (CUresult(err), pctx)

@cython.embedsignature(True)
def cuCtxSetCurrent(ctx):
    """ Binds the specified CUDA context to the calling CPU thread.

    Binds the specified CUDA context to the calling CPU thread. If `ctx` is
    NULL then the CUDA context previously bound to the calling CPU thread
    is unbound and :py:obj:`~.CUDA_SUCCESS` is returned.

    If there exists a CUDA context stack on the calling CPU thread, this
    will replace the top of that stack with `ctx`. If `ctx` is NULL then
    this will be equivalent to popping the top of the calling CPU thread's
    CUDA context stack (or a no-op if the calling CPU thread's CUDA context
    stack is empty).

    Parameters
    ----------
    ctx : :py:obj:`~.CUcontext`
        Context to bind to the calling CPU thread

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`

    See Also
    --------
    :py:obj:`~.cuCtxGetCurrent`, :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cudaSetDevice`
    """
    cdef ccuda.CUcontext cctx
    if ctx is None:
        cctx = <ccuda.CUcontext><void_ptr>0
    elif isinstance(ctx, (CUcontext,)):
        pctx = int(ctx)
        cctx = <ccuda.CUcontext><void_ptr>pctx
    else:
        pctx = int(CUcontext(ctx))
        cctx = <ccuda.CUcontext><void_ptr>pctx
    err = ccuda.cuCtxSetCurrent(cctx)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuCtxGetCurrent():
    """ Returns the CUDA context bound to the calling CPU thread.

    Returns in `*pctx` the CUDA context bound to the calling CPU thread. If
    no context is bound to the calling CPU thread then `*pctx` is set to
    NULL and :py:obj:`~.CUDA_SUCCESS` is returned.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`,
    pctx : :py:obj:`~.CUcontext`
        Returned context handle

    See Also
    --------
    :py:obj:`~.cuCtxSetCurrent`, :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cudaGetDevice`
    """
    cdef CUcontext pctx = CUcontext()
    err = ccuda.cuCtxGetCurrent(<ccuda.CUcontext*>pctx._ptr)
    return (CUresult(err), pctx)

@cython.embedsignature(True)
def cuCtxGetDevice():
    """ Returns the device ID for the current context.

    Returns in `*device` the ordinal of the current context's device.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`,
    device : :py:obj:`~.CUdevice`
        Returned device ID for the current context

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`, :py:obj:`~.cudaGetDevice`
    """
    cdef CUdevice device = CUdevice()
    err = ccuda.cuCtxGetDevice(<ccuda.CUdevice*>device._ptr)
    return (CUresult(err), device)

@cython.embedsignature(True)
def cuCtxGetFlags():
    """ Returns the flags for the current context.

    Returns in `*flags` the flags of the current context. See
    :py:obj:`~.cuCtxCreate` for flag values.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`,
    flags : unsigned int
        Pointer to store flags of current context

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetCurrent`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxGetSharedMemConfig`, :py:obj:`~.cuCtxGetStreamPriorityRange`, :py:obj:`~.cuCtxSetFlags`, :py:obj:`~.cudaGetDeviceFlags`
    """
    cdef unsigned int flags = 0
    err = ccuda.cuCtxGetFlags(&flags)
    return (CUresult(err), flags)

@cython.embedsignature(True)
def cuCtxSetFlags(unsigned int flags):
    """ Sets the flags for the current context.

    Sets the flags for the current context overwriting previously set ones.
    See :py:obj:`~.cuDevicePrimaryCtxSetFlags` for flag values.

    Parameters
    ----------
    flags : unsigned int
        Flags to set on the current context

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`,

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetCurrent`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxGetSharedMemConfig`, :py:obj:`~.cuCtxGetStreamPriorityRange`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cudaGetDeviceFlags`, :py:obj:`~.cuDevicePrimaryCtxSetFlags`,
    """
    err = ccuda.cuCtxSetFlags(flags)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuCtxGetId(ctx):
    """ Returns the unique Id associated with the context supplied.

    Returns in `ctxId` the unique Id which is associated with a given
    context. The Id is unique for the life of the program for this instance
    of CUDA. If context is supplied as NULL and there is one current, the
    Id of the current context is returned.

    Parameters
    ----------
    ctx : :py:obj:`~.CUcontext`
        Context for which to obtain the Id

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_CONTEXT_IS_DESTROYED`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    ctxId : unsigned long long
        Pointer to store the Id of the context

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPushCurrent`
    """
    cdef ccuda.CUcontext cctx
    if ctx is None:
        cctx = <ccuda.CUcontext><void_ptr>0
    elif isinstance(ctx, (CUcontext,)):
        pctx = int(ctx)
        cctx = <ccuda.CUcontext><void_ptr>pctx
    else:
        pctx = int(CUcontext(ctx))
        cctx = <ccuda.CUcontext><void_ptr>pctx
    cdef unsigned long long ctxId = 0
    err = ccuda.cuCtxGetId(cctx, &ctxId)
    return (CUresult(err), ctxId)

@cython.embedsignature(True)
def cuCtxSynchronize():
    """ Block for a context's tasks to complete.

    Blocks until the device has completed all preceding requested tasks.
    :py:obj:`~.cuCtxSynchronize()` returns an error if one of the preceding
    tasks failed. If the context was created with the
    :py:obj:`~.CU_CTX_SCHED_BLOCKING_SYNC` flag, the CPU thread will block
    until the GPU context has finished its work.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cudaDeviceSynchronize`
    """
    err = ccuda.cuCtxSynchronize()
    return (CUresult(err),)

@cython.embedsignature(True)
def cuCtxSetLimit(limit not None : CUlimit, size_t value):
    """ Set resource limits.

    Setting `limit` to `value` is a request by the application to update
    the current limit maintained by the context. The driver is free to
    modify the requested value to meet h/w requirements (this could be
    clamping to minimum or maximum values, rounding up to nearest element
    size, etc). The application can use :py:obj:`~.cuCtxGetLimit()` to find
    out exactly what the limit has been set to.

    Setting each :py:obj:`~.CUlimit` has its own specific restrictions, so
    each is discussed here.

    - :py:obj:`~.CU_LIMIT_STACK_SIZE` controls the stack size in bytes of
      each GPU thread. The driver automatically increases the per-thread
      stack size for each kernel launch as needed. This size isn't reset
      back to the original value after each launch. Setting this value will
      take effect immediately, and if necessary, the device will block
      until all preceding requested tasks are complete.

    - :py:obj:`~.CU_LIMIT_PRINTF_FIFO_SIZE` controls the size in bytes of
      the FIFO used by the :py:obj:`~.printf()` device system call. Setting
      :py:obj:`~.CU_LIMIT_PRINTF_FIFO_SIZE` must be performed before
      launching any kernel that uses the :py:obj:`~.printf()` device system
      call, otherwise :py:obj:`~.CUDA_ERROR_INVALID_VALUE` will be
      returned.

    - :py:obj:`~.CU_LIMIT_MALLOC_HEAP_SIZE` controls the size in bytes of
      the heap used by the :py:obj:`~.malloc()` and :py:obj:`~.free()`
      device system calls. Setting :py:obj:`~.CU_LIMIT_MALLOC_HEAP_SIZE`
      must be performed before launching any kernel that uses the
      :py:obj:`~.malloc()` or :py:obj:`~.free()` device system calls,
      otherwise :py:obj:`~.CUDA_ERROR_INVALID_VALUE` will be returned.

    - :py:obj:`~.CU_LIMIT_DEV_RUNTIME_SYNC_DEPTH` controls the maximum
      nesting depth of a grid at which a thread can safely call
      :py:obj:`~.cudaDeviceSynchronize()`. Setting this limit must be
      performed before any launch of a kernel that uses the device runtime
      and calls :py:obj:`~.cudaDeviceSynchronize()` above the default sync
      depth, two levels of grids. Calls to
      :py:obj:`~.cudaDeviceSynchronize()` will fail with error code
      :py:obj:`~.cudaErrorSyncDepthExceeded` if the limitation is violated.
      This limit can be set smaller than the default or up the maximum
      launch depth of 24. When setting this limit, keep in mind that
      additional levels of sync depth require the driver to reserve large
      amounts of device memory which can no longer be used for user
      allocations. If these reservations of device memory fail,
      :py:obj:`~.cuCtxSetLimit()` will return
      :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, and the limit can be reset to a
      lower value. This limit is only applicable to devices of compute
      capability < 9.0. Attempting to set this limit on devices of other
      compute capability versions will result in the error
      :py:obj:`~.CUDA_ERROR_UNSUPPORTED_LIMIT` being returned.

    - :py:obj:`~.CU_LIMIT_DEV_RUNTIME_PENDING_LAUNCH_COUNT` controls the
      maximum number of outstanding device runtime launches that can be
      made from the current context. A grid is outstanding from the point
      of launch up until the grid is known to have been completed. Device
      runtime launches which violate this limitation fail and return
      :py:obj:`~.cudaErrorLaunchPendingCountExceeded` when
      :py:obj:`~.cudaGetLastError()` is called after launch. If more
      pending launches than the default (2048 launches) are needed for a
      module using the device runtime, this limit can be increased. Keep in
      mind that being able to sustain additional pending launches will
      require the driver to reserve larger amounts of device memory upfront
      which can no longer be used for allocations. If these reservations
      fail, :py:obj:`~.cuCtxSetLimit()` will return
      :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, and the limit can be reset to a
      lower value. This limit is only applicable to devices of compute
      capability 3.5 and higher. Attempting to set this limit on devices of
      compute capability less than 3.5 will result in the error
      :py:obj:`~.CUDA_ERROR_UNSUPPORTED_LIMIT` being returned.

    - :py:obj:`~.CU_LIMIT_MAX_L2_FETCH_GRANULARITY` controls the L2 cache
      fetch granularity. Values can range from 0B to 128B. This is purely a
      performance hint and it can be ignored or clamped depending on the
      platform.

    - :py:obj:`~.CU_LIMIT_PERSISTING_L2_CACHE_SIZE` controls size in bytes
      available for persisting L2 cache. This is purely a performance hint
      and it can be ignored or clamped depending on the platform.

    Parameters
    ----------
    limit : :py:obj:`~.CUlimit`
        Limit to set
    value : size_t
        Size of limit

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_LIMIT`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSynchronize`, :py:obj:`~.cudaDeviceSetLimit`
    """
    cdef ccuda.CUlimit climit = limit.value
    err = ccuda.cuCtxSetLimit(climit, value)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuCtxGetLimit(limit not None : CUlimit):
    """ Returns resource limits.

    Returns in `*pvalue` the current size of `limit`. The supported
    :py:obj:`~.CUlimit` values are:

    - :py:obj:`~.CU_LIMIT_STACK_SIZE`: stack size in bytes of each GPU
      thread.

    - :py:obj:`~.CU_LIMIT_PRINTF_FIFO_SIZE`: size in bytes of the FIFO used
      by the :py:obj:`~.printf()` device system call.

    - :py:obj:`~.CU_LIMIT_MALLOC_HEAP_SIZE`: size in bytes of the heap used
      by the :py:obj:`~.malloc()` and :py:obj:`~.free()` device system
      calls.

    - :py:obj:`~.CU_LIMIT_DEV_RUNTIME_SYNC_DEPTH`: maximum grid depth at
      which a thread can issue the device runtime call
      :py:obj:`~.cudaDeviceSynchronize()` to wait on child grid launches to
      complete.

    - :py:obj:`~.CU_LIMIT_DEV_RUNTIME_PENDING_LAUNCH_COUNT`: maximum number
      of outstanding device runtime launches that can be made from this
      context.

    - :py:obj:`~.CU_LIMIT_MAX_L2_FETCH_GRANULARITY`: L2 cache fetch
      granularity.

    - :py:obj:`~.CU_LIMIT_PERSISTING_L2_CACHE_SIZE`: Persisting L2 cache
      size in bytes

    Parameters
    ----------
    limit : :py:obj:`~.CUlimit`
        Limit to query

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_LIMIT`
    pvalue : int
        Returned size of limit

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`, :py:obj:`~.cudaDeviceGetLimit`
    """
    cdef size_t pvalue = 0
    cdef ccuda.CUlimit climit = limit.value
    err = ccuda.cuCtxGetLimit(&pvalue, climit)
    return (CUresult(err), pvalue)

@cython.embedsignature(True)
def cuCtxGetCacheConfig():
    """ Returns the preferred cache configuration for the current context.

    On devices where the L1 cache and shared memory use the same hardware
    resources, this function returns through `pconfig` the preferred cache
    configuration for the current context. This is only a preference. The
    driver will use the requested configuration if possible, but it is free
    to choose a different configuration if required to execute functions.

    This will return a `pconfig` of :py:obj:`~.CU_FUNC_CACHE_PREFER_NONE`
    on devices where the size of the L1 cache and shared memory are fixed.

    The supported cache configurations are:

    - :py:obj:`~.CU_FUNC_CACHE_PREFER_NONE`: no preference for shared
      memory or L1 (default)

    - :py:obj:`~.CU_FUNC_CACHE_PREFER_SHARED`: prefer larger shared memory
      and smaller L1 cache

    - :py:obj:`~.CU_FUNC_CACHE_PREFER_L1`: prefer larger L1 cache and
      smaller shared memory

    - :py:obj:`~.CU_FUNC_CACHE_PREFER_EQUAL`: prefer equal sized L1 cache
      and shared memory

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pconfig : :py:obj:`~.CUfunc_cache`
        Returned cache configuration

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`, :py:obj:`~.cuFuncSetCacheConfig`, :py:obj:`~.cudaDeviceGetCacheConfig`
    """
    cdef ccuda.CUfunc_cache pconfig
    err = ccuda.cuCtxGetCacheConfig(&pconfig)
    return (CUresult(err), CUfunc_cache(pconfig))

@cython.embedsignature(True)
def cuCtxSetCacheConfig(config not None : CUfunc_cache):
    """ Sets the preferred cache configuration for the current context.

    On devices where the L1 cache and shared memory use the same hardware
    resources, this sets through `config` the preferred cache configuration
    for the current context. This is only a preference. The driver will use
    the requested configuration if possible, but it is free to choose a
    different configuration if required to execute the function. Any
    function preference set via :py:obj:`~.cuFuncSetCacheConfig()` or
    :py:obj:`~.cuKernelSetCacheConfig()` will be preferred over this
    context-wide setting. Setting the context-wide cache configuration to
    :py:obj:`~.CU_FUNC_CACHE_PREFER_NONE` will cause subsequent kernel
    launches to prefer to not change the cache configuration unless
    required to launch the kernel.

    This setting does nothing on devices where the size of the L1 cache and
    shared memory are fixed.

    Launching a kernel with a different preference than the most recent
    preference setting may insert a device-side synchronization point.

    The supported cache configurations are:

    - :py:obj:`~.CU_FUNC_CACHE_PREFER_NONE`: no preference for shared
      memory or L1 (default)

    - :py:obj:`~.CU_FUNC_CACHE_PREFER_SHARED`: prefer larger shared memory
      and smaller L1 cache

    - :py:obj:`~.CU_FUNC_CACHE_PREFER_L1`: prefer larger L1 cache and
      smaller shared memory

    - :py:obj:`~.CU_FUNC_CACHE_PREFER_EQUAL`: prefer equal sized L1 cache
      and shared memory

    Parameters
    ----------
    config : :py:obj:`~.CUfunc_cache`
        Requested cache configuration

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`, :py:obj:`~.cuFuncSetCacheConfig`, :py:obj:`~.cudaDeviceSetCacheConfig`, :py:obj:`~.cuKernelSetCacheConfig`
    """
    cdef ccuda.CUfunc_cache cconfig = config.value
    err = ccuda.cuCtxSetCacheConfig(cconfig)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuCtxGetApiVersion(ctx):
    """ Gets the context's API version.

    Returns a version number in `version` corresponding to the capabilities
    of the context (e.g. 3010 or 3020), which library developers can use to
    direct callers to a specific API version. If `ctx` is NULL, returns the
    API version used to create the currently bound context.

    Note that new API versions are only introduced when context
    capabilities are changed that break binary compatibility, so the API
    version and driver version may be different. For example, it is valid
    for the API version to be 3020 while the driver version is 4020.

    Parameters
    ----------
    ctx : :py:obj:`~.CUcontext`
        Context to check

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_UNKNOWN`
    version : unsigned int
        Pointer to version

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`
    """
    cdef ccuda.CUcontext cctx
    if ctx is None:
        cctx = <ccuda.CUcontext><void_ptr>0
    elif isinstance(ctx, (CUcontext,)):
        pctx = int(ctx)
        cctx = <ccuda.CUcontext><void_ptr>pctx
    else:
        pctx = int(CUcontext(ctx))
        cctx = <ccuda.CUcontext><void_ptr>pctx
    cdef unsigned int version = 0
    err = ccuda.cuCtxGetApiVersion(cctx, &version)
    return (CUresult(err), version)

@cython.embedsignature(True)
def cuCtxGetStreamPriorityRange():
    """ Returns numerical values that correspond to the least and greatest stream priorities.

    Returns in `*leastPriority` and `*greatestPriority` the numerical
    values that correspond to the least and greatest stream priorities
    respectively. Stream priorities follow a convention where lower numbers
    imply greater priorities. The range of meaningful stream priorities is
    given by [`*greatestPriority`, `*leastPriority`]. If the user attempts
    to create a stream with a priority value that is outside the meaningful
    range as specified by this API, the priority is automatically clamped
    down or up to either `*leastPriority` or `*greatestPriority`
    respectively. See :py:obj:`~.cuStreamCreateWithPriority` for details on
    creating a priority stream. A NULL may be passed in for
    `*leastPriority` or `*greatestPriority` if the value is not desired.

    This function will return '0' in both `*leastPriority` and
    `*greatestPriority` if the current context's device does not support
    stream priorities (see :py:obj:`~.cuDeviceGetAttribute`).

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`,
    leastPriority : int
        Pointer to an int in which the numerical value for least stream
        priority is returned
    greatestPriority : int
        Pointer to an int in which the numerical value for greatest stream
        priority is returned

    See Also
    --------
    :py:obj:`~.cuStreamCreateWithPriority`, :py:obj:`~.cuStreamGetPriority`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`, :py:obj:`~.cudaDeviceGetStreamPriorityRange`
    """
    cdef int leastPriority = 0
    cdef int greatestPriority = 0
    err = ccuda.cuCtxGetStreamPriorityRange(&leastPriority, &greatestPriority)
    return (CUresult(err), leastPriority, greatestPriority)

@cython.embedsignature(True)
def cuCtxResetPersistingL2Cache():
    """ Resets all persisting lines in cache to normal status.

    :py:obj:`~.cuCtxResetPersistingL2Cache` Resets all persisting lines in
    cache to normal status. Takes effect on function return.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`

    See Also
    --------
    :py:obj:`~.CUaccessPolicyWindow`
    """
    err = ccuda.cuCtxResetPersistingL2Cache()
    return (CUresult(err),)

@cython.embedsignature(True)
def cuCtxGetExecAffinity(typename not None : CUexecAffinityType):
    """ Returns the execution affinity setting for the current context.

    Returns in `*pExecAffinity` the current value of `typename`. The
    supported :py:obj:`~.CUexecAffinityType` values are:

    - :py:obj:`~.CU_EXEC_AFFINITY_TYPE_SM_COUNT`: number of SMs the context
      is limited to use.

    Parameters
    ----------
    typename : :py:obj:`~.CUexecAffinityType`
        Execution affinity type to query

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_EXEC_AFFINITY`
    pExecAffinity : :py:obj:`~.CUexecAffinityParam`
        Returned execution affinity

    See Also
    --------
    :py:obj:`~.CUexecAffinityParam`
    """
    cdef CUexecAffinityParam pExecAffinity = CUexecAffinityParam()
    cdef ccuda.CUexecAffinityType ctypename = typename.value
    err = ccuda.cuCtxGetExecAffinity(<ccuda.CUexecAffinityParam*>pExecAffinity._ptr, ctypename)
    return (CUresult(err), pExecAffinity)

@cython.embedsignature(True)
def cuCtxAttach(unsigned int flags):
    """ Increment a context's usage-count.

    [Deprecated]

    Note that this function is deprecated and should not be used.

    Increments the usage count of the context and passes back a context
    handle in `*pctx` that must be passed to :py:obj:`~.cuCtxDetach()` when
    the application is done with the context. :py:obj:`~.cuCtxAttach()`
    fails if there is no context current to the thread.

    Currently, the `flags` parameter must be 0.

    Parameters
    ----------
    flags : unsigned int
        Context attach flags (must be 0)

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pctx : :py:obj:`~.CUcontext`
        Returned context handle of the current context

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxDetach`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`
    """
    cdef CUcontext pctx = CUcontext()
    err = ccuda.cuCtxAttach(<ccuda.CUcontext*>pctx._ptr, flags)
    return (CUresult(err), pctx)

@cython.embedsignature(True)
def cuCtxDetach(ctx):
    """ Decrement a context's usage-count.

    [Deprecated]

    Note that this function is deprecated and should not be used.

    Decrements the usage count of the context `ctx`, and destroys the
    context if the usage count goes to 0. The context must be a handle that
    was passed back by :py:obj:`~.cuCtxCreate()` or
    :py:obj:`~.cuCtxAttach()`, and must be current to the calling thread.

    Parameters
    ----------
    ctx : :py:obj:`~.CUcontext`
        Context to destroy

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`
    """
    cdef ccuda.CUcontext cctx
    if ctx is None:
        cctx = <ccuda.CUcontext><void_ptr>0
    elif isinstance(ctx, (CUcontext,)):
        pctx = int(ctx)
        cctx = <ccuda.CUcontext><void_ptr>pctx
    else:
        pctx = int(CUcontext(ctx))
        cctx = <ccuda.CUcontext><void_ptr>pctx
    err = ccuda.cuCtxDetach(cctx)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuCtxGetSharedMemConfig():
    """ Returns the current shared memory configuration for the current context.

    [Deprecated]

    This function will return in `pConfig` the current size of shared
    memory banks in the current context. On devices with configurable
    shared memory banks, :py:obj:`~.cuCtxSetSharedMemConfig` can be used to
    change this setting, so that all subsequent kernel launches will by
    default use the new bank size. When :py:obj:`~.cuCtxGetSharedMemConfig`
    is called on devices without configurable shared memory, it will return
    the fixed bank size of the hardware.

    The returned bank configurations can be either:

    - :py:obj:`~.CU_SHARED_MEM_CONFIG_FOUR_BYTE_BANK_SIZE`: shared memory
      bank width is four bytes.

    - :py:obj:`~.CU_SHARED_MEM_CONFIG_EIGHT_BYTE_BANK_SIZE`: shared memory
      bank width will eight bytes.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pConfig : :py:obj:`~.CUsharedconfig`
        returned shared memory configuration

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`, :py:obj:`~.cuCtxGetSharedMemConfig`, :py:obj:`~.cuFuncSetCacheConfig`, :py:obj:`~.cudaDeviceGetSharedMemConfig`
    """
    cdef ccuda.CUsharedconfig pConfig
    err = ccuda.cuCtxGetSharedMemConfig(&pConfig)
    return (CUresult(err), CUsharedconfig(pConfig))

@cython.embedsignature(True)
def cuCtxSetSharedMemConfig(config not None : CUsharedconfig):
    """ Sets the shared memory configuration for the current context.

    [Deprecated]

    On devices with configurable shared memory banks, this function will
    set the context's shared memory bank size which is used for subsequent
    kernel launches.

    Changed the shared memory configuration between launches may insert a
    device side synchronization point between those launches.

    Changing the shared memory bank size will not increase shared memory
    usage or affect occupancy of kernels, but may have major effects on
    performance. Larger bank sizes will allow for greater potential
    bandwidth to shared memory, but will change what kinds of accesses to
    shared memory will result in bank conflicts.

    This function will do nothing on devices with fixed shared memory bank
    size.

    The supported bank configurations are:

    - :py:obj:`~.CU_SHARED_MEM_CONFIG_DEFAULT_BANK_SIZE`: set bank width to
      the default initial setting (currently, four bytes).

    - :py:obj:`~.CU_SHARED_MEM_CONFIG_FOUR_BYTE_BANK_SIZE`: set shared
      memory bank width to be natively four bytes.

    - :py:obj:`~.CU_SHARED_MEM_CONFIG_EIGHT_BYTE_BANK_SIZE`: set shared
      memory bank width to be natively eight bytes.

    Parameters
    ----------
    config : :py:obj:`~.CUsharedconfig`
        requested shared memory configuration

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`, :py:obj:`~.cuCtxGetSharedMemConfig`, :py:obj:`~.cuFuncSetCacheConfig`, :py:obj:`~.cudaDeviceSetSharedMemConfig`
    """
    cdef ccuda.CUsharedconfig cconfig = config.value
    err = ccuda.cuCtxSetSharedMemConfig(cconfig)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuModuleLoad(char* fname):
    """ Loads a compute module.

    Takes a filename `fname` and loads the corresponding module `module`
    into the current context. The CUDA driver API does not attempt to
    lazily allocate the resources needed by a module; if the memory for
    functions and data (constant and global) needed by the module cannot be
    allocated, :py:obj:`~.cuModuleLoad()` fails. The file should be a
    `cubin` file as output by nvcc, or a `PTX` file either as output by
    nvcc or handwritten, or a `fatbin` file as output by nvcc from
    toolchain 4.0 or later.

    Parameters
    ----------
    fname : bytes
        Filename of module to load

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_PTX`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_PTX_VERSION`, :py:obj:`~.CUDA_ERROR_NOT_FOUND`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_FILE_NOT_FOUND`, :py:obj:`~.CUDA_ERROR_NO_BINARY_FOR_GPU`, :py:obj:`~.CUDA_ERROR_SHARED_OBJECT_SYMBOL_NOT_FOUND`, :py:obj:`~.CUDA_ERROR_SHARED_OBJECT_INIT_FAILED`, :py:obj:`~.CUDA_ERROR_JIT_COMPILER_NOT_FOUND`
    module : :py:obj:`~.CUmodule`
        Returned module

    See Also
    --------
    :py:obj:`~.cuModuleGetFunction`, :py:obj:`~.cuModuleGetGlobal`, :py:obj:`~.cuModuleGetTexRef`, :py:obj:`~.cuModuleLoadData`, :py:obj:`~.cuModuleLoadDataEx`, :py:obj:`~.cuModuleLoadFatBinary`, :py:obj:`~.cuModuleUnload`
    """
    cdef CUmodule module = CUmodule()
    err = ccuda.cuModuleLoad(<ccuda.CUmodule*>module._ptr, fname)
    return (CUresult(err), module)

@cython.embedsignature(True)
def cuModuleLoadData(image):
    """ Load a module's data.

    Takes a pointer `image` and loads the corresponding module `module`
    into the current context. The `image` may be a `cubin` or `fatbin` as
    output by nvcc, or a NULL-terminated `PTX`, either as output by nvcc or
    hand-written.

    Parameters
    ----------
    image : Any
        Module data to load

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_PTX`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_PTX_VERSION`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_NO_BINARY_FOR_GPU`, :py:obj:`~.CUDA_ERROR_SHARED_OBJECT_SYMBOL_NOT_FOUND`, :py:obj:`~.CUDA_ERROR_SHARED_OBJECT_INIT_FAILED`, :py:obj:`~.CUDA_ERROR_JIT_COMPILER_NOT_FOUND`
    module : :py:obj:`~.CUmodule`
        Returned module

    See Also
    --------
    :py:obj:`~.cuModuleGetFunction`, :py:obj:`~.cuModuleGetGlobal`, :py:obj:`~.cuModuleGetTexRef`, :py:obj:`~.cuModuleLoad`, :py:obj:`~.cuModuleLoadDataEx`, :py:obj:`~.cuModuleLoadFatBinary`, :py:obj:`~.cuModuleUnload`
    """
    cdef CUmodule module = CUmodule()
    cimage = utils.HelperInputVoidPtr(image)
    cdef void* cimage_ptr = <void*><void_ptr>cimage.cptr
    err = ccuda.cuModuleLoadData(<ccuda.CUmodule*>module._ptr, cimage_ptr)
    return (CUresult(err), module)

@cython.embedsignature(True)
def cuModuleLoadDataEx(image, unsigned int numOptions, options : Optional[List[CUjit_option]], optionValues : Optional[List[Any]]):
    """ Load a module's data with options.

    Takes a pointer `image` and loads the corresponding module `module`
    into the current context. The `image` may be a `cubin` or `fatbin` as
    output by nvcc, or a NULL-terminated `PTX`, either as output by nvcc or
    hand-written.

    Parameters
    ----------
    image : Any
        Module data to load
    numOptions : unsigned int
        Number of options
    options : List[:py:obj:`~.CUjit_option`]
        Options for JIT
    optionValues : List[Any]
        Option values for JIT

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_PTX`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_PTX_VERSION`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_NO_BINARY_FOR_GPU`, :py:obj:`~.CUDA_ERROR_SHARED_OBJECT_SYMBOL_NOT_FOUND`, :py:obj:`~.CUDA_ERROR_SHARED_OBJECT_INIT_FAILED`, :py:obj:`~.CUDA_ERROR_JIT_COMPILER_NOT_FOUND`
    module : :py:obj:`~.CUmodule`
        Returned module

    See Also
    --------
    :py:obj:`~.cuModuleGetFunction`, :py:obj:`~.cuModuleGetGlobal`, :py:obj:`~.cuModuleGetTexRef`, :py:obj:`~.cuModuleLoad`, :py:obj:`~.cuModuleLoadData`, :py:obj:`~.cuModuleLoadFatBinary`, :py:obj:`~.cuModuleUnload`
    """
    optionValues = [] if optionValues is None else optionValues
    options = [] if options is None else options
    if not all(isinstance(_x, (CUjit_option)) for _x in options):
        raise TypeError("Argument 'options' is not instance of type (expected List[ccuda.CUjit_option]")
    cdef CUmodule module = CUmodule()
    cimage = utils.HelperInputVoidPtr(image)
    cdef void* cimage_ptr = <void*><void_ptr>cimage.cptr
    if numOptions > len(options): raise RuntimeError("List is too small: " + str(len(options)) + " < " + str(numOptions))
    if numOptions > len(optionValues): raise RuntimeError("List is too small: " + str(len(optionValues)) + " < " + str(numOptions))
    cdef vector[ccuda.CUjit_option] coptions = [pyoptions.value for pyoptions in (options)]
    pylist = [utils.HelperCUjit_option(pyoptions, pyoptionValues) for pyoptions, pyoptionValues in zip(options, optionValues)]
    cdef utils.InputVoidPtrPtrHelper voidStarHelperoptionValues = utils.InputVoidPtrPtrHelper(pylist)
    err = ccuda.cuModuleLoadDataEx(<ccuda.CUmodule*>module._ptr, cimage_ptr, numOptions, coptions.data(), <void**><void_ptr>voidStarHelperoptionValues.cptr)
    return (CUresult(err), module)

@cython.embedsignature(True)
def cuModuleLoadFatBinary(fatCubin):
    """ Load a module's data.

    Takes a pointer `fatCubin` and loads the corresponding module `module`
    into the current context. The pointer represents a `fat binary` object,
    which is a collection of different `cubin` and/or `PTX` files, all
    representing the same device code, but compiled and optimized for
    different architectures.

    Prior to CUDA 4.0, there was no documented API for constructing and
    using fat binary objects by programmers. Starting with CUDA 4.0, fat
    binary objects can be constructed by providing the `-fatbin option` to
    nvcc. More information can be found in the nvcc document.

    Parameters
    ----------
    fatCubin : Any
        Fat binary to load

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_PTX`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_PTX_VERSION`, :py:obj:`~.CUDA_ERROR_NOT_FOUND`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_NO_BINARY_FOR_GPU`, :py:obj:`~.CUDA_ERROR_SHARED_OBJECT_SYMBOL_NOT_FOUND`, :py:obj:`~.CUDA_ERROR_SHARED_OBJECT_INIT_FAILED`, :py:obj:`~.CUDA_ERROR_JIT_COMPILER_NOT_FOUND`
    module : :py:obj:`~.CUmodule`
        Returned module

    See Also
    --------
    :py:obj:`~.cuModuleGetFunction`, :py:obj:`~.cuModuleGetGlobal`, :py:obj:`~.cuModuleGetTexRef`, :py:obj:`~.cuModuleLoad`, :py:obj:`~.cuModuleLoadData`, :py:obj:`~.cuModuleLoadDataEx`, :py:obj:`~.cuModuleUnload`
    """
    cdef CUmodule module = CUmodule()
    cfatCubin = utils.HelperInputVoidPtr(fatCubin)
    cdef void* cfatCubin_ptr = <void*><void_ptr>cfatCubin.cptr
    err = ccuda.cuModuleLoadFatBinary(<ccuda.CUmodule*>module._ptr, cfatCubin_ptr)
    return (CUresult(err), module)

@cython.embedsignature(True)
def cuModuleUnload(hmod):
    """ Unloads a module.

    Unloads a module `hmod` from the current context. Attempting to unload
    a module which was obtained from the Library Management API such as
    :py:obj:`~.cuLibraryGetModule` will return
    :py:obj:`~.CUDA_ERROR_NOT_PERMITTED`.

    Parameters
    ----------
    hmod : :py:obj:`~.CUmodule`
        Module to unload

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_PERMITTED`

    See Also
    --------
    :py:obj:`~.cuModuleGetFunction`, :py:obj:`~.cuModuleGetGlobal`, :py:obj:`~.cuModuleGetTexRef`, :py:obj:`~.cuModuleLoad`, :py:obj:`~.cuModuleLoadData`, :py:obj:`~.cuModuleLoadDataEx`, :py:obj:`~.cuModuleLoadFatBinary`
    """
    cdef ccuda.CUmodule chmod
    if hmod is None:
        chmod = <ccuda.CUmodule><void_ptr>0
    elif isinstance(hmod, (CUmodule,)):
        phmod = int(hmod)
        chmod = <ccuda.CUmodule><void_ptr>phmod
    else:
        phmod = int(CUmodule(hmod))
        chmod = <ccuda.CUmodule><void_ptr>phmod
    err = ccuda.cuModuleUnload(chmod)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuModuleGetLoadingMode():
    """ Query lazy loading mode.

    Returns lazy loading mode Module loading mode is controlled by
    CUDA_MODULE_LOADING env variable

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`,
    mode : :py:obj:`~.CUmoduleLoadingMode`
        Returns the lazy loading mode

    See Also
    --------
    :py:obj:`~.cuModuleLoad`,
    """
    cdef ccuda.CUmoduleLoadingMode mode
    err = ccuda.cuModuleGetLoadingMode(&mode)
    return (CUresult(err), CUmoduleLoadingMode(mode))

@cython.embedsignature(True)
def cuModuleGetFunction(hmod, char* name):
    """ Returns a function handle.

    Returns in `*hfunc` the handle of the function of name `name` located
    in module `hmod`. If no function of that name exists,
    :py:obj:`~.cuModuleGetFunction()` returns
    :py:obj:`~.CUDA_ERROR_NOT_FOUND`.

    Parameters
    ----------
    hmod : :py:obj:`~.CUmodule`
        Module to retrieve function from
    name : bytes
        Name of function to retrieve

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_FOUND`
    hfunc : :py:obj:`~.CUfunction`
        Returned function handle

    See Also
    --------
    :py:obj:`~.cuModuleGetGlobal`, :py:obj:`~.cuModuleGetTexRef`, :py:obj:`~.cuModuleLoad`, :py:obj:`~.cuModuleLoadData`, :py:obj:`~.cuModuleLoadDataEx`, :py:obj:`~.cuModuleLoadFatBinary`, :py:obj:`~.cuModuleUnload`
    """
    cdef ccuda.CUmodule chmod
    if hmod is None:
        chmod = <ccuda.CUmodule><void_ptr>0
    elif isinstance(hmod, (CUmodule,)):
        phmod = int(hmod)
        chmod = <ccuda.CUmodule><void_ptr>phmod
    else:
        phmod = int(CUmodule(hmod))
        chmod = <ccuda.CUmodule><void_ptr>phmod
    cdef CUfunction hfunc = CUfunction()
    err = ccuda.cuModuleGetFunction(<ccuda.CUfunction*>hfunc._ptr, chmod, name)
    return (CUresult(err), hfunc)

@cython.embedsignature(True)
def cuModuleGetFunctionCount(mod):
    """ Returns the number of functions within a module.

    Returns in `count` the number of functions in `mod`.

    Parameters
    ----------
    mod : :py:obj:`~.CUmodule`
        Module to query

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    count : unsigned int
        Number of functions found within the module
    """
    cdef ccuda.CUmodule cmod
    if mod is None:
        cmod = <ccuda.CUmodule><void_ptr>0
    elif isinstance(mod, (CUmodule,)):
        pmod = int(mod)
        cmod = <ccuda.CUmodule><void_ptr>pmod
    else:
        pmod = int(CUmodule(mod))
        cmod = <ccuda.CUmodule><void_ptr>pmod
    cdef unsigned int count = 0
    err = ccuda.cuModuleGetFunctionCount(&count, cmod)
    return (CUresult(err), count)

@cython.embedsignature(True)
def cuModuleEnumerateFunctions(unsigned int numFunctions, mod):
    """ Returns the function handles within a module.

    Returns in `functions` a maximum number of `numFunctions` function
    handles within `mod`. When function loading mode is set to LAZY the
    function retrieved may be partially loaded. The loading state of a
    function can be queried using :py:obj:`~.cuFunctionIsLoaded`. CUDA APIs
    may load the function automatically when called with partially loaded
    function handle which may incur additional latency. Alternatively,
    :py:obj:`~.cuFunctionLoad` can be used to explicitly load a function.
    The returned function handles become invalid when the module is
    unloaded.

    Parameters
    ----------
    numFunctions : unsigned int
        Maximum number of function handles may be returned to the buffer
    mod : :py:obj:`~.CUmodule`
        Module to query from

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    functions : List[:py:obj:`~.CUfunction`]
        Buffer where the function handles are returned to

    See Also
    --------
    :py:obj:`~.cuModuleGetFunction`, :py:obj:`~.cuModuleGetFunctionCount`, :py:obj:`~.cuFuncIsLoaded`, :py:obj:`~.cuFuncLoad`
    """
    cdef ccuda.CUmodule cmod
    if mod is None:
        cmod = <ccuda.CUmodule><void_ptr>0
    elif isinstance(mod, (CUmodule,)):
        pmod = int(mod)
        cmod = <ccuda.CUmodule><void_ptr>pmod
    else:
        pmod = int(CUmodule(mod))
        cmod = <ccuda.CUmodule><void_ptr>pmod
    cdef ccuda.CUfunction* cfunctions = NULL
    pyfunctions = []
    if numFunctions != 0:
        cfunctions = <ccuda.CUfunction*>calloc(numFunctions, sizeof(ccuda.CUfunction))
        if cfunctions is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(numFunctions) + 'x' + str(sizeof(ccuda.CUfunction)))
    err = ccuda.cuModuleEnumerateFunctions(cfunctions, numFunctions, cmod)
    if CUresult(err) == CUresult(0):
        pyfunctions = [CUfunction(init_value=<void_ptr>cfunctions[idx]) for idx in range(numFunctions)]
    if cfunctions is not NULL:
        free(cfunctions)
    return (CUresult(err), pyfunctions)

@cython.embedsignature(True)
def cuModuleGetGlobal(hmod, char* name):
    """ Returns a global pointer from a module.

    Returns in `*dptr` and `*bytes` the base pointer and size of the global
    of name `name` located in module `hmod`. If no variable of that name
    exists, :py:obj:`~.cuModuleGetGlobal()` returns
    :py:obj:`~.CUDA_ERROR_NOT_FOUND`. One of the parameters `dptr` or
    `numbytes` (not both) can be NULL in which case it is ignored.

    Parameters
    ----------
    hmod : :py:obj:`~.CUmodule`
        Module to retrieve global from
    name : bytes
        Name of global to retrieve

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_FOUND`
    dptr : :py:obj:`~.CUdeviceptr`
        Returned global device pointer
    numbytes : int
        Returned global size in bytes

    See Also
    --------
    :py:obj:`~.cuModuleGetFunction`, :py:obj:`~.cuModuleGetTexRef`, :py:obj:`~.cuModuleLoad`, :py:obj:`~.cuModuleLoadData`, :py:obj:`~.cuModuleLoadDataEx`, :py:obj:`~.cuModuleLoadFatBinary`, :py:obj:`~.cuModuleUnload`, :py:obj:`~.cudaGetSymbolAddress`, :py:obj:`~.cudaGetSymbolSize`
    """
    cdef ccuda.CUmodule chmod
    if hmod is None:
        chmod = <ccuda.CUmodule><void_ptr>0
    elif isinstance(hmod, (CUmodule,)):
        phmod = int(hmod)
        chmod = <ccuda.CUmodule><void_ptr>phmod
    else:
        phmod = int(CUmodule(hmod))
        chmod = <ccuda.CUmodule><void_ptr>phmod
    cdef CUdeviceptr dptr = CUdeviceptr()
    cdef size_t numbytes = 0
    err = ccuda.cuModuleGetGlobal(<ccuda.CUdeviceptr*>dptr._ptr, &numbytes, chmod, name)
    return (CUresult(err), dptr, numbytes)

@cython.embedsignature(True)
def cuLinkCreate(unsigned int numOptions, options : Optional[List[CUjit_option]], optionValues : Optional[List[Any]]):
    """ Creates a pending JIT linker invocation.

    If the call is successful, the caller owns the returned CUlinkState,
    which should eventually be destroyed with :py:obj:`~.cuLinkDestroy`.
    The device code machine size (32 or 64 bit) will match the calling
    application.

    Both linker and compiler options may be specified. Compiler options
    will be applied to inputs to this linker action which must be compiled
    from PTX. The options :py:obj:`~.CU_JIT_WALL_TIME`,
    :py:obj:`~.CU_JIT_INFO_LOG_BUFFER_SIZE_BYTES`, and
    :py:obj:`~.CU_JIT_ERROR_LOG_BUFFER_SIZE_BYTES` will accumulate data
    until the CUlinkState is destroyed.

    `optionValues` must remain valid for the life of the CUlinkState if
    output options are used. No other references to inputs are maintained
    after this call returns.

    Parameters
    ----------
    numOptions : unsigned int
        Size of options arrays
    options : List[:py:obj:`~.CUjit_option`]
        Array of linker and compiler options
    optionValues : List[Any]
        Array of option values, each cast to void *

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_JIT_COMPILER_NOT_FOUND`
    stateOut : :py:obj:`~.CUlinkState`
        On success, this will contain a CUlinkState to specify and complete
        this action

    See Also
    --------
    :py:obj:`~.cuLinkAddData`, :py:obj:`~.cuLinkAddFile`, :py:obj:`~.cuLinkComplete`, :py:obj:`~.cuLinkDestroy`

    Notes
    -----
    For LTO-IR input, only LTO-IR compiled with toolkits prior to CUDA 12.0 will be accepted
    """
    optionValues = [] if optionValues is None else optionValues
    options = [] if options is None else options
    if not all(isinstance(_x, (CUjit_option)) for _x in options):
        raise TypeError("Argument 'options' is not instance of type (expected List[ccuda.CUjit_option]")
    if numOptions > len(options): raise RuntimeError("List is too small: " + str(len(options)) + " < " + str(numOptions))
    if numOptions > len(optionValues): raise RuntimeError("List is too small: " + str(len(optionValues)) + " < " + str(numOptions))
    cdef vector[ccuda.CUjit_option] coptions = [pyoptions.value for pyoptions in (options)]
    pylist = [utils.HelperCUjit_option(pyoptions, pyoptionValues) for pyoptions, pyoptionValues in zip(options, optionValues)]
    cdef utils.InputVoidPtrPtrHelper voidStarHelperoptionValues = utils.InputVoidPtrPtrHelper(pylist)
    cdef CUlinkState stateOut = CUlinkState()
    err = ccuda.cuLinkCreate(numOptions, coptions.data(), <void**><void_ptr>voidStarHelperoptionValues.cptr, stateOut._ptr)
    stateOut._keepalive.append(voidStarHelperoptionValues)
    for option in pylist:
        stateOut._keepalive.append(option)
    return (CUresult(err), stateOut)

@cython.embedsignature(True)
def cuLinkAddData(state, typename not None : CUjitInputType, data, size_t size, char* name, unsigned int numOptions, options : Optional[List[CUjit_option]], optionValues : Optional[List[Any]]):
    """ Add an input to a pending linker invocation.

    Ownership of `data` is retained by the caller. No reference is retained
    to any inputs after this call returns.

    This method accepts only compiler options, which are used if the data
    must be compiled from PTX, and does not accept any of
    :py:obj:`~.CU_JIT_WALL_TIME`, :py:obj:`~.CU_JIT_INFO_LOG_BUFFER`,
    :py:obj:`~.CU_JIT_ERROR_LOG_BUFFER`,
    :py:obj:`~.CU_JIT_TARGET_FROM_CUCONTEXT`, or :py:obj:`~.CU_JIT_TARGET`.

    Parameters
    ----------
    state : :py:obj:`~.CUlinkState`
        A pending linker action.
    typename : :py:obj:`~.CUjitInputType`
        The type of the input data.
    data : Any
        The input data. PTX must be NULL-terminated.
    size : size_t
        The length of the input data.
    name : bytes
        An optional name for this input in log messages.
    numOptions : unsigned int
        Size of options.
    options : List[:py:obj:`~.CUjit_option`]
        Options to be applied only for this input (overrides options from
        :py:obj:`~.cuLinkCreate`).
    optionValues : List[Any]
        Array of option values, each cast to void *.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_IMAGE`, :py:obj:`~.CUDA_ERROR_INVALID_PTX`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_PTX_VERSION`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_NO_BINARY_FOR_GPU`

    See Also
    --------
    :py:obj:`~.cuLinkCreate`, :py:obj:`~.cuLinkAddFile`, :py:obj:`~.cuLinkComplete`, :py:obj:`~.cuLinkDestroy`

    Notes
    -----
    For LTO-IR input, only LTO-IR compiled with toolkits prior to CUDA 12.0 will be accepted
    """
    optionValues = [] if optionValues is None else optionValues
    options = [] if options is None else options
    if not all(isinstance(_x, (CUjit_option)) for _x in options):
        raise TypeError("Argument 'options' is not instance of type (expected List[ccuda.CUjit_option]")
    cdef ccuda.CUlinkState cstate
    if state is None:
        cstate = <ccuda.CUlinkState><void_ptr>0
    elif isinstance(state, (CUlinkState,)):
        pstate = int(state)
        cstate = <ccuda.CUlinkState><void_ptr>pstate
    else:
        pstate = int(CUlinkState(state))
        cstate = <ccuda.CUlinkState><void_ptr>pstate
    cdef ccuda.CUjitInputType ctypename = typename.value
    cdata = utils.HelperInputVoidPtr(data)
    cdef void* cdata_ptr = <void*><void_ptr>cdata.cptr
    if numOptions > len(options): raise RuntimeError("List is too small: " + str(len(options)) + " < " + str(numOptions))
    if numOptions > len(optionValues): raise RuntimeError("List is too small: " + str(len(optionValues)) + " < " + str(numOptions))
    cdef vector[ccuda.CUjit_option] coptions = [pyoptions.value for pyoptions in (options)]
    pylist = [utils.HelperCUjit_option(pyoptions, pyoptionValues) for pyoptions, pyoptionValues in zip(options, optionValues)]
    cdef utils.InputVoidPtrPtrHelper voidStarHelperoptionValues = utils.InputVoidPtrPtrHelper(pylist)
    err = ccuda.cuLinkAddData(cstate, ctypename, cdata_ptr, size, name, numOptions, coptions.data(), <void**><void_ptr>voidStarHelperoptionValues.cptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuLinkAddFile(state, typename not None : CUjitInputType, char* path, unsigned int numOptions, options : Optional[List[CUjit_option]], optionValues : Optional[List[Any]]):
    """ Add a file input to a pending linker invocation.

    No reference is retained to any inputs after this call returns.

    This method accepts only compiler options, which are used if the input
    must be compiled from PTX, and does not accept any of
    :py:obj:`~.CU_JIT_WALL_TIME`, :py:obj:`~.CU_JIT_INFO_LOG_BUFFER`,
    :py:obj:`~.CU_JIT_ERROR_LOG_BUFFER`,
    :py:obj:`~.CU_JIT_TARGET_FROM_CUCONTEXT`, or :py:obj:`~.CU_JIT_TARGET`.

    This method is equivalent to invoking :py:obj:`~.cuLinkAddData` on the
    contents of the file.

    Parameters
    ----------
    state : :py:obj:`~.CUlinkState`
        A pending linker action
    typename : :py:obj:`~.CUjitInputType`
        The type of the input data
    path : bytes
        Path to the input file
    numOptions : unsigned int
        Size of options
    options : List[:py:obj:`~.CUjit_option`]
        Options to be applied only for this input (overrides options from
        :py:obj:`~.cuLinkCreate`)
    optionValues : List[Any]
        Array of option values, each cast to void *

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_FILE_NOT_FOUND` :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_IMAGE`, :py:obj:`~.CUDA_ERROR_INVALID_PTX`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_PTX_VERSION`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_NO_BINARY_FOR_GPU`

    See Also
    --------
    :py:obj:`~.cuLinkCreate`, :py:obj:`~.cuLinkAddData`, :py:obj:`~.cuLinkComplete`, :py:obj:`~.cuLinkDestroy`

    Notes
    -----
    For LTO-IR input, only LTO-IR compiled with toolkits prior to CUDA 12.0 will be accepted
    """
    optionValues = [] if optionValues is None else optionValues
    options = [] if options is None else options
    if not all(isinstance(_x, (CUjit_option)) for _x in options):
        raise TypeError("Argument 'options' is not instance of type (expected List[ccuda.CUjit_option]")
    cdef ccuda.CUlinkState cstate
    if state is None:
        cstate = <ccuda.CUlinkState><void_ptr>0
    elif isinstance(state, (CUlinkState,)):
        pstate = int(state)
        cstate = <ccuda.CUlinkState><void_ptr>pstate
    else:
        pstate = int(CUlinkState(state))
        cstate = <ccuda.CUlinkState><void_ptr>pstate
    cdef ccuda.CUjitInputType ctypename = typename.value
    if numOptions > len(options): raise RuntimeError("List is too small: " + str(len(options)) + " < " + str(numOptions))
    if numOptions > len(optionValues): raise RuntimeError("List is too small: " + str(len(optionValues)) + " < " + str(numOptions))
    cdef vector[ccuda.CUjit_option] coptions = [pyoptions.value for pyoptions in (options)]
    pylist = [utils.HelperCUjit_option(pyoptions, pyoptionValues) for pyoptions, pyoptionValues in zip(options, optionValues)]
    cdef utils.InputVoidPtrPtrHelper voidStarHelperoptionValues = utils.InputVoidPtrPtrHelper(pylist)
    err = ccuda.cuLinkAddFile(cstate, ctypename, path, numOptions, coptions.data(), <void**><void_ptr>voidStarHelperoptionValues.cptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuLinkComplete(state):
    """ Complete a pending linker invocation.

    Completes the pending linker action and returns the cubin image for the
    linked device code, which can be used with
    :py:obj:`~.cuModuleLoadData`. The cubin is owned by `state`, so it
    should be loaded before `state` is destroyed via
    :py:obj:`~.cuLinkDestroy`. This call does not destroy `state`.

    Parameters
    ----------
    state : :py:obj:`~.CUlinkState`
        A pending linker invocation

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    cubinOut : Any
        On success, this will point to the output image
    sizeOut : int
        Optional parameter to receive the size of the generated image

    See Also
    --------
    :py:obj:`~.cuLinkCreate`, :py:obj:`~.cuLinkAddData`, :py:obj:`~.cuLinkAddFile`, :py:obj:`~.cuLinkDestroy`, :py:obj:`~.cuModuleLoadData`
    """
    cdef ccuda.CUlinkState cstate
    if state is None:
        cstate = <ccuda.CUlinkState><void_ptr>0
    elif isinstance(state, (CUlinkState,)):
        pstate = int(state)
        cstate = <ccuda.CUlinkState><void_ptr>pstate
    else:
        pstate = int(CUlinkState(state))
        cstate = <ccuda.CUlinkState><void_ptr>pstate
    cdef void_ptr cubinOut = 0
    cdef size_t sizeOut = 0
    err = ccuda.cuLinkComplete(cstate, <void**>&cubinOut, &sizeOut)
    return (CUresult(err), cubinOut, sizeOut)

@cython.embedsignature(True)
def cuLinkDestroy(state):
    """ Destroys state for a JIT linker invocation.

    Parameters
    ----------
    state : :py:obj:`~.CUlinkState`
        State object for the linker invocation

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuLinkCreate`
    """
    cdef ccuda.CUlinkState cstate
    if state is None:
        cstate = <ccuda.CUlinkState><void_ptr>0
    elif isinstance(state, (CUlinkState,)):
        pstate = int(state)
        cstate = <ccuda.CUlinkState><void_ptr>pstate
    else:
        pstate = int(CUlinkState(state))
        cstate = <ccuda.CUlinkState><void_ptr>pstate
    err = ccuda.cuLinkDestroy(cstate)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuModuleGetTexRef(hmod, char* name):
    """ Returns a handle to a texture reference.

    [Deprecated]

    Returns in `*pTexRef` the handle of the texture reference of name
    `name` in the module `hmod`. If no texture reference of that name
    exists, :py:obj:`~.cuModuleGetTexRef()` returns
    :py:obj:`~.CUDA_ERROR_NOT_FOUND`. This texture reference handle should
    not be destroyed, since it will be destroyed when the module is
    unloaded.

    Parameters
    ----------
    hmod : :py:obj:`~.CUmodule`
        Module to retrieve texture reference from
    name : bytes
        Name of texture reference to retrieve

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_FOUND`
    pTexRef : :py:obj:`~.CUtexref`
        Returned texture reference

    See Also
    --------
    :py:obj:`~.cuModuleGetFunction`, :py:obj:`~.cuModuleGetGlobal`, :py:obj:`~.cuModuleGetSurfRef`, :py:obj:`~.cuModuleLoad`, :py:obj:`~.cuModuleLoadData`, :py:obj:`~.cuModuleLoadDataEx`, :py:obj:`~.cuModuleLoadFatBinary`, :py:obj:`~.cuModuleUnload`
    """
    cdef ccuda.CUmodule chmod
    if hmod is None:
        chmod = <ccuda.CUmodule><void_ptr>0
    elif isinstance(hmod, (CUmodule,)):
        phmod = int(hmod)
        chmod = <ccuda.CUmodule><void_ptr>phmod
    else:
        phmod = int(CUmodule(hmod))
        chmod = <ccuda.CUmodule><void_ptr>phmod
    cdef CUtexref pTexRef = CUtexref()
    err = ccuda.cuModuleGetTexRef(<ccuda.CUtexref*>pTexRef._ptr, chmod, name)
    return (CUresult(err), pTexRef)

@cython.embedsignature(True)
def cuModuleGetSurfRef(hmod, char* name):
    """ Returns a handle to a surface reference.

    [Deprecated]

    Returns in `*pSurfRef` the handle of the surface reference of name
    `name` in the module `hmod`. If no surface reference of that name
    exists, :py:obj:`~.cuModuleGetSurfRef()` returns
    :py:obj:`~.CUDA_ERROR_NOT_FOUND`.

    Parameters
    ----------
    hmod : :py:obj:`~.CUmodule`
        Module to retrieve surface reference from
    name : bytes
        Name of surface reference to retrieve

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_FOUND`
    pSurfRef : :py:obj:`~.CUsurfref`
        Returned surface reference

    See Also
    --------
    :py:obj:`~.cuModuleGetFunction`, :py:obj:`~.cuModuleGetGlobal`, :py:obj:`~.cuModuleGetTexRef`, :py:obj:`~.cuModuleLoad`, :py:obj:`~.cuModuleLoadData`, :py:obj:`~.cuModuleLoadDataEx`, :py:obj:`~.cuModuleLoadFatBinary`, :py:obj:`~.cuModuleUnload`
    """
    cdef ccuda.CUmodule chmod
    if hmod is None:
        chmod = <ccuda.CUmodule><void_ptr>0
    elif isinstance(hmod, (CUmodule,)):
        phmod = int(hmod)
        chmod = <ccuda.CUmodule><void_ptr>phmod
    else:
        phmod = int(CUmodule(hmod))
        chmod = <ccuda.CUmodule><void_ptr>phmod
    cdef CUsurfref pSurfRef = CUsurfref()
    err = ccuda.cuModuleGetSurfRef(<ccuda.CUsurfref*>pSurfRef._ptr, chmod, name)
    return (CUresult(err), pSurfRef)

@cython.embedsignature(True)
def cuLibraryLoadData(code, jitOptions : Optional[List[CUjit_option]], jitOptionsValues : Optional[List[Any]], unsigned int numJitOptions, libraryOptions : Optional[List[CUlibraryOption]], libraryOptionValues : Optional[List[Any]], unsigned int numLibraryOptions):
    """ Load a library with specified code and options.

    Takes a pointer `code` and loads the corresponding library `library`
    based on the application defined library loading mode:

    - If module loading is set to EAGER, via the environment variables
      described in "Module loading", `library` is loaded eagerly into all
      contexts at the time of the call and future contexts at the time of
      creation until the library is unloaded with
      :py:obj:`~.cuLibraryUnload()`.

    - If the environment variables are set to LAZY, `library` is not
      immediately loaded onto all existent contexts and will only be loaded
      when a function is needed for that context, such as a kernel launch.

    These environment variables are described in the CUDA programming guide
    under the "CUDA environment variables" section.

    The `code` may be a `cubin` or `fatbin` as output by nvcc, or a NULL-
    terminated `PTX`, either as output by nvcc or hand-written.

    Options are passed as an array via `jitOptions` and any corresponding
    parameters are passed in `jitOptionsValues`. The number of total JIT
    options is supplied via `numJitOptions`. Any outputs will be returned
    via `jitOptionsValues`.

    Library load options are passed as an array via `libraryOptions` and
    any corresponding parameters are passed in `libraryOptionValues`. The
    number of total library load options is supplied via
    `numLibraryOptions`.

    Parameters
    ----------
    code : Any
        Code to load
    jitOptions : List[:py:obj:`~.CUjit_option`]
        Options for JIT
    jitOptionsValues : List[Any]
        Option values for JIT
    numJitOptions : unsigned int
        Number of options
    libraryOptions : List[:py:obj:`~.CUlibraryOption`]
        Options for loading
    libraryOptionValues : List[Any]
        Option values for loading
    numLibraryOptions : unsigned int
        Number of options for loading

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_PTX`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_PTX_VERSION`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_NO_BINARY_FOR_GPU`, :py:obj:`~.CUDA_ERROR_SHARED_OBJECT_SYMBOL_NOT_FOUND`, :py:obj:`~.CUDA_ERROR_SHARED_OBJECT_INIT_FAILED`, :py:obj:`~.CUDA_ERROR_JIT_COMPILER_NOT_FOUND`
    library : :py:obj:`~.CUlibrary`
        Returned library

    See Also
    --------
    :py:obj:`~.cuLibraryLoadFromFile`, :py:obj:`~.cuLibraryUnload`, :py:obj:`~.cuModuleLoad`, :py:obj:`~.cuModuleLoadData`, :py:obj:`~.cuModuleLoadDataEx`
    """
    libraryOptionValues = [] if libraryOptionValues is None else libraryOptionValues
    libraryOptions = [] if libraryOptions is None else libraryOptions
    if not all(isinstance(_x, (CUlibraryOption)) for _x in libraryOptions):
        raise TypeError("Argument 'libraryOptions' is not instance of type (expected List[ccuda.CUlibraryOption]")
    jitOptionsValues = [] if jitOptionsValues is None else jitOptionsValues
    jitOptions = [] if jitOptions is None else jitOptions
    if not all(isinstance(_x, (CUjit_option)) for _x in jitOptions):
        raise TypeError("Argument 'jitOptions' is not instance of type (expected List[ccuda.CUjit_option]")
    cdef CUlibrary library = CUlibrary()
    ccode = utils.HelperInputVoidPtr(code)
    cdef void* ccode_ptr = <void*><void_ptr>ccode.cptr
    cdef vector[ccuda.CUjit_option] cjitOptions = [pyjitOptions.value for pyjitOptions in (jitOptions)]
    pylist = [utils.HelperCUjit_option(pyoptions, pyoptionValues) for pyoptions, pyoptionValues in zip(numJitOptions, jitOptionsValues)]
    cdef utils.InputVoidPtrPtrHelper voidStarHelperjitOptionsValues = utils.InputVoidPtrPtrHelper(pylist)
    if numJitOptions > len(jitOptions): raise RuntimeError("List is too small: " + str(len(jitOptions)) + " < " + str(numJitOptions))
    if numJitOptions > len(jitOptionsValues): raise RuntimeError("List is too small: " + str(len(jitOptionsValues)) + " < " + str(numJitOptions))
    cdef vector[ccuda.CUlibraryOption] clibraryOptions = [pylibraryOptions.value for pylibraryOptions in (libraryOptions)]
    pylist = [utils.HelperCUlibraryOption(pyoptions, pyoptionValues) for pyoptions, pyoptionValues in zip(numLibraryOptions, libraryOptionValues)]
    cdef utils.InputVoidPtrPtrHelper voidStarHelperlibraryOptionValues = utils.InputVoidPtrPtrHelper(pylist)
    if numLibraryOptions > len(libraryOptions): raise RuntimeError("List is too small: " + str(len(libraryOptions)) + " < " + str(numLibraryOptions))
    if numLibraryOptions > len(libraryOptionValues): raise RuntimeError("List is too small: " + str(len(libraryOptionValues)) + " < " + str(numLibraryOptions))
    err = ccuda.cuLibraryLoadData(<ccuda.CUlibrary*>library._ptr, ccode_ptr, cjitOptions.data(), <void**><void_ptr>voidStarHelperjitOptionsValues.cptr, numJitOptions, clibraryOptions.data(), <void**><void_ptr>voidStarHelperlibraryOptionValues.cptr, numLibraryOptions)
    return (CUresult(err), library)

@cython.embedsignature(True)
def cuLibraryLoadFromFile(char* fileName, jitOptions : Optional[List[CUjit_option]], jitOptionsValues : Optional[List[Any]], unsigned int numJitOptions, libraryOptions : Optional[List[CUlibraryOption]], libraryOptionValues : Optional[List[Any]], unsigned int numLibraryOptions):
    """ Load a library with specified file and options.

    Takes a pointer `code` and loads the corresponding library `library`
    based on the application defined library loading mode:

    - If module loading is set to EAGER, via the environment variables
      described in "Module loading", `library` is loaded eagerly into all
      contexts at the time of the call and future contexts at the time of
      creation until the library is unloaded with
      :py:obj:`~.cuLibraryUnload()`.

    - If the environment variables are set to LAZY, `library` is not
      immediately loaded onto all existent contexts and will only be loaded
      when a function is needed for that context, such as a kernel launch.

    These environment variables are described in the CUDA programming guide
    under the "CUDA environment variables" section.

    The file should be a `cubin` file as output by nvcc, or a `PTX` file
    either as output by nvcc or handwritten, or a `fatbin` file as output
    by nvcc.

    Options are passed as an array via `jitOptions` and any corresponding
    parameters are passed in `jitOptionsValues`. The number of total
    options is supplied via `numJitOptions`. Any outputs will be returned
    via `jitOptionsValues`.

    Library load options are passed as an array via `libraryOptions` and
    any corresponding parameters are passed in `libraryOptionValues`. The
    number of total library load options is supplied via
    `numLibraryOptions`.

    Parameters
    ----------
    fileName : bytes
        File to load from
    jitOptions : List[:py:obj:`~.CUjit_option`]
        Options for JIT
    jitOptionsValues : List[Any]
        Option values for JIT
    numJitOptions : unsigned int
        Number of options
    libraryOptions : List[:py:obj:`~.CUlibraryOption`]
        Options for loading
    libraryOptionValues : List[Any]
        Option values for loading
    numLibraryOptions : unsigned int
        Number of options for loading

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_PTX`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_PTX_VERSION`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_NO_BINARY_FOR_GPU`, :py:obj:`~.CUDA_ERROR_SHARED_OBJECT_SYMBOL_NOT_FOUND`, :py:obj:`~.CUDA_ERROR_SHARED_OBJECT_INIT_FAILED`, :py:obj:`~.CUDA_ERROR_JIT_COMPILER_NOT_FOUND`
    library : :py:obj:`~.CUlibrary`
        Returned library

    See Also
    --------
    :py:obj:`~.cuLibraryLoadData`, :py:obj:`~.cuLibraryUnload`, :py:obj:`~.cuModuleLoad`, :py:obj:`~.cuModuleLoadData`, :py:obj:`~.cuModuleLoadDataEx`
    """
    libraryOptionValues = [] if libraryOptionValues is None else libraryOptionValues
    libraryOptions = [] if libraryOptions is None else libraryOptions
    if not all(isinstance(_x, (CUlibraryOption)) for _x in libraryOptions):
        raise TypeError("Argument 'libraryOptions' is not instance of type (expected List[ccuda.CUlibraryOption]")
    jitOptionsValues = [] if jitOptionsValues is None else jitOptionsValues
    jitOptions = [] if jitOptions is None else jitOptions
    if not all(isinstance(_x, (CUjit_option)) for _x in jitOptions):
        raise TypeError("Argument 'jitOptions' is not instance of type (expected List[ccuda.CUjit_option]")
    cdef CUlibrary library = CUlibrary()
    cdef vector[ccuda.CUjit_option] cjitOptions = [pyjitOptions.value for pyjitOptions in (jitOptions)]
    pylist = [utils.HelperCUjit_option(pyoptions, pyoptionValues) for pyoptions, pyoptionValues in zip(numJitOptions, jitOptionsValues)]
    cdef utils.InputVoidPtrPtrHelper voidStarHelperjitOptionsValues = utils.InputVoidPtrPtrHelper(pylist)
    if numJitOptions > len(jitOptions): raise RuntimeError("List is too small: " + str(len(jitOptions)) + " < " + str(numJitOptions))
    if numJitOptions > len(jitOptionsValues): raise RuntimeError("List is too small: " + str(len(jitOptionsValues)) + " < " + str(numJitOptions))
    cdef vector[ccuda.CUlibraryOption] clibraryOptions = [pylibraryOptions.value for pylibraryOptions in (libraryOptions)]
    pylist = [utils.HelperCUlibraryOption(pyoptions, pyoptionValues) for pyoptions, pyoptionValues in zip(numLibraryOptions, libraryOptionValues)]
    cdef utils.InputVoidPtrPtrHelper voidStarHelperlibraryOptionValues = utils.InputVoidPtrPtrHelper(pylist)
    if numLibraryOptions > len(libraryOptions): raise RuntimeError("List is too small: " + str(len(libraryOptions)) + " < " + str(numLibraryOptions))
    if numLibraryOptions > len(libraryOptionValues): raise RuntimeError("List is too small: " + str(len(libraryOptionValues)) + " < " + str(numLibraryOptions))
    err = ccuda.cuLibraryLoadFromFile(<ccuda.CUlibrary*>library._ptr, fileName, cjitOptions.data(), <void**><void_ptr>voidStarHelperjitOptionsValues.cptr, numJitOptions, clibraryOptions.data(), <void**><void_ptr>voidStarHelperlibraryOptionValues.cptr, numLibraryOptions)
    return (CUresult(err), library)

@cython.embedsignature(True)
def cuLibraryUnload(library):
    """ Unloads a library.

    Unloads the library specified with `library`

    Parameters
    ----------
    library : :py:obj:`~.CUlibrary`
        Library to unload

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuLibraryLoadData`, :py:obj:`~.cuLibraryLoadFromFile`, :py:obj:`~.cuModuleUnload`
    """
    cdef ccuda.CUlibrary clibrary
    if library is None:
        clibrary = <ccuda.CUlibrary><void_ptr>0
    elif isinstance(library, (CUlibrary,)):
        plibrary = int(library)
        clibrary = <ccuda.CUlibrary><void_ptr>plibrary
    else:
        plibrary = int(CUlibrary(library))
        clibrary = <ccuda.CUlibrary><void_ptr>plibrary
    err = ccuda.cuLibraryUnload(clibrary)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuLibraryGetKernel(library, char* name):
    """ Returns a kernel handle.

    Returns in `pKernel` the handle of the kernel with name `name` located
    in library `library`. If kernel handle is not found, the call returns
    :py:obj:`~.CUDA_ERROR_NOT_FOUND`.

    Parameters
    ----------
    library : :py:obj:`~.CUlibrary`
        Library to retrieve kernel from
    name : bytes
        Name of kernel to retrieve

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_NOT_FOUND`
    pKernel : :py:obj:`~.CUkernel`
        Returned kernel handle

    See Also
    --------
    :py:obj:`~.cuLibraryLoadData`, :py:obj:`~.cuLibraryLoadFromFile`, :py:obj:`~.cuLibraryUnload`, :py:obj:`~.cuKernelGetFunction`, :py:obj:`~.cuLibraryGetModule`, :py:obj:`~.cuModuleGetFunction`
    """
    cdef ccuda.CUlibrary clibrary
    if library is None:
        clibrary = <ccuda.CUlibrary><void_ptr>0
    elif isinstance(library, (CUlibrary,)):
        plibrary = int(library)
        clibrary = <ccuda.CUlibrary><void_ptr>plibrary
    else:
        plibrary = int(CUlibrary(library))
        clibrary = <ccuda.CUlibrary><void_ptr>plibrary
    cdef CUkernel pKernel = CUkernel()
    err = ccuda.cuLibraryGetKernel(<ccuda.CUkernel*>pKernel._ptr, clibrary, name)
    return (CUresult(err), pKernel)

@cython.embedsignature(True)
def cuLibraryGetKernelCount(lib):
    """ Returns the number of kernels within a library.

    Returns in `count` the number of kernels in `lib`.

    Parameters
    ----------
    lib : :py:obj:`~.CUlibrary`
        Library to query

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    count : unsigned int
        Number of kernels found within the library
    """
    cdef ccuda.CUlibrary clib
    if lib is None:
        clib = <ccuda.CUlibrary><void_ptr>0
    elif isinstance(lib, (CUlibrary,)):
        plib = int(lib)
        clib = <ccuda.CUlibrary><void_ptr>plib
    else:
        plib = int(CUlibrary(lib))
        clib = <ccuda.CUlibrary><void_ptr>plib
    cdef unsigned int count = 0
    err = ccuda.cuLibraryGetKernelCount(&count, clib)
    return (CUresult(err), count)

@cython.embedsignature(True)
def cuLibraryEnumerateKernels(unsigned int numKernels, lib):
    """ Retrieve the kernel handles within a library.

    Returns in `kernels` a maximum number of `numKernels` kernel handles
    within `lib`. The returned kernel handle becomes invalid when the
    library is unloaded.

    Parameters
    ----------
    numKernels : unsigned int
        Maximum number of kernel handles may be returned to the buffer
    lib : :py:obj:`~.CUlibrary`
        Library to query from

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    kernels : List[:py:obj:`~.CUkernel`]
        Buffer where the kernel handles are returned to

    See Also
    --------
    :py:obj:`~.cuLibraryGetKernelCount`
    """
    cdef ccuda.CUlibrary clib
    if lib is None:
        clib = <ccuda.CUlibrary><void_ptr>0
    elif isinstance(lib, (CUlibrary,)):
        plib = int(lib)
        clib = <ccuda.CUlibrary><void_ptr>plib
    else:
        plib = int(CUlibrary(lib))
        clib = <ccuda.CUlibrary><void_ptr>plib
    cdef ccuda.CUkernel* ckernels = NULL
    pykernels = []
    if numKernels != 0:
        ckernels = <ccuda.CUkernel*>calloc(numKernels, sizeof(ccuda.CUkernel))
        if ckernels is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(numKernels) + 'x' + str(sizeof(ccuda.CUkernel)))
    err = ccuda.cuLibraryEnumerateKernels(ckernels, numKernels, clib)
    if CUresult(err) == CUresult(0):
        pykernels = [CUkernel(init_value=<void_ptr>ckernels[idx]) for idx in range(numKernels)]
    if ckernels is not NULL:
        free(ckernels)
    return (CUresult(err), pykernels)

@cython.embedsignature(True)
def cuLibraryGetModule(library):
    """ Returns a module handle.

    Returns in `pMod` the module handle associated with the current context
    located in library `library`. If module handle is not found, the call
    returns :py:obj:`~.CUDA_ERROR_NOT_FOUND`.

    Parameters
    ----------
    library : :py:obj:`~.CUlibrary`
        Library to retrieve module from

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_NOT_FOUND`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_CONTEXT_IS_DESTROYED`
    pMod : :py:obj:`~.CUmodule`
        Returned module handle

    See Also
    --------
    :py:obj:`~.cuLibraryLoadData`, :py:obj:`~.cuLibraryLoadFromFile`, :py:obj:`~.cuLibraryUnload`, :py:obj:`~.cuModuleGetFunction`
    """
    cdef ccuda.CUlibrary clibrary
    if library is None:
        clibrary = <ccuda.CUlibrary><void_ptr>0
    elif isinstance(library, (CUlibrary,)):
        plibrary = int(library)
        clibrary = <ccuda.CUlibrary><void_ptr>plibrary
    else:
        plibrary = int(CUlibrary(library))
        clibrary = <ccuda.CUlibrary><void_ptr>plibrary
    cdef CUmodule pMod = CUmodule()
    err = ccuda.cuLibraryGetModule(<ccuda.CUmodule*>pMod._ptr, clibrary)
    return (CUresult(err), pMod)

@cython.embedsignature(True)
def cuKernelGetFunction(kernel):
    """ Returns a function handle.

    Returns in `pFunc` the handle of the function for the requested kernel
    `kernel` and the current context. If function handle is not found, the
    call returns :py:obj:`~.CUDA_ERROR_NOT_FOUND`.

    Parameters
    ----------
    kernel : :py:obj:`~.CUkernel`
        Kernel to retrieve function for the requested context

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_NOT_FOUND`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_CONTEXT_IS_DESTROYED`
    pFunc : :py:obj:`~.CUfunction`
        Returned function handle

    See Also
    --------
    :py:obj:`~.cuLibraryLoadData`, :py:obj:`~.cuLibraryLoadFromFile`, :py:obj:`~.cuLibraryUnload`, :py:obj:`~.cuLibraryGetKernel`, :py:obj:`~.cuLibraryGetModule`, :py:obj:`~.cuModuleGetFunction`
    """
    cdef ccuda.CUkernel ckernel
    if kernel is None:
        ckernel = <ccuda.CUkernel><void_ptr>0
    elif isinstance(kernel, (CUkernel,)):
        pkernel = int(kernel)
        ckernel = <ccuda.CUkernel><void_ptr>pkernel
    else:
        pkernel = int(CUkernel(kernel))
        ckernel = <ccuda.CUkernel><void_ptr>pkernel
    cdef CUfunction pFunc = CUfunction()
    err = ccuda.cuKernelGetFunction(<ccuda.CUfunction*>pFunc._ptr, ckernel)
    return (CUresult(err), pFunc)

@cython.embedsignature(True)
def cuLibraryGetGlobal(library, char* name):
    """ Returns a global device pointer.

    Returns in `*dptr` and `*bytes` the base pointer and size of the global
    with name `name` for the requested library `library` and the current
    context. If no global for the requested name `name` exists, the call
    returns :py:obj:`~.CUDA_ERROR_NOT_FOUND`. One of the parameters `dptr`
    or `numbytes` (not both) can be NULL in which case it is ignored.

    Parameters
    ----------
    library : :py:obj:`~.CUlibrary`
        Library to retrieve global from
    name : bytes
        Name of global to retrieve

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_NOT_FOUND`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_CONTEXT_IS_DESTROYED`
    dptr : :py:obj:`~.CUdeviceptr`
        Returned global device pointer for the requested context
    numbytes : int
        Returned global size in bytes

    See Also
    --------
    :py:obj:`~.cuLibraryLoadData`, :py:obj:`~.cuLibraryLoadFromFile`, :py:obj:`~.cuLibraryUnload`, :py:obj:`~.cuLibraryGetModule`, :py:obj:`~.cuModuleGetGlobal`
    """
    cdef ccuda.CUlibrary clibrary
    if library is None:
        clibrary = <ccuda.CUlibrary><void_ptr>0
    elif isinstance(library, (CUlibrary,)):
        plibrary = int(library)
        clibrary = <ccuda.CUlibrary><void_ptr>plibrary
    else:
        plibrary = int(CUlibrary(library))
        clibrary = <ccuda.CUlibrary><void_ptr>plibrary
    cdef CUdeviceptr dptr = CUdeviceptr()
    cdef size_t numbytes = 0
    err = ccuda.cuLibraryGetGlobal(<ccuda.CUdeviceptr*>dptr._ptr, &numbytes, clibrary, name)
    return (CUresult(err), dptr, numbytes)

@cython.embedsignature(True)
def cuLibraryGetManaged(library, char* name):
    """ Returns a pointer to managed memory.

    Returns in `*dptr` and `*bytes` the base pointer and size of the
    managed memory with name `name` for the requested library `library`. If
    no managed memory with the requested name `name` exists, the call
    returns :py:obj:`~.CUDA_ERROR_NOT_FOUND`. One of the parameters `dptr`
    or `numbytes` (not both) can be NULL in which case it is ignored. Note
    that managed memory for library `library` is shared across devices and
    is registered when the library is loaded into atleast one context.

    Parameters
    ----------
    library : :py:obj:`~.CUlibrary`
        Library to retrieve managed memory from
    name : bytes
        Name of managed memory to retrieve

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_NOT_FOUND`
    dptr : :py:obj:`~.CUdeviceptr`
        Returned pointer to the managed memory
    numbytes : int
        Returned memory size in bytes

    See Also
    --------
    :py:obj:`~.cuLibraryLoadData`, :py:obj:`~.cuLibraryLoadFromFile`, :py:obj:`~.cuLibraryUnload`

    Notes
    -----
    The API requires a CUDA context to be present and initialized on at least one device. If no context is present, the call returns :py:obj:`~.CUDA_ERROR_NOT_FOUND`.
    """
    cdef ccuda.CUlibrary clibrary
    if library is None:
        clibrary = <ccuda.CUlibrary><void_ptr>0
    elif isinstance(library, (CUlibrary,)):
        plibrary = int(library)
        clibrary = <ccuda.CUlibrary><void_ptr>plibrary
    else:
        plibrary = int(CUlibrary(library))
        clibrary = <ccuda.CUlibrary><void_ptr>plibrary
    cdef CUdeviceptr dptr = CUdeviceptr()
    cdef size_t numbytes = 0
    err = ccuda.cuLibraryGetManaged(<ccuda.CUdeviceptr*>dptr._ptr, &numbytes, clibrary, name)
    return (CUresult(err), dptr, numbytes)

@cython.embedsignature(True)
def cuLibraryGetUnifiedFunction(library, char* symbol):
    """ Returns a pointer to a unified function.

    Returns in `*fptr` the function pointer to a unified function denoted
    by `symbol`. If no unified function with name `symbol` exists, the call
    returns :py:obj:`~.CUDA_ERROR_NOT_FOUND`. If there is no device with
    attribute :py:obj:`~.CU_DEVICE_ATTRIBUTE_UNIFIED_FUNCTION_POINTERS`
    present in the system, the call may return
    :py:obj:`~.CUDA_ERROR_NOT_FOUND`.

    Parameters
    ----------
    library : :py:obj:`~.CUlibrary`
        Library to retrieve function pointer memory from
    symbol : bytes
        Name of function pointer to retrieve

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_NOT_FOUND`
    fptr : Any
        Returned pointer to a unified function

    See Also
    --------
    :py:obj:`~.cuLibraryLoadData`, :py:obj:`~.cuLibraryLoadFromFile`, :py:obj:`~.cuLibraryUnload`
    """
    cdef ccuda.CUlibrary clibrary
    if library is None:
        clibrary = <ccuda.CUlibrary><void_ptr>0
    elif isinstance(library, (CUlibrary,)):
        plibrary = int(library)
        clibrary = <ccuda.CUlibrary><void_ptr>plibrary
    else:
        plibrary = int(CUlibrary(library))
        clibrary = <ccuda.CUlibrary><void_ptr>plibrary
    cdef void_ptr fptr = 0
    err = ccuda.cuLibraryGetUnifiedFunction(<void**>&fptr, clibrary, symbol)
    return (CUresult(err), fptr)

@cython.embedsignature(True)
def cuKernelGetAttribute(attrib not None : CUfunction_attribute, kernel, dev):
    """ Returns information about a kernel.

    Returns in `*pi` the integer value of the attribute `attrib` for the
    kernel `kernel` for the requested device `dev`. The supported
    attributes are:

    - :py:obj:`~.CU_FUNC_ATTRIBUTE_MAX_THREADS_PER_BLOCK`: The maximum
      number of threads per block, beyond which a launch of the kernel
      would fail. This number depends on both the kernel and the requested
      device.

    - :py:obj:`~.CU_FUNC_ATTRIBUTE_SHARED_SIZE_BYTES`: The size in bytes of
      statically-allocated shared memory per block required by this kernel.
      This does not include dynamically-allocated shared memory requested
      by the user at runtime.

    - :py:obj:`~.CU_FUNC_ATTRIBUTE_CONST_SIZE_BYTES`: The size in bytes of
      user-allocated constant memory required by this kernel.

    - :py:obj:`~.CU_FUNC_ATTRIBUTE_LOCAL_SIZE_BYTES`: The size in bytes of
      local memory used by each thread of this kernel.

    - :py:obj:`~.CU_FUNC_ATTRIBUTE_NUM_REGS`: The number of registers used
      by each thread of this kernel.

    - :py:obj:`~.CU_FUNC_ATTRIBUTE_PTX_VERSION`: The PTX virtual
      architecture version for which the kernel was compiled. This value is
      the major PTX version * 10

      - the minor PTX version, so a PTX version 1.3 function would return
        the value 13. Note that this may return the undefined value of 0
        for cubins compiled prior to CUDA 3.0.

    - :py:obj:`~.CU_FUNC_ATTRIBUTE_BINARY_VERSION`: The binary architecture
      version for which the kernel was compiled. This value is the major
      binary version * 10 + the minor binary version, so a binary version
      1.3 function would return the value 13. Note that this will return a
      value of 10 for legacy cubins that do not have a properly-encoded
      binary architecture version.

    - :py:obj:`~.CU_FUNC_CACHE_MODE_CA`: The attribute to indicate whether
      the kernel has been compiled with user specified option "-Xptxas
      --dlcm=ca" set.

    - :py:obj:`~.CU_FUNC_ATTRIBUTE_MAX_DYNAMIC_SHARED_SIZE_BYTES`: The
      maximum size in bytes of dynamically-allocated shared memory.

    - :py:obj:`~.CU_FUNC_ATTRIBUTE_PREFERRED_SHARED_MEMORY_CARVEOUT`:
      Preferred shared memory-L1 cache split ratio in percent of total
      shared memory.

    - :py:obj:`~.CU_FUNC_ATTRIBUTE_CLUSTER_SIZE_MUST_BE_SET`: If this
      attribute is set, the kernel must launch with a valid cluster size
      specified.

    - :py:obj:`~.CU_FUNC_ATTRIBUTE_REQUIRED_CLUSTER_WIDTH`: The required
      cluster width in blocks.

    - :py:obj:`~.CU_FUNC_ATTRIBUTE_REQUIRED_CLUSTER_HEIGHT`: The required
      cluster height in blocks.

    - :py:obj:`~.CU_FUNC_ATTRIBUTE_REQUIRED_CLUSTER_DEPTH`: The required
      cluster depth in blocks.

    - :py:obj:`~.CU_FUNC_ATTRIBUTE_NON_PORTABLE_CLUSTER_SIZE_ALLOWED`:
      Indicates whether the function can be launched with non-portable
      cluster size. 1 is allowed, 0 is disallowed. A non-portable cluster
      size may only function on the specific SKUs the program is tested on.
      The launch might fail if the program is run on a different hardware
      platform. CUDA API provides cudaOccupancyMaxActiveClusters to assist
      with checking whether the desired size can be launched on the current
      device. A portable cluster size is guaranteed to be functional on all
      compute capabilities higher than the target compute capability. The
      portable cluster size for sm_90 is 8 blocks per cluster. This value
      may increase for future compute capabilities. The specific hardware
      unit may support higher cluster sizes that’s not guaranteed to be
      portable.

    - :py:obj:`~.CU_FUNC_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE`:
      The block scheduling policy of a function. The value type is
      CUclusterSchedulingPolicy.

    Parameters
    ----------
    attrib : :py:obj:`~.CUfunction_attribute`
        Attribute requested
    kernel : :py:obj:`~.CUkernel`
        Kernel to query attribute of
    dev : :py:obj:`~.CUdevice`
        Device to query attribute of

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    pi : int
        Returned attribute value

    See Also
    --------
    :py:obj:`~.cuLibraryLoadData`, :py:obj:`~.cuLibraryLoadFromFile`, :py:obj:`~.cuLibraryUnload`, :py:obj:`~.cuKernelSetAttribute`, :py:obj:`~.cuLibraryGetKernel`, :py:obj:`~.cuLaunchKernel`, :py:obj:`~.cuKernelGetFunction`, :py:obj:`~.cuLibraryGetModule`, :py:obj:`~.cuModuleGetFunction`, :py:obj:`~.cuFuncGetAttribute`

    Notes
    -----
    If another thread is trying to set the same attribute on the same device using :py:obj:`~.cuKernelSetAttribute()` simultaneously, the attribute query will give the old or new value depending on the interleavings chosen by the OS scheduler and memory consistency.
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    cdef ccuda.CUkernel ckernel
    if kernel is None:
        ckernel = <ccuda.CUkernel><void_ptr>0
    elif isinstance(kernel, (CUkernel,)):
        pkernel = int(kernel)
        ckernel = <ccuda.CUkernel><void_ptr>pkernel
    else:
        pkernel = int(CUkernel(kernel))
        ckernel = <ccuda.CUkernel><void_ptr>pkernel
    cdef int pi = 0
    cdef ccuda.CUfunction_attribute cattrib = attrib.value
    err = ccuda.cuKernelGetAttribute(&pi, cattrib, ckernel, cdev)
    return (CUresult(err), pi)

@cython.embedsignature(True)
def cuKernelSetAttribute(attrib not None : CUfunction_attribute, int val, kernel, dev):
    """ Sets information about a kernel.

    This call sets the value of a specified attribute `attrib` on the
    kernel `kernel` for the requested device `dev` to an integer value
    specified by `val`. This function returns CUDA_SUCCESS if the new value
    of the attribute could be successfully set. If the set fails, this call
    will return an error. Not all attributes can have values set.
    Attempting to set a value on a read-only attribute will result in an
    error (CUDA_ERROR_INVALID_VALUE)

    Note that attributes set using :py:obj:`~.cuFuncSetAttribute()` will
    override the attribute set by this API irrespective of whether the call
    to :py:obj:`~.cuFuncSetAttribute()` is made before or after this API
    call. However, :py:obj:`~.cuKernelGetAttribute()` will always return
    the attribute value set by this API.

    Supported attributes are:

    - :py:obj:`~.CU_FUNC_ATTRIBUTE_MAX_DYNAMIC_SHARED_SIZE_BYTES`: This is
      the maximum size in bytes of dynamically-allocated shared memory. The
      value should contain the requested maximum size of dynamically-
      allocated shared memory. The sum of this value and the function
      attribute :py:obj:`~.CU_FUNC_ATTRIBUTE_SHARED_SIZE_BYTES` cannot
      exceed the device attribute
      :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_BLOCK_OPTIN`.
      The maximal size of requestable dynamic shared memory may differ by
      GPU architecture.

    - :py:obj:`~.CU_FUNC_ATTRIBUTE_PREFERRED_SHARED_MEMORY_CARVEOUT`: On
      devices where the L1 cache and shared memory use the same hardware
      resources, this sets the shared memory carveout preference, in
      percent of the total shared memory. See
      :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_MULTIPROCESSOR`
      This is only a hint, and the driver can choose a different ratio if
      required to execute the function.

    - :py:obj:`~.CU_FUNC_ATTRIBUTE_REQUIRED_CLUSTER_WIDTH`: The required
      cluster width in blocks. The width, height, and depth values must
      either all be 0 or all be positive. The validity of the cluster
      dimensions is checked at launch time. If the value is set during
      compile time, it cannot be set at runtime. Setting it at runtime will
      return CUDA_ERROR_NOT_PERMITTED.

    - :py:obj:`~.CU_FUNC_ATTRIBUTE_REQUIRED_CLUSTER_HEIGHT`: The required
      cluster height in blocks. The width, height, and depth values must
      either all be 0 or all be positive. The validity of the cluster
      dimensions is checked at launch time. If the value is set during
      compile time, it cannot be set at runtime. Setting it at runtime will
      return CUDA_ERROR_NOT_PERMITTED.

    - :py:obj:`~.CU_FUNC_ATTRIBUTE_REQUIRED_CLUSTER_DEPTH`: The required
      cluster depth in blocks. The width, height, and depth values must
      either all be 0 or all be positive. The validity of the cluster
      dimensions is checked at launch time. If the value is set during
      compile time, it cannot be set at runtime. Setting it at runtime will
      return CUDA_ERROR_NOT_PERMITTED.

    - :py:obj:`~.CU_FUNC_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE`:
      The block scheduling policy of a function. The value type is
      CUclusterSchedulingPolicy.

    Parameters
    ----------
    attrib : :py:obj:`~.CUfunction_attribute`
        Attribute requested
    val : int
        Value to set
    kernel : :py:obj:`~.CUkernel`
        Kernel to set attribute of
    dev : :py:obj:`~.CUdevice`
        Device to set attribute of

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`

    See Also
    --------
    :py:obj:`~.cuLibraryLoadData`, :py:obj:`~.cuLibraryLoadFromFile`, :py:obj:`~.cuLibraryUnload`, :py:obj:`~.cuKernelGetAttribute`, :py:obj:`~.cuLibraryGetKernel`, :py:obj:`~.cuLaunchKernel`, :py:obj:`~.cuKernelGetFunction`, :py:obj:`~.cuLibraryGetModule`, :py:obj:`~.cuModuleGetFunction`, :py:obj:`~.cuFuncSetAttribute`

    Notes
    -----
    The API has stricter locking requirements in comparison to its legacy counterpart :py:obj:`~.cuFuncSetAttribute()` due to device-wide semantics. If multiple threads are trying to set the same attribute on the same device simultaneously, the attribute setting will depend on the interleavings chosen by the OS scheduler and memory consistency.
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    cdef ccuda.CUkernel ckernel
    if kernel is None:
        ckernel = <ccuda.CUkernel><void_ptr>0
    elif isinstance(kernel, (CUkernel,)):
        pkernel = int(kernel)
        ckernel = <ccuda.CUkernel><void_ptr>pkernel
    else:
        pkernel = int(CUkernel(kernel))
        ckernel = <ccuda.CUkernel><void_ptr>pkernel
    cdef ccuda.CUfunction_attribute cattrib = attrib.value
    err = ccuda.cuKernelSetAttribute(cattrib, val, ckernel, cdev)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuKernelSetCacheConfig(kernel, config not None : CUfunc_cache, dev):
    """ Sets the preferred cache configuration for a device kernel.

    On devices where the L1 cache and shared memory use the same hardware
    resources, this sets through `config` the preferred cache configuration
    for the device kernel `kernel` on the requested device `dev`. This is
    only a preference. The driver will use the requested configuration if
    possible, but it is free to choose a different configuration if
    required to execute `kernel`. Any context-wide preference set via
    :py:obj:`~.cuCtxSetCacheConfig()` will be overridden by this per-kernel
    setting.

    Note that attributes set using :py:obj:`~.cuFuncSetCacheConfig()` will
    override the attribute set by this API irrespective of whether the call
    to :py:obj:`~.cuFuncSetCacheConfig()` is made before or after this API
    call.

    This setting does nothing on devices where the size of the L1 cache and
    shared memory are fixed.

    Launching a kernel with a different preference than the most recent
    preference setting may insert a device-side synchronization point.

    The supported cache configurations are:

    - :py:obj:`~.CU_FUNC_CACHE_PREFER_NONE`: no preference for shared
      memory or L1 (default)

    - :py:obj:`~.CU_FUNC_CACHE_PREFER_SHARED`: prefer larger shared memory
      and smaller L1 cache

    - :py:obj:`~.CU_FUNC_CACHE_PREFER_L1`: prefer larger L1 cache and
      smaller shared memory

    - :py:obj:`~.CU_FUNC_CACHE_PREFER_EQUAL`: prefer equal sized L1 cache
      and shared memory

    Parameters
    ----------
    kernel : :py:obj:`~.CUkernel`
        Kernel to configure cache for
    config : :py:obj:`~.CUfunc_cache`
        Requested cache configuration
    dev : :py:obj:`~.CUdevice`
        Device to set attribute of

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`

    See Also
    --------
    :py:obj:`~.cuLibraryLoadData`, :py:obj:`~.cuLibraryLoadFromFile`, :py:obj:`~.cuLibraryUnload`, :py:obj:`~.cuLibraryGetKernel`, :py:obj:`~.cuKernelGetFunction`, :py:obj:`~.cuLibraryGetModule`, :py:obj:`~.cuModuleGetFunction`, :py:obj:`~.cuFuncSetCacheConfig`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuLaunchKernel`

    Notes
    -----
    The API has stricter locking requirements in comparison to its legacy counterpart :py:obj:`~.cuFuncSetCacheConfig()` due to device-wide semantics. If multiple threads are trying to set a config on the same device simultaneously, the cache config setting will depend on the interleavings chosen by the OS scheduler and memory consistency.
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    cdef ccuda.CUkernel ckernel
    if kernel is None:
        ckernel = <ccuda.CUkernel><void_ptr>0
    elif isinstance(kernel, (CUkernel,)):
        pkernel = int(kernel)
        ckernel = <ccuda.CUkernel><void_ptr>pkernel
    else:
        pkernel = int(CUkernel(kernel))
        ckernel = <ccuda.CUkernel><void_ptr>pkernel
    cdef ccuda.CUfunc_cache cconfig = config.value
    err = ccuda.cuKernelSetCacheConfig(ckernel, cconfig, cdev)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuKernelGetName(hfunc):
    """ Returns the function name for a :py:obj:`~.CUkernel` handle.

    Returns in `**name` the function name associated with the kernel handle
    `hfunc` . The function name is returned as a null-terminated string.
    The returned name is only valid when the kernel handle is valid. If the
    library is unloaded or reloaded, one must call the API again to get the
    updated name. This API may return a mangled name if the function is not
    declared as having C linkage. If either `**name` or `hfunc` is NULL,
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE` is returned.

    Parameters
    ----------
    hfunc : :py:obj:`~.CUkernel`
        The function handle to retrieve the name for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    name : bytes
        The returned name of the function
    """
    cdef ccuda.CUkernel chfunc
    if hfunc is None:
        chfunc = <ccuda.CUkernel><void_ptr>0
    elif isinstance(hfunc, (CUkernel,)):
        phfunc = int(hfunc)
        chfunc = <ccuda.CUkernel><void_ptr>phfunc
    else:
        phfunc = int(CUkernel(hfunc))
        chfunc = <ccuda.CUkernel><void_ptr>phfunc
    cdef const char* name = NULL
    err = ccuda.cuKernelGetName(&name, chfunc)
    return (CUresult(err), <bytes>name)

@cython.embedsignature(True)
def cuKernelGetParamInfo(kernel, size_t paramIndex):
    """ Returns the offset and size of a kernel parameter in the device-side parameter layout.

    Queries the kernel parameter at `paramIndex` into `kernel's` list of
    parameters, and returns in `paramOffset` and `paramSize` the offset and
    size, respectively, where the parameter will reside in the device-side
    parameter layout. This information can be used to update kernel node
    parameters from the device via
    :py:obj:`~.cudaGraphKernelNodeSetParam()` and
    :py:obj:`~.cudaGraphKernelNodeUpdatesApply()`. `paramIndex` must be
    less than the number of parameters that `kernel` takes. `paramSize` can
    be set to NULL if only the parameter offset is desired.

    Parameters
    ----------
    kernel : :py:obj:`~.CUkernel`
        The kernel to query
    paramIndex : size_t
        The parameter index to query

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`,
    paramOffset : int
        Returns the offset into the device-side parameter layout at which
        the parameter resides
    paramSize : int
        Optionally returns the size of the parameter in the device-side
        parameter layout

    See Also
    --------
    :py:obj:`~.cuFuncGetParamInfo`
    """
    cdef ccuda.CUkernel ckernel
    if kernel is None:
        ckernel = <ccuda.CUkernel><void_ptr>0
    elif isinstance(kernel, (CUkernel,)):
        pkernel = int(kernel)
        ckernel = <ccuda.CUkernel><void_ptr>pkernel
    else:
        pkernel = int(CUkernel(kernel))
        ckernel = <ccuda.CUkernel><void_ptr>pkernel
    cdef size_t paramOffset = 0
    cdef size_t paramSize = 0
    err = ccuda.cuKernelGetParamInfo(ckernel, paramIndex, &paramOffset, &paramSize)
    return (CUresult(err), paramOffset, paramSize)

@cython.embedsignature(True)
def cuMemGetInfo():
    """ Gets free and total memory.

    Returns in `*total` the total amount of memory available to the the
    current context. Returns in `*free` the amount of memory on the device
    that is free according to the OS. CUDA is not guaranteed to be able to
    allocate all of the memory that the OS reports as free. In a multi-
    tenet situation, free estimate returned is prone to race condition
    where a new allocation/free done by a different process or a different
    thread in the same process between the time when free memory was
    estimated and reported, will result in deviation in free value reported
    and actual free memory.

    The integrated GPU on Tegra shares memory with CPU and other component
    of the SoC. The free and total values returned by the API excludes the
    SWAP memory space maintained by the OS on some platforms. The OS may
    move some of the memory pages into swap area as the GPU or CPU allocate
    or access memory. See Tegra app note on how to calculate total and free
    memory on Tegra.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    free : int
        Returned free memory in bytes
    total : int
        Returned total memory in bytes

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemGetInfo`
    """
    cdef size_t free = 0
    cdef size_t total = 0
    err = ccuda.cuMemGetInfo(&free, &total)
    return (CUresult(err), free, total)

@cython.embedsignature(True)
def cuMemAlloc(size_t bytesize):
    """ Allocates device memory.

    Allocates `bytesize` bytes of linear memory on the device and returns
    in `*dptr` a pointer to the allocated memory. The allocated memory is
    suitably aligned for any kind of variable. The memory is not cleared.
    If `bytesize` is 0, :py:obj:`~.cuMemAlloc()` returns
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE`.

    Parameters
    ----------
    bytesize : size_t
        Requested allocation size in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    dptr : :py:obj:`~.CUdeviceptr`
        Returned device pointer

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMalloc`
    """
    cdef CUdeviceptr dptr = CUdeviceptr()
    err = ccuda.cuMemAlloc(<ccuda.CUdeviceptr*>dptr._ptr, bytesize)
    return (CUresult(err), dptr)

@cython.embedsignature(True)
def cuMemAllocPitch(size_t WidthInBytes, size_t Height, unsigned int ElementSizeBytes):
    """ Allocates pitched device memory.

    Allocates at least `WidthInBytes` * `Height` bytes of linear memory on
    the device and returns in `*dptr` a pointer to the allocated memory.
    The function may pad the allocation to ensure that corresponding
    pointers in any given row will continue to meet the alignment
    requirements for coalescing as the address is updated from row to row.
    `ElementSizeBytes` specifies the size of the largest reads and writes
    that will be performed on the memory range. `ElementSizeBytes` may be
    4, 8 or 16 (since coalesced memory transactions are not possible on
    other data sizes). If `ElementSizeBytes` is smaller than the actual
    read/write size of a kernel, the kernel will run correctly, but
    possibly at reduced speed. The pitch returned in `*pPitch` by
    :py:obj:`~.cuMemAllocPitch()` is the width in bytes of the allocation.
    The intended usage of pitch is as a separate parameter of the
    allocation, used to compute addresses within the 2D array. Given the
    row and column of an array element of type T, the address is computed
    as:

    **View CUDA Toolkit Documentation for a C++ code example**

    The pitch returned by :py:obj:`~.cuMemAllocPitch()` is guaranteed to
    work with :py:obj:`~.cuMemcpy2D()` under all circumstances. For
    allocations of 2D arrays, it is recommended that programmers consider
    performing pitch allocations using :py:obj:`~.cuMemAllocPitch()`. Due
    to alignment restrictions in the hardware, this is especially true if
    the application will be performing 2D memory copies between different
    regions of device memory (whether linear memory or CUDA arrays).

    The byte alignment of the pitch returned by
    :py:obj:`~.cuMemAllocPitch()` is guaranteed to match or exceed the
    alignment requirement for texture binding with
    :py:obj:`~.cuTexRefSetAddress2D()`.

    Parameters
    ----------
    WidthInBytes : size_t
        Requested allocation width in bytes
    Height : size_t
        Requested allocation height in rows
    ElementSizeBytes : unsigned int
        Size of largest reads/writes for range

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    dptr : :py:obj:`~.CUdeviceptr`
        Returned device pointer
    pPitch : int
        Returned pitch of allocation in bytes

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMallocPitch`
    """
    cdef CUdeviceptr dptr = CUdeviceptr()
    cdef size_t pPitch = 0
    err = ccuda.cuMemAllocPitch(<ccuda.CUdeviceptr*>dptr._ptr, &pPitch, WidthInBytes, Height, ElementSizeBytes)
    return (CUresult(err), dptr, pPitch)

@cython.embedsignature(True)
def cuMemFree(dptr):
    """ Frees device memory.

    Frees the memory space pointed to by `dptr`, which must have been
    returned by a previous call to one of the following memory allocation
    APIs - :py:obj:`~.cuMemAlloc()`, :py:obj:`~.cuMemAllocPitch()`,
    :py:obj:`~.cuMemAllocManaged()`, :py:obj:`~.cuMemAllocAsync()`,
    :py:obj:`~.cuMemAllocFromPoolAsync()`

    Note - This API will not perform any implict synchronization when the
    pointer was allocated with :py:obj:`~.cuMemAllocAsync` or
    :py:obj:`~.cuMemAllocFromPoolAsync`. Callers must ensure that all
    accesses to the pointer have completed before invoking
    :py:obj:`~.cuMemFree`. For best performance and memory reuse, users
    should use :py:obj:`~.cuMemFreeAsync` to free memory allocated via the
    stream ordered memory allocator.

    Parameters
    ----------
    dptr : :py:obj:`~.CUdeviceptr`
        Pointer to memory to free

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemAllocManaged`, :py:obj:`~.cuMemAllocAsync`, :py:obj:`~.cuMemAllocFromPoolAsync`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemFreeAsync`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaFree`
    """
    cdef ccuda.CUdeviceptr cdptr
    if dptr is None:
        cdptr = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dptr, (CUdeviceptr,)):
        pdptr = int(dptr)
        cdptr = <ccuda.CUdeviceptr><void_ptr>pdptr
    else:
        pdptr = int(CUdeviceptr(dptr))
        cdptr = <ccuda.CUdeviceptr><void_ptr>pdptr
    err = ccuda.cuMemFree(cdptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemGetAddressRange(dptr):
    """ Get information on memory allocations.

    Returns the base address in `*pbase` and size in `*psize` of the
    allocation by :py:obj:`~.cuMemAlloc()` or :py:obj:`~.cuMemAllocPitch()`
    that contains the input pointer `dptr`. Both parameters `pbase` and
    `psize` are optional. If one of them is NULL, it is ignored.

    Parameters
    ----------
    dptr : :py:obj:`~.CUdeviceptr`
        Device pointer to query

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_NOT_FOUND`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pbase : :py:obj:`~.CUdeviceptr`
        Returned base address
    psize : int
        Returned size of device memory allocation

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`
    """
    cdef ccuda.CUdeviceptr cdptr
    if dptr is None:
        cdptr = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dptr, (CUdeviceptr,)):
        pdptr = int(dptr)
        cdptr = <ccuda.CUdeviceptr><void_ptr>pdptr
    else:
        pdptr = int(CUdeviceptr(dptr))
        cdptr = <ccuda.CUdeviceptr><void_ptr>pdptr
    cdef CUdeviceptr pbase = CUdeviceptr()
    cdef size_t psize = 0
    err = ccuda.cuMemGetAddressRange(<ccuda.CUdeviceptr*>pbase._ptr, &psize, cdptr)
    return (CUresult(err), pbase, psize)

@cython.embedsignature(True)
def cuMemAllocHost(size_t bytesize):
    """ Allocates page-locked host memory.

    Allocates `bytesize` bytes of host memory that is page-locked and
    accessible to the device. The driver tracks the virtual memory ranges
    allocated with this function and automatically accelerates calls to
    functions such as :py:obj:`~.cuMemcpy()`. Since the memory can be
    accessed directly by the device, it can be read or written with much
    higher bandwidth than pageable memory obtained with functions such as
    :py:obj:`~.malloc()`.

    On systems where
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS_USES_HOST_PAGE_TABLES`
    is true, :py:obj:`~.cuMemAllocHost` may not page-lock the allocated
    memory.

    Page-locking excessive amounts of memory with
    :py:obj:`~.cuMemAllocHost()` may degrade system performance, since it
    reduces the amount of memory available to the system for paging. As a
    result, this function is best used sparingly to allocate staging areas
    for data exchange between host and device.

    Note all host memory allocated using :py:obj:`~.cuMemAllocHost()` will
    automatically be immediately accessible to all contexts on all devices
    which support unified addressing (as may be queried using
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_UNIFIED_ADDRESSING`). The device pointer
    that may be used to access this host memory from those contexts is
    always equal to the returned host pointer `*pp`. See :py:obj:`~.Unified
    Addressing` for additional details.

    Parameters
    ----------
    bytesize : size_t
        Requested allocation size in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    pp : Any
        Returned pointer to host memory

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMallocHost`
    """
    cdef void_ptr pp = 0
    err = ccuda.cuMemAllocHost(<void**>&pp, bytesize)
    return (CUresult(err), pp)

@cython.embedsignature(True)
def cuMemFreeHost(p):
    """ Frees page-locked host memory.

    Frees the memory space pointed to by `p`, which must have been returned
    by a previous call to :py:obj:`~.cuMemAllocHost()`.

    Parameters
    ----------
    p : Any
        Pointer to memory to free

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaFreeHost`
    """
    cp = utils.HelperInputVoidPtr(p)
    cdef void* cp_ptr = <void*><void_ptr>cp.cptr
    err = ccuda.cuMemFreeHost(cp_ptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemHostAlloc(size_t bytesize, unsigned int Flags):
    """ Allocates page-locked host memory.

    Allocates `bytesize` bytes of host memory that is page-locked and
    accessible to the device. The driver tracks the virtual memory ranges
    allocated with this function and automatically accelerates calls to
    functions such as :py:obj:`~.cuMemcpyHtoD()`. Since the memory can be
    accessed directly by the device, it can be read or written with much
    higher bandwidth than pageable memory obtained with functions such as
    :py:obj:`~.malloc()`.

    On systems where
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS_USES_HOST_PAGE_TABLES`
    is true, :py:obj:`~.cuMemHostAlloc` may not page-lock the allocated
    memory.

    Page-locking excessive amounts of memory may degrade system
    performance, since it reduces the amount of memory available to the
    system for paging. As a result, this function is best used sparingly to
    allocate staging areas for data exchange between host and device.

    The `Flags` parameter enables different options to be specified that
    affect the allocation, as follows.

    - :py:obj:`~.CU_MEMHOSTALLOC_PORTABLE`: The memory returned by this
      call will be considered as pinned memory by all CUDA contexts, not
      just the one that performed the allocation.

    - :py:obj:`~.CU_MEMHOSTALLOC_DEVICEMAP`: Maps the allocation into the
      CUDA address space. The device pointer to the memory may be obtained
      by calling :py:obj:`~.cuMemHostGetDevicePointer()`.

    - :py:obj:`~.CU_MEMHOSTALLOC_WRITECOMBINED`: Allocates the memory as
      write-combined (WC). WC memory can be transferred across the PCI
      Express bus more quickly on some system configurations, but cannot be
      read efficiently by most CPUs. WC memory is a good option for buffers
      that will be written by the CPU and read by the GPU via mapped pinned
      memory or host->device transfers.

    All of these flags are orthogonal to one another: a developer may
    allocate memory that is portable, mapped and/or write-combined with no
    restrictions.

    The :py:obj:`~.CU_MEMHOSTALLOC_DEVICEMAP` flag may be specified on CUDA
    contexts for devices that do not support mapped pinned memory. The
    failure is deferred to :py:obj:`~.cuMemHostGetDevicePointer()` because
    the memory may be mapped into other CUDA contexts via the
    :py:obj:`~.CU_MEMHOSTALLOC_PORTABLE` flag.

    The memory allocated by this function must be freed with
    :py:obj:`~.cuMemFreeHost()`.

    Note all host memory allocated using :py:obj:`~.cuMemHostAlloc()` will
    automatically be immediately accessible to all contexts on all devices
    which support unified addressing (as may be queried using
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_UNIFIED_ADDRESSING`). Unless the flag
    :py:obj:`~.CU_MEMHOSTALLOC_WRITECOMBINED` is specified, the device
    pointer that may be used to access this host memory from those contexts
    is always equal to the returned host pointer `*pp`. If the flag
    :py:obj:`~.CU_MEMHOSTALLOC_WRITECOMBINED` is specified, then the
    function :py:obj:`~.cuMemHostGetDevicePointer()` must be used to query
    the device pointer, even if the context supports unified addressing.
    See :py:obj:`~.Unified Addressing` for additional details.

    Parameters
    ----------
    bytesize : size_t
        Requested allocation size in bytes
    Flags : unsigned int
        Flags for allocation request

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    pp : Any
        Returned pointer to host memory

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaHostAlloc`
    """
    cdef void_ptr pp = 0
    err = ccuda.cuMemHostAlloc(<void**>&pp, bytesize, Flags)
    return (CUresult(err), pp)

@cython.embedsignature(True)
def cuMemHostGetDevicePointer(p, unsigned int Flags):
    """ Passes back device pointer of mapped pinned memory.

    Passes back the device pointer `pdptr` corresponding to the mapped,
    pinned host buffer `p` allocated by :py:obj:`~.cuMemHostAlloc`.

    :py:obj:`~.cuMemHostGetDevicePointer()` will fail if the
    :py:obj:`~.CU_MEMHOSTALLOC_DEVICEMAP` flag was not specified at the
    time the memory was allocated, or if the function is called on a GPU
    that does not support mapped pinned memory.

    For devices that have a non-zero value for the device attribute
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_CAN_USE_HOST_POINTER_FOR_REGISTERED_MEM`,
    the memory can also be accessed from the device using the host pointer
    `p`. The device pointer returned by
    :py:obj:`~.cuMemHostGetDevicePointer()` may or may not match the
    original host pointer `p` and depends on the devices visible to the
    application. If all devices visible to the application have a non-zero
    value for the device attribute, the device pointer returned by
    :py:obj:`~.cuMemHostGetDevicePointer()` will match the original pointer
    `p`. If any device visible to the application has a zero value for the
    device attribute, the device pointer returned by
    :py:obj:`~.cuMemHostGetDevicePointer()` will not match the original
    host pointer `p`, but it will be suitable for use on all devices
    provided Unified Virtual Addressing is enabled. In such systems, it is
    valid to access the memory using either pointer on devices that have a
    non-zero value for the device attribute. Note however that such devices
    should access the memory using only one of the two pointers and not
    both.

    `Flags` provides for future releases. For now, it must be set to 0.

    Parameters
    ----------
    p : Any
        Host pointer
    Flags : unsigned int
        Options (must be 0)

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pdptr : :py:obj:`~.CUdeviceptr`
        Returned device pointer

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaHostGetDevicePointer`
    """
    cdef CUdeviceptr pdptr = CUdeviceptr()
    cp = utils.HelperInputVoidPtr(p)
    cdef void* cp_ptr = <void*><void_ptr>cp.cptr
    err = ccuda.cuMemHostGetDevicePointer(<ccuda.CUdeviceptr*>pdptr._ptr, cp_ptr, Flags)
    return (CUresult(err), pdptr)

@cython.embedsignature(True)
def cuMemHostGetFlags(p):
    """ Passes back flags that were used for a pinned allocation.

    Passes back the flags `pFlags` that were specified when allocating the
    pinned host buffer `p` allocated by :py:obj:`~.cuMemHostAlloc`.

    :py:obj:`~.cuMemHostGetFlags()` will fail if the pointer does not
    reside in an allocation performed by :py:obj:`~.cuMemAllocHost()` or
    :py:obj:`~.cuMemHostAlloc()`.

    Parameters
    ----------
    p : Any
        Host pointer

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pFlags : unsigned int
        Returned flags word

    See Also
    --------
    :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cudaHostGetFlags`
    """
    cdef unsigned int pFlags = 0
    cp = utils.HelperInputVoidPtr(p)
    cdef void* cp_ptr = <void*><void_ptr>cp.cptr
    err = ccuda.cuMemHostGetFlags(&pFlags, cp_ptr)
    return (CUresult(err), pFlags)

@cython.embedsignature(True)
def cuMemAllocManaged(size_t bytesize, unsigned int flags):
    """ Allocates memory that will be automatically managed by the Unified Memory system.

    Allocates `bytesize` bytes of managed memory on the device and returns
    in `*dptr` a pointer to the allocated memory. If the device doesn't
    support allocating managed memory, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`
    is returned. Support for managed memory can be queried using the device
    attribute :py:obj:`~.CU_DEVICE_ATTRIBUTE_MANAGED_MEMORY`. The allocated
    memory is suitably aligned for any kind of variable. The memory is not
    cleared. If `bytesize` is 0, :py:obj:`~.cuMemAllocManaged` returns
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE`. The pointer is valid on the CPU
    and on all GPUs in the system that support managed memory. All accesses
    to this pointer must obey the Unified Memory programming model.

    `flags` specifies the default stream association for this allocation.
    `flags` must be one of :py:obj:`~.CU_MEM_ATTACH_GLOBAL` or
    :py:obj:`~.CU_MEM_ATTACH_HOST`. If :py:obj:`~.CU_MEM_ATTACH_GLOBAL` is
    specified, then this memory is accessible from any stream on any
    device. If :py:obj:`~.CU_MEM_ATTACH_HOST` is specified, then the
    allocation should not be accessed from devices that have a zero value
    for the device attribute
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS`; an explicit
    call to :py:obj:`~.cuStreamAttachMemAsync` will be required to enable
    access on such devices.

    If the association is later changed via
    :py:obj:`~.cuStreamAttachMemAsync` to a single stream, the default
    association as specified during :py:obj:`~.cuMemAllocManaged` is
    restored when that stream is destroyed. For managed variables, the
    default association is always :py:obj:`~.CU_MEM_ATTACH_GLOBAL`. Note
    that destroying a stream is an asynchronous operation, and as a result,
    the change to default association won't happen until all work in the
    stream has completed.

    Memory allocated with :py:obj:`~.cuMemAllocManaged` should be released
    with :py:obj:`~.cuMemFree`.

    Device memory oversubscription is possible for GPUs that have a non-
    zero value for the device attribute
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS`. Managed
    memory on such GPUs may be evicted from device memory to host memory at
    any time by the Unified Memory driver in order to make room for other
    allocations.

    In a system where all GPUs have a non-zero value for the device
    attribute :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS`,
    managed memory may not be populated when this API returns and instead
    may be populated on access. In such systems, managed memory can migrate
    to any processor's memory at any time. The Unified Memory driver will
    employ heuristics to maintain data locality and prevent excessive page
    faults to the extent possible. The application can also guide the
    driver about memory usage patterns via :py:obj:`~.cuMemAdvise`. The
    application can also explicitly migrate memory to a desired processor's
    memory via :py:obj:`~.cuMemPrefetchAsync`.

    In a multi-GPU system where all of the GPUs have a zero value for the
    device attribute
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS` and all the
    GPUs have peer-to-peer support with each other, the physical storage
    for managed memory is created on the GPU which is active at the time
    :py:obj:`~.cuMemAllocManaged` is called. All other GPUs will reference
    the data at reduced bandwidth via peer mappings over the PCIe bus. The
    Unified Memory driver does not migrate memory among such GPUs.

    In a multi-GPU system where not all GPUs have peer-to-peer support with
    each other and where the value of the device attribute
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS` is zero for
    at least one of those GPUs, the location chosen for physical storage of
    managed memory is system-dependent.

    - On Linux, the location chosen will be device memory as long as the
      current set of active contexts are on devices that either have peer-
      to-peer support with each other or have a non-zero value for the
      device attribute
      :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS`. If there
      is an active context on a GPU that does not have a non-zero value for
      that device attribute and it does not have peer-to-peer support with
      the other devices that have active contexts on them, then the
      location for physical storage will be 'zero-copy' or host memory.
      Note that this means that managed memory that is located in device
      memory is migrated to host memory if a new context is created on a
      GPU that doesn't have a non-zero value for the device attribute and
      does not support peer-to-peer with at least one of the other devices
      that has an active context. This in turn implies that context
      creation may fail if there is insufficient host memory to migrate all
      managed allocations.

    - On Windows, the physical storage is always created in 'zero-copy' or
      host memory. All GPUs will reference the data at reduced bandwidth
      over the PCIe bus. In these circumstances, use of the environment
      variable CUDA_VISIBLE_DEVICES is recommended to restrict CUDA to only
      use those GPUs that have peer-to-peer support. Alternatively, users
      can also set CUDA_MANAGED_FORCE_DEVICE_ALLOC to a non-zero value to
      force the driver to always use device memory for physical storage.
      When this environment variable is set to a non-zero value, all
      contexts created in that process on devices that support managed
      memory have to be peer-to-peer compatible with each other. Context
      creation will fail if a context is created on a device that supports
      managed memory and is not peer-to-peer compatible with any of the
      other managed memory supporting devices on which contexts were
      previously created, even if those contexts have been destroyed. These
      environment variables are described in the CUDA programming guide
      under the "CUDA environment variables" section.

    - On ARM, managed memory is not available on discrete gpu with Drive
      PX-2.

    Parameters
    ----------
    bytesize : size_t
        Requested allocation size in bytes
    flags : unsigned int
        Must be one of :py:obj:`~.CU_MEM_ATTACH_GLOBAL` or
        :py:obj:`~.CU_MEM_ATTACH_HOST`

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    dptr : :py:obj:`~.CUdeviceptr`
        Returned device pointer

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuStreamAttachMemAsync`, :py:obj:`~.cudaMallocManaged`
    """
    cdef CUdeviceptr dptr = CUdeviceptr()
    err = ccuda.cuMemAllocManaged(<ccuda.CUdeviceptr*>dptr._ptr, bytesize, flags)
    return (CUresult(err), dptr)

@cython.embedsignature(True)
def cuDeviceRegisterAsyncNotification(device, callbackFunc, userData):
    """ Registers a callback function to receive async notifications.

    Registers `callbackFunc` to receive async notifications.

    The `userData` parameter is passed to the callback function at async
    notification time.   Likewise, `callback` is also passed to the
    callback function to distinguish between multiple registered callbacks.

    The callback function being registered should be designed to return
    quickly (~10ms).   Any long running tasks should be queued for
    execution on an application thread.

    Callbacks may not call cuDeviceRegisterAsyncNotification or
    cuDeviceUnregisterAsyncNotification. Doing so will result in
    :py:obj:`~.CUDA_ERROR_NOT_PERMITTED`. Async notification callbacks
    execute in an undefined order and may be serialized.

    Returns in `*callback` a handle representing the registered callback
    instance.

    Parameters
    ----------
    device : :py:obj:`~.CUdevice`
        The device on which to register the callback
    callbackFunc : :py:obj:`~.CUasyncCallback`
        The function to register as a callback
    userData : Any
        A generic pointer to user data. This is passed into the callback
        function.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS` :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED` :py:obj:`~.CUDA_ERROR_INVALID_DEVICE` :py:obj:`~.CUDA_ERROR_INVALID_VALUE` :py:obj:`~.CUDA_ERROR_NOT_PERMITTED` :py:obj:`~.CUDA_ERROR_UNKNOWN`
    callback : :py:obj:`~.CUasyncCallbackHandle`
        A handle representing the registered callback instance

    See Also
    --------
    :py:obj:`~.cuDeviceUnregisterAsyncNotification`
    """
    cdef ccuda.CUasyncCallback ccallbackFunc
    if callbackFunc is None:
        ccallbackFunc = <ccuda.CUasyncCallback><void_ptr>0
    elif isinstance(callbackFunc, (CUasyncCallback,)):
        pcallbackFunc = int(callbackFunc)
        ccallbackFunc = <ccuda.CUasyncCallback><void_ptr>pcallbackFunc
    else:
        pcallbackFunc = int(CUasyncCallback(callbackFunc))
        ccallbackFunc = <ccuda.CUasyncCallback><void_ptr>pcallbackFunc
    cdef ccuda.CUdevice cdevice
    if device is None:
        cdevice = <ccuda.CUdevice>0
    elif isinstance(device, (CUdevice,)):
        pdevice = int(device)
        cdevice = <ccuda.CUdevice>pdevice
    else:
        pdevice = int(CUdevice(device))
        cdevice = <ccuda.CUdevice>pdevice
    cuserData = utils.HelperInputVoidPtr(userData)
    cdef void* cuserData_ptr = <void*><void_ptr>cuserData.cptr
    cdef CUasyncCallbackHandle callback = CUasyncCallbackHandle()
    err = ccuda.cuDeviceRegisterAsyncNotification(cdevice, ccallbackFunc, cuserData_ptr, <ccuda.CUasyncCallbackHandle*>callback._ptr)
    return (CUresult(err), callback)

@cython.embedsignature(True)
def cuDeviceUnregisterAsyncNotification(device, callback):
    """ Unregisters an async notification callback.

    Unregisters `callback` so that the corresponding callback function will
    stop receiving async notifications.

    Parameters
    ----------
    device : :py:obj:`~.CUdevice`
        The device from which to remove `callback`.
    callback : :py:obj:`~.CUasyncCallbackHandle`
        The callback instance to unregister from receiving async
        notifications.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS` :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED` :py:obj:`~.CUDA_ERROR_INVALID_DEVICE` :py:obj:`~.CUDA_ERROR_INVALID_VALUE` :py:obj:`~.CUDA_ERROR_NOT_PERMITTED` :py:obj:`~.CUDA_ERROR_UNKNOWN`

    See Also
    --------
    :py:obj:`~.cuDeviceRegisterAsyncNotification`
    """
    cdef ccuda.CUasyncCallbackHandle ccallback
    if callback is None:
        ccallback = <ccuda.CUasyncCallbackHandle><void_ptr>0
    elif isinstance(callback, (CUasyncCallbackHandle,)):
        pcallback = int(callback)
        ccallback = <ccuda.CUasyncCallbackHandle><void_ptr>pcallback
    else:
        pcallback = int(CUasyncCallbackHandle(callback))
        ccallback = <ccuda.CUasyncCallbackHandle><void_ptr>pcallback
    cdef ccuda.CUdevice cdevice
    if device is None:
        cdevice = <ccuda.CUdevice>0
    elif isinstance(device, (CUdevice,)):
        pdevice = int(device)
        cdevice = <ccuda.CUdevice>pdevice
    else:
        pdevice = int(CUdevice(device))
        cdevice = <ccuda.CUdevice>pdevice
    err = ccuda.cuDeviceUnregisterAsyncNotification(cdevice, ccallback)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuDeviceGetByPCIBusId(char* pciBusId):
    """ Returns a handle to a compute device.

    Returns in `*device` a device handle given a PCI bus ID string.

    where `domain`, `bus`, `device`, and `function` are all hexadecimal
    values

    Parameters
    ----------
    pciBusId : bytes
        String in one of the following forms:

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    dev : :py:obj:`~.CUdevice`
        Returned device handle

    See Also
    --------
    :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetPCIBusId`, :py:obj:`~.cudaDeviceGetByPCIBusId`
    """
    cdef CUdevice dev = CUdevice()
    err = ccuda.cuDeviceGetByPCIBusId(<ccuda.CUdevice*>dev._ptr, pciBusId)
    return (CUresult(err), dev)

@cython.embedsignature(True)
def cuDeviceGetPCIBusId(int length, dev):
    """ Returns a PCI Bus Id string for the device.

    Returns an ASCII string identifying the device `dev` in the NULL-
    terminated string pointed to by `pciBusId`. `length` specifies the
    maximum length of the string that may be returned.

    where `domain`, `bus`, `device`, and `function` are all hexadecimal
    values. pciBusId should be large enough to store 13 characters
    including the NULL-terminator.

    Parameters
    ----------
    length : int
        Maximum length of string to store in `name`
    dev : :py:obj:`~.CUdevice`
        Device to get identifier string for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    pciBusId : bytes
        Returned identifier string for the device in the following format

    See Also
    --------
    :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetByPCIBusId`, :py:obj:`~.cudaDeviceGetPCIBusId`
    """
    cdef ccuda.CUdevice cdev
    if dev is None:
        cdev = <ccuda.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cdev = <ccuda.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cdev = <ccuda.CUdevice>pdev
    pypciBusId = b" " * length
    cdef char* pciBusId = pypciBusId
    err = ccuda.cuDeviceGetPCIBusId(pciBusId, length, cdev)
    return (CUresult(err), pypciBusId)

@cython.embedsignature(True)
def cuIpcGetEventHandle(event):
    """ Gets an interprocess handle for a previously allocated event.

    Takes as input a previously allocated event. This event must have been
    created with the :py:obj:`~.CU_EVENT_INTERPROCESS` and
    :py:obj:`~.CU_EVENT_DISABLE_TIMING` flags set. This opaque handle may
    be copied into other processes and opened with
    :py:obj:`~.cuIpcOpenEventHandle` to allow efficient hardware
    synchronization between GPU work in different processes.

    After the event has been opened in the importing process,
    :py:obj:`~.cuEventRecord`, :py:obj:`~.cuEventSynchronize`,
    :py:obj:`~.cuStreamWaitEvent` and :py:obj:`~.cuEventQuery` may be used
    in either process. Performing operations on the imported event after
    the exported event has been freed with :py:obj:`~.cuEventDestroy` will
    result in undefined behavior.

    IPC functionality is restricted to devices with support for unified
    addressing on Linux and Windows operating systems. IPC functionality on
    Windows is restricted to GPUs in TCC mode Users can test their device
    for IPC functionality by calling :py:obj:`~.cuapiDeviceGetAttribute`
    with :py:obj:`~.CU_DEVICE_ATTRIBUTE_IPC_EVENT_SUPPORTED`

    Parameters
    ----------
    event : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Event allocated with :py:obj:`~.CU_EVENT_INTERPROCESS` and
        :py:obj:`~.CU_EVENT_DISABLE_TIMING` flags.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_MAP_FAILED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pHandle : :py:obj:`~.CUipcEventHandle`
        Pointer to a user allocated CUipcEventHandle in which to return the
        opaque event handle

    See Also
    --------
    :py:obj:`~.cuEventCreate`, :py:obj:`~.cuEventDestroy`, :py:obj:`~.cuEventSynchronize`, :py:obj:`~.cuEventQuery`, :py:obj:`~.cuStreamWaitEvent`, :py:obj:`~.cuIpcOpenEventHandle`, :py:obj:`~.cuIpcGetMemHandle`, :py:obj:`~.cuIpcOpenMemHandle`, :py:obj:`~.cuIpcCloseMemHandle`, :py:obj:`~.cudaIpcGetEventHandle`
    """
    cdef ccuda.CUevent cevent
    if event is None:
        cevent = <ccuda.CUevent><void_ptr>0
    elif isinstance(event, (CUevent,)):
        pevent = int(event)
        cevent = <ccuda.CUevent><void_ptr>pevent
    else:
        pevent = int(CUevent(event))
        cevent = <ccuda.CUevent><void_ptr>pevent
    cdef CUipcEventHandle pHandle = CUipcEventHandle()
    err = ccuda.cuIpcGetEventHandle(<ccuda.CUipcEventHandle*>pHandle._ptr, cevent)
    return (CUresult(err), pHandle)

@cython.embedsignature(True)
def cuIpcOpenEventHandle(handle not None : CUipcEventHandle):
    """ Opens an interprocess event handle for use in the current process.

    Opens an interprocess event handle exported from another process with
    :py:obj:`~.cuIpcGetEventHandle`. This function returns a
    :py:obj:`~.CUevent` that behaves like a locally created event with the
    :py:obj:`~.CU_EVENT_DISABLE_TIMING` flag specified. This event must be
    freed with :py:obj:`~.cuEventDestroy`.

    Performing operations on the imported event after the exported event
    has been freed with :py:obj:`~.cuEventDestroy` will result in undefined
    behavior.

    IPC functionality is restricted to devices with support for unified
    addressing on Linux and Windows operating systems. IPC functionality on
    Windows is restricted to GPUs in TCC mode Users can test their device
    for IPC functionality by calling :py:obj:`~.cuapiDeviceGetAttribute`
    with :py:obj:`~.CU_DEVICE_ATTRIBUTE_IPC_EVENT_SUPPORTED`

    Parameters
    ----------
    handle : :py:obj:`~.CUipcEventHandle`
        Interprocess handle to open

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_MAP_FAILED`, :py:obj:`~.CUDA_ERROR_PEER_ACCESS_UNSUPPORTED`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    phEvent : :py:obj:`~.CUevent`
        Returns the imported event

    See Also
    --------
    :py:obj:`~.cuEventCreate`, :py:obj:`~.cuEventDestroy`, :py:obj:`~.cuEventSynchronize`, :py:obj:`~.cuEventQuery`, :py:obj:`~.cuStreamWaitEvent`, :py:obj:`~.cuIpcGetEventHandle`, :py:obj:`~.cuIpcGetMemHandle`, :py:obj:`~.cuIpcOpenMemHandle`, :py:obj:`~.cuIpcCloseMemHandle`, :py:obj:`~.cudaIpcOpenEventHandle`
    """
    cdef CUevent phEvent = CUevent()
    err = ccuda.cuIpcOpenEventHandle(<ccuda.CUevent*>phEvent._ptr, handle._ptr[0])
    return (CUresult(err), phEvent)

@cython.embedsignature(True)
def cuIpcGetMemHandle(dptr):
    """ Gets an interprocess memory handle for an existing device memory allocation.

    Takes a pointer to the base of an existing device memory allocation
    created with :py:obj:`~.cuMemAlloc` and exports it for use in another
    process. This is a lightweight operation and may be called multiple
    times on an allocation without adverse effects.

    If a region of memory is freed with :py:obj:`~.cuMemFree` and a
    subsequent call to :py:obj:`~.cuMemAlloc` returns memory with the same
    device address, :py:obj:`~.cuIpcGetMemHandle` will return a unique
    handle for the new memory.

    IPC functionality is restricted to devices with support for unified
    addressing on Linux and Windows operating systems. IPC functionality on
    Windows is restricted to GPUs in TCC mode Users can test their device
    for IPC functionality by calling :py:obj:`~.cuapiDeviceGetAttribute`
    with :py:obj:`~.CU_DEVICE_ATTRIBUTE_IPC_EVENT_SUPPORTED`

    Parameters
    ----------
    dptr : :py:obj:`~.CUdeviceptr`
        Base pointer to previously allocated device memory

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_MAP_FAILED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pHandle : :py:obj:`~.CUipcMemHandle`
        Pointer to user allocated :py:obj:`~.CUipcMemHandle` to return the
        handle in.

    See Also
    --------
    :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuIpcGetEventHandle`, :py:obj:`~.cuIpcOpenEventHandle`, :py:obj:`~.cuIpcOpenMemHandle`, :py:obj:`~.cuIpcCloseMemHandle`, :py:obj:`~.cudaIpcGetMemHandle`
    """
    cdef ccuda.CUdeviceptr cdptr
    if dptr is None:
        cdptr = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dptr, (CUdeviceptr,)):
        pdptr = int(dptr)
        cdptr = <ccuda.CUdeviceptr><void_ptr>pdptr
    else:
        pdptr = int(CUdeviceptr(dptr))
        cdptr = <ccuda.CUdeviceptr><void_ptr>pdptr
    cdef CUipcMemHandle pHandle = CUipcMemHandle()
    err = ccuda.cuIpcGetMemHandle(<ccuda.CUipcMemHandle*>pHandle._ptr, cdptr)
    return (CUresult(err), pHandle)

@cython.embedsignature(True)
def cuIpcOpenMemHandle(handle not None : CUipcMemHandle, unsigned int Flags):
    """ Opens an interprocess memory handle exported from another process and returns a device pointer usable in the local process.

    Maps memory exported from another process with
    :py:obj:`~.cuIpcGetMemHandle` into the current device address space.
    For contexts on different devices :py:obj:`~.cuIpcOpenMemHandle` can
    attempt to enable peer access between the devices as if the user called
    :py:obj:`~.cuCtxEnablePeerAccess`. This behavior is controlled by the
    :py:obj:`~.CU_IPC_MEM_LAZY_ENABLE_PEER_ACCESS` flag.
    :py:obj:`~.cuDeviceCanAccessPeer` can determine if a mapping is
    possible.

    Contexts that may open :py:obj:`~.CUipcMemHandles` are restricted in
    the following way. :py:obj:`~.CUipcMemHandles` from each
    :py:obj:`~.CUdevice` in a given process may only be opened by one
    :py:obj:`~.CUcontext` per :py:obj:`~.CUdevice` per other process.

    If the memory handle has already been opened by the current context,
    the reference count on the handle is incremented by 1 and the existing
    device pointer is returned.

    Memory returned from :py:obj:`~.cuIpcOpenMemHandle` must be freed with
    :py:obj:`~.cuIpcCloseMemHandle`.

    Calling :py:obj:`~.cuMemFree` on an exported memory region before
    calling :py:obj:`~.cuIpcCloseMemHandle` in the importing context will
    result in undefined behavior.

    IPC functionality is restricted to devices with support for unified
    addressing on Linux and Windows operating systems. IPC functionality on
    Windows is restricted to GPUs in TCC mode Users can test their device
    for IPC functionality by calling :py:obj:`~.cuapiDeviceGetAttribute`
    with :py:obj:`~.CU_DEVICE_ATTRIBUTE_IPC_EVENT_SUPPORTED`

    Parameters
    ----------
    handle : :py:obj:`~.CUipcMemHandle`
        :py:obj:`~.CUipcMemHandle` to open
    Flags : unsigned int
        Flags for this operation. Must be specified as
        :py:obj:`~.CU_IPC_MEM_LAZY_ENABLE_PEER_ACCESS`

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_MAP_FAILED`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_TOO_MANY_PEERS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pdptr : :py:obj:`~.CUdeviceptr`
        Returned device pointer

    See Also
    --------
    :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuIpcGetEventHandle`, :py:obj:`~.cuIpcOpenEventHandle`, :py:obj:`~.cuIpcGetMemHandle`, :py:obj:`~.cuIpcCloseMemHandle`, :py:obj:`~.cuCtxEnablePeerAccess`, :py:obj:`~.cuDeviceCanAccessPeer`, :py:obj:`~.cudaIpcOpenMemHandle`

    Notes
    -----
    No guarantees are made about the address returned in `*pdptr`. In particular, multiple processes may not receive the same address for the same `handle`.
    """
    cdef CUdeviceptr pdptr = CUdeviceptr()
    err = ccuda.cuIpcOpenMemHandle(<ccuda.CUdeviceptr*>pdptr._ptr, handle._ptr[0], Flags)
    return (CUresult(err), pdptr)

@cython.embedsignature(True)
def cuIpcCloseMemHandle(dptr):
    """ Attempts to close memory mapped with :py:obj:`~.cuIpcOpenMemHandle`.

    Decrements the reference count of the memory returned by
    :py:obj:`~.cuIpcOpenMemHandle` by 1. When the reference count reaches
    0, this API unmaps the memory. The original allocation in the exporting
    process as well as imported mappings in other processes will be
    unaffected.

    Any resources used to enable peer access will be freed if this is the
    last mapping using them.

    IPC functionality is restricted to devices with support for unified
    addressing on Linux and Windows operating systems. IPC functionality on
    Windows is restricted to GPUs in TCC mode Users can test their device
    for IPC functionality by calling :py:obj:`~.cuapiDeviceGetAttribute`
    with :py:obj:`~.CU_DEVICE_ATTRIBUTE_IPC_EVENT_SUPPORTED`

    Parameters
    ----------
    dptr : :py:obj:`~.CUdeviceptr`
        Device pointer returned by :py:obj:`~.cuIpcOpenMemHandle`

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_MAP_FAILED`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuIpcGetEventHandle`, :py:obj:`~.cuIpcOpenEventHandle`, :py:obj:`~.cuIpcGetMemHandle`, :py:obj:`~.cuIpcOpenMemHandle`, :py:obj:`~.cudaIpcCloseMemHandle`
    """
    cdef ccuda.CUdeviceptr cdptr
    if dptr is None:
        cdptr = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dptr, (CUdeviceptr,)):
        pdptr = int(dptr)
        cdptr = <ccuda.CUdeviceptr><void_ptr>pdptr
    else:
        pdptr = int(CUdeviceptr(dptr))
        cdptr = <ccuda.CUdeviceptr><void_ptr>pdptr
    err = ccuda.cuIpcCloseMemHandle(cdptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemHostRegister(p, size_t bytesize, unsigned int Flags):
    """ Registers an existing host memory range for use by CUDA.

    Page-locks the memory range specified by `p` and `bytesize` and maps it
    for the device(s) as specified by `Flags`. This memory range also is
    added to the same tracking mechanism as :py:obj:`~.cuMemHostAlloc` to
    automatically accelerate calls to functions such as
    :py:obj:`~.cuMemcpyHtoD()`. Since the memory can be accessed directly
    by the device, it can be read or written with much higher bandwidth
    than pageable memory that has not been registered. Page-locking
    excessive amounts of memory may degrade system performance, since it
    reduces the amount of memory available to the system for paging. As a
    result, this function is best used sparingly to register staging areas
    for data exchange between host and device.

    On systems where
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS_USES_HOST_PAGE_TABLES`
    is true, :py:obj:`~.cuMemHostRegister` will not page-lock the memory
    range specified by `ptr` but only populate unpopulated pages.

    The `Flags` parameter enables different options to be specified that
    affect the allocation, as follows.

    - :py:obj:`~.CU_MEMHOSTREGISTER_PORTABLE`: The memory returned by this
      call will be considered as pinned memory by all CUDA contexts, not
      just the one that performed the allocation.

    - :py:obj:`~.CU_MEMHOSTREGISTER_DEVICEMAP`: Maps the allocation into
      the CUDA address space. The device pointer to the memory may be
      obtained by calling :py:obj:`~.cuMemHostGetDevicePointer()`.

    - :py:obj:`~.CU_MEMHOSTREGISTER_IOMEMORY`: The pointer is treated as
      pointing to some I/O memory space, e.g. the PCI Express resource of a
      3rd party device.

    - :py:obj:`~.CU_MEMHOSTREGISTER_READ_ONLY`: The pointer is treated as
      pointing to memory that is considered read-only by the device. On
      platforms without
      :py:obj:`~.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS_USES_HOST_PAGE_TABLES`,
      this flag is required in order to register memory mapped to the CPU
      as read-only. Support for the use of this flag can be queried from
      the device attribute
      :py:obj:`~.CU_DEVICE_ATTRIBUTE_READ_ONLY_HOST_REGISTER_SUPPORTED`.
      Using this flag with a current context associated with a device that
      does not have this attribute set will cause
      :py:obj:`~.cuMemHostRegister` to error with CUDA_ERROR_NOT_SUPPORTED.

    All of these flags are orthogonal to one another: a developer may page-
    lock memory that is portable or mapped with no restrictions.

    The :py:obj:`~.CU_MEMHOSTREGISTER_DEVICEMAP` flag may be specified on
    CUDA contexts for devices that do not support mapped pinned memory. The
    failure is deferred to :py:obj:`~.cuMemHostGetDevicePointer()` because
    the memory may be mapped into other CUDA contexts via the
    :py:obj:`~.CU_MEMHOSTREGISTER_PORTABLE` flag.

    For devices that have a non-zero value for the device attribute
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_CAN_USE_HOST_POINTER_FOR_REGISTERED_MEM`,
    the memory can also be accessed from the device using the host pointer
    `p`. The device pointer returned by
    :py:obj:`~.cuMemHostGetDevicePointer()` may or may not match the
    original host pointer `ptr` and depends on the devices visible to the
    application. If all devices visible to the application have a non-zero
    value for the device attribute, the device pointer returned by
    :py:obj:`~.cuMemHostGetDevicePointer()` will match the original pointer
    `ptr`. If any device visible to the application has a zero value for
    the device attribute, the device pointer returned by
    :py:obj:`~.cuMemHostGetDevicePointer()` will not match the original
    host pointer `ptr`, but it will be suitable for use on all devices
    provided Unified Virtual Addressing is enabled. In such systems, it is
    valid to access the memory using either pointer on devices that have a
    non-zero value for the device attribute. Note however that such devices
    should access the memory using only of the two pointers and not both.

    The memory page-locked by this function must be unregistered with
    :py:obj:`~.cuMemHostUnregister()`.

    Parameters
    ----------
    p : Any
        Host pointer to memory to page-lock
    bytesize : size_t
        Size in bytes of the address range to page-lock
    Flags : unsigned int
        Flags for allocation request

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_HOST_MEMORY_ALREADY_REGISTERED`, :py:obj:`~.CUDA_ERROR_NOT_PERMITTED`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`

    See Also
    --------
    :py:obj:`~.cuMemHostUnregister`, :py:obj:`~.cuMemHostGetFlags`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cudaHostRegister`
    """
    cp = utils.HelperInputVoidPtr(p)
    cdef void* cp_ptr = <void*><void_ptr>cp.cptr
    err = ccuda.cuMemHostRegister(cp_ptr, bytesize, Flags)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemHostUnregister(p):
    """ Unregisters a memory range that was registered with cuMemHostRegister.

    Unmaps the memory range whose base address is specified by `p`, and
    makes it pageable again.

    The base address must be the same one specified to
    :py:obj:`~.cuMemHostRegister()`.

    Parameters
    ----------
    p : Any
        Host pointer to memory to unregister

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_HOST_MEMORY_NOT_REGISTERED`,

    See Also
    --------
    :py:obj:`~.cuMemHostRegister`, :py:obj:`~.cudaHostUnregister`
    """
    cp = utils.HelperInputVoidPtr(p)
    cdef void* cp_ptr = <void*><void_ptr>cp.cptr
    err = ccuda.cuMemHostUnregister(cp_ptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpy(dst, src, size_t ByteCount):
    """ Copies memory.

    Copies data between two pointers. `dst` and `src` are base pointers of
    the destination and source, respectively. `ByteCount` specifies the
    number of bytes to copy. Note that this function infers the type of the
    transfer (host to host, host to device, device to device, or device to
    host) from the pointer values. This function is only allowed in
    contexts which support unified addressing.

    Parameters
    ----------
    dst : :py:obj:`~.CUdeviceptr`
        Destination unified virtual address space pointer
    src : :py:obj:`~.CUdeviceptr`
        Source unified virtual address space pointer
    ByteCount : size_t
        Size of memory copy in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpyToSymbol`, :py:obj:`~.cudaMemcpyFromSymbol`
    """
    cdef ccuda.CUdeviceptr csrc
    if src is None:
        csrc = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(src, (CUdeviceptr,)):
        psrc = int(src)
        csrc = <ccuda.CUdeviceptr><void_ptr>psrc
    else:
        psrc = int(CUdeviceptr(src))
        csrc = <ccuda.CUdeviceptr><void_ptr>psrc
    cdef ccuda.CUdeviceptr cdst
    if dst is None:
        cdst = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dst, (CUdeviceptr,)):
        pdst = int(dst)
        cdst = <ccuda.CUdeviceptr><void_ptr>pdst
    else:
        pdst = int(CUdeviceptr(dst))
        cdst = <ccuda.CUdeviceptr><void_ptr>pdst
    err = ccuda.cuMemcpy(cdst, csrc, ByteCount)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyPeer(dstDevice, dstContext, srcDevice, srcContext, size_t ByteCount):
    """ Copies device memory between two contexts.

    Copies from device memory in one context to device memory in another
    context. `dstDevice` is the base device pointer of the destination
    memory and `dstContext` is the destination context. `srcDevice` is the
    base device pointer of the source memory and `srcContext` is the source
    pointer. `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    dstContext : :py:obj:`~.CUcontext`
        Destination context
    srcDevice : :py:obj:`~.CUdeviceptr`
        Source device pointer
    srcContext : :py:obj:`~.CUcontext`
        Source context
    ByteCount : size_t
        Size of memory copy in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpy3DPeer`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyPeerAsync`, :py:obj:`~.cuMemcpy3DPeerAsync`, :py:obj:`~.cudaMemcpyPeer`
    """
    cdef ccuda.CUcontext csrcContext
    if srcContext is None:
        csrcContext = <ccuda.CUcontext><void_ptr>0
    elif isinstance(srcContext, (CUcontext,)):
        psrcContext = int(srcContext)
        csrcContext = <ccuda.CUcontext><void_ptr>psrcContext
    else:
        psrcContext = int(CUcontext(srcContext))
        csrcContext = <ccuda.CUcontext><void_ptr>psrcContext
    cdef ccuda.CUdeviceptr csrcDevice
    if srcDevice is None:
        csrcDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(srcDevice, (CUdeviceptr,)):
        psrcDevice = int(srcDevice)
        csrcDevice = <ccuda.CUdeviceptr><void_ptr>psrcDevice
    else:
        psrcDevice = int(CUdeviceptr(srcDevice))
        csrcDevice = <ccuda.CUdeviceptr><void_ptr>psrcDevice
    cdef ccuda.CUcontext cdstContext
    if dstContext is None:
        cdstContext = <ccuda.CUcontext><void_ptr>0
    elif isinstance(dstContext, (CUcontext,)):
        pdstContext = int(dstContext)
        cdstContext = <ccuda.CUcontext><void_ptr>pdstContext
    else:
        pdstContext = int(CUcontext(dstContext))
        cdstContext = <ccuda.CUcontext><void_ptr>pdstContext
    cdef ccuda.CUdeviceptr cdstDevice
    if dstDevice is None:
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    err = ccuda.cuMemcpyPeer(cdstDevice, cdstContext, csrcDevice, csrcContext, ByteCount)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyHtoD(dstDevice, srcHost, size_t ByteCount):
    """ Copies memory from Host to Device.

    Copies from host memory to device memory. `dstDevice` and `srcHost` are
    the base addresses of the destination and source, respectively.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    srcHost : Any
        Source host pointer
    ByteCount : size_t
        Size of memory copy in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpyToSymbol`
    """
    cdef ccuda.CUdeviceptr cdstDevice
    if dstDevice is None:
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    csrcHost = utils.HelperInputVoidPtr(srcHost)
    cdef void* csrcHost_ptr = <void*><void_ptr>csrcHost.cptr
    err = ccuda.cuMemcpyHtoD(cdstDevice, csrcHost_ptr, ByteCount)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyDtoH(dstHost, srcDevice, size_t ByteCount):
    """ Copies memory from Device to Host.

    Copies from device to host memory. `dstHost` and `srcDevice` specify
    the base pointers of the destination and source, respectively.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstHost : Any
        Destination host pointer
    srcDevice : :py:obj:`~.CUdeviceptr`
        Source device pointer
    ByteCount : size_t
        Size of memory copy in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpyFromSymbol`
    """
    cdef ccuda.CUdeviceptr csrcDevice
    if srcDevice is None:
        csrcDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(srcDevice, (CUdeviceptr,)):
        psrcDevice = int(srcDevice)
        csrcDevice = <ccuda.CUdeviceptr><void_ptr>psrcDevice
    else:
        psrcDevice = int(CUdeviceptr(srcDevice))
        csrcDevice = <ccuda.CUdeviceptr><void_ptr>psrcDevice
    cdstHost = utils.HelperInputVoidPtr(dstHost)
    cdef void* cdstHost_ptr = <void*><void_ptr>cdstHost.cptr
    err = ccuda.cuMemcpyDtoH(cdstHost_ptr, csrcDevice, ByteCount)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyDtoD(dstDevice, srcDevice, size_t ByteCount):
    """ Copies memory from Device to Device.

    Copies from device memory to device memory. `dstDevice` and `srcDevice`
    are the base pointers of the destination and source, respectively.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    srcDevice : :py:obj:`~.CUdeviceptr`
        Source device pointer
    ByteCount : size_t
        Size of memory copy in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpyToSymbol`, :py:obj:`~.cudaMemcpyFromSymbol`
    """
    cdef ccuda.CUdeviceptr csrcDevice
    if srcDevice is None:
        csrcDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(srcDevice, (CUdeviceptr,)):
        psrcDevice = int(srcDevice)
        csrcDevice = <ccuda.CUdeviceptr><void_ptr>psrcDevice
    else:
        psrcDevice = int(CUdeviceptr(srcDevice))
        csrcDevice = <ccuda.CUdeviceptr><void_ptr>psrcDevice
    cdef ccuda.CUdeviceptr cdstDevice
    if dstDevice is None:
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    err = ccuda.cuMemcpyDtoD(cdstDevice, csrcDevice, ByteCount)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyDtoA(dstArray, size_t dstOffset, srcDevice, size_t ByteCount):
    """ Copies memory from Device to Array.

    Copies from device memory to a 1D CUDA array. `dstArray` and
    `dstOffset` specify the CUDA array handle and starting index of the
    destination data. `srcDevice` specifies the base pointer of the source.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstArray : :py:obj:`~.CUarray`
        Destination array
    dstOffset : size_t
        Offset in bytes of destination array
    srcDevice : :py:obj:`~.CUdeviceptr`
        Source device pointer
    ByteCount : size_t
        Size of memory copy in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpyToArray`
    """
    cdef ccuda.CUdeviceptr csrcDevice
    if srcDevice is None:
        csrcDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(srcDevice, (CUdeviceptr,)):
        psrcDevice = int(srcDevice)
        csrcDevice = <ccuda.CUdeviceptr><void_ptr>psrcDevice
    else:
        psrcDevice = int(CUdeviceptr(srcDevice))
        csrcDevice = <ccuda.CUdeviceptr><void_ptr>psrcDevice
    cdef ccuda.CUarray cdstArray
    if dstArray is None:
        cdstArray = <ccuda.CUarray><void_ptr>0
    elif isinstance(dstArray, (CUarray,)):
        pdstArray = int(dstArray)
        cdstArray = <ccuda.CUarray><void_ptr>pdstArray
    else:
        pdstArray = int(CUarray(dstArray))
        cdstArray = <ccuda.CUarray><void_ptr>pdstArray
    err = ccuda.cuMemcpyDtoA(cdstArray, dstOffset, csrcDevice, ByteCount)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyAtoD(dstDevice, srcArray, size_t srcOffset, size_t ByteCount):
    """ Copies memory from Array to Device.

    Copies from one 1D CUDA array to device memory. `dstDevice` specifies
    the base pointer of the destination and must be naturally aligned with
    the CUDA array elements. `srcArray` and `srcOffset` specify the CUDA
    array handle and the offset in bytes into the array where the copy is
    to begin. `ByteCount` specifies the number of bytes to copy and must be
    evenly divisible by the array element size.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    srcArray : :py:obj:`~.CUarray`
        Source array
    srcOffset : size_t
        Offset in bytes of source array
    ByteCount : size_t
        Size of memory copy in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpyFromArray`
    """
    cdef ccuda.CUarray csrcArray
    if srcArray is None:
        csrcArray = <ccuda.CUarray><void_ptr>0
    elif isinstance(srcArray, (CUarray,)):
        psrcArray = int(srcArray)
        csrcArray = <ccuda.CUarray><void_ptr>psrcArray
    else:
        psrcArray = int(CUarray(srcArray))
        csrcArray = <ccuda.CUarray><void_ptr>psrcArray
    cdef ccuda.CUdeviceptr cdstDevice
    if dstDevice is None:
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    err = ccuda.cuMemcpyAtoD(cdstDevice, csrcArray, srcOffset, ByteCount)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyHtoA(dstArray, size_t dstOffset, srcHost, size_t ByteCount):
    """ Copies memory from Host to Array.

    Copies from host memory to a 1D CUDA array. `dstArray` and `dstOffset`
    specify the CUDA array handle and starting offset in bytes of the
    destination data. `pSrc` specifies the base address of the source.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstArray : :py:obj:`~.CUarray`
        Destination array
    dstOffset : size_t
        Offset in bytes of destination array
    srcHost : Any
        Source host pointer
    ByteCount : size_t
        Size of memory copy in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpyToArray`
    """
    cdef ccuda.CUarray cdstArray
    if dstArray is None:
        cdstArray = <ccuda.CUarray><void_ptr>0
    elif isinstance(dstArray, (CUarray,)):
        pdstArray = int(dstArray)
        cdstArray = <ccuda.CUarray><void_ptr>pdstArray
    else:
        pdstArray = int(CUarray(dstArray))
        cdstArray = <ccuda.CUarray><void_ptr>pdstArray
    csrcHost = utils.HelperInputVoidPtr(srcHost)
    cdef void* csrcHost_ptr = <void*><void_ptr>csrcHost.cptr
    err = ccuda.cuMemcpyHtoA(cdstArray, dstOffset, csrcHost_ptr, ByteCount)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyAtoH(dstHost, srcArray, size_t srcOffset, size_t ByteCount):
    """ Copies memory from Array to Host.

    Copies from one 1D CUDA array to host memory. `dstHost` specifies the
    base pointer of the destination. `srcArray` and `srcOffset` specify the
    CUDA array handle and starting offset in bytes of the source data.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstHost : Any
        Destination device pointer
    srcArray : :py:obj:`~.CUarray`
        Source array
    srcOffset : size_t
        Offset in bytes of source array
    ByteCount : size_t
        Size of memory copy in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpyFromArray`
    """
    cdef ccuda.CUarray csrcArray
    if srcArray is None:
        csrcArray = <ccuda.CUarray><void_ptr>0
    elif isinstance(srcArray, (CUarray,)):
        psrcArray = int(srcArray)
        csrcArray = <ccuda.CUarray><void_ptr>psrcArray
    else:
        psrcArray = int(CUarray(srcArray))
        csrcArray = <ccuda.CUarray><void_ptr>psrcArray
    cdstHost = utils.HelperInputVoidPtr(dstHost)
    cdef void* cdstHost_ptr = <void*><void_ptr>cdstHost.cptr
    err = ccuda.cuMemcpyAtoH(cdstHost_ptr, csrcArray, srcOffset, ByteCount)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyAtoA(dstArray, size_t dstOffset, srcArray, size_t srcOffset, size_t ByteCount):
    """ Copies memory from Array to Array.

    Copies from one 1D CUDA array to another. `dstArray` and `srcArray`
    specify the handles of the destination and source CUDA arrays for the
    copy, respectively. `dstOffset` and `srcOffset` specify the destination
    and source offsets in bytes into the CUDA arrays. `ByteCount` is the
    number of bytes to be copied. The size of the elements in the CUDA
    arrays need not be the same format, but the elements must be the same
    size; and count must be evenly divisible by that size.

    Parameters
    ----------
    dstArray : :py:obj:`~.CUarray`
        Destination array
    dstOffset : size_t
        Offset in bytes of destination array
    srcArray : :py:obj:`~.CUarray`
        Source array
    srcOffset : size_t
        Offset in bytes of source array
    ByteCount : size_t
        Size of memory copy in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpyArrayToArray`
    """
    cdef ccuda.CUarray csrcArray
    if srcArray is None:
        csrcArray = <ccuda.CUarray><void_ptr>0
    elif isinstance(srcArray, (CUarray,)):
        psrcArray = int(srcArray)
        csrcArray = <ccuda.CUarray><void_ptr>psrcArray
    else:
        psrcArray = int(CUarray(srcArray))
        csrcArray = <ccuda.CUarray><void_ptr>psrcArray
    cdef ccuda.CUarray cdstArray
    if dstArray is None:
        cdstArray = <ccuda.CUarray><void_ptr>0
    elif isinstance(dstArray, (CUarray,)):
        pdstArray = int(dstArray)
        cdstArray = <ccuda.CUarray><void_ptr>pdstArray
    else:
        pdstArray = int(CUarray(dstArray))
        cdstArray = <ccuda.CUarray><void_ptr>pdstArray
    err = ccuda.cuMemcpyAtoA(cdstArray, dstOffset, csrcArray, srcOffset, ByteCount)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpy2D(pCopy : Optional[CUDA_MEMCPY2D]):
    """ Copies memory for 2D arrays.

    Perform a 2D memory copy according to the parameters specified in
    `pCopy`. The :py:obj:`~.CUDA_MEMCPY2D` structure is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where:

    - :py:obj:`~.srcMemoryType` and :py:obj:`~.dstMemoryType` specify the
      type of memory of the source and destination, respectively;
      :py:obj:`~.CUmemorytype_enum` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_UNIFIED`,
    :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` specify the (unified
    virtual address space) base address of the source data and the bytes
    per row to apply. :py:obj:`~.srcArray` is ignored. This value may be
    used only if unified addressing is supported in the calling context.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_HOST`,
    :py:obj:`~.srcHost` and :py:obj:`~.srcPitch` specify the (host) base
    address of the source data and the bytes per row to apply.
    :py:obj:`~.srcArray` is ignored.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_DEVICE`,
    :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` specify the (device)
    base address of the source data and the bytes per row to apply.
    :py:obj:`~.srcArray` is ignored.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_ARRAY`,
    :py:obj:`~.srcArray` specifies the handle of the source data.
    :py:obj:`~.srcHost`, :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` are
    ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_HOST`,
    :py:obj:`~.dstHost` and :py:obj:`~.dstPitch` specify the (host) base
    address of the destination data and the bytes per row to apply.
    :py:obj:`~.dstArray` is ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_UNIFIED`,
    :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` specify the (unified
    virtual address space) base address of the source data and the bytes
    per row to apply. :py:obj:`~.dstArray` is ignored. This value may be
    used only if unified addressing is supported in the calling context.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_DEVICE`,
    :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` specify the (device)
    base address of the destination data and the bytes per row to apply.
    :py:obj:`~.dstArray` is ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_ARRAY`,
    :py:obj:`~.dstArray` specifies the handle of the destination data.
    :py:obj:`~.dstHost`, :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` are
    ignored.

    - :py:obj:`~.srcXInBytes` and :py:obj:`~.srcY` specify the base address
      of the source data for the copy.

    For host pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For device pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For CUDA arrays, :py:obj:`~.srcXInBytes` must be evenly divisible by
    the array element size.

    - :py:obj:`~.dstXInBytes` and :py:obj:`~.dstY` specify the base address
      of the destination data for the copy.

    For host pointers, the base address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For device pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For CUDA arrays, :py:obj:`~.dstXInBytes` must be evenly divisible by
    the array element size.

    - :py:obj:`~.WidthInBytes` and :py:obj:`~.Height` specify the width (in
      bytes) and height of the 2D copy being performed.

    - If specified, :py:obj:`~.srcPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + :py:obj:`~.srcXInBytes`, and
      :py:obj:`~.dstPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + dstXInBytes.

    :py:obj:`~.cuMemcpy2D()` returns an error if any pitch is greater than
    the maximum allowed (:py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_PITCH`).
    :py:obj:`~.cuMemAllocPitch()` passes back pitches that always work with
    :py:obj:`~.cuMemcpy2D()`. On intra-device memory copies (device to
    device, CUDA array to device, CUDA array to CUDA array),
    :py:obj:`~.cuMemcpy2D()` may fail for pitches not computed by
    :py:obj:`~.cuMemAllocPitch()`. :py:obj:`~.cuMemcpy2DUnaligned()` does
    not have this restriction, but may run significantly slower in the
    cases where :py:obj:`~.cuMemcpy2D()` would have returned an error code.

    Parameters
    ----------
    pCopy : :py:obj:`~.CUDA_MEMCPY2D`
        Parameters for the memory copy

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpy2D`, :py:obj:`~.cudaMemcpy2DToArray`, :py:obj:`~.cudaMemcpy2DFromArray`
    """
    cdef ccuda.CUDA_MEMCPY2D* cpCopy_ptr = pCopy._ptr if pCopy != None else NULL
    err = ccuda.cuMemcpy2D(cpCopy_ptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpy2DUnaligned(pCopy : Optional[CUDA_MEMCPY2D]):
    """ Copies memory for 2D arrays.

    Perform a 2D memory copy according to the parameters specified in
    `pCopy`. The :py:obj:`~.CUDA_MEMCPY2D` structure is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where:

    - :py:obj:`~.srcMemoryType` and :py:obj:`~.dstMemoryType` specify the
      type of memory of the source and destination, respectively;
      :py:obj:`~.CUmemorytype_enum` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_UNIFIED`,
    :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` specify the (unified
    virtual address space) base address of the source data and the bytes
    per row to apply. :py:obj:`~.srcArray` is ignored. This value may be
    used only if unified addressing is supported in the calling context.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_HOST`,
    :py:obj:`~.srcHost` and :py:obj:`~.srcPitch` specify the (host) base
    address of the source data and the bytes per row to apply.
    :py:obj:`~.srcArray` is ignored.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_DEVICE`,
    :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` specify the (device)
    base address of the source data and the bytes per row to apply.
    :py:obj:`~.srcArray` is ignored.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_ARRAY`,
    :py:obj:`~.srcArray` specifies the handle of the source data.
    :py:obj:`~.srcHost`, :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` are
    ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_UNIFIED`,
    :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` specify the (unified
    virtual address space) base address of the source data and the bytes
    per row to apply. :py:obj:`~.dstArray` is ignored. This value may be
    used only if unified addressing is supported in the calling context.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_HOST`,
    :py:obj:`~.dstHost` and :py:obj:`~.dstPitch` specify the (host) base
    address of the destination data and the bytes per row to apply.
    :py:obj:`~.dstArray` is ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_DEVICE`,
    :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` specify the (device)
    base address of the destination data and the bytes per row to apply.
    :py:obj:`~.dstArray` is ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_ARRAY`,
    :py:obj:`~.dstArray` specifies the handle of the destination data.
    :py:obj:`~.dstHost`, :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` are
    ignored.

    - :py:obj:`~.srcXInBytes` and :py:obj:`~.srcY` specify the base address
      of the source data for the copy.

    For host pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For device pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For CUDA arrays, :py:obj:`~.srcXInBytes` must be evenly divisible by
    the array element size.

    - :py:obj:`~.dstXInBytes` and :py:obj:`~.dstY` specify the base address
      of the destination data for the copy.

    For host pointers, the base address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For device pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For CUDA arrays, :py:obj:`~.dstXInBytes` must be evenly divisible by
    the array element size.

    - :py:obj:`~.WidthInBytes` and :py:obj:`~.Height` specify the width (in
      bytes) and height of the 2D copy being performed.

    - If specified, :py:obj:`~.srcPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + :py:obj:`~.srcXInBytes`, and
      :py:obj:`~.dstPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + dstXInBytes.

    :py:obj:`~.cuMemcpy2D()` returns an error if any pitch is greater than
    the maximum allowed (:py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_PITCH`).
    :py:obj:`~.cuMemAllocPitch()` passes back pitches that always work with
    :py:obj:`~.cuMemcpy2D()`. On intra-device memory copies (device to
    device, CUDA array to device, CUDA array to CUDA array),
    :py:obj:`~.cuMemcpy2D()` may fail for pitches not computed by
    :py:obj:`~.cuMemAllocPitch()`. :py:obj:`~.cuMemcpy2DUnaligned()` does
    not have this restriction, but may run significantly slower in the
    cases where :py:obj:`~.cuMemcpy2D()` would have returned an error code.

    Parameters
    ----------
    pCopy : :py:obj:`~.CUDA_MEMCPY2D`
        Parameters for the memory copy

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpy2D`, :py:obj:`~.cudaMemcpy2DToArray`, :py:obj:`~.cudaMemcpy2DFromArray`
    """
    cdef ccuda.CUDA_MEMCPY2D* cpCopy_ptr = pCopy._ptr if pCopy != None else NULL
    err = ccuda.cuMemcpy2DUnaligned(cpCopy_ptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpy3D(pCopy : Optional[CUDA_MEMCPY3D]):
    """ Copies memory for 3D arrays.

    Perform a 3D memory copy according to the parameters specified in
    `pCopy`. The :py:obj:`~.CUDA_MEMCPY3D` structure is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where:

    - :py:obj:`~.srcMemoryType` and :py:obj:`~.dstMemoryType` specify the
      type of memory of the source and destination, respectively;
      :py:obj:`~.CUmemorytype_enum` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_UNIFIED`,
    :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` specify the (unified
    virtual address space) base address of the source data and the bytes
    per row to apply. :py:obj:`~.srcArray` is ignored. This value may be
    used only if unified addressing is supported in the calling context.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_HOST`,
    :py:obj:`~.srcHost`, :py:obj:`~.srcPitch` and :py:obj:`~.srcHeight`
    specify the (host) base address of the source data, the bytes per row,
    and the height of each 2D slice of the 3D array. :py:obj:`~.srcArray`
    is ignored.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_DEVICE`,
    :py:obj:`~.srcDevice`, :py:obj:`~.srcPitch` and :py:obj:`~.srcHeight`
    specify the (device) base address of the source data, the bytes per
    row, and the height of each 2D slice of the 3D array.
    :py:obj:`~.srcArray` is ignored.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_ARRAY`,
    :py:obj:`~.srcArray` specifies the handle of the source data.
    :py:obj:`~.srcHost`, :py:obj:`~.srcDevice`, :py:obj:`~.srcPitch` and
    :py:obj:`~.srcHeight` are ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_UNIFIED`,
    :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` specify the (unified
    virtual address space) base address of the source data and the bytes
    per row to apply. :py:obj:`~.dstArray` is ignored. This value may be
    used only if unified addressing is supported in the calling context.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_HOST`,
    :py:obj:`~.dstHost` and :py:obj:`~.dstPitch` specify the (host) base
    address of the destination data, the bytes per row, and the height of
    each 2D slice of the 3D array. :py:obj:`~.dstArray` is ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_DEVICE`,
    :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` specify the (device)
    base address of the destination data, the bytes per row, and the height
    of each 2D slice of the 3D array. :py:obj:`~.dstArray` is ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_ARRAY`,
    :py:obj:`~.dstArray` specifies the handle of the destination data.
    :py:obj:`~.dstHost`, :py:obj:`~.dstDevice`, :py:obj:`~.dstPitch` and
    :py:obj:`~.dstHeight` are ignored.

    - :py:obj:`~.srcXInBytes`, :py:obj:`~.srcY` and :py:obj:`~.srcZ`
      specify the base address of the source data for the copy.

    For host pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For device pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For CUDA arrays, :py:obj:`~.srcXInBytes` must be evenly divisible by
    the array element size.

    - dstXInBytes, :py:obj:`~.dstY` and :py:obj:`~.dstZ` specify the base
      address of the destination data for the copy.

    For host pointers, the base address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For device pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For CUDA arrays, :py:obj:`~.dstXInBytes` must be evenly divisible by
    the array element size.

    - :py:obj:`~.WidthInBytes`, :py:obj:`~.Height` and :py:obj:`~.Depth`
      specify the width (in bytes), height and depth of the 3D copy being
      performed.

    - If specified, :py:obj:`~.srcPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + :py:obj:`~.srcXInBytes`, and
      :py:obj:`~.dstPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + dstXInBytes.

    - If specified, :py:obj:`~.srcHeight` must be greater than or equal to
      :py:obj:`~.Height` + :py:obj:`~.srcY`, and :py:obj:`~.dstHeight` must
      be greater than or equal to :py:obj:`~.Height` + :py:obj:`~.dstY`.

    :py:obj:`~.cuMemcpy3D()` returns an error if any pitch is greater than
    the maximum allowed (:py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_PITCH`).

    The :py:obj:`~.srcLOD` and :py:obj:`~.dstLOD` members of the
    :py:obj:`~.CUDA_MEMCPY3D` structure must be set to 0.

    Parameters
    ----------
    pCopy : :py:obj:`~.CUDA_MEMCPY3D`
        Parameters for the memory copy

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpy3D`
    """
    cdef ccuda.CUDA_MEMCPY3D* cpCopy_ptr = pCopy._ptr if pCopy != None else NULL
    err = ccuda.cuMemcpy3D(cpCopy_ptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpy3DPeer(pCopy : Optional[CUDA_MEMCPY3D_PEER]):
    """ Copies memory between contexts.

    Perform a 3D memory copy according to the parameters specified in
    `pCopy`. See the definition of the :py:obj:`~.CUDA_MEMCPY3D_PEER`
    structure for documentation of its parameters.

    Parameters
    ----------
    pCopy : :py:obj:`~.CUDA_MEMCPY3D_PEER`
        Parameters for the memory copy

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyPeer`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyPeerAsync`, :py:obj:`~.cuMemcpy3DPeerAsync`, :py:obj:`~.cudaMemcpy3DPeer`
    """
    cdef ccuda.CUDA_MEMCPY3D_PEER* cpCopy_ptr = pCopy._ptr if pCopy != None else NULL
    err = ccuda.cuMemcpy3DPeer(cpCopy_ptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyAsync(dst, src, size_t ByteCount, hStream):
    """ Copies memory asynchronously.

    Copies data between two pointers. `dst` and `src` are base pointers of
    the destination and source, respectively. `ByteCount` specifies the
    number of bytes to copy. Note that this function infers the type of the
    transfer (host to host, host to device, device to device, or device to
    host) from the pointer values. This function is only allowed in
    contexts which support unified addressing.

    Parameters
    ----------
    dst : :py:obj:`~.CUdeviceptr`
        Destination unified virtual address space pointer
    src : :py:obj:`~.CUdeviceptr`
        Source unified virtual address space pointer
    ByteCount : size_t
        Size of memory copy in bytes
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpyToSymbolAsync`, :py:obj:`~.cudaMemcpyFromSymbolAsync`
    """
    cdef ccuda.CUstream chStream
    if hStream is None:
        chStream = <ccuda.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        chStream = <ccuda.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        chStream = <ccuda.CUstream><void_ptr>phStream
    cdef ccuda.CUdeviceptr csrc
    if src is None:
        csrc = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(src, (CUdeviceptr,)):
        psrc = int(src)
        csrc = <ccuda.CUdeviceptr><void_ptr>psrc
    else:
        psrc = int(CUdeviceptr(src))
        csrc = <ccuda.CUdeviceptr><void_ptr>psrc
    cdef ccuda.CUdeviceptr cdst
    if dst is None:
        cdst = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dst, (CUdeviceptr,)):
        pdst = int(dst)
        cdst = <ccuda.CUdeviceptr><void_ptr>pdst
    else:
        pdst = int(CUdeviceptr(dst))
        cdst = <ccuda.CUdeviceptr><void_ptr>pdst
    err = ccuda.cuMemcpyAsync(cdst, csrc, ByteCount, chStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyPeerAsync(dstDevice, dstContext, srcDevice, srcContext, size_t ByteCount, hStream):
    """ Copies device memory between two contexts asynchronously.

    Copies from device memory in one context to device memory in another
    context. `dstDevice` is the base device pointer of the destination
    memory and `dstContext` is the destination context. `srcDevice` is the
    base device pointer of the source memory and `srcContext` is the source
    pointer. `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    dstContext : :py:obj:`~.CUcontext`
        Destination context
    srcDevice : :py:obj:`~.CUdeviceptr`
        Source device pointer
    srcContext : :py:obj:`~.CUcontext`
        Source context
    ByteCount : size_t
        Size of memory copy in bytes
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyPeer`, :py:obj:`~.cuMemcpy3DPeer`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpy3DPeerAsync`, :py:obj:`~.cudaMemcpyPeerAsync`
    """
    cdef ccuda.CUstream chStream
    if hStream is None:
        chStream = <ccuda.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        chStream = <ccuda.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        chStream = <ccuda.CUstream><void_ptr>phStream
    cdef ccuda.CUcontext csrcContext
    if srcContext is None:
        csrcContext = <ccuda.CUcontext><void_ptr>0
    elif isinstance(srcContext, (CUcontext,)):
        psrcContext = int(srcContext)
        csrcContext = <ccuda.CUcontext><void_ptr>psrcContext
    else:
        psrcContext = int(CUcontext(srcContext))
        csrcContext = <ccuda.CUcontext><void_ptr>psrcContext
    cdef ccuda.CUdeviceptr csrcDevice
    if srcDevice is None:
        csrcDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(srcDevice, (CUdeviceptr,)):
        psrcDevice = int(srcDevice)
        csrcDevice = <ccuda.CUdeviceptr><void_ptr>psrcDevice
    else:
        psrcDevice = int(CUdeviceptr(srcDevice))
        csrcDevice = <ccuda.CUdeviceptr><void_ptr>psrcDevice
    cdef ccuda.CUcontext cdstContext
    if dstContext is None:
        cdstContext = <ccuda.CUcontext><void_ptr>0
    elif isinstance(dstContext, (CUcontext,)):
        pdstContext = int(dstContext)
        cdstContext = <ccuda.CUcontext><void_ptr>pdstContext
    else:
        pdstContext = int(CUcontext(dstContext))
        cdstContext = <ccuda.CUcontext><void_ptr>pdstContext
    cdef ccuda.CUdeviceptr cdstDevice
    if dstDevice is None:
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    err = ccuda.cuMemcpyPeerAsync(cdstDevice, cdstContext, csrcDevice, csrcContext, ByteCount, chStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyHtoDAsync(dstDevice, srcHost, size_t ByteCount, hStream):
    """ Copies memory from Host to Device.

    Copies from host memory to device memory. `dstDevice` and `srcHost` are
    the base addresses of the destination and source, respectively.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    srcHost : Any
        Source host pointer
    ByteCount : size_t
        Size of memory copy in bytes
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpyToSymbolAsync`
    """
    cdef ccuda.CUstream chStream
    if hStream is None:
        chStream = <ccuda.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        chStream = <ccuda.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        chStream = <ccuda.CUstream><void_ptr>phStream
    cdef ccuda.CUdeviceptr cdstDevice
    if dstDevice is None:
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    csrcHost = utils.HelperInputVoidPtr(srcHost)
    cdef void* csrcHost_ptr = <void*><void_ptr>csrcHost.cptr
    err = ccuda.cuMemcpyHtoDAsync(cdstDevice, csrcHost_ptr, ByteCount, chStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyDtoHAsync(dstHost, srcDevice, size_t ByteCount, hStream):
    """ Copies memory from Device to Host.

    Copies from device to host memory. `dstHost` and `srcDevice` specify
    the base pointers of the destination and source, respectively.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstHost : Any
        Destination host pointer
    srcDevice : :py:obj:`~.CUdeviceptr`
        Source device pointer
    ByteCount : size_t
        Size of memory copy in bytes
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpyFromSymbolAsync`
    """
    cdef ccuda.CUstream chStream
    if hStream is None:
        chStream = <ccuda.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        chStream = <ccuda.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        chStream = <ccuda.CUstream><void_ptr>phStream
    cdef ccuda.CUdeviceptr csrcDevice
    if srcDevice is None:
        csrcDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(srcDevice, (CUdeviceptr,)):
        psrcDevice = int(srcDevice)
        csrcDevice = <ccuda.CUdeviceptr><void_ptr>psrcDevice
    else:
        psrcDevice = int(CUdeviceptr(srcDevice))
        csrcDevice = <ccuda.CUdeviceptr><void_ptr>psrcDevice
    cdstHost = utils.HelperInputVoidPtr(dstHost)
    cdef void* cdstHost_ptr = <void*><void_ptr>cdstHost.cptr
    err = ccuda.cuMemcpyDtoHAsync(cdstHost_ptr, csrcDevice, ByteCount, chStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyDtoDAsync(dstDevice, srcDevice, size_t ByteCount, hStream):
    """ Copies memory from Device to Device.

    Copies from device memory to device memory. `dstDevice` and `srcDevice`
    are the base pointers of the destination and source, respectively.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    srcDevice : :py:obj:`~.CUdeviceptr`
        Source device pointer
    ByteCount : size_t
        Size of memory copy in bytes
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpyToSymbolAsync`, :py:obj:`~.cudaMemcpyFromSymbolAsync`
    """
    cdef ccuda.CUstream chStream
    if hStream is None:
        chStream = <ccuda.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        chStream = <ccuda.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        chStream = <ccuda.CUstream><void_ptr>phStream
    cdef ccuda.CUdeviceptr csrcDevice
    if srcDevice is None:
        csrcDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(srcDevice, (CUdeviceptr,)):
        psrcDevice = int(srcDevice)
        csrcDevice = <ccuda.CUdeviceptr><void_ptr>psrcDevice
    else:
        psrcDevice = int(CUdeviceptr(srcDevice))
        csrcDevice = <ccuda.CUdeviceptr><void_ptr>psrcDevice
    cdef ccuda.CUdeviceptr cdstDevice
    if dstDevice is None:
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    err = ccuda.cuMemcpyDtoDAsync(cdstDevice, csrcDevice, ByteCount, chStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyHtoAAsync(dstArray, size_t dstOffset, srcHost, size_t ByteCount, hStream):
    """ Copies memory from Host to Array.

    Copies from host memory to a 1D CUDA array. `dstArray` and `dstOffset`
    specify the CUDA array handle and starting offset in bytes of the
    destination data. `srcHost` specifies the base address of the source.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstArray : :py:obj:`~.CUarray`
        Destination array
    dstOffset : size_t
        Offset in bytes of destination array
    srcHost : Any
        Source host pointer
    ByteCount : size_t
        Size of memory copy in bytes
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemcpyToArrayAsync`
    """
    cdef ccuda.CUstream chStream
    if hStream is None:
        chStream = <ccuda.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        chStream = <ccuda.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        chStream = <ccuda.CUstream><void_ptr>phStream
    cdef ccuda.CUarray cdstArray
    if dstArray is None:
        cdstArray = <ccuda.CUarray><void_ptr>0
    elif isinstance(dstArray, (CUarray,)):
        pdstArray = int(dstArray)
        cdstArray = <ccuda.CUarray><void_ptr>pdstArray
    else:
        pdstArray = int(CUarray(dstArray))
        cdstArray = <ccuda.CUarray><void_ptr>pdstArray
    csrcHost = utils.HelperInputVoidPtr(srcHost)
    cdef void* csrcHost_ptr = <void*><void_ptr>csrcHost.cptr
    err = ccuda.cuMemcpyHtoAAsync(cdstArray, dstOffset, csrcHost_ptr, ByteCount, chStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyAtoHAsync(dstHost, srcArray, size_t srcOffset, size_t ByteCount, hStream):
    """ Copies memory from Array to Host.

    Copies from one 1D CUDA array to host memory. `dstHost` specifies the
    base pointer of the destination. `srcArray` and `srcOffset` specify the
    CUDA array handle and starting offset in bytes of the source data.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstHost : Any
        Destination pointer
    srcArray : :py:obj:`~.CUarray`
        Source array
    srcOffset : size_t
        Offset in bytes of source array
    ByteCount : size_t
        Size of memory copy in bytes
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemcpyFromArrayAsync`
    """
    cdef ccuda.CUstream chStream
    if hStream is None:
        chStream = <ccuda.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        chStream = <ccuda.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        chStream = <ccuda.CUstream><void_ptr>phStream
    cdef ccuda.CUarray csrcArray
    if srcArray is None:
        csrcArray = <ccuda.CUarray><void_ptr>0
    elif isinstance(srcArray, (CUarray,)):
        psrcArray = int(srcArray)
        csrcArray = <ccuda.CUarray><void_ptr>psrcArray
    else:
        psrcArray = int(CUarray(srcArray))
        csrcArray = <ccuda.CUarray><void_ptr>psrcArray
    cdstHost = utils.HelperInputVoidPtr(dstHost)
    cdef void* cdstHost_ptr = <void*><void_ptr>cdstHost.cptr
    err = ccuda.cuMemcpyAtoHAsync(cdstHost_ptr, csrcArray, srcOffset, ByteCount, chStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpy2DAsync(pCopy : Optional[CUDA_MEMCPY2D], hStream):
    """ Copies memory for 2D arrays.

    Perform a 2D memory copy according to the parameters specified in
    `pCopy`. The :py:obj:`~.CUDA_MEMCPY2D` structure is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where:

    - :py:obj:`~.srcMemoryType` and :py:obj:`~.dstMemoryType` specify the
      type of memory of the source and destination, respectively;
      :py:obj:`~.CUmemorytype_enum` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_HOST`,
    :py:obj:`~.srcHost` and :py:obj:`~.srcPitch` specify the (host) base
    address of the source data and the bytes per row to apply.
    :py:obj:`~.srcArray` is ignored.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_UNIFIED`,
    :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` specify the (unified
    virtual address space) base address of the source data and the bytes
    per row to apply. :py:obj:`~.srcArray` is ignored. This value may be
    used only if unified addressing is supported in the calling context.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_DEVICE`,
    :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` specify the (device)
    base address of the source data and the bytes per row to apply.
    :py:obj:`~.srcArray` is ignored.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_ARRAY`,
    :py:obj:`~.srcArray` specifies the handle of the source data.
    :py:obj:`~.srcHost`, :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` are
    ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_UNIFIED`,
    :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` specify the (unified
    virtual address space) base address of the source data and the bytes
    per row to apply. :py:obj:`~.dstArray` is ignored. This value may be
    used only if unified addressing is supported in the calling context.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_HOST`,
    :py:obj:`~.dstHost` and :py:obj:`~.dstPitch` specify the (host) base
    address of the destination data and the bytes per row to apply.
    :py:obj:`~.dstArray` is ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_DEVICE`,
    :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` specify the (device)
    base address of the destination data and the bytes per row to apply.
    :py:obj:`~.dstArray` is ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_ARRAY`,
    :py:obj:`~.dstArray` specifies the handle of the destination data.
    :py:obj:`~.dstHost`, :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` are
    ignored.

    - :py:obj:`~.srcXInBytes` and :py:obj:`~.srcY` specify the base address
      of the source data for the copy.

    For host pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For device pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For CUDA arrays, :py:obj:`~.srcXInBytes` must be evenly divisible by
    the array element size.

    - :py:obj:`~.dstXInBytes` and :py:obj:`~.dstY` specify the base address
      of the destination data for the copy.

    For host pointers, the base address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For device pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For CUDA arrays, :py:obj:`~.dstXInBytes` must be evenly divisible by
    the array element size.

    - :py:obj:`~.WidthInBytes` and :py:obj:`~.Height` specify the width (in
      bytes) and height of the 2D copy being performed.

    - If specified, :py:obj:`~.srcPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + :py:obj:`~.srcXInBytes`, and
      :py:obj:`~.dstPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + dstXInBytes.

    - If specified, :py:obj:`~.srcPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + :py:obj:`~.srcXInBytes`, and
      :py:obj:`~.dstPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + dstXInBytes.

    - If specified, :py:obj:`~.srcHeight` must be greater than or equal to
      :py:obj:`~.Height` + :py:obj:`~.srcY`, and :py:obj:`~.dstHeight` must
      be greater than or equal to :py:obj:`~.Height` + :py:obj:`~.dstY`.

    :py:obj:`~.cuMemcpy2DAsync()` returns an error if any pitch is greater
    than the maximum allowed (:py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_PITCH`).
    :py:obj:`~.cuMemAllocPitch()` passes back pitches that always work with
    :py:obj:`~.cuMemcpy2D()`. On intra-device memory copies (device to
    device, CUDA array to device, CUDA array to CUDA array),
    :py:obj:`~.cuMemcpy2DAsync()` may fail for pitches not computed by
    :py:obj:`~.cuMemAllocPitch()`.

    Parameters
    ----------
    pCopy : :py:obj:`~.CUDA_MEMCPY2D`
        Parameters for the memory copy
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemcpy2DAsync`, :py:obj:`~.cudaMemcpy2DToArrayAsync`, :py:obj:`~.cudaMemcpy2DFromArrayAsync`
    """
    cdef ccuda.CUstream chStream
    if hStream is None:
        chStream = <ccuda.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        chStream = <ccuda.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        chStream = <ccuda.CUstream><void_ptr>phStream
    cdef ccuda.CUDA_MEMCPY2D* cpCopy_ptr = pCopy._ptr if pCopy != None else NULL
    err = ccuda.cuMemcpy2DAsync(cpCopy_ptr, chStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpy3DAsync(pCopy : Optional[CUDA_MEMCPY3D], hStream):
    """ Copies memory for 3D arrays.

    Perform a 3D memory copy according to the parameters specified in
    `pCopy`. The :py:obj:`~.CUDA_MEMCPY3D` structure is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where:

    - :py:obj:`~.srcMemoryType` and :py:obj:`~.dstMemoryType` specify the
      type of memory of the source and destination, respectively;
      :py:obj:`~.CUmemorytype_enum` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_UNIFIED`,
    :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` specify the (unified
    virtual address space) base address of the source data and the bytes
    per row to apply. :py:obj:`~.srcArray` is ignored. This value may be
    used only if unified addressing is supported in the calling context.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_HOST`,
    :py:obj:`~.srcHost`, :py:obj:`~.srcPitch` and :py:obj:`~.srcHeight`
    specify the (host) base address of the source data, the bytes per row,
    and the height of each 2D slice of the 3D array. :py:obj:`~.srcArray`
    is ignored.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_DEVICE`,
    :py:obj:`~.srcDevice`, :py:obj:`~.srcPitch` and :py:obj:`~.srcHeight`
    specify the (device) base address of the source data, the bytes per
    row, and the height of each 2D slice of the 3D array.
    :py:obj:`~.srcArray` is ignored.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_ARRAY`,
    :py:obj:`~.srcArray` specifies the handle of the source data.
    :py:obj:`~.srcHost`, :py:obj:`~.srcDevice`, :py:obj:`~.srcPitch` and
    :py:obj:`~.srcHeight` are ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_UNIFIED`,
    :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` specify the (unified
    virtual address space) base address of the source data and the bytes
    per row to apply. :py:obj:`~.dstArray` is ignored. This value may be
    used only if unified addressing is supported in the calling context.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_HOST`,
    :py:obj:`~.dstHost` and :py:obj:`~.dstPitch` specify the (host) base
    address of the destination data, the bytes per row, and the height of
    each 2D slice of the 3D array. :py:obj:`~.dstArray` is ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_DEVICE`,
    :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` specify the (device)
    base address of the destination data, the bytes per row, and the height
    of each 2D slice of the 3D array. :py:obj:`~.dstArray` is ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_ARRAY`,
    :py:obj:`~.dstArray` specifies the handle of the destination data.
    :py:obj:`~.dstHost`, :py:obj:`~.dstDevice`, :py:obj:`~.dstPitch` and
    :py:obj:`~.dstHeight` are ignored.

    - :py:obj:`~.srcXInBytes`, :py:obj:`~.srcY` and :py:obj:`~.srcZ`
      specify the base address of the source data for the copy.

    For host pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For device pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For CUDA arrays, :py:obj:`~.srcXInBytes` must be evenly divisible by
    the array element size.

    - dstXInBytes, :py:obj:`~.dstY` and :py:obj:`~.dstZ` specify the base
      address of the destination data for the copy.

    For host pointers, the base address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For device pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For CUDA arrays, :py:obj:`~.dstXInBytes` must be evenly divisible by
    the array element size.

    - :py:obj:`~.WidthInBytes`, :py:obj:`~.Height` and :py:obj:`~.Depth`
      specify the width (in bytes), height and depth of the 3D copy being
      performed.

    - If specified, :py:obj:`~.srcPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + :py:obj:`~.srcXInBytes`, and
      :py:obj:`~.dstPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + dstXInBytes.

    - If specified, :py:obj:`~.srcHeight` must be greater than or equal to
      :py:obj:`~.Height` + :py:obj:`~.srcY`, and :py:obj:`~.dstHeight` must
      be greater than or equal to :py:obj:`~.Height` + :py:obj:`~.dstY`.

    :py:obj:`~.cuMemcpy3DAsync()` returns an error if any pitch is greater
    than the maximum allowed (:py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_PITCH`).

    The :py:obj:`~.srcLOD` and :py:obj:`~.dstLOD` members of the
    :py:obj:`~.CUDA_MEMCPY3D` structure must be set to 0.

    Parameters
    ----------
    pCopy : :py:obj:`~.CUDA_MEMCPY3D`
        Parameters for the memory copy
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemcpy3DAsync`
    """
    cdef ccuda.CUstream chStream
    if hStream is None:
        chStream = <ccuda.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        chStream = <ccuda.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        chStream = <ccuda.CUstream><void_ptr>phStream
    cdef ccuda.CUDA_MEMCPY3D* cpCopy_ptr = pCopy._ptr if pCopy != None else NULL
    err = ccuda.cuMemcpy3DAsync(cpCopy_ptr, chStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpy3DPeerAsync(pCopy : Optional[CUDA_MEMCPY3D_PEER], hStream):
    """ Copies memory between contexts asynchronously.

    Perform a 3D memory copy according to the parameters specified in
    `pCopy`. See the definition of the :py:obj:`~.CUDA_MEMCPY3D_PEER`
    structure for documentation of its parameters.

    Parameters
    ----------
    pCopy : :py:obj:`~.CUDA_MEMCPY3D_PEER`
        Parameters for the memory copy
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyPeer`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyPeerAsync`, :py:obj:`~.cuMemcpy3DPeerAsync`, :py:obj:`~.cudaMemcpy3DPeerAsync`
    """
    cdef ccuda.CUstream chStream
    if hStream is None:
        chStream = <ccuda.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        chStream = <ccuda.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        chStream = <ccuda.CUstream><void_ptr>phStream
    cdef ccuda.CUDA_MEMCPY3D_PEER* cpCopy_ptr = pCopy._ptr if pCopy != None else NULL
    err = ccuda.cuMemcpy3DPeerAsync(cpCopy_ptr, chStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD8(dstDevice, unsigned char uc, size_t N):
    """ Initializes device memory.

    Sets the memory range of `N` 8-bit values to the specified value `uc`.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    uc : unsigned char
        Value to set
    N : size_t
        Number of elements

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemset`
    """
    cdef ccuda.CUdeviceptr cdstDevice
    if dstDevice is None:
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    err = ccuda.cuMemsetD8(cdstDevice, uc, N)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD16(dstDevice, unsigned short us, size_t N):
    """ Initializes device memory.

    Sets the memory range of `N` 16-bit values to the specified value `us`.
    The `dstDevice` pointer must be two byte aligned.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    us : unsigned short
        Value to set
    N : size_t
        Number of elements

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemset`
    """
    cdef ccuda.CUdeviceptr cdstDevice
    if dstDevice is None:
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    err = ccuda.cuMemsetD16(cdstDevice, us, N)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD32(dstDevice, unsigned int ui, size_t N):
    """ Initializes device memory.

    Sets the memory range of `N` 32-bit values to the specified value `ui`.
    The `dstDevice` pointer must be four byte aligned.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    ui : unsigned int
        Value to set
    N : size_t
        Number of elements

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemset`
    """
    cdef ccuda.CUdeviceptr cdstDevice
    if dstDevice is None:
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    err = ccuda.cuMemsetD32(cdstDevice, ui, N)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD2D8(dstDevice, size_t dstPitch, unsigned char uc, size_t Width, size_t Height):
    """ Initializes device memory.

    Sets the 2D memory range of `Width` 8-bit values to the specified value
    `uc`. `Height` specifies the number of rows to set, and `dstPitch`
    specifies the number of bytes between each row. This function performs
    fastest when the pitch is one that has been passed back by
    :py:obj:`~.cuMemAllocPitch()`.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    dstPitch : size_t
        Pitch of destination device pointer(Unused if `Height` is 1)
    uc : unsigned char
        Value to set
    Width : size_t
        Width of row
    Height : size_t
        Number of rows

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemset2D`
    """
    cdef ccuda.CUdeviceptr cdstDevice
    if dstDevice is None:
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    err = ccuda.cuMemsetD2D8(cdstDevice, dstPitch, uc, Width, Height)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD2D16(dstDevice, size_t dstPitch, unsigned short us, size_t Width, size_t Height):
    """ Initializes device memory.

    Sets the 2D memory range of `Width` 16-bit values to the specified
    value `us`. `Height` specifies the number of rows to set, and
    `dstPitch` specifies the number of bytes between each row. The
    `dstDevice` pointer and `dstPitch` offset must be two byte aligned.
    This function performs fastest when the pitch is one that has been
    passed back by :py:obj:`~.cuMemAllocPitch()`.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    dstPitch : size_t
        Pitch of destination device pointer(Unused if `Height` is 1)
    us : unsigned short
        Value to set
    Width : size_t
        Width of row
    Height : size_t
        Number of rows

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemset2D`
    """
    cdef ccuda.CUdeviceptr cdstDevice
    if dstDevice is None:
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    err = ccuda.cuMemsetD2D16(cdstDevice, dstPitch, us, Width, Height)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD2D32(dstDevice, size_t dstPitch, unsigned int ui, size_t Width, size_t Height):
    """ Initializes device memory.

    Sets the 2D memory range of `Width` 32-bit values to the specified
    value `ui`. `Height` specifies the number of rows to set, and
    `dstPitch` specifies the number of bytes between each row. The
    `dstDevice` pointer and `dstPitch` offset must be four byte aligned.
    This function performs fastest when the pitch is one that has been
    passed back by :py:obj:`~.cuMemAllocPitch()`.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    dstPitch : size_t
        Pitch of destination device pointer(Unused if `Height` is 1)
    ui : unsigned int
        Value to set
    Width : size_t
        Width of row
    Height : size_t
        Number of rows

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemset2D`
    """
    cdef ccuda.CUdeviceptr cdstDevice
    if dstDevice is None:
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    err = ccuda.cuMemsetD2D32(cdstDevice, dstPitch, ui, Width, Height)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD8Async(dstDevice, unsigned char uc, size_t N, hStream):
    """ Sets device memory.

    Sets the memory range of `N` 8-bit values to the specified value `uc`.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    uc : unsigned char
        Value to set
    N : size_t
        Number of elements
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemsetAsync`
    """
    cdef ccuda.CUstream chStream
    if hStream is None:
        chStream = <ccuda.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        chStream = <ccuda.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        chStream = <ccuda.CUstream><void_ptr>phStream
    cdef ccuda.CUdeviceptr cdstDevice
    if dstDevice is None:
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    err = ccuda.cuMemsetD8Async(cdstDevice, uc, N, chStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD16Async(dstDevice, unsigned short us, size_t N, hStream):
    """ Sets device memory.

    Sets the memory range of `N` 16-bit values to the specified value `us`.
    The `dstDevice` pointer must be two byte aligned.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    us : unsigned short
        Value to set
    N : size_t
        Number of elements
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemsetAsync`
    """
    cdef ccuda.CUstream chStream
    if hStream is None:
        chStream = <ccuda.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        chStream = <ccuda.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        chStream = <ccuda.CUstream><void_ptr>phStream
    cdef ccuda.CUdeviceptr cdstDevice
    if dstDevice is None:
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    err = ccuda.cuMemsetD16Async(cdstDevice, us, N, chStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD32Async(dstDevice, unsigned int ui, size_t N, hStream):
    """ Sets device memory.

    Sets the memory range of `N` 32-bit values to the specified value `ui`.
    The `dstDevice` pointer must be four byte aligned.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    ui : unsigned int
        Value to set
    N : size_t
        Number of elements
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemsetAsync`
    """
    cdef ccuda.CUstream chStream
    if hStream is None:
        chStream = <ccuda.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        chStream = <ccuda.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        chStream = <ccuda.CUstream><void_ptr>phStream
    cdef ccuda.CUdeviceptr cdstDevice
    if dstDevice is None:
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    err = ccuda.cuMemsetD32Async(cdstDevice, ui, N, chStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD2D8Async(dstDevice, size_t dstPitch, unsigned char uc, size_t Width, size_t Height, hStream):
    """ Sets device memory.

    Sets the 2D memory range of `Width` 8-bit values to the specified value
    `uc`. `Height` specifies the number of rows to set, and `dstPitch`
    specifies the number of bytes between each row. This function performs
    fastest when the pitch is one that has been passed back by
    :py:obj:`~.cuMemAllocPitch()`.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    dstPitch : size_t
        Pitch of destination device pointer(Unused if `Height` is 1)
    uc : unsigned char
        Value to set
    Width : size_t
        Width of row
    Height : size_t
        Number of rows
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemset2DAsync`
    """
    cdef ccuda.CUstream chStream
    if hStream is None:
        chStream = <ccuda.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        chStream = <ccuda.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        chStream = <ccuda.CUstream><void_ptr>phStream
    cdef ccuda.CUdeviceptr cdstDevice
    if dstDevice is None:
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    err = ccuda.cuMemsetD2D8Async(cdstDevice, dstPitch, uc, Width, Height, chStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD2D16Async(dstDevice, size_t dstPitch, unsigned short us, size_t Width, size_t Height, hStream):
    """ Sets device memory.

    Sets the 2D memory range of `Width` 16-bit values to the specified
    value `us`. `Height` specifies the number of rows to set, and
    `dstPitch` specifies the number of bytes between each row. The
    `dstDevice` pointer and `dstPitch` offset must be two byte aligned.
    This function performs fastest when the pitch is one that has been
    passed back by :py:obj:`~.cuMemAllocPitch()`.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    dstPitch : size_t
        Pitch of destination device pointer(Unused if `Height` is 1)
    us : unsigned short
        Value to set
    Width : size_t
        Width of row
    Height : size_t
        Number of rows
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemset2DAsync`
    """
    cdef ccuda.CUstream chStream
    if hStream is None:
        chStream = <ccuda.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        chStream = <ccuda.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        chStream = <ccuda.CUstream><void_ptr>phStream
    cdef ccuda.CUdeviceptr cdstDevice
    if dstDevice is None:
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    err = ccuda.cuMemsetD2D16Async(cdstDevice, dstPitch, us, Width, Height, chStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD2D32Async(dstDevice, size_t dstPitch, unsigned int ui, size_t Width, size_t Height, hStream):
    """ Sets device memory.

    Sets the 2D memory range of `Width` 32-bit values to the specified
    value `ui`. `Height` specifies the number of rows to set, and
    `dstPitch` specifies the number of bytes between each row. The
    `dstDevice` pointer and `dstPitch` offset must be four byte aligned.
    This function performs fastest when the pitch is one that has been
    passed back by :py:obj:`~.cuMemAllocPitch()`.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    dstPitch : size_t
        Pitch of destination device pointer(Unused if `Height` is 1)
    ui : unsigned int
        Value to set
    Width : size_t
        Width of row
    Height : size_t
        Number of rows
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemset2DAsync`
    """
    cdef ccuda.CUstream chStream
    if hStream is None:
        chStream = <ccuda.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        chStream = <ccuda.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        chStream = <ccuda.CUstream><void_ptr>phStream
    cdef ccuda.CUdeviceptr cdstDevice
    if dstDevice is None:
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cdstDevice = <ccuda.CUdeviceptr><void_ptr>pdstDevice
    err = ccuda.cuMemsetD2D32Async(cdstDevice, dstPitch, ui, Width, Height, chStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuArrayCreate(pAllocateArray : Optional[CUDA_ARRAY_DESCRIPTOR]):
    """ Creates a 1D or 2D CUDA array.

    Creates a CUDA array according to the :py:obj:`~.CUDA_ARRAY_DESCRIPTOR`
    structure `pAllocateArray` and returns a handle to the new CUDA array
    in `*pHandle`. The :py:obj:`~.CUDA_ARRAY_DESCRIPTOR` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where:

    - `Width`, and `Height` are the width, and height of the CUDA array (in
      elements); the CUDA array is one-dimensional if height is 0, two-
      dimensional otherwise;

    - :py:obj:`~.Format` specifies the format of the elements;
      :py:obj:`~.CUarray_format` is defined as:

    - **View CUDA Toolkit Documentation for a C++ code example**

    - `NumChannels` specifies the number of packed components per CUDA
      array element; it may be 1, 2, or 4;

    Here are examples of CUDA array descriptions:

    Description for a CUDA array of 2048 floats:

    **View CUDA Toolkit Documentation for a C++ code example**

    Description for a 64 x 64 CUDA array of floats:

    **View CUDA Toolkit Documentation for a C++ code example**

    Description for a `width` x `height` CUDA array of 64-bit, 4x16-bit
    float16's:

    **View CUDA Toolkit Documentation for a C++ code example**

    Description for a `width` x `height` CUDA array of 16-bit elements,
    each of which is two 8-bit unsigned chars:

    **View CUDA Toolkit Documentation for a C++ code example**

    Parameters
    ----------
    pAllocateArray : :py:obj:`~.CUDA_ARRAY_DESCRIPTOR`
        Array descriptor

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_UNKNOWN`
    pHandle : :py:obj:`~.CUarray`
        Returned array

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMallocArray`
    """
    cdef CUarray pHandle = CUarray()
    cdef ccuda.CUDA_ARRAY_DESCRIPTOR* cpAllocateArray_ptr = pAllocateArray._ptr if pAllocateArray != None else NULL
    err = ccuda.cuArrayCreate(<ccuda.CUarray*>pHandle._ptr, cpAllocateArray_ptr)
    return (CUresult(err), pHandle)

@cython.embedsignature(True)
def cuArrayGetDescriptor(hArray):
    """ Get a 1D or 2D CUDA array descriptor.

    Returns in `*pArrayDescriptor` a descriptor containing information on
    the format and dimensions of the CUDA array `hArray`. It is useful for
    subroutines that have been passed a CUDA array, but need to know the
    CUDA array parameters for validation or other purposes.

    Parameters
    ----------
    hArray : :py:obj:`~.CUarray`
        Array to get descriptor of

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`
    pArrayDescriptor : :py:obj:`~.CUDA_ARRAY_DESCRIPTOR`
        Returned array descriptor

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaArrayGetInfo`
    """
    cdef ccuda.CUarray chArray
    if hArray is None:
        chArray = <ccuda.CUarray><void_ptr>0
    elif isinstance(hArray, (CUarray,)):
        phArray = int(hArray)
        chArray = <ccuda.CUarray><void_ptr>phArray
    else:
        phArray = int(CUarray(hArray))
        chArray = <ccuda.CUarray><void_ptr>phArray
    cdef CUDA_ARRAY_DESCRIPTOR pArrayDescriptor = CUDA_ARRAY_DESCRIPTOR()
    err = ccuda.cuArrayGetDescriptor(<ccuda.CUDA_ARRAY_DESCRIPTOR*>pArrayDescriptor._ptr, chArray)
    return (CUresult(err), pArrayDescriptor)

@cython.embedsignature(True)
def cuArrayGetSparseProperties(array):
    """ Returns the layout properties of a sparse CUDA array.

    Returns the layout properties of a sparse CUDA array in
    `sparseProperties` If the CUDA array is not allocated with flag
    :py:obj:`~.CUDA_ARRAY3D_SPARSE` :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    will be returned.

    If the returned value in :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES.flags`
    contains :py:obj:`~.CU_ARRAY_SPARSE_PROPERTIES_SINGLE_MIPTAIL`, then
    :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES.miptailSize` represents the
    total size of the array. Otherwise, it will be zero. Also, the returned
    value in :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES.miptailFirstLevel` is
    always zero. Note that the `array` must have been allocated using
    :py:obj:`~.cuArrayCreate` or :py:obj:`~.cuArray3DCreate`. For CUDA
    arrays obtained using :py:obj:`~.cuMipmappedArrayGetLevel`,
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE` will be returned. Instead,
    :py:obj:`~.cuMipmappedArrayGetSparseProperties` must be used to obtain
    the sparse properties of the entire CUDA mipmapped array to which
    `array` belongs to.

    Parameters
    ----------
    array : :py:obj:`~.CUarray`
        CUDA array to get the sparse properties of

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS` :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    sparseProperties : :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES`
        Pointer to :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES`

    See Also
    --------
    :py:obj:`~.cuMipmappedArrayGetSparseProperties`, :py:obj:`~.cuMemMapArrayAsync`
    """
    cdef ccuda.CUarray carray
    if array is None:
        carray = <ccuda.CUarray><void_ptr>0
    elif isinstance(array, (CUarray,)):
        parray = int(array)
        carray = <ccuda.CUarray><void_ptr>parray
    else:
        parray = int(CUarray(array))
        carray = <ccuda.CUarray><void_ptr>parray
    cdef CUDA_ARRAY_SPARSE_PROPERTIES sparseProperties = CUDA_ARRAY_SPARSE_PROPERTIES()
    err = ccuda.cuArrayGetSparseProperties(<ccuda.CUDA_ARRAY_SPARSE_PROPERTIES*>sparseProperties._ptr, carray)
    return (CUresult(err), sparseProperties)

@cython.embedsignature(True)
def cuMipmappedArrayGetSparseProperties(mipmap):
    """ Returns the layout properties of a sparse CUDA mipmapped array.

    Returns the sparse array layout properties in `sparseProperties` If the
    CUDA mipmapped array is not allocated with flag
    :py:obj:`~.CUDA_ARRAY3D_SPARSE` :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    will be returned.

    For non-layered CUDA mipmapped arrays,
    :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES.miptailSize` returns the size
    of the mip tail region. The mip tail region includes all mip levels
    whose width, height or depth is less than that of the tile. For layered
    CUDA mipmapped arrays, if
    :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES.flags` contains
    :py:obj:`~.CU_ARRAY_SPARSE_PROPERTIES_SINGLE_MIPTAIL`, then
    :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES.miptailSize` specifies the size
    of the mip tail of all layers combined. Otherwise,
    :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES.miptailSize` specifies mip tail
    size per layer. The returned value of
    :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES.miptailFirstLevel` is valid
    only if :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES.miptailSize` is non-
    zero.

    Parameters
    ----------
    mipmap : :py:obj:`~.CUmipmappedArray`
        CUDA mipmapped array to get the sparse properties of

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS` :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    sparseProperties : :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES`
        Pointer to :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES`

    See Also
    --------
    :py:obj:`~.cuArrayGetSparseProperties`, :py:obj:`~.cuMemMapArrayAsync`
    """
    cdef ccuda.CUmipmappedArray cmipmap
    if mipmap is None:
        cmipmap = <ccuda.CUmipmappedArray><void_ptr>0
    elif isinstance(mipmap, (CUmipmappedArray,)):
        pmipmap = int(mipmap)
        cmipmap = <ccuda.CUmipmappedArray><void_ptr>pmipmap
    else:
        pmipmap = int(CUmipmappedArray(mipmap))
        cmipmap = <ccuda.CUmipmappedArray><void_ptr>pmipmap
    cdef CUDA_ARRAY_SPARSE_PROPERTIES sparseProperties = CUDA_ARRAY_SPARSE_PROPERTIES()
    err = ccuda.cuMipmappedArrayGetSparseProperties(<ccuda.CUDA_ARRAY_SPARSE_PROPERTIES*>sparseProperties._ptr, cmipmap)
    return (CUresult(err), sparseProperties)

@cython.embedsignature(True)
def cuArrayGetMemoryRequirements(array, device):
    """ Returns the memory requirements of a CUDA array.

    Returns the memory requirements of a CUDA array in `memoryRequirements`
    If the CUDA array is not allocated with flag
    :py:obj:`~.CUDA_ARRAY3D_DEFERRED_MAPPING`
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE` will be returned.

    The returned value in :py:obj:`~.CUDA_ARRAY_MEMORY_REQUIREMENTS.size`
    represents the total size of the CUDA array. The returned value in
    :py:obj:`~.CUDA_ARRAY_MEMORY_REQUIREMENTS.alignment` represents the
    alignment necessary for mapping the CUDA array.

    Parameters
    ----------
    array : :py:obj:`~.CUarray`
        CUDA array to get the memory requirements of
    device : :py:obj:`~.CUdevice`
        Device to get the memory requirements for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS` :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    memoryRequirements : :py:obj:`~.CUDA_ARRAY_MEMORY_REQUIREMENTS`
        Pointer to :py:obj:`~.CUDA_ARRAY_MEMORY_REQUIREMENTS`

    See Also
    --------
    :py:obj:`~.cuMipmappedArrayGetMemoryRequirements`, :py:obj:`~.cuMemMapArrayAsync`
    """
    cdef ccuda.CUdevice cdevice
    if device is None:
        cdevice = <ccuda.CUdevice>0
    elif isinstance(device, (CUdevice,)):
        pdevice = int(device)
        cdevice = <ccuda.CUdevice>pdevice
    else:
        pdevice = int(CUdevice(device))
        cdevice = <ccuda.CUdevice>pdevice
    cdef ccuda.CUarray carray
    if array is None:
        carray = <ccuda.CUarray><void_ptr>0
    elif isinstance(array, (CUarray,)):
        parray = int(array)
        carray = <ccuda.CUarray><void_ptr>parray
    else:
        parray = int(CUarray(array))
        carray = <ccuda.CUarray><void_ptr>parray
    cdef CUDA_ARRAY_MEMORY_REQUIREMENTS memoryRequirements = CUDA_ARRAY_MEMORY_REQUIREMENTS()
    err = ccuda.cuArrayGetMemoryRequirements(<ccuda.CUDA_ARRAY_MEMORY_REQUIREMENTS*>memoryRequirements._ptr, carray, cdevice)
    return (CUresult(err), memoryRequirements)

@cython.embedsignature(True)
def cuMipmappedArrayGetMemoryRequirements(mipmap, device):
    """ Returns the memory requirements of a CUDA mipmapped array.

    Returns the memory requirements of a CUDA mipmapped array in
    `memoryRequirements` If the CUDA mipmapped array is not allocated with
    flag :py:obj:`~.CUDA_ARRAY3D_DEFERRED_MAPPING`
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE` will be returned.

    The returned value in :py:obj:`~.CUDA_ARRAY_MEMORY_REQUIREMENTS.size`
    represents the total size of the CUDA mipmapped array. The returned
    value in :py:obj:`~.CUDA_ARRAY_MEMORY_REQUIREMENTS.alignment`
    represents the alignment necessary for mapping the CUDA mipmapped
    array.

    Parameters
    ----------
    mipmap : :py:obj:`~.CUmipmappedArray`
        CUDA mipmapped array to get the memory requirements of
    device : :py:obj:`~.CUdevice`
        Device to get the memory requirements for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS` :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    memoryRequirements : :py:obj:`~.CUDA_ARRAY_MEMORY_REQUIREMENTS`
        Pointer to :py:obj:`~.CUDA_ARRAY_MEMORY_REQUIREMENTS`

    See Also
    --------
    :py:obj:`~.cuArrayGetMemoryRequirements`, :py:obj:`~.cuMemMapArrayAsync`
    """
    cdef ccuda.CUdevice cdevice
    if device is None:
        cdevice = <ccuda.CUdevice>0
    elif isinstance(device, (CUdevice,)):
        pdevice = int(device)
        cdevice = <ccuda.CUdevice>pdevice
    else:
        pdevice = int(CUdevice(device))
        cdevice = <ccuda.CUdevice>pdevice
    cdef ccuda.CUmipmappedArray cmipmap
    if mipmap is None:
        cmipmap = <ccuda.CUmipmappedArray><void_ptr>0
    elif isinstance(mipmap, (CUmipmappedArray,)):
        pmipmap = int(mipmap)
        cmipmap = <ccuda.CUmipmappedArray><void_ptr>pmipmap
    else:
        pmipmap = int(CUmipmappedArray(mipmap))
        cmipmap = <ccuda.CUmipmappedArray><void_ptr>pmipmap
    cdef CUDA_ARRAY_MEMORY_REQUIREMENTS memoryRequirements = CUDA_ARRAY_MEMORY_REQUIREMENTS()
    err = ccuda.cuMipmappedArrayGetMemoryRequirements(<ccuda.CUDA_ARRAY_MEMORY_REQUIREMENTS*>memoryRequirements._ptr, cmipmap, cdevice)
    return (CUresult(err), memoryRequirements)

@cython.embedsignature(True)
def cuArrayGetPlane(hArray, unsigned int planeIdx):
    """ Gets a CUDA array plane from a CUDA array.

    Returns in `pPlaneArray` a CUDA array that represents a single format
    plane of the CUDA array `hArray`.

    If `planeIdx` is greater than the maximum number of planes in this
    array or if the array does not have a multi-planar format e.g:
    :py:obj:`~.CU_AD_FORMAT_NV12`, then
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE` is returned.

    Note that if the `hArray` has format :py:obj:`~.CU_AD_FORMAT_NV12`,
    then passing in 0 for `planeIdx` returns a CUDA array of the same size
    as `hArray` but with one channel and
    :py:obj:`~.CU_AD_FORMAT_UNSIGNED_INT8` as its format. If 1 is passed
    for `planeIdx`, then the returned CUDA array has half the height and
    width of `hArray` with two channels and
    :py:obj:`~.CU_AD_FORMAT_UNSIGNED_INT8` as its format.

    Parameters
    ----------
    hArray : :py:obj:`~.CUarray`
        Multiplanar CUDA array
    planeIdx : unsigned int
        Plane index

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`
    pPlaneArray : :py:obj:`~.CUarray`
        Returned CUDA array referenced by the `planeIdx`

    See Also
    --------
    :py:obj:`~.cuArrayCreate`, :py:obj:`~.cudaArrayGetPlane`
    """
    cdef ccuda.CUarray chArray
    if hArray is None:
        chArray = <ccuda.CUarray><void_ptr>0
    elif isinstance(hArray, (CUarray,)):
        phArray = int(hArray)
        chArray = <ccuda.CUarray><void_ptr>phArray
    else:
        phArray = int(CUarray(hArray))
        chArray = <ccuda.CUarray><void_ptr>phArray
    cdef CUarray pPlaneArray = CUarray()
    err = ccuda.cuArrayGetPlane(<ccuda.CUarray*>pPlaneArray._ptr, chArray, planeIdx)
    return (CUresult(err), pPlaneArray)

@cython.embedsignature(True)
def cuArrayDestroy(hArray):
    """ Destroys a CUDA array.

    Destroys the CUDA array `hArray`.

    Parameters
    ----------
    hArray : :py:obj:`~.CUarray`
        Array to destroy

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_ARRAY_IS_MAPPED`, :py:obj:`~.CUDA_ERROR_CONTEXT_IS_DESTROYED`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaFreeArray`
    """
    cdef ccuda.CUarray chArray
    if hArray is None:
        chArray = <ccuda.CUarray><void_ptr>0
    elif isinstance(hArray, (CUarray,)):
        phArray = int(hArray)
        chArray = <ccuda.CUarray><void_ptr>phArray
    else:
        phArray = int(CUarray(hArray))
        chArray = <ccuda.CUarray><void_ptr>phArray
    err = ccuda.cuArrayDestroy(chArray)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuArray3DCreate(pAllocateArray : Optional[CUDA_ARRAY3D_DESCRIPTOR]):
    """ Creates a 3D CUDA array.

    Creates a CUDA array according to the
    :py:obj:`~.CUDA_ARRAY3D_DESCRIPTOR` structure `pAllocateArray` and
    returns a handle to the new CUDA array in `*pHandle`. The
    :py:obj:`~.CUDA_ARRAY3D_DESCRIPTOR` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where:

    - `Width`, `Height`, and `Depth` are the width, height, and depth of
      the CUDA array (in elements); the following types of CUDA arrays can
      be allocated:

      - A 1D array is allocated if `Height` and `Depth` extents are both
        zero.

      - A 2D array is allocated if only `Depth` extent is zero.

      - A 3D array is allocated if all three extents are non-zero.

      - A 1D layered CUDA array is allocated if only `Height` is zero and
        the :py:obj:`~.CUDA_ARRAY3D_LAYERED` flag is set. Each layer is a
        1D array. The number of layers is determined by the depth extent.

      - A 2D layered CUDA array is allocated if all three extents are non-
        zero and the :py:obj:`~.CUDA_ARRAY3D_LAYERED` flag is set. Each
        layer is a 2D array. The number of layers is determined by the
        depth extent.

      - A cubemap CUDA array is allocated if all three extents are non-zero
        and the :py:obj:`~.CUDA_ARRAY3D_CUBEMAP` flag is set. `Width` must
        be equal to `Height`, and `Depth` must be six. A cubemap is a
        special type of 2D layered CUDA array, where the six layers
        represent the six faces of a cube. The order of the six layers in
        memory is the same as that listed in
        :py:obj:`~.CUarray_cubemap_face`.

      - A cubemap layered CUDA array is allocated if all three extents are
        non-zero, and both, :py:obj:`~.CUDA_ARRAY3D_CUBEMAP` and
        :py:obj:`~.CUDA_ARRAY3D_LAYERED` flags are set. `Width` must be
        equal to `Height`, and `Depth` must be a multiple of six. A cubemap
        layered CUDA array is a special type of 2D layered CUDA array that
        consists of a collection of cubemaps. The first six layers
        represent the first cubemap, the next six layers form the second
        cubemap, and so on.

    - :py:obj:`~.Format` specifies the format of the elements;
      :py:obj:`~.CUarray_format` is defined as:

    - **View CUDA Toolkit Documentation for a C++ code example**

    - `NumChannels` specifies the number of packed components per CUDA
      array element; it may be 1, 2, or 4;

    - :py:obj:`~.Flags` may be set to

      - :py:obj:`~.CUDA_ARRAY3D_LAYERED` to enable creation of layered CUDA
        arrays. If this flag is set, `Depth` specifies the number of
        layers, not the depth of a 3D array.

      - :py:obj:`~.CUDA_ARRAY3D_SURFACE_LDST` to enable surface references
        to be bound to the CUDA array. If this flag is not set,
        :py:obj:`~.cuSurfRefSetArray` will fail when attempting to bind the
        CUDA array to a surface reference.

      - :py:obj:`~.CUDA_ARRAY3D_CUBEMAP` to enable creation of cubemaps. If
        this flag is set, `Width` must be equal to `Height`, and `Depth`
        must be six. If the :py:obj:`~.CUDA_ARRAY3D_LAYERED` flag is also
        set, then `Depth` must be a multiple of six.

      - :py:obj:`~.CUDA_ARRAY3D_TEXTURE_GATHER` to indicate that the CUDA
        array will be used for texture gather. Texture gather can only be
        performed on 2D CUDA arrays.

    `Width`, `Height` and `Depth` must meet certain size requirements as
    listed in the following table. All values are specified in elements.
    Note that for brevity's sake, the full name of the device attribute is
    not specified. For ex., TEXTURE1D_WIDTH refers to the device attribute
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_WIDTH`.

    Note that 2D CUDA arrays have different size requirements if the
    :py:obj:`~.CUDA_ARRAY3D_TEXTURE_GATHER` flag is set. `Width` and
    `Height` must not be greater than
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_GATHER_WIDTH` and
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_GATHER_HEIGHT`
    respectively, in that case.

    **View CUDA Toolkit Documentation for a table example**

    Here are examples of CUDA array descriptions:

    Description for a CUDA array of 2048 floats:

    **View CUDA Toolkit Documentation for a C++ code example**

    Description for a 64 x 64 CUDA array of floats:

    **View CUDA Toolkit Documentation for a C++ code example**

    Description for a `width` x `height` x `depth` CUDA array of 64-bit,
    4x16-bit float16's:

    **View CUDA Toolkit Documentation for a C++ code example**

    Parameters
    ----------
    pAllocateArray : :py:obj:`~.CUDA_ARRAY3D_DESCRIPTOR`
        3D array descriptor

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_UNKNOWN`
    pHandle : :py:obj:`~.CUarray`
        Returned array

    See Also
    --------
    :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMalloc3DArray`
    """
    cdef CUarray pHandle = CUarray()
    cdef ccuda.CUDA_ARRAY3D_DESCRIPTOR* cpAllocateArray_ptr = pAllocateArray._ptr if pAllocateArray != None else NULL
    err = ccuda.cuArray3DCreate(<ccuda.CUarray*>pHandle._ptr, cpAllocateArray_ptr)
    return (CUresult(err), pHandle)

@cython.embedsignature(True)
def cuArray3DGetDescriptor(hArray):
    """ Get a 3D CUDA array descriptor.

    Returns in `*pArrayDescriptor` a descriptor containing information on
    the format and dimensions of the CUDA array `hArray`. It is useful for
    subroutines that have been passed a CUDA array, but need to know the
    CUDA array parameters for validation or other purposes.

    This function may be called on 1D and 2D arrays, in which case the
    `Height` and/or `Depth` members of the descriptor struct will be set to
    0.

    Parameters
    ----------
    hArray : :py:obj:`~.CUarray`
        3D array to get descriptor of

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_CONTEXT_IS_DESTROYED`
    pArrayDescriptor : :py:obj:`~.CUDA_ARRAY3D_DESCRIPTOR`
        Returned 3D array descriptor

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaArrayGetInfo`
    """
    cdef ccuda.CUarray chArray
    if hArray is None:
        chArray = <ccuda.CUarray><void_ptr>0
    elif isinstance(hArray, (CUarray,)):
        phArray = int(hArray)
        chArray = <ccuda.CUarray><void_ptr>phArray
    else:
        phArray = int(CUarray(hArray))
        chArray = <ccuda.CUarray><void_ptr>phArray
    cdef CUDA_ARRAY3D_DESCRIPTOR pArrayDescriptor = CUDA_ARRAY3D_DESCRIPTOR()
    err = ccuda.cuArray3DGetDescriptor(<ccuda.CUDA_ARRAY3D_DESCRIPTOR*>pArrayDescriptor._ptr, chArray)
    return (CUresult(err), pArrayDescriptor)

@cython.embedsignature(True)
def cuMipmappedArrayCreate(pMipmappedArrayDesc : Optional[CUDA_ARRAY3D_DESCRIPTOR], unsigned int numMipmapLevels):
    """ Creates a CUDA mipmapped array.

    Creates a CUDA mipmapped array according to the
    :py:obj:`~.CUDA_ARRAY3D_DESCRIPTOR` structure `pMipmappedArrayDesc` and
    returns a handle to the new CUDA mipmapped array in `*pHandle`.
    `numMipmapLevels` specifies the number of mipmap levels to be
    allocated. This value is clamped to the range [1, 1 +
    floor(log2(max(width, height, depth)))].

    The :py:obj:`~.CUDA_ARRAY3D_DESCRIPTOR` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where:

    - `Width`, `Height`, and `Depth` are the width, height, and depth of
      the CUDA array (in elements); the following types of CUDA arrays can
      be allocated:

      - A 1D mipmapped array is allocated if `Height` and `Depth` extents
        are both zero.

      - A 2D mipmapped array is allocated if only `Depth` extent is zero.

      - A 3D mipmapped array is allocated if all three extents are non-
        zero.

      - A 1D layered CUDA mipmapped array is allocated if only `Height` is
        zero and the :py:obj:`~.CUDA_ARRAY3D_LAYERED` flag is set. Each
        layer is a 1D array. The number of layers is determined by the
        depth extent.

      - A 2D layered CUDA mipmapped array is allocated if all three extents
        are non-zero and the :py:obj:`~.CUDA_ARRAY3D_LAYERED` flag is set.
        Each layer is a 2D array. The number of layers is determined by the
        depth extent.

      - A cubemap CUDA mipmapped array is allocated if all three extents
        are non-zero and the :py:obj:`~.CUDA_ARRAY3D_CUBEMAP` flag is set.
        `Width` must be equal to `Height`, and `Depth` must be six. A
        cubemap is a special type of 2D layered CUDA array, where the six
        layers represent the six faces of a cube. The order of the six
        layers in memory is the same as that listed in
        :py:obj:`~.CUarray_cubemap_face`.

      - A cubemap layered CUDA mipmapped array is allocated if all three
        extents are non-zero, and both, :py:obj:`~.CUDA_ARRAY3D_CUBEMAP`
        and :py:obj:`~.CUDA_ARRAY3D_LAYERED` flags are set. `Width` must be
        equal to `Height`, and `Depth` must be a multiple of six. A cubemap
        layered CUDA array is a special type of 2D layered CUDA array that
        consists of a collection of cubemaps. The first six layers
        represent the first cubemap, the next six layers form the second
        cubemap, and so on.

    - :py:obj:`~.Format` specifies the format of the elements;
      :py:obj:`~.CUarray_format` is defined as:

    - **View CUDA Toolkit Documentation for a C++ code example**

    - `NumChannels` specifies the number of packed components per CUDA
      array element; it may be 1, 2, or 4;

    - :py:obj:`~.Flags` may be set to

      - :py:obj:`~.CUDA_ARRAY3D_LAYERED` to enable creation of layered CUDA
        mipmapped arrays. If this flag is set, `Depth` specifies the number
        of layers, not the depth of a 3D array.

      - :py:obj:`~.CUDA_ARRAY3D_SURFACE_LDST` to enable surface references
        to be bound to individual mipmap levels of the CUDA mipmapped
        array. If this flag is not set, :py:obj:`~.cuSurfRefSetArray` will
        fail when attempting to bind a mipmap level of the CUDA mipmapped
        array to a surface reference.

    - :py:obj:`~.CUDA_ARRAY3D_CUBEMAP` to enable creation of mipmapped
    cubemaps. If this flag is set, `Width` must be equal to `Height`, and
    `Depth` must be six. If the :py:obj:`~.CUDA_ARRAY3D_LAYERED` flag is
    also set, then `Depth` must be a multiple of six.

      - :py:obj:`~.CUDA_ARRAY3D_TEXTURE_GATHER` to indicate that the CUDA
        mipmapped array will be used for texture gather. Texture gather can
        only be performed on 2D CUDA mipmapped arrays.

    `Width`, `Height` and `Depth` must meet certain size requirements as
    listed in the following table. All values are specified in elements.
    Note that for brevity's sake, the full name of the device attribute is
    not specified. For ex., TEXTURE1D_MIPMAPPED_WIDTH refers to the device
    attribute
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_MIPMAPPED_WIDTH`.

    **View CUDA Toolkit Documentation for a table example**

    Parameters
    ----------
    pMipmappedArrayDesc : :py:obj:`~.CUDA_ARRAY3D_DESCRIPTOR`
        mipmapped array descriptor
    numMipmapLevels : unsigned int
        Number of mipmap levels

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_UNKNOWN`
    pHandle : :py:obj:`~.CUmipmappedArray`
        Returned mipmapped array

    See Also
    --------
    :py:obj:`~.cuMipmappedArrayDestroy`, :py:obj:`~.cuMipmappedArrayGetLevel`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cudaMallocMipmappedArray`
    """
    cdef CUmipmappedArray pHandle = CUmipmappedArray()
    cdef ccuda.CUDA_ARRAY3D_DESCRIPTOR* cpMipmappedArrayDesc_ptr = pMipmappedArrayDesc._ptr if pMipmappedArrayDesc != None else NULL
    err = ccuda.cuMipmappedArrayCreate(<ccuda.CUmipmappedArray*>pHandle._ptr, cpMipmappedArrayDesc_ptr, numMipmapLevels)
    return (CUresult(err), pHandle)

@cython.embedsignature(True)
def cuMipmappedArrayGetLevel(hMipmappedArray, unsigned int level):
    """ Gets a mipmap level of a CUDA mipmapped array.

    Returns in `*pLevelArray` a CUDA array that represents a single mipmap
    level of the CUDA mipmapped array `hMipmappedArray`.

    If `level` is greater than the maximum number of levels in this
    mipmapped array, :py:obj:`~.CUDA_ERROR_INVALID_VALUE` is returned.

    Parameters
    ----------
    hMipmappedArray : :py:obj:`~.CUmipmappedArray`
        CUDA mipmapped array
    level : unsigned int
        Mipmap level

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`
    pLevelArray : :py:obj:`~.CUarray`
        Returned mipmap level CUDA array

    See Also
    --------
    :py:obj:`~.cuMipmappedArrayCreate`, :py:obj:`~.cuMipmappedArrayDestroy`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cudaGetMipmappedArrayLevel`
    """
    cdef ccuda.CUmipmappedArray chMipmappedArray
    if hMipmappedArray is None:
        chMipmappedArray = <ccuda.CUmipmappedArray><void_ptr>0
    elif isinstance(hMipmappedArray, (CUmipmappedArray,)):
        phMipmappedArray = int(hMipmappedArray)
        chMipmappedArray = <ccuda.CUmipmappedArray><void_ptr>phMipmappedArray
    else:
        phMipmappedArray = int(CUmipmappedArray(hMipmappedArray))
        chMipmappedArray = <ccuda.CUmipmappedArray><void_ptr>phMipmappedArray
    cdef CUarray pLevelArray = CUarray()
    err = ccuda.cuMipmappedArrayGetLevel(<ccuda.CUarray*>pLevelArray._ptr, chMipmappedArray, level)
    return (CUresult(err), pLevelArray)

@cython.embedsignature(True)
def cuMipmappedArrayDestroy(hMipmappedArray):
    """ Destroys a CUDA mipmapped array.

    Destroys the CUDA mipmapped array `hMipmappedArray`.

    Parameters
    ----------
    hMipmappedArray : :py:obj:`~.CUmipmappedArray`
        Mipmapped array to destroy

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_ARRAY_IS_MAPPED`, :py:obj:`~.CUDA_ERROR_CONTEXT_IS_DESTROYED`

    See Also
    --------
    :py:obj:`~.cuMipmappedArrayCreate`, :py:obj:`~.cuMipmappedArrayGetLevel`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cudaFreeMipmappedArray`
    """
    cdef ccuda.CUmipmappedArray chMipmappedArray
    if hMipmappedArray is None:
        chMipmappedArray = <ccuda.CUmipmappedArray><void_ptr>0