'use strict';

const node_module = require('node:module');

const NodeBuiltinModules = [].concat(
  node_module.Module.builtinModules,
  [
    "assert/strict",
    "fs/promises",
    "path/posix",
    "path/win32",
    "stream/consumers",
    "stream/promises",
    "stream/web",
    "timers/promises",
    "util/types"
  ]
);
function mapArrToVal(val, arr) {
  return Object.fromEntries(arr.map((c) => [c, val]));
}

const node = {
  alias: {
    "node-fetch": "unenv/runtime/npm/node-fetch",
    "cross-fetch": "unenv/runtime/npm/cross-fetch",
    "cross-fetch/polyfill": "unenv/runtime/mock/empty",
    "isomorphic-fetch": "unenv/runtime/mock/empty"
  },
  polyfill: [],
  external: [...NodeBuiltinModules]
};

const nodeless = {
  alias: {
    // Generic mock for built-ins
    ...mapArrToVal("unenv/runtime/mock/proxy-cjs", NodeBuiltinModules),
    // Built-ins implemented by unenv
    "buffer/index.js": "buffer",
    ...Object.fromEntries(
      [
        "assert",
        "assert/strict",
        "async_hooks",
        "buffer",
        "console",
        "child_process",
        "constants",
        "cluster",
        "crypto",
        "dgram",
        "diagnostics_channel",
        "dns",
        "dns/promises",
        "domain",
        "events",
        "fs",
        "fs/promises",
        "http",
        "https",
        "http2",
        "inspector",
        "module",
        "net",
        "os",
        "path",
        "punycode",
        "perf_hooks",
        "process",
        "querystring",
        "readline",
        "readline/promises",
        "stream",
        "stream/promises",
        "stream/consumers",
        "stream/web",
        "string_decoder",
        "trace_events",
        "timers",
        "timers/promises",
        "tls",
        "tty",
        "url",
        "util",
        "util/types",
        "v8",
        "vm",
        "wasi",
        "worker_threads",
        "zlib"
      ].map((m) => [m, `unenv/runtime/node/${m}/index`])
    ),
    "path/posix": "unenv/runtime/node/path/index",
    "path/win32": "unenv/runtime/node/path/index",
    "inspector/promises": "unenv/runtime/node/inspector/index",
    // The sys module is deprecated and has been renamed util
    // https://github.com/nodejs/node/blob/main/lib/sys.js#L27
    sys: "unenv/runtime/node/util/index",
    // npm
    fsevents: "unenv/runtime/npm/fsevents",
    "node-fetch": "unenv/runtime/npm/node-fetch",
    "node-fetch-native": "unenv/runtime/npm/node-fetch",
    "node-fetch-native/polyfill": "unenv/runtime/mock/empty",
    "cross-fetch": "unenv/runtime/npm/cross-fetch",
    "cross-fetch/polyfill": "unenv/runtime/mock/empty",
    "isomorphic-fetch": "unenv/runtime/mock/empty",
    inherits: "unenv/runtime/npm/inherits"
  },
  inject: {
    global: "unenv/runtime/node/_global",
    process: "unenv/runtime/node/process",
    Buffer: ["buffer", "Buffer"],
    performance: "unenv/runtime/polyfill/performance"
  },
  polyfill: [
    "unenv/runtime/polyfill/node-global",
    "unenv/runtime/polyfill/process",
    "unenv/runtime/polyfill/performance"
  ]
};
for (const m of NodeBuiltinModules) {
  nodeless.alias[`node:${m}`] = nodeless.alias[m];
}
const nodeless$1 = nodeless;

const cloudflareNodeCompatModules = [
  "_stream_duplex",
  "_stream_passthrough",
  "_stream_readable",
  "_stream_transform",
  "_stream_writable",
  "assert",
  "assert/strict",
  "buffer",
  "diagnostics_channel",
  "events",
  "path",
  "path/posix",
  "path/win32",
  "querystring",
  "stream",
  "stream/consumers",
  "stream/promises",
  "stream/web",
  "string_decoder",
  "url",
  "util/types",
  "zlib"
];
const hybridNodeCompatModules = [
  "async_hooks",
  "console",
  "crypto",
  "module",
  "process",
  "timers",
  "util"
];
const cloudflarePreset = {
  alias: {
    ...Object.fromEntries(
      cloudflareNodeCompatModules.flatMap((p) => [
        [p, p],
        [`node:${p}`, `node:${p}`]
      ])
    ),
    // The `node:sys` module is just a deprecated alias for `node:util`
    // Keep this until util is a full compat module.
    sys: "unenv/runtime/node/util/$cloudflare",
    "node:sys": "unenv/runtime/node/util/$cloudflare",
    // define aliases for hybrid modules
    ...Object.fromEntries(
      hybridNodeCompatModules.flatMap((m) => [
        [m, `unenv/runtime/node/${m}/$cloudflare`],
        [`node:${m}`, `unenv/runtime/node/${m}/$cloudflare`]
      ])
    ),
    // TODO: this is a hotfix and breaks unenv/fetch
    // https://github.com/unjs/unenv/issues/364
    "unenv/runtime/node/stream/index": "node:stream"
  },
  inject: {
    // workerd already defines `global` and `Buffer`
    // override the previous presets so that we use the native implementation
    Buffer: false,
    global: false,
    console: "unenv/runtime/node/console/$cloudflare",
    process: "unenv/runtime/node/process/$cloudflare",
    setImmediate: ["unenv/runtime/node/timers/$cloudflare", "setImmediate"],
    clearImmediate: ["unenv/runtime/node/timers/$cloudflare", "clearImmediate"]
  },
  polyfill: [],
  external: cloudflareNodeCompatModules.flatMap((p) => [p, `node:${p}`])
};
const cloudflarePreset$1 = cloudflarePreset;

function env(...presets) {
  const _env = {
    alias: {},
    inject: {},
    polyfill: [],
    external: []
  };
  for (const preset of presets) {
    if (preset.alias) {
      const aliases = Object.keys(preset.alias).sort(
        (a, b) => b.split("/").length - a.split("/").length || b.length - a.length
      );
      for (const from of aliases) {
        _env.alias[from] = preset.alias[from];
      }
    }
    if (preset.inject) {
      for (const global in preset.inject) {
        const globalValue = preset.inject[global];
        if (Array.isArray(globalValue)) {
          const [id, ...path] = globalValue;
          _env.inject[global] = [id, ...path];
        } else if (globalValue === false) {
          delete _env.inject[global];
        } else {
          _env.inject[global] = globalValue;
        }
      }
    }
    if (preset.polyfill) {
      _env.polyfill.push(...preset.polyfill.filter(Boolean));
    }
    if (preset.external) {
      _env.external.push(...preset.external);
    }
  }
  return _env;
}

exports.NodeBuiltinModules = NodeBuiltinModules;
exports.cloudflare = cloudflarePreset$1;
exports.env = env;
exports.mapArrToVal = mapArrToVal;
exports.node = node;
exports.nodeless = nodeless$1;
